require "./spec_helper"

describe Luce do
  it "correctly inserts all at 0 index with smaller original" do
    arr1 = [2]

    exp = [0, 1, 2]

    arr1.insert_all(0, [0, 1])

    arr1.should eq exp
  end

  it "correctly inserts all at 0 index with larger original" do
    arr1 = [1, 2, 3, 4, 5, 6]

    exp = [0, 1, 2, 3, 4, 5, 6]

    arr1.insert_all(0, [0])

    arr1.should eq exp
  end

  it "correctly inserts all at end of array" do
    arr1 = [1, 2, 3]

    exp = [1, 2, 3, 7, 8, 9]

    arr1.insert_all(3, [7, 8, 9])

    arr1.should eq exp
    arr1.size.should eq exp.size
  end

  it "correctly inserts all within the array" do
    numbers = [1, 2, 3, 4]
    insert_items = [10, 11]

    exp = [1, 2, 10, 11, 3, 4]

    numbers.insert_all(2, insert_items)

    numbers.should eq exp
  end

  it "correctly inserts all within the array when other is larger" do
    numbers = [1, 2]
    numbers_mid = [3, 4, 5, 6]
    numbers_end = [7, 8, 9, 10]

    expected_partial = [1, 2, 7, 8, 9, 10]
    expected_full = [1, 2, 3, 4, 5, 6, 7, 8, 9, 10]

    numbers.insert_all(2, numbers_end)
    numbers.should eq expected_partial
    numbers.size.should eq expected_partial.size

    numbers.insert_all(2, numbers_mid)
    numbers.should eq expected_full
    numbers.size.should eq expected_full.size
  end

  it "correctly inserts all as shown in the documentation example" do
    arr = [1, 2, 3, 7]

    arr.insert_all(4, [8, 9])
    arr.should eq [1, 2, 3, 7, 8, 9]

    arr.insert_all(3, [4, 5, 6])
    arr.should eq [1, 2, 3, 4, 5, 6, 7, 8, 9]
  end

  it "should throw IndexError on negative number and out-of-bounds" do
    arr = [1, 2, 3]

    expect_raises(IndexError) { arr.insert_all(-1, [4, 5, 6]) }
    expect_raises(IndexError) { arr.insert_all(6, [7, 8, 9]) }
  end
end

test_directory "original"

# Block syntax extensions.
test_file "extensions/fenced_blockquotes.unit",
  block_syntaxes: [Luce::FencedBlockquoteSyntax.new]
test_file "extensions/fenced_code_blocks.unit",
  block_syntaxes: [Luce::FencedCodeBlockSyntax.new]
test_file "extensions/headers_with_ids.unit",
  block_syntaxes: [Luce::HeaderWithIdSyntax.new]
test_file "extensions/ordered_list_with_checkboxes.unit",
  block_syntaxes: [Luce::OrderedListWithCheckboxSyntax.new]
test_file "extensions/setext_headers_with_ids.unit",
  block_syntaxes: [Luce::SetextHeaderWithIdSyntax.new]
test_file "extensions/tables.unit",
  block_syntaxes: [Luce::TableSyntax.new]
test_file "extensions/unordered_list_with_checkboxes.unit",
  block_syntaxes: [Luce::UnorderedListWithCheckboxSyntax.new]

# Inline syntax extensions
test_file "extensions/emojis.unit",
  inline_syntaxes: [Luce::EmojiSyntax.new]
test_file "extensions/inline_html.unit",
  inline_syntaxes: [Luce::InlineHTMLSyntax.new]
test_file "extensions/strikethrough.unit",
  inline_syntaxes: [Luce::StrikethroughSyntax.new]

test_directory "common_mark"
test_directory "gfm"

describe "corner cases" do
  validate_core("Incorrect Links",
    "5 Ethernet ([Music](",
    "<p>5 Ethernet ([Music](</p>\n")

  validate_core("Escaping code block language",
    %q{```"/><a/href="url">arbitrary_html</a>},
    "<pre><code class=\"language-&quot;/&gt;&lt;a/href=&quot;url&quot;&gt;arbitrary_html&lt;/a&gt;\"></code></pre>\n")

  validate_core("Unicode ellipsis as punctuation",
    %{"Connecting dot **A** to **B.**…"},
    "<p>&quot;Connecting dot <strong>A</strong> to <strong>B.</strong>…&quot;</p>\n")

  describe Luce::Resolver do
    nyan_resolver = Luce::Resolver.new do |text, _|
      text.empty? ? nil : Luce::Text.new("~=[,,_#{text}_,,]:3").as Luce::Node
    end

    validate_core("simple link resolver",
      "resolve [this] thing",
      "<p>resolve ~=[,,_this_,,]:3 thing</p>\n",
      link_resolver: nyan_resolver)

    validate_core("simple image resolver",
      "resolve ![this] thing",
      "<p>resolve ~=[,,_this_,,]:3 thing</p>\n",
      image_link_resolver: nyan_resolver)

    validate_core("can resolve link containing inline tags",
      "resolve [*star* _underline_] thing",
      "<p>resolve ~=[,,_*star* _underline__,,]:3 thing</p>\n",
      link_resolver: nyan_resolver)

    validate_core("link resolver uses un-normalized link label",
      "resolve [TH IS] thing",
      "<p>resolve ~=[,,_TH IS_,,]:3 thing</p>\n",
      link_resolver: nyan_resolver)

    validate_core("can resolve escaped brackets",
      %q{resolve [\[\]] thing},
      "<p>resolve ~=[,,_[]_,,]:3 thing</p>\n",
      link_resolver: nyan_resolver)

    validate_core("can choose to _not_ resolve something, like an empty link",
      "resolve [[]] thing",
      "<p>resolve ~=[,,_[]_,,]:3 thing</p>\n",
      link_resolver: nyan_resolver)
  end

  describe "Custom inline syntax" do
    nyan_syntax = [] of Luce::InlineSyntax
    nyan_syntax << Luce::TextSyntax.new("nyan", sub: "~=[,,_,,]:3")

    link_resolver = Luce::Resolver.new do |text, _|
      Luce::Element.text("a", text.gsub("<", "&lt;"))
    end

    validate_core("simple inline syntax",
      "nyan",
      "<p>~=[,,_,,]:3</p>\n",
      inline_syntaxes: nyan_syntax)

    validate_core("dart custom links",
      "links [are<foo>] awesome",
      "<p>links <a>are&lt;foo></a> awesome</p>\n",
      link_resolver: link_resolver)

    # TODO(amouravski): need more tests here for custom syntaxes, as some
    # things are not quite working properly. The regexps are sometime a little
    # too greedy, I think.
  end

  describe "Inline only" do
    validate_core("simple line",
      "This would normally create a paragraph",
      "This would normally create a paragraph",
      inline_only: true)

    validate_core("strong and em",
      "This would _normally_ create a **paragraph**",
      "This would <em>normally</em> create a <strong>paragraph</strong>",
      inline_only: true)

    validate_core("link",
      "This [link](http://example.com/) will work normally",
      "This <a href=\"http://example.com/\">link</a> will work normally",
      inline_only: true)

    validate_core("references do not work",
      "[This][] shouldn't work, though.",
      "[This][] shouldn't work, though.",
      inline_only: true)

    validate_core("less than and ampersand are escaped",
      "< &",
      "&lt; &amp;",
      inline_only: true)

    validate_core("keep newlines",
      "This paragraph
      continues after a newline",
      "This paragraph
      continues after a newline",
      inline_only: true)

    validate_core("ignores block-level markdown syntax",
      "1. This will not be an <ol>.",
      "1. This will not be an <ol>.",
      inline_only: true)
  end
end
