#
# Copyright (c) 2023 supercell
#
# SPDX-License-Identifier: BSD-3-Clause
#
require "../patterns"

module Luce
  # Parses setext-style headers.
  class SetextHeaderSyntax < BlockSyntax
    def pattern : Regex
      Luce.dummy_pattern
    end

    def can_parse?(parser : BlockParser) : Bool
      return false unless interperable_as_paragraph(parser.current)
      i = 1
      loop do
        next_line = parser.peek(i)
        # We never reached an underline
        return false if next_line.nil?
        return true if Luce.setext_pattern.matches? next_line
        # Ensure that we're still in something like paragraph text.
        return false unless interperable_as_paragraph(next_line)
        i += 1
      end
    end

    def parse(parser : BlockParser) : Node
      lines = [] of String
      tag : String?
      while !parser.done?
        match = Luce.setext_pattern.match(parser.current)
        if match.nil?
          # More text
          lines << parser.current
          parser.advance
          next
        else
          # The underline
          tag = (match[1][0] == '=') ? "h1" : "h2"
          parser.advance
          break
        end
      end

      contents = UnparsedContent.new(lines.join("\n").rstrip)
      Element.new(tag.not_nil!, [contents] of Node)
    end

    private def interperable_as_paragraph(line : String) : Bool
      !(Luce.indent_pattern.matches?(line) ||
        Luce.code_fence_pattern.matches?(line) ||
        Luce.header_pattern.matches?(line) ||
        Luce.blockquote_pattern.matches?(line) ||
        Luce.hr_pattern.matches?(line) ||
        Luce.ul_pattern.matches?(line) ||
        Luce.ol_pattern.matches?(line) ||
        Luce.empty_pattern.matches?(line))
    end
  end
end
