#
# Copyright (c) 2023 supercell
#
# SPDX-License-Identifier: BSD-3-Clause
#

module Luce
  # Parse HTML blocks.
  class HTMLBlockSyntax < BlockSyntax
    # There are seven kinds of HTML block defined in the CommonMark spec:
    # https://spec.commonmark.org/0.30/#html-blocks.
    # These matching conditions and HTML block types mentioned in this syntax
    # correspond to these ones in the CommonMark spec.

    @@end_conditions = [
      # For condition 1, it does not need to match the start tag, see
      # https://spec.commonmark.org/0.30/#end-condition
      Regex.new("</(?:pre|script|style|textarea)>", Regex::Options::IGNORE_CASE),
      Regex.new("-->"),
      Regex.new(%q{\?>}),
      Regex.new(">"),
      Regex.new("]]>"),
      Luce.empty_pattern,
      Luce.empty_pattern,
    ]

    def pattern : Regex
      Luce.html_block_pattern
    end

    def can_end_block?(parser : BlockParser) : Bool
      # All types of HTML blocks except type 7 may interrupt a paragraph, see the
      # second paragraph after https://spec.commonmark.org/0.30/#example-148 for
      # more detail.
      pattern.match(parser.current).not_nil!["condition_7"]?.nil?
    end

    def parse_child_lines(parser : BlockParser) : Array(String)
      lines = [] of String

      match = pattern.match(parser.current)
      matched_condition = 0
      i = 0
      while i < match.not_nil!.group_size
        if match.not_nil![i + 1]?.nil?
          i += 1
        else
          matched_condition = i
          break
        end
      end

      end_condition = @@end_conditions[matched_condition]
      if end_condition == Luce.empty_pattern
        lines << parser.current
        parser.advance

        while (false == parser.done?) && (false == end_condition.matches?(parser.current))
          lines << parser.current
          parser.advance
        end
      else
        until parser.done?
          lines << parser.current
          break if end_condition.matches? parser.current
          parser.advance
        end
        parser.advance
      end

      # If the following lines start an HTML block again, put them together with
      # current HTML block.
      if (false == parser.done?) && parser.next != nil && pattern.matches?(parser.next.not_nil!)
        parser.advance
        lines.concat(parse_child_lines(parser))
      end

      lines
    end

    def parse(parser : BlockParser) : Node
      child_lines = parse_child_lines(parser)
      Text.new(child_lines.join("\n").rstrip)
    end
  end
end
