#
# Copyright (c) 2021, 2023 supercell
#
# SPDX-License-Identifier: BSD-3-Clause
#
require "html"

module Luce
  # Maintains the internal state needed to parse a series of lines into
  # blocks of Markdown suitable for further inline parsing.
  class BlockParser
    getter lines : Array(String)

    # The Markdown document this parser is parsing
    getter document : Document

    # The enabled block syntaxes
    #
    # To turn a series of lines into blocks, each of these will be
    # tried in turn. Order matters here.
    getter block_syntaxes = [] of BlockSyntax

    # Index of the current line
    @pos : Int32 = 0

    # Whether the parser has encountered a blank line between two
    # block-level elements.
    @[Deprecated("Use `#encountered_blank_line?` instead. Remove at version 1.0.")]
    def encountered_blank_line : Bool
      encountered_blank_line?
    end

    # Whether the parser has encountered a blank line between two
    # block-level elements.
    property? encountered_blank_line : Bool = false

    # The collection of built-in block parsers
    getter standard_block_syntaxes = [
      Luce::EmptyBlockSyntax.new,
      Luce::HTMLBlockSyntax.new,
      Luce::SetextHeaderSyntax.new,
      Luce::HeaderSyntax.new,
      Luce::CodeBlockSyntax.new,
      Luce::BlockquoteSyntax.new,
      Luce::HorizontalRuleSyntax.new,
      Luce::UnorderedListSyntax.new,
      Luce::OrderedListSyntax.new,
      Luce::ParagraphSyntax.new,
    ]

    def initialize(@lines : Array(String), @document : Document)
      block_syntaxes.concat document.block_syntaxes

      if document.with_default_block_syntaxes?
        block_syntaxes.concat standard_block_syntaxes
      else
        block_syntaxes << DummyBlockSyntax.new
      end
    end

    # Return the current line
    def current : String
      @lines[@pos]
    end

    # Return the line after the current one or `nil` if there is none.
    def next : String?
      # Don't read past the end
      return nil if @pos >= @lines.size - 1
      @lines[@pos + 1]
    end

    # Return the line that is *lines_ahead* lines ahead of the current
    # one, or `nil` if there is none.
    #
    # Note that `peek(0)` is equivalent to `current`, and `peek(1)` is
    # equivalent to `next`.
    def peek(lines_ahead : Int32) : String?
      if lines_ahead < 0
        raise ArgumentError.new("Invalid lines_ahead #{lines_ahead}; must be >= 0.")
      end
      # Don't read past the end.
      return nil if @pos >= (@lines.size - lines_ahead)
      @lines[@pos + lines_ahead]
    end

    def advance : Nil
      @pos += 1
    end

    def retreat : Nil
      @pos -= 1
    end

    def done? : Bool
      @pos >= @lines.size
    end

    # Return if the current line matches the given *regex* or not.
    def matches?(regex : Regex) : Bool
      return false if done?
      regex.matches?(current)
    end

    # Return if the next line matches the given *regex* or not.
    def matches_next?(regex : Regex) : Bool
      return false if self.next.nil?
      regex.matches?(self.next.not_nil!)
    end

    def parse_lines : Array(Node)
      blocks = [] of Node

      # If the @pos does not change before and after `parse()`, never try to
      # parse the line at @pos with the same syntax again.
      # For example, the `TableSyntax` might not advance the @pos in `parse`
      # method, because the header row does not match the delimiter row in the
      # number of cells, which makes the table like structure not be recognized.
      never_match : BlockSyntax? = nil

      until done?
        block_syntaxes.each do |syntax|
          next if never_match == syntax

          if syntax.can_parse? self
            position_before = @pos
            block = syntax.parse self
            blocks << block unless block.nil?
            never_match = @pos != position_before ? nil : syntax
            break
          end
        end
      end

      blocks
    end
  end
end
