﻿//  MeCab -- Yet Another Part-of-Speech and Morphological Analyzer
//
//  Copyright(C) 2001-2006 Taku Kudo <taku@chasen.org>
//  Copyright(C) 2004-2006 Nippon Telegraph and Telephone Corporation

using System;
using System.IO;
using System.Text;
using NMeCab.Core;

namespace NMeCab
{
    public class MeCabParam
    {
        public const float DefaultTheta = 0.75f;

        public const string DefaultRcFile = "dicrc";

        /// <summary>
        ///     コンストラクタ
        /// </summary>
        public MeCabParam()
        {
            Theta = DefaultTheta;
            RcFile = DefaultRcFile;

            DicDir = AppDomain.CurrentDomain.BaseDirectory;
            UserDic = new string[0];
            OutputFormatType = "lattice";
        }

        public string DicDir { get; set; }

        public string[] UserDic { get; set; }

        public int MaxGroupingSize { get; set; }

        /// <summary>
        ///     文頭, 文末の素性
        /// </summary>
        public string BosFeature { get; set; }

        public string UnkFeature { get; set; }

        public bool AlloCateSentence { get; set; }

        /// <summary>
        ///     コスト値に変換するときのスケーリングファクター
        /// </summary>
        public int CostFactor { get; set; }

        /// <summary>
        ///     ソフト分かち書きの温度パラメータ
        /// </summary>
        public float Theta { get; set; }

        /// <summary>
        ///     ラティスレベル(どの程度のラティス情報を解析時に構築するか)
        /// </summary>
        public MeCabLatticeLevel LatticeLevel { get; set; }

        /// <summary>
        ///     部分解析
        /// </summary>
        public bool Partial { get; set; }

        /// <summary>
        ///     出力モード
        /// </summary>
        /// <value>
        ///     true: 全出力
        ///     false: ベスト解のみ
        /// </value>
        public bool AllMorphs { get; set; }

        //public bool AllocateSentence { get; set; }

        public string OutputFormatType { get; set; }

        public string RcFile { get; set; }

        public void LoadDicRC()
        {
            var rc = Path.Combine(DicDir, RcFile);
            Load(rc);
        }

        public void Load(string path)
        {
            var ini = new IniParser();
            ini.Load(path, Encoding.ASCII);

            CostFactor = int.Parse(ini["cost-factor"] ?? "0");
            BosFeature = ini["bos-feature"];
        }

        private string[] SplitStringArray(string configStr, char separator)
        {
            if (string.IsNullOrEmpty(configStr)) return new string[0];

            var ret = configStr.Split(separator);

            for (var i = 0; i < ret.Length; i++)
                ret[i] = ret[i].Trim();

            return ret;
        }
    }
}