# mq-select-activate.sh
# ------------------------------------------------------------------------------
#
# Supplementary shell script fragment; it handles all "qselect" requests to
# modify the active guard state, whether they be to activate any new set of
# guards, or to deactivate all currently active guards (if any).
#
# ------------------------------------------------------------------------------
#
# $Id$
#
# Written by Keith Marshall <keith@users.osdn.me>
# Copyright (C) 2019, Keith Marshall
#
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# ------------------------------------------------------------------------------
#
# This script fragment needs to parse all of the patch queue "guards",
# "series", and "status" files, so identify their path names.
#
  mq_map_control_file_refs "$mq_patchdir" guards series status

# To accommodate reporting of guards status changes, we provide this
# helper function, called both before, and after changing the guards
# state, so we can detect any change in the number of unguarded, and
# unapplied patches.
#
  mq_count() {
    awk "$mq_guards $mq_series $mq_status"' END {
      for( idx = 0; entries > idx; ++idx )
      { $0 = series[idx]; if( state[$1] == "'"$1"'" ) ++count; }
      print 0 + count;
    }' "$mq_guards_file" "$mq_series_file" "$mq_status_file"
  }

# Use it, to capture the "before" unguarded, unapplied patch count;
# note that this script fragment should ALWAYS be invoked within the
# scope of an "mq_require" function call, so we may safely capture
# the count into a local variable.
#
  local mq_initial_count=`mq_count "U"`

# When compiling the "before" count, the "guards" file may not yet
# exist, in which case we will have compiled the count from a parse
# of /dev/null, but while updating the guards state, and before we
# compile the "after" count, we must create the actual file, so
# ensure that we use the proper path name.
#
  mq_guards_file="$mq_patchdir/guards"

# When called with arguments, then we write the "guards" file, to
# make each argument represent an active guard; note that this will
# take precedence over the "--none" action, which will NOT have any
# effect, even if the "--none" option is specified.  While it makes
# no sense to specify "--none" together with arguments, we silently
# ignore the conflict, and ignore "--none" itself; this behaviour
# is consistent with that of Mercurial (as in version 2.7.2).
#
  if test $# -gt 0
  then for mq_arg; do echo "$mq_arg"; done | sort | uniq > "$mq_guards_file"

# Conversely, when there are no arguments, the only reason for this
# to have been invoked is that the "--none" option was specified; in
# this case, we clear all content from the "guards" file, and report
# deactivation of all previously active guards.
#
  else true > "$mq_guards_file"
    echo "$mq_facility: all guards have been deactivated"
  fi

# Finally, compile the "after" unguarded, unapplied patches count,
# compare it to the "before" count, and report if changed.
#
  local mq_final_count=`mq_count "U"`
  test $mq_initial_count -eq $mq_final_count || {
    mq_qualified() {
      test "$1" -eq 1 && echo "$2${4+ $4} $1" || echo "$3${4+ $4} $1"
    }
    mq_final_count=`mq_qualified $mq_final_count is are now`
    mq_initial_count=`mq_qualified $mq_initial_count was were originally`
    echo "$mq_facility: number of unguarded, unapplied patches has changed"
    echo "$mq_facility: there $mq_initial_count; there $mq_final_count"
  }
#
# ------------------------------------------------------------------------------
# $RCSfile$: end of file
