﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Windows.Forms; // Add
using System.Management;    // Add
using System.Diagnostics;   // Add
using VixCOM;               // Add
using System.IO;                        // Add
using System.Text.RegularExpressions;   // Add
using System.Runtime.InteropServices; // Add DLL Import




namespace CtrlVix
{
    public static class Constants
    {
        public enum POWEROP : int           // VMOperation()
        {
            POWER_ON_GUI,
            POWER_ON_NOGUI,
            SUSPEND,
            ACPI_RESET,
            ACPI_POWER_OFF,
            HW_RESET,
            HW_POWER_OFF
        };
        public enum VMSTATE : int           // GetVMState()
        {
            POWERED_ON = 1,
            PROCESSING,
            POWERED_OFF,
            ERROR = 0xffff
        };
        public enum RUNNING_MODE : int      // CheckRunningMode()
        {
            CANNOT_DETERMINE = 0,
            GUI,
            NOGUI,
            POWERED_OFF = 0xffff
        };
        public enum POWER_OFF_MODE : int    // CheckPowerOffMode()
        {
            SUSPENDED,
            POWERED_OFF,
            ERROR = 0xffff
        };
        public enum TOOLS_RUNNING : int     // CheckToolsRunnnging()
        {
            NOT_RUNNING,
            RUNNING,
            ERROR = 0xffff
        };
        public static string WORKINGDIR = "workingDir";         // GetVMWorkingDir()
        public static string WORKINGDIR_ERR = "WORKINGDIR_ERR";
    }

    public class CtrlVIX
    {
        private enum VixVMPowerOpOptions : int
        {
            VIX_VMPOWEROP_NORMAL = 0,
            VIX_VMPOWEROP_FROM_GUEST = 0x0004,
            VIX_VMPOWEROP_SUPPRESS_SNAPSHOT_POWERON = 0x0080,
            VIX_VMPOWEROP_LAUNCH_GUI = 0x0200,
            VIX_VMPOWEROP_START_VM_PAUSED = 0x1000,
        };

        private enum VixPowerState : int
        {
            VIX_POWERSTATE_POWERING_OFF = 0x0001,
            VIX_POWERSTATE_POWERED_OFF = 0x0002,
            VIX_POWERSTATE_POWERING_ON = 0x0004,
            VIX_POWERSTATE_POWERED_ON = 0x0008,
            VIX_POWERSTATE_SUSPENDING = 0x0010,
            VIX_POWERSTATE_SUSPENDED = 0x0020,
            VIX_POWERSTATE_TOOLS_RUNNING = 0x0040,
            VIX_POWERSTATE_RESETTING = 0x0080,
            VIX_POWERSTATE_BLOCKED_ON_MSG = 0x0100,
            VIX_POWERSTATE_PAUSED = 0x0200,
            VIX_POWERSTATE_RESUMING = 0x0800,
        };

        private enum VixToolsState : int
        {
            VIX_TOOLSSTATE_UNKNOWN = 0x0001,
            VIX_TOOLSSTATE_RUNNING = 0x0002,
            VIX_TOOLSSTATE_NOT_INSTALLED = 0x0004,
        };

        private enum VixPropertyID : int
        {
            VIX_PROPERTY_NONE = 0,

            /* Properties used by several handle types. */
            VIX_PROPERTY_META_DATA_CONTAINER = 2,

            /* VIX_HANDLETYPE_HOST properties */
            VIX_PROPERTY_HOST_HOSTTYPE = 50,
            VIX_PROPERTY_HOST_API_VERSION = 51,
            VIX_PROPERTY_HOST_SOFTWARE_VERSION = 52,

            /* VIX_HANDLETYPE_VM properties */
            VIX_PROPERTY_VM_NUM_VCPUS = 101,
            VIX_PROPERTY_VM_VMX_PATHNAME = 103,
            VIX_PROPERTY_VM_VMTEAM_PATHNAME = 105,
            VIX_PROPERTY_VM_MEMORY_SIZE = 106,
            VIX_PROPERTY_VM_READ_ONLY = 107,
            VIX_PROPERTY_VM_NAME = 108,
            VIX_PROPERTY_VM_GUESTOS = 109,
            VIX_PROPERTY_VM_IN_VMTEAM = 128,
            VIX_PROPERTY_VM_POWER_STATE = 129,
            VIX_PROPERTY_VM_TOOLS_STATE = 152,
            VIX_PROPERTY_VM_IS_RUNNING = 196,
            VIX_PROPERTY_VM_SUPPORTED_FEATURES = 197,
            /* VIX_PROPERTY_VM_IS_RECORDING                       = 236, Removed in version 1.11 */
            /* VIX_PROPERTY_VM_IS_REPLAYING                       = 237, Removed in version 1.11 */
            VIX_PROPERTY_VM_SSL_ERROR = 293,

            /* Result properties; these are returned by various procedures */
            VIX_PROPERTY_JOB_RESULT_ERROR_CODE = 3000,
            VIX_PROPERTY_JOB_RESULT_VM_IN_GROUP = 3001,
            VIX_PROPERTY_JOB_RESULT_USER_MESSAGE = 3002,
            VIX_PROPERTY_JOB_RESULT_EXIT_CODE = 3004,
            VIX_PROPERTY_JOB_RESULT_COMMAND_OUTPUT = 3005,
            VIX_PROPERTY_JOB_RESULT_HANDLE = 3010,
            VIX_PROPERTY_JOB_RESULT_GUEST_OBJECT_EXISTS = 3011,
            VIX_PROPERTY_JOB_RESULT_GUEST_PROGRAM_ELAPSED_TIME = 3017,
            VIX_PROPERTY_JOB_RESULT_GUEST_PROGRAM_EXIT_CODE = 3018,
            VIX_PROPERTY_JOB_RESULT_ITEM_NAME = 3035,
            VIX_PROPERTY_JOB_RESULT_FOUND_ITEM_DESCRIPTION = 3036,
            VIX_PROPERTY_JOB_RESULT_SHARED_FOLDER_COUNT = 3046,
            VIX_PROPERTY_JOB_RESULT_SHARED_FOLDER_HOST = 3048,
            VIX_PROPERTY_JOB_RESULT_SHARED_FOLDER_FLAGS = 3049,
            VIX_PROPERTY_JOB_RESULT_PROCESS_ID = 3051,
            VIX_PROPERTY_JOB_RESULT_PROCESS_OWNER = 3052,
            VIX_PROPERTY_JOB_RESULT_PROCESS_COMMAND = 3053,
            VIX_PROPERTY_JOB_RESULT_FILE_FLAGS = 3054,
            VIX_PROPERTY_JOB_RESULT_PROCESS_START_TIME = 3055,
            VIX_PROPERTY_JOB_RESULT_VM_VARIABLE_STRING = 3056,
            VIX_PROPERTY_JOB_RESULT_PROCESS_BEING_DEBUGGED = 3057,
            VIX_PROPERTY_JOB_RESULT_SCREEN_IMAGE_SIZE = 3058,
            VIX_PROPERTY_JOB_RESULT_SCREEN_IMAGE_DATA = 3059,
            VIX_PROPERTY_JOB_RESULT_FILE_SIZE = 3061,
            VIX_PROPERTY_JOB_RESULT_FILE_MOD_TIME = 3062,
            VIX_PROPERTY_JOB_RESULT_EXTRA_ERROR_INFO = 3084,

            /* Event properties; these are sent in the moreEventInfo for some events. */
            VIX_PROPERTY_FOUND_ITEM_LOCATION = 4010,

            /* VIX_HANDLETYPE_SNAPSHOT properties */
            VIX_PROPERTY_SNAPSHOT_DISPLAYNAME = 4200,
            VIX_PROPERTY_SNAPSHOT_DESCRIPTION = 4201,
            VIX_PROPERTY_SNAPSHOT_POWERSTATE = 4205,
            /* VIX_PROPERTY_SNAPSHOT_IS_REPLAYABLE                = 4207, Removed in version 1.11 */

            VIX_PROPERTY_GUEST_SHAREDFOLDERS_SHARES_PATH = 4525,

            /* Virtual machine encryption properties */
            VIX_PROPERTY_VM_ENCRYPTION_PASSWORD = 7001,
        };

        private enum VixServiceProvider : int
        {
            VIX_SERVICEPROVIDER_DEFAULT = 1,
            VIX_SERVICEPROVIDER_VMWARE_SERVER = 2,
            VIX_SERVICEPROVIDER_VMWARE_WORKSTATION = 3,
            VIX_SERVICEPROVIDER_VMWARE_PLAYER = 4,
            VIX_SERVICEPROVIDER_VMWARE_VI_SERVER = 10,
            VIX_SERVICEPROVIDER_VMWARE_WORKSTATION_SHARED = 11,
        /*
         * VIX_API_VERSION tells VixHost_Connect to use the latest API version 
         * that is available for the product specified in the VixServiceProvider 
         * parameter.
         */
            VIX_API_VERSION = -1
        };


        private enum VixErr : UInt64
        {
            VIX_OK = 0,

            /* General errors */
            VIX_E_FAIL = 1,
            VIX_E_OUT_OF_MEMORY = 2,
            VIX_E_INVALID_ARG = 3,
            VIX_E_FILE_NOT_FOUND = 4,
            VIX_E_OBJECT_IS_BUSY = 5,
            VIX_E_NOT_SUPPORTED = 6,
            VIX_E_FILE_ERROR = 7,
            VIX_E_DISK_FULL = 8,
            VIX_E_INCORRECT_FILE_TYPE = 9,
            VIX_E_CANCELLED = 10,
            VIX_E_FILE_READ_ONLY = 11,
            VIX_E_FILE_ALREADY_EXISTS = 12,
            VIX_E_FILE_ACCESS_ERROR = 13,
            VIX_E_REQUIRES_LARGE_FILES = 14,
            VIX_E_FILE_ALREADY_LOCKED = 15,
            VIX_E_VMDB = 16,
            VIX_E_NOT_SUPPORTED_ON_REMOTE_OBJECT = 20,
            VIX_E_FILE_TOO_BIG = 21,
            VIX_E_FILE_NAME_INVALID = 22,
            VIX_E_ALREADY_EXISTS = 23,
            VIX_E_BUFFER_TOOSMALL = 24,
            VIX_E_OBJECT_NOT_FOUND = 25,
            VIX_E_HOST_NOT_CONNECTED = 26,
            VIX_E_INVALID_UTF8_STRING = 27,
            VIX_E_OPERATION_ALREADY_IN_PROGRESS = 31,
            VIX_E_UNFINISHED_JOB = 29,
            VIX_E_NEED_KEY = 30,
            VIX_E_LICENSE = 32,
            VIX_E_VM_HOST_DISCONNECTED = 34,
            VIX_E_AUTHENTICATION_FAIL = 35,
            VIX_E_HOST_CONNECTION_LOST = 36,
            VIX_E_DUPLICATE_NAME = 41,
            VIX_E_ARGUMENT_TOO_BIG = 44,

            /* Handle Errors */
            VIX_E_INVALID_HANDLE = 1000,
            VIX_E_NOT_SUPPORTED_ON_HANDLE_TYPE = 1001,
            VIX_E_TOO_MANY_HANDLES = 1002,

            /* XML errors */
            VIX_E_NOT_FOUND = 2000,
            VIX_E_TYPE_MISMATCH = 2001,
            VIX_E_INVALID_XML = 2002,

            /* VM Control Errors */
            VIX_E_TIMEOUT_WAITING_FOR_TOOLS = 3000,
            VIX_E_UNRECOGNIZED_COMMAND = 3001,
            VIX_E_OP_NOT_SUPPORTED_ON_GUEST = 3003,
            VIX_E_PROGRAM_NOT_STARTED = 3004,
            VIX_E_CANNOT_START_READ_ONLY_VM = 3005,
            VIX_E_VM_NOT_RUNNING = 3006,
            VIX_E_VM_IS_RUNNING = 3007,
            VIX_E_CANNOT_CONNECT_TO_VM = 3008,
            VIX_E_POWEROP_SCRIPTS_NOT_AVAILABLE = 3009,
            VIX_E_NO_GUEST_OS_INSTALLED = 3010,
            VIX_E_VM_INSUFFICIENT_HOST_MEMORY = 3011,
            VIX_E_SUSPEND_ERROR = 3012,
            VIX_E_VM_NOT_ENOUGH_CPUS = 3013,
            VIX_E_HOST_USER_PERMISSIONS = 3014,
            VIX_E_GUEST_USER_PERMISSIONS = 3015,
            VIX_E_TOOLS_NOT_RUNNING = 3016,
            VIX_E_GUEST_OPERATIONS_PROHIBITED = 3017,
            VIX_E_ANON_GUEST_OPERATIONS_PROHIBITED = 3018,
            VIX_E_ROOT_GUEST_OPERATIONS_PROHIBITED = 3019,
            VIX_E_MISSING_ANON_GUEST_ACCOUNT = 3023,
            VIX_E_CANNOT_AUTHENTICATE_WITH_GUEST = 3024,
            VIX_E_UNRECOGNIZED_COMMAND_IN_GUEST = 3025,
            VIX_E_CONSOLE_GUEST_OPERATIONS_PROHIBITED = 3026,
            VIX_E_MUST_BE_CONSOLE_USER = 3027,
            VIX_E_VMX_MSG_DIALOG_AND_NO_UI = 3028,
            /* VIX_E_NOT_ALLOWED_DURING_VM_RECORDING        = 3029, Removed in version 1.11 */
            /* VIX_E_NOT_ALLOWED_DURING_VM_REPLAY           = 3030, Removed in version 1.11 */
            VIX_E_OPERATION_NOT_ALLOWED_FOR_LOGIN_TYPE = 3031,
            VIX_E_LOGIN_TYPE_NOT_SUPPORTED = 3032,
            VIX_E_EMPTY_PASSWORD_NOT_ALLOWED_IN_GUEST = 3033,
            VIX_E_INTERACTIVE_SESSION_NOT_PRESENT = 3034,
            VIX_E_INTERACTIVE_SESSION_USER_MISMATCH = 3035,
            /* VIX_E_UNABLE_TO_REPLAY_VM                    = 3039, Removed in version 1.11 */
            VIX_E_CANNOT_POWER_ON_VM = 3041,
            VIX_E_NO_DISPLAY_SERVER = 3043,
            /* VIX_E_VM_NOT_RECORDING                       = 3044, Removed in version 1.11 */
            /* VIX_E_VM_NOT_REPLAYING                       = 3045, Removed in version 1.11 */
            VIX_E_TOO_MANY_LOGONS = 3046,
            VIX_E_INVALID_AUTHENTICATION_SESSION = 3047,

            /* VM Errors */
            VIX_E_VM_NOT_FOUND = 4000,
            VIX_E_NOT_SUPPORTED_FOR_VM_VERSION = 4001,
            VIX_E_CANNOT_READ_VM_CONFIG = 4002,
            VIX_E_TEMPLATE_VM = 4003,
            VIX_E_VM_ALREADY_LOADED = 4004,
            VIX_E_VM_ALREADY_UP_TO_DATE = 4006,
            VIX_E_VM_UNSUPPORTED_GUEST = 4011,

            /* Property Errors */
            VIX_E_UNRECOGNIZED_PROPERTY = 6000,
            VIX_E_INVALID_PROPERTY_VALUE = 6001,
            VIX_E_READ_ONLY_PROPERTY = 6002,
            VIX_E_MISSING_REQUIRED_PROPERTY = 6003,
            VIX_E_INVALID_SERIALIZED_DATA = 6004,
            VIX_E_PROPERTY_TYPE_MISMATCH = 6005,

            /* Completion Errors */
            VIX_E_BAD_VM_INDEX = 8000,

            /* Message errors */
            VIX_E_INVALID_MESSAGE_HEADER = 10000,
            VIX_E_INVALID_MESSAGE_BODY = 10001,

            /* Snapshot errors */
            VIX_E_SNAPSHOT_INVAL = 13000,
            VIX_E_SNAPSHOT_DUMPER = 13001,
            VIX_E_SNAPSHOT_DISKLIB = 13002,
            VIX_E_SNAPSHOT_NOTFOUND = 13003,
            VIX_E_SNAPSHOT_EXISTS = 13004,
            VIX_E_SNAPSHOT_VERSION = 13005,
            VIX_E_SNAPSHOT_NOPERM = 13006,
            VIX_E_SNAPSHOT_CONFIG = 13007,
            VIX_E_SNAPSHOT_NOCHANGE = 13008,
            VIX_E_SNAPSHOT_CHECKPOINT = 13009,
            VIX_E_SNAPSHOT_LOCKED = 13010,
            VIX_E_SNAPSHOT_INCONSISTENT = 13011,
            VIX_E_SNAPSHOT_NAMETOOLONG = 13012,
            VIX_E_SNAPSHOT_VIXFILE = 13013,
            VIX_E_SNAPSHOT_DISKLOCKED = 13014,
            VIX_E_SNAPSHOT_DUPLICATEDDISK = 13015,
            VIX_E_SNAPSHOT_INDEPENDENTDISK = 13016,
            VIX_E_SNAPSHOT_NONUNIQUE_NAME = 13017,
            VIX_E_SNAPSHOT_MEMORY_ON_INDEPENDENT_DISK = 13018,
            VIX_E_SNAPSHOT_MAXSNAPSHOTS = 13019,
            VIX_E_SNAPSHOT_MIN_FREE_SPACE = 13020,
            VIX_E_SNAPSHOT_HIERARCHY_TOODEEP = 13021,
            // DEPRECRATED VIX_E_SNAPSHOT_RRSUSPEND                     = 13022,
            VIX_E_SNAPSHOT_NOT_REVERTABLE = 13024,

            /* Host Errors */
            VIX_E_HOST_DISK_INVALID_VALUE = 14003,
            VIX_E_HOST_DISK_SECTORSIZE = 14004,
            VIX_E_HOST_FILE_ERROR_EOF = 14005,
            VIX_E_HOST_NETBLKDEV_HANDSHAKE = 14006,
            VIX_E_HOST_SOCKET_CREATION_ERROR = 14007,
            VIX_E_HOST_SERVER_NOT_FOUND = 14008,
            VIX_E_HOST_NETWORK_CONN_REFUSED = 14009,
            VIX_E_HOST_TCP_SOCKET_ERROR = 14010,
            VIX_E_HOST_TCP_CONN_LOST = 14011,
            VIX_E_HOST_NBD_HASHFILE_VOLUME = 14012,
            VIX_E_HOST_NBD_HASHFILE_INIT = 14013,

            /* Disklib errors */
            VIX_E_DISK_INVAL = 16000,
            VIX_E_DISK_NOINIT = 16001,
            VIX_E_DISK_NOIO = 16002,
            VIX_E_DISK_PARTIALCHAIN = 16003,
            VIX_E_DISK_NEEDSREPAIR = 16006,
            VIX_E_DISK_OUTOFRANGE = 16007,
            VIX_E_DISK_CID_MISMATCH = 16008,
            VIX_E_DISK_CANTSHRINK = 16009,
            VIX_E_DISK_PARTMISMATCH = 16010,
            VIX_E_DISK_UNSUPPORTEDDISKVERSION = 16011,
            VIX_E_DISK_OPENPARENT = 16012,
            VIX_E_DISK_NOTSUPPORTED = 16013,
            VIX_E_DISK_NEEDKEY = 16014,
            VIX_E_DISK_NOKEYOVERRIDE = 16015,
            VIX_E_DISK_NOTENCRYPTED = 16016,
            VIX_E_DISK_NOKEY = 16017,
            VIX_E_DISK_INVALIDPARTITIONTABLE = 16018,
            VIX_E_DISK_NOTNORMAL = 16019,
            VIX_E_DISK_NOTENCDESC = 16020,
            VIX_E_DISK_NEEDVMFS = 16022,
            VIX_E_DISK_RAWTOOBIG = 16024,
            VIX_E_DISK_TOOMANYOPENFILES = 16027,
            VIX_E_DISK_TOOMANYREDO = 16028,
            VIX_E_DISK_RAWTOOSMALL = 16029,
            VIX_E_DISK_INVALIDCHAIN = 16030,
            VIX_E_DISK_KEY_NOTFOUND = 16052, // metadata key is not found
            VIX_E_DISK_SUBSYSTEM_INIT_FAIL = 16053,
            VIX_E_DISK_INVALID_CONNECTION = 16054,
            VIX_E_DISK_ENCODING = 16061,
            VIX_E_DISK_CANTREPAIR = 16062,
            VIX_E_DISK_INVALIDDISK = 16063,
            VIX_E_DISK_NOLICENSE = 16064,
            VIX_E_DISK_NODEVICE = 16065,
            VIX_E_DISK_UNSUPPORTEDDEVICE = 16066,
            VIX_E_DISK_CAPACITY_MISMATCH = 16067,
            VIX_E_DISK_PARENT_NOTALLOWED = 16068,
            VIX_E_DISK_ATTACH_ROOTLINK = 16069,

            /* Crypto Library Errors */
            VIX_E_CRYPTO_UNKNOWN_ALGORITHM = 17000,
            VIX_E_CRYPTO_BAD_BUFFER_SIZE = 17001,
            VIX_E_CRYPTO_INVALID_OPERATION = 17002,
            VIX_E_CRYPTO_RANDOM_DEVICE = 17003,
            VIX_E_CRYPTO_NEED_PASSWORD = 17004,
            VIX_E_CRYPTO_BAD_PASSWORD = 17005,
            VIX_E_CRYPTO_NOT_IN_DICTIONARY = 17006,
            VIX_E_CRYPTO_NO_CRYPTO = 17007,
            VIX_E_CRYPTO_ERROR = 17008,
            VIX_E_CRYPTO_BAD_FORMAT = 17009,
            VIX_E_CRYPTO_LOCKED = 17010,
            VIX_E_CRYPTO_EMPTY = 17011,
            VIX_E_CRYPTO_KEYSAFE_LOCATOR = 17012,

            /* Remoting Errors. */
            VIX_E_CANNOT_CONNECT_TO_HOST = 18000,
            VIX_E_NOT_FOR_REMOTE_HOST = 18001,
            VIX_E_INVALID_HOSTNAME_SPECIFICATION = 18002,

            /* Screen Capture Errors. */
            VIX_E_SCREEN_CAPTURE_ERROR = 19000,
            VIX_E_SCREEN_CAPTURE_BAD_FORMAT = 19001,
            VIX_E_SCREEN_CAPTURE_COMPRESSION_FAIL = 19002,
            VIX_E_SCREEN_CAPTURE_LARGE_DATA = 19003,

            /* Guest Errors */
            VIX_E_GUEST_VOLUMES_NOT_FROZEN = 20000,
            VIX_E_NOT_A_FILE = 20001,
            VIX_E_NOT_A_DIRECTORY = 20002,
            VIX_E_NO_SUCH_PROCESS = 20003,
            VIX_E_FILE_NAME_TOO_LONG = 20004,
            VIX_E_OPERATION_DISABLED = 20005,

            /* Tools install errors */
            VIX_E_TOOLS_INSTALL_NO_IMAGE = 21000,
            VIX_E_TOOLS_INSTALL_IMAGE_INACCESIBLE = 21001,
            VIX_E_TOOLS_INSTALL_NO_DEVICE = 21002,
            VIX_E_TOOLS_INSTALL_DEVICE_NOT_CONNECTED = 21003,
            VIX_E_TOOLS_INSTALL_CANCELLED = 21004,
            VIX_E_TOOLS_INSTALL_INIT_FAILED = 21005,
            VIX_E_TOOLS_INSTALL_AUTO_NOT_SUPPORTED = 21006,
            VIX_E_TOOLS_INSTALL_GUEST_NOT_READY = 21007,
            /* VIX_E_TOOLS_INSTALL_SIG_CHECK_FAILED         = 21008, Removed in version 1.15.0 */
            VIX_E_TOOLS_INSTALL_ERROR = 21009,
            VIX_E_TOOLS_INSTALL_ALREADY_UP_TO_DATE = 21010,
            VIX_E_TOOLS_INSTALL_IN_PROGRESS = 21011,
            VIX_E_TOOLS_INSTALL_IMAGE_COPY_FAILED = 21012,

            /* Wrapper Errors */
            VIX_E_WRAPPER_WORKSTATION_NOT_INSTALLED = 22001,
            VIX_E_WRAPPER_VERSION_NOT_FOUND = 22002,
            VIX_E_WRAPPER_SERVICEPROVIDER_NOT_FOUND = 22003,
            VIX_E_WRAPPER_PLAYER_NOT_INSTALLED = 22004,
            VIX_E_WRAPPER_RUNTIME_NOT_INSTALLED = 22005,
            VIX_E_WRAPPER_MULTIPLE_SERVICEPROVIDERS = 22006,

            /* FuseMnt errors*/
            VIX_E_MNTAPI_MOUNTPT_NOT_FOUND = 24000,
            VIX_E_MNTAPI_MOUNTPT_IN_USE = 24001,
            VIX_E_MNTAPI_DISK_NOT_FOUND = 24002,
            VIX_E_MNTAPI_DISK_NOT_MOUNTED = 24003,
            VIX_E_MNTAPI_DISK_IS_MOUNTED = 24004,
            VIX_E_MNTAPI_DISK_NOT_SAFE = 24005,
            VIX_E_MNTAPI_DISK_CANT_OPEN = 24006,
            VIX_E_MNTAPI_CANT_READ_PARTS = 24007,
            VIX_E_MNTAPI_UMOUNT_APP_NOT_FOUND = 24008,
            VIX_E_MNTAPI_UMOUNT = 24009,
            VIX_E_MNTAPI_NO_MOUNTABLE_PARTITONS = 24010,
            VIX_E_MNTAPI_PARTITION_RANGE = 24011,
            VIX_E_MNTAPI_PERM = 24012,
            VIX_E_MNTAPI_DICT = 24013,
            VIX_E_MNTAPI_DICT_LOCKED = 24014,
            VIX_E_MNTAPI_OPEN_HANDLES = 24015,
            VIX_E_MNTAPI_CANT_MAKE_VAR_DIR = 24016,
            VIX_E_MNTAPI_NO_ROOT = 24017,
            VIX_E_MNTAPI_LOOP_FAILED = 24018,
            VIX_E_MNTAPI_DAEMON = 24019,
            VIX_E_MNTAPI_INTERNAL = 24020,
            VIX_E_MNTAPI_SYSTEM = 24021,
            VIX_E_MNTAPI_NO_CONNECTION_DETAILS = 24022,
            /* FuseMnt errors: Do not exceed 24299 */

            /* VixMntapi errors*/
            VIX_E_MNTAPI_INCOMPATIBLE_VERSION = 24300,
            VIX_E_MNTAPI_OS_ERROR = 24301,
            VIX_E_MNTAPI_DRIVE_LETTER_IN_USE = 24302,
            VIX_E_MNTAPI_DRIVE_LETTER_ALREADY_ASSIGNED = 24303,
            VIX_E_MNTAPI_VOLUME_NOT_MOUNTED = 24304,
            VIX_E_MNTAPI_VOLUME_ALREADY_MOUNTED = 24305,
            VIX_E_MNTAPI_FORMAT_FAILURE = 24306,
            VIX_E_MNTAPI_NO_DRIVER = 24307,
            VIX_E_MNTAPI_ALREADY_OPENED = 24308,
            VIX_E_MNTAPI_ITEM_NOT_FOUND = 24309,
            VIX_E_MNTAPI_UNSUPPROTED_BOOT_LOADER = 24310,
            VIX_E_MNTAPI_UNSUPPROTED_OS = 24311,
            VIX_E_MNTAPI_CODECONVERSION = 24312,
            VIX_E_MNTAPI_REGWRITE_ERROR = 24313,
            VIX_E_MNTAPI_UNSUPPORTED_FT_VOLUME = 24314,
            VIX_E_MNTAPI_PARTITION_NOT_FOUND = 24315,
            VIX_E_MNTAPI_PUTFILE_ERROR = 24316,
            VIX_E_MNTAPI_GETFILE_ERROR = 24317,
            VIX_E_MNTAPI_REG_NOT_OPENED = 24318,
            VIX_E_MNTAPI_REGDELKEY_ERROR = 24319,
            VIX_E_MNTAPI_CREATE_PARTITIONTABLE_ERROR = 24320,
            VIX_E_MNTAPI_OPEN_FAILURE = 24321,
            VIX_E_MNTAPI_VOLUME_NOT_WRITABLE = 24322,

            /* Success on operation that completes asynchronously */
            VIX_ASYNC = 25000,

            /* Async errors */
            VIX_E_ASYNC_MIXEDMODE_UNSUPPORTED = 26000,

            /* Network Errors */
            VIX_E_NET_HTTP_UNSUPPORTED_PROTOCOL = 30001,
            VIX_E_NET_HTTP_URL_MALFORMAT = 30003,
            VIX_E_NET_HTTP_COULDNT_RESOLVE_PROXY = 30005,
            VIX_E_NET_HTTP_COULDNT_RESOLVE_HOST = 30006,
            VIX_E_NET_HTTP_COULDNT_CONNECT = 30007,
            VIX_E_NET_HTTP_HTTP_RETURNED_ERROR = 30022,
            VIX_E_NET_HTTP_OPERATION_TIMEDOUT = 30028,
            VIX_E_NET_HTTP_SSL_CONNECT_ERROR = 30035,
            VIX_E_NET_HTTP_TOO_MANY_REDIRECTS = 30047,
            VIX_E_NET_HTTP_TRANSFER = 30200,
            VIX_E_NET_HTTP_SSL_SECURITY = 30201,
            VIX_E_NET_HTTP_GENERIC = 30202
        }

        public bool VMOperation(string vmxFilePath, CtrlVix.Constants.POWEROP powerOP)
        {
            VixCOM.IVixLib IVixLib = new VixCOM.VixLib();
            VixCOM.IHost IHost;
            VixCOM.IVM2 IVM;
            VixCOM.IJob IJob;

            ulong err;
#if DEBUG
            Debug.WriteLine("VIX: Start VMOperation.");
#endif
            // Open the virtual machine and get a handle...
            if ((IHost = GetHandle()) == null)
            {
                return false;       // error
            }
            if ((IVM = OpenVM(IHost, vmxFilePath)) == null)
            {
                IHost.Disconnect();
                return false;       // error
            }
#if DEBUG
//            Debug.WriteLine("VIX: Start VIX command.");
#endif
            switch (powerOP)
            {
                case Constants.POWEROP.POWER_ON_GUI:
                    IJob = IVM.PowerOn((int)VixVMPowerOpOptions.VIX_VMPOWEROP_LAUNCH_GUI, null, null);
                    break;
                case Constants.POWEROP.POWER_ON_NOGUI:
                    IJob = IVM.PowerOn((int)VixVMPowerOpOptions.VIX_VMPOWEROP_NORMAL, null, null);
                    break;
                case Constants.POWEROP.SUSPEND:
                    IJob = IVM.Suspend((int)VixVMPowerOpOptions.VIX_VMPOWEROP_NORMAL, null);
                    break;
                case Constants.POWEROP.ACPI_RESET:
                    IJob = IVM.Reset((int)VixVMPowerOpOptions.VIX_VMPOWEROP_FROM_GUEST, null);
                    break;
                case Constants.POWEROP.ACPI_POWER_OFF:
                    IJob = IVM.PowerOff((int)VixVMPowerOpOptions.VIX_VMPOWEROP_FROM_GUEST, null);
                    break;
                case Constants.POWEROP.HW_RESET:
                    IJob = IVM.Reset((int)VixVMPowerOpOptions.VIX_VMPOWEROP_NORMAL, null);
                    break;
                case Constants.POWEROP.HW_POWER_OFF:
                    IJob = IVM.PowerOff((int)VixVMPowerOpOptions.VIX_VMPOWEROP_NORMAL, null);
                    break;
                default:
                    return false;
            }
#if DEBUG
            Debug.WriteLine("VIX: Done VIX command.");
#endif
            System.Threading.Thread.Sleep(1000);    // 1sec wait
            err = IJob.WaitWithoutResults();
#if DEBUG
            Debug.WriteLine("VIX: Finish JobWaitWithoutResults.");
#endif
            if (IVixLib.ErrorIndicatesFailure(err))
            {
                // VIX's bug? ACPI Reset/Power Offをしたとき3009エラーで実行されないことがある
                if (((powerOP == Constants.POWEROP.ACPI_RESET) || (powerOP == Constants.POWEROP.ACPI_POWER_OFF)) 
                                    && (IVixLib.ErrorCode(err) == (short)VixErr.VIX_E_POWEROP_SCRIPTS_NOT_AVAILABLE))
                {
                    CloseVixObject(IJob);
                    CloseVixObject(IVM);
                    IHost.Disconnect();

                    // もう一度実行する
#if DEBUG
                    Debug.WriteLine("VIX: ReRunVMControl");
#endif
                    System.Threading.Thread.Sleep(5000);    // 5sec wait
                    if (reRunVMOperation(vmxFilePath, powerOP)) return true;
                    
                    if (CheckToolsRunning(vmxFilePath) != Constants.TOOLS_RUNNING.RUNNING)
                    {
                        // toolが動いていないときはメッセージは表示するがエラーとはしない
                        ErrMessage((ulong)VixErr.VIX_E_POWEROP_SCRIPTS_NOT_AVAILABLE);
                        return true;
                    }
                    // toolが動いてるときもメッセージは表示するがエラーとはしない
                    ErrMessage((ulong)VixErr.VIX_E_TIMEOUT_WAITING_FOR_TOOLS);
                    return true;
                }
                // SUSPED させたとき VM_NOT_RUNNNINGはエラーとはしない
                else if ((powerOP == Constants.POWEROP.SUSPEND) &&
                                                (IVixLib.ErrorCode(err) == (short)VixErr.VIX_E_VM_NOT_RUNNING))
                {
                    // NOP
                }
                // for Player 12.0.0 NOGUI させたとき VIX_E_CANCELLEDはメッセージは表示するがエラーとはしない
                else if ((powerOP == Constants.POWEROP.POWER_ON_NOGUI) &&
                                                (IVixLib.ErrorCode(err) == (short)VixErr.VIX_E_CANCELLED))
                {
                    ErrMessage(err);
                }
                else
                {
                    // Error Handler
                    ErrMessage(err);
                    CloseVixObject(IJob);
                    CloseVixObject(IVM);
                    IHost.Disconnect();
                    return false;
                }
            }
            CloseVixObject(IJob);
            CloseVixObject(IVM);
            IHost.Disconnect();
#if DEBUG
            Debug.WriteLine("VIX: Finish VMOperation.");
#endif
            return true;
        }

        private bool reRunVMOperation(string vmxFilePath, CtrlVix.Constants.POWEROP powerOP)
        {
            VixCOM.IVixLib IVixLib = new VixCOM.VixLib();
            VixCOM.IHost IHost;
            VixCOM.IVM2 IVM;
            VixCOM.IJob IJob;

            ulong err;
#if DEBUG
            Debug.WriteLine("VIX: Re:Start VMOperation.");
#endif
            // Open the virtual machine and get a handle...
            if ((IHost = GetHandle()) == null)
            {
                return false;       // error
            }
            if ((IVM = OpenVM(IHost, vmxFilePath)) == null)
            {
                IHost.Disconnect();
                return false;       // error
            }
#if DEBUG
            Debug.WriteLine("VIX: Re:Start VIX command.");
#endif
            switch (powerOP)
            {
                case Constants.POWEROP.ACPI_RESET:
                    IJob = IVM.Reset((int)VixVMPowerOpOptions.VIX_VMPOWEROP_FROM_GUEST, null);
                    break;
                case Constants.POWEROP.ACPI_POWER_OFF:
                    IJob = IVM.PowerOff((int)VixVMPowerOpOptions.VIX_VMPOWEROP_FROM_GUEST, null);
                    break;
                default:
                    return false;
            }
#if DEBUG
            Debug.WriteLine("VIX: Re:Done VIX command.");
#endif
            err = IJob.WaitWithoutResults();
#if DEBUG
            Debug.WriteLine("VIX: Re:Finish JobWaitWithoutResults.");
#endif
            if (IVixLib.ErrorIndicatesFailure(err))
            {
                // Error Handler
                CloseVixObject(IJob);
                CloseVixObject(IVM);
                IHost.Disconnect();
                return false;
             }
            CloseVixObject(IJob);
            CloseVixObject(IVM);
            IHost.Disconnect();
            return true;
        }

        public Constants.VMSTATE GetVMState(string vmxFilePath)
        {
            // Processing (0x0895)
            const int processing = (int)VixPowerState.VIX_POWERSTATE_POWERING_OFF // 0x0001
                                 | (int)VixPowerState.VIX_POWERSTATE_POWERING_ON  // 0x0004
                                 | (int)VixPowerState.VIX_POWERSTATE_SUSPENDING   // 0x0010
                                 | (int)VixPowerState.VIX_POWERSTATE_RESETTING    // 0x0080
                                 | (int)VixPowerState.VIX_POWERSTATE_RESUMING;    // 0x0800
            // Unknown status (0x0300)
            const int unknown = (int)VixPowerState.VIX_POWERSTATE_BLOCKED_ON_MSG  // 0x0100
                              | (int)VixPowerState.VIX_POWERSTATE_PAUSED;         // 0x0200
            // Powered OFF (0x0022)
            const int powered_off = (int)VixPowerState.VIX_POWERSTATE_POWERED_OFF // 0x0002
                                  | (int)VixPowerState.VIX_POWERSTATE_SUSPENDED;  // 0x0020
            // Powered ON (0x0008)
            const int powered_on = (int)VixPowerState.VIX_POWERSTATE_POWERED_ON;  // 0x0008
#if DEBUG
            // Tooles running (0x0040)
            const int tools_running = (int)VixPowerState.VIX_POWERSTATE_TOOLS_RUNNING;    // 0x0040
            // Undefined
            const int undefined = 0xffff ^ (processing | unknown | powered_off | powered_on | tools_running);
#endif
            int vmState;

            vmState = getVMState(vmxFilePath);
            if (vmState == 0xffff)
            {
#if DEBUG
                Debug.WriteLine("VIX: Detect VMSTATE.ERROR");
#endif
                return Constants.VMSTATE.ERROR;         // error
            }
#if DEBUG
            if ((vmState & undefined) != 0)
            {
                // 何もしない(debug)用
                Debug.WriteLine("VIX: " + vmState.ToString("X") + " Detect undefined flag in vmstate.");
            }
            if ((vmState & tools_running) != 0)
            {
                // 何もしない(debug)用
                Debug.WriteLine("VIX: Detect tools running flag in vmstate.");
            }
#endif
            // Powered OFF
            if ((vmState & powered_off) != 0)
            {
#if DEBUG
                if ((vmState & (int)VixPowerState.VIX_POWERSTATE_SUSPENDED) != 0)
                {
                    Debug.WriteLine("VIX: Detect VIX_POWERSTATE_SUSPENDED.");
                }
#endif
                // VIXを長時間動かしているとPoweredOFFになってしまうバグの回避のため
                if (CheckPowerOnByProcess(vmxFilePath))
                {
#if DEBUG
                    Debug.WriteLine("VIX: Rewrite the vmState to power on cause of detecting the VIX's bug.");
                    Debug.WriteLine("     However, some cases go through this routing by remaining the process.");
#endif
                    vmState = (vmState ^ (vmState & powered_off)) | powered_on; // powered_offを落としてpowered_onを立てて後の処理に任せる
                }
                else return Constants.VMSTATE.POWERED_OFF;   // Powered OFF
            }
            // Processing
            if (( vmState & (processing | unknown)) != 0)
            {
#if DEBUG
                if ((vmState & (int)VixPowerState.VIX_POWERSTATE_BLOCKED_ON_MSG) != 0)
                {
                    Debug.WriteLine("VIX: Detect VIX_POWERSTATE_BLOCKED_ON_MSG.");
                }
                if ((vmState & (int)VixPowerState.VIX_POWERSTATE_PAUSED) != 0)
                {
                    Debug.WriteLine("VIX: Detect VIX_POWERSTATE_PAUSED.");
                }
                if ((vmState & (int)VixPowerState.VIX_POWERSTATE_RESUMING) != 0)
                {
                    Debug.WriteLine("VIX: Detect VIX_POWERSTATE_RESUMING.");
                }
#endif
                return Constants.VMSTATE.PROCESSING;    // Processing
            }
            // Powered ON
            if ((vmState & powered_on) != 0)
            {
                return Constants.VMSTATE.POWERED_ON;    // Powered ON
            }
            return Constants.VMSTATE.ERROR;      // error
        }

        public Constants.RUNNING_MODE CheckRunningMode(string vmxFilePath)
        {
            bool vmxflag = true;
            string vmxName = Path.GetFileNameWithoutExtension(vmxFilePath);  // VM 名を作成

            ManagementPath mngmntPath = new ManagementPath("Win32_Process");
            ManagementClass mngmntClass = new ManagementClass(mngmntPath);

            ManagementObjectCollection mngmntObjClctn = mngmntClass.GetInstances();
            foreach (ManagementObject obj in mngmntObjClctn)
            {
                object commandLineObj = obj["CommandLine"];
                if (commandLineObj != null)
                {
                    string commandLine = commandLineObj.ToString();
                    if (0 <= commandLine.IndexOf("vmplayer"))
                    {
                        if (0 <= commandLine.IndexOf(vmxName))
                        {
                            return Constants.RUNNING_MODE.GUI;  // "vmplayer + vmxName" runnning 
                        }
                        if (0 >= commandLine.IndexOf(".vmx"))
                        {
                            vmxflag = false;
                        }
                    }
                }
            }
            if (!vmxflag)
            {
                return Constants.RUNNING_MODE.CANNOT_DETERMINE;  // "vmplayer" only running
            }
            return Constants.RUNNING_MODE.NOGUI;    // "vmplayer" not runnning or "vmplayer + other vmx" runnning
        }

        public Constants.TOOLS_RUNNING CheckToolsRunning(string vmxFilePath)
        {
            int vmState;

            vmState = getVMState(vmxFilePath);
            if (vmState == 0xffff)
            {
#if DEBUG
                Debug.WriteLine("VIX: Detect TOOLS_RUNNING.ERROR");
#endif
                return Constants.TOOLS_RUNNING.ERROR;         // error
            }
            if ((vmState & (int)VixPowerState.VIX_POWERSTATE_POWERED_ON) != 0)
            {
                if ((vmState & (int)VixPowerState.VIX_POWERSTATE_TOOLS_RUNNING) != 0)
                {
                    return Constants.TOOLS_RUNNING.RUNNING;
                }
            }
#if DEBUG
            Debug.WriteLine("VIX: Detect TOOLS_RUNNING.NOT_RUNNING");
#endif
            return Constants.TOOLS_RUNNING.NOT_RUNNING;
        }

        public Constants.POWER_OFF_MODE CheckPowerOffMode(string vmxFilePath)
        {
            int vmState;
            string Dir;
            string VMname;
            string VMSSname;
            string[] files;

            Dir = getVMWorkingDir(vmxFilePath);

            vmState = getVMState(vmxFilePath);
            if (vmState == 0xffff)
            {
                return Constants.POWER_OFF_MODE.ERROR;         // error
            }
            // VIXからこのフラグは戻らないようだが念のためチェックする
            if ((vmState & (int)VixPowerState.VIX_POWERSTATE_SUSPENDED) != 0)
            {
#if DEBUG
                Debug.WriteLine("VIX: Returned VM Suspended flag.");
#endif
                return Constants.POWER_OFF_MODE.SUSPENDED;
            }
            // suspend時に"VMXname-????????.vmss"ファイルが作成されることを利用して判定
            // エラーでファイルが残った場合などはうまく判定できない可能性あり
            VMname = Path.GetFileNameWithoutExtension(vmxFilePath);
            VMSSname = VMname + "-*.vmss";
            try
            {
                files = Directory.GetFiles(Dir, VMSSname, SearchOption.TopDirectoryOnly);
                if (files.Length == 0)
                {
                    return Constants.POWER_OFF_MODE.POWERED_OFF;
                }
                return Constants.POWER_OFF_MODE.SUSPENDED;
            }
            catch (IOException ex)
            {
#if DEBUG
                Debug.WriteLine("VIX: Catch System.IO.IOException");
#endif
                MessageBox.Show("Inside CtrlVix error: " + ex.Message,
                              "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return Constants.POWER_OFF_MODE.ERROR;
            }
            catch (Exception ex)
            {
#if DEBUG
                Debug.WriteLine("VIX: Catch System.Exception");
#endif
                MessageBox.Show("Inside CtrlVix error: " + ex.Message,
                              "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return Constants.POWER_OFF_MODE.ERROR;
            }
        }

        private int getVMState(string vmxFilePath)
        {
            // VMwareの動作状態をチェック
            // These values are bitwise flags.
            //const int VIX_POWERSTATE_POWERING_OFF    = 0x0001;
            // Indicates that VixVM_PowerOff() has been called, but the operation itself has not completed. 
            //const int VIX_POWERSTATE_POWERED_OFF     = 0x0002;
            // Indicates that the virtual machine is not running. 
            //const int VIX_POWERSTATE_POWERING_ON     = 0x0004;
            // Indicates that VixVM_PowerOn() has been called, but the operation itself has not completed. 
            //const int VIX_POWERSTATE_POWERED_ON      = 0x0008;
            // Indicates that the virtual machine is running. 
            //const int VIX_POWERSTATE_SUSPENDING      = 0x0010;
            // Indicates that VixVM_Suspend() has been called, but the operation itself has not completed. 
            //const int VIX_POWERSTATE_SUSPENDED       = 0x0020;
            // Indicates that the virtual machine is suspended. Use VixVM_PowerOn() to resume the virtual machine. 
            //const int VIX_POWERSTATE_TOOLS_RUNNING   = 0x0040;
            // Indicates that the virtual machine is running and the VMware Tools suite is active. See also the VixToolsState property. 
            //const int VIX_POWERSTATE_RESETTING       = 0x0080;
            // Indicates that VixVM_Reset() has been called, but the operation itself has not completed. 
            //const int VIX_POWERSTATE_BLOCKED_ON_MSG  = 0x0100;
            // Indicates that a virtual machine state change is blocked, waiting for user interaction
            //const int VIX_POWERSTATE_PAUSED          = 0x0200;
            //const int VIX_POWERSTATE_RESUMING        = 0x0800;

            // Get Properties
            VixCOM.IVixLib IVixLib = new VixCOM.VixLib();
            VixCOM.IHost IHost;
            VixCOM.IVM2 IVM;
            VixCOM.IVixHandle IVixHandle;

            ulong err;
            object propertiesArray;
            int[] propertyIDs;
            int vmState;
            int toolsState;

            // Open the virtual machine and get a handle...
            if ((IHost = GetHandle()) == null)
            {
                return 0xffff;      // error
            }
            if ((IVM = OpenVM(IHost, vmxFilePath)) == null)
            {
                IHost.Disconnect();
                return 0xffff;      // error
            }
            // Get Properties
            // propertyIDs = new int[] { (int)VixPropertyID.VIX_PROPERTY_VM_POWER_STATE,
            //                                    (int)VixPropertyID.VIX_PROPERTY_VM_VMX_PATHNAME };
            // propertyIDs = new int[] { (int)VixPropertyID.VIX_PROPERTY_VM_POWER_STATE };
            propertyIDs = new int[] { (int)VixPropertyID.VIX_PROPERTY_VM_POWER_STATE,
                                                 (int)VixPropertyID.VIX_PROPERTY_VM_TOOLS_STATE };
            propertiesArray = null;
            IVixHandle = (VixCOM.IVixHandle)IVM;
            err = IVixHandle.GetProperties(propertyIDs, ref propertiesArray);
            if (IVixLib.ErrorIndicatesFailure(err))
            {
                // Error Handler
                ErrMessage(err);
                CloseVixObject(IVM);
                IHost.Disconnect();
                return 0xffff;      // error
            }
            vmState = (int)((object[])propertiesArray)[0];     // VIX_PROPERTY_VM_POWER_STATE
            // vmPathName = (string)((object[])propertiesArray)[1];    // VIX_PROPERTY_VM_VMX_PATHNAME
            toolsState = (int)((object[])propertiesArray)[1];    // VIX_PROPERTY_VM_TOOLS_STATE
#if DEBUG
             Debug.WriteLine("VIX: vmState = " + vmState.ToString("X") + "  toolsState = " + toolsState.ToString("X"));
#endif
            if ((toolsState & (int)VixToolsState.VIX_TOOLSSTATE_RUNNING) == (int)VixToolsState.VIX_TOOLSSTATE_RUNNING)
            {
                // POWER_STATEプロパティにTOOLS_RUNNINGが入ってこないので、TOOLS_STATEを見て手動でフラグを立てる
                // ただし、正しくは判定されない。
                vmState = vmState | (int)VixPowerState.VIX_POWERSTATE_TOOLS_RUNNING;
            }
            CloseVixObject(IVM);
            IHost.Disconnect();
            return vmState;
        }

        private VixCOM.IHost GetHandle()
        {
            int vmxApiVersion = (int)VixServiceProvider.VIX_API_VERSION;
            int vmxHostType = (int)VixServiceProvider.VIX_SERVICEPROVIDER_VMWARE_PLAYER;
            string hostName = "";
            string userName = "";
            string password = "";

            VixCOM.IVixLib IVixLib = new VixCOM.VixLib();
            VixCOM.IJob IJob;
            VixCOM.IHost IHost;

            ulong err;
            object propertiesArray;
            int[] propertyIDs;

            // Open the virtual machine and get a handle...
            IJob = IVixLib.Connect(vmxApiVersion, vmxHostType, hostName, 0, userName, password, 0, null, null);
            propertiesArray = null;
            propertyIDs = new int[] { (int)VixPropertyID.VIX_PROPERTY_JOB_RESULT_HANDLE };
            err = IJob.Wait(propertyIDs, ref propertiesArray);
            if (IVixLib.ErrorIndicatesFailure(err))
            {
                // Error Handler
                ErrMessage(err);
                IHost = null;
            }
            else
            {
                IHost = (VixCOM.IHost)((object[])propertiesArray)[0];
                CloseVixObject(IJob);
            }
            return IHost;
        }

        private VixCOM.IVM2 OpenVM(VixCOM.IHost IHost, string vmxFilePath)
        {
            VixCOM.IVixLib IVixLib = new VixCOM.VixLib();
            VixCOM.IJob IJob;
            VixCOM.IVM2 IVM;

            ulong err;
            object propertiesArray;
            int[] propertyIDs;

            // Open the virtual machine
            IJob = IHost.OpenVM(vmxFilePath, null);
            propertiesArray = null;
            propertyIDs = new int[] { (int)VixPropertyID.VIX_PROPERTY_JOB_RESULT_HANDLE };
            err = IJob.Wait(propertyIDs, ref propertiesArray);
            if (IVixLib.ErrorIndicatesFailure(err))
            {
                // Error Handler
                ErrMessage(err);
                IVM = null;
            }
            else
            {
                IVM = (VixCOM.IVM2)((object[])propertiesArray)[0];
                CloseVixObject(IJob);
            }
            return IVM;
        }

        private static void CloseVixObject(Object vixObject)
        {
            try
            {
                ((VixCOM.IVixHandle2)vixObject).Close();
            }
            catch (Exception)
            {
                // Close is not supported in this version of Vix COM - Ignore
            }
        }

        private string getVMWorkingDir(string vmxFilePath)
        {
            foreach (var section in ReadVMX(vmxFilePath))
            {
                // section.Key は処理しない
                foreach (var pair in section.Value)
                {
                    if (pair.Key == Constants.WORKINGDIR_ERR) return Constants.WORKINGDIR_ERR;
                    if (pair.Key == Constants.WORKINGDIR) return (pair.Value.Substring(1, pair.Value.Length - 2));
                }
            }
            return Path.GetDirectoryName(vmxFilePath);
        }

        private static Dictionary<string, Dictionary<string, string>> ReadVMX(string file)
        {
            var sections = new Dictionary<string, Dictionary<string, string>>(StringComparer.Ordinal);
            var regexSection = new Regex(@"^\s*\[(?<section>[^\]]+)\].*$", RegexOptions.Singleline
                                                                        | RegexOptions.CultureInvariant);
            var regexNameValue = new Regex(@"^\s*(?<name>[^=]+)=(?<value>.*?)(\s+;(?<comment>.*))?$",
                                                RegexOptions.Singleline | RegexOptions.CultureInvariant);
            var currentSection = string.Empty;

            // セクション名が明示されていない先頭部分のセクション名を""として扱う
            sections[string.Empty] = new Dictionary<string, string>();   
            try
            {
                using (var reader = new StreamReader(file))
                {
                    for (; ; )
                    {
                        var line = reader.ReadLine();

                        if (line == null) break;
                        if (line.Length == 0) continue;     // 空行は読み飛ばす
                        if (line.StartsWith(";", StringComparison.Ordinal)) continue;       // ; コメント行は読み飛ばす
                        else if (line.StartsWith("#", StringComparison.Ordinal)) continue;  // # コメント行は読み飛ばす

                        var matchNameValue = regexNameValue.Match(line);
                        if (matchNameValue.Success)
                        {
                            sections[currentSection][matchNameValue.Groups["name"].Value.Trim()]
                                                            = matchNameValue.Groups["value"].Value.Trim();  // name=valueの行
                            continue;
                        }

                        var matchSection = regexSection.Match(line);
                        if (matchSection.Success)
                        {
                            currentSection = matchSection.Groups["section"].Value;  // [section]の行
                            if (!sections.ContainsKey(currentSection)) sections[currentSection]
                                                                = new Dictionary<string, string>(); continue;
                        }
                    }
                    return sections;
                }
            }
            catch (IOException ex)
            {
#if DEBUG
                Debug.WriteLine("VIX: Catch System.IO.IOException");
#endif
                MessageBox.Show("Inside CtrlVix error: " + ex.Message,
                              "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                sections[string.Empty][Constants.WORKINGDIR_ERR] = Constants.WORKINGDIR_ERR;
                return sections;
            }
            catch (Exception ex)
            {
#if DEBUG
                Debug.WriteLine("VIX: Catch System.Exception");
#endif
                MessageBox.Show("Inside CtrlVix error: " + ex.Message,
                              "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                sections[string.Empty][Constants.WORKINGDIR_ERR] = Constants.WORKINGDIR_ERR;
                return sections;
            }
        }

        private void ErrMessage(ulong err)
        {
            VixCOM.IVixLib IVixLib = new VixCOM.VixLib();
            string errMsg = IVixLib.GetErrorText(err, null);

            switch (err)
            {
#if !DEBUG
                case (ulong)VixErr.VIX_E_FAIL:                                   // 1
                    errMsg = "VIX_E_FAIL";
                    break;
                case (ulong)VixErr.VIX_E_OUT_OF_MEMORY:                          // 2
                    errMsg = "VIX_E_OUT_OF_MEMORY";
                    break;
                case (ulong)VixErr.VIX_E_INVALID_ARG:                            // 3
                    errMsg = "VIX_E_INVALID_ARG";
                    break;
                case (ulong)VixErr.VIX_E_FILE_NOT_FOUND:                         // 4
                    errMsg = "VIX_E_FILE_NOT_FOUND";
                    break;
                case (ulong)VixErr.VIX_E_OBJECT_IS_BUSY:                         // 5
                    errMsg = "VIX_E_OBJECT_IS_BUSY";
                    break;
                case (ulong)VixErr.VIX_E_NOT_SUPPORTED:                          // 6
                    errMsg = "VIX_E_NOT_SUPPORTED";
                    break;
                case (ulong)VixErr.VIX_E_FILE_ERROR:                             // 7
                    errMsg = "VIX_E_FILE_ERROR";
                    break;
                case (ulong)VixErr.VIX_E_DISK_FULL:                              // 8
                    errMsg = "VIX_E_DISK_FULL";
                    break;
                case (ulong)VixErr.VIX_E_INCORRECT_FILE_TYPE:                    // 9
                    errMsg = "VIX_E_INCORRECT_FILE_TYPE";
                    break;
#endif
                case (ulong)VixErr.VIX_E_CANCELLED:                              // 10
                    errMsg = "Operation cancelled.";
                    break;
#if !DEBUG
                case (ulong)VixErr.VIX_E_FILE_READ_ONLY:                         // 11
                    errMsg = "VIX_E_FILE_READ_ONLY";
                    break;
                case (ulong)VixErr.VIX_E_FILE_ALREADY_EXISTS:                    // 12
                    errMsg = "VIX_E_FILE_ALREADY_EXISTS";
                    break;
                case (ulong)VixErr.VIX_E_FILE_ACCESS_ERROR:                      // 13
                    errMsg = "VIX_E_FILE_ACCESS_ERROR";
                    break;
                case (ulong)VixErr.VIX_E_REQUIRES_LARGE_FILES:                   // 14
                    errMsg = "VIX_E_REQUIRES_LARGE_FILES";
                    break;
                case (ulong)VixErr.VIX_E_FILE_ALREADY_LOCKED:                    // 15
                    errMsg = "VIX_E_FILE_ALREADY_LOCKED";
                    break;
                case (ulong)VixErr.VIX_E_VMDB:                                   // 16
                    errMsg = "VIX_E_VMDB";
                    break;
                case (ulong)VixErr.VIX_E_NOT_SUPPORTED_ON_REMOTE_OBJECT:         // 20
                    errMsg = "VIX_E_NOT_SUPPORTED_ON_REMOTE_OBJECT";
                    break;
                case (ulong)VixErr.VIX_E_FILE_TOO_BIG:                           // 21
                    errMsg = "VIX_E_FILE_TOO_BIG";
                    break;
                case (ulong)VixErr.VIX_E_FILE_NAME_INVALID:                      // 22
                    errMsg = "VIX_E_FILE_NAME_INVALID";
                    break;
                case (ulong)VixErr.VIX_E_ALREADY_EXISTS:                         // 23
                    errMsg = "VIX_E_ALREADY_EXISTS";
                    break;
                case (ulong)VixErr.VIX_E_BUFFER_TOOSMALL:                        // 24
                    errMsg = "VIX_E_BUFFER_TOOSMALL";
                    break;
                case (ulong)VixErr.VIX_E_OBJECT_NOT_FOUND:                       // 25
                    errMsg = "VIX_E_OBJECT_NOT_FOUND";
                    break;
                case (ulong)VixErr.VIX_E_HOST_NOT_CONNECTED:                     // 26
                    errMsg = "VIX_E_HOST_NOT_CONNECTED";
                    break;
                case (ulong)VixErr.VIX_E_INVALID_UTF8_STRING:                    // 27
                    errMsg = "VIX_E_INVALID_UTF8_STRING";
                    break;
                case (ulong)VixErr.VIX_E_OPERATION_ALREADY_IN_PROGRESS:          // 31
                    errMsg = "VIX_E_OPERATION_ALREADY_IN_PROGRESS";
                    break;
                case (ulong)VixErr.VIX_E_UNFINISHED_JOB:                         // 29
                    errMsg = "VIX_E_UNFINISHED_JOB";
                    break;
                case (ulong)VixErr.VIX_E_NEED_KEY:                               // 30
                    errMsg = "VIX_E_NEED_KEY";
                    break;
                case (ulong)VixErr.VIX_E_LICENSE:                                // 32
                    errMsg = "VIX_E_LICENSE";
                    break;
                case (ulong)VixErr.VIX_E_VM_HOST_DISCONNECTED:                   // 34
                    errMsg = "VIX_E_VM_HOST_DISCONNECTED";
                    break;
                case (ulong)VixErr.VIX_E_AUTHENTICATION_FAIL:                    // 35
                    errMsg = "VIX_E_AUTHENTICATION_FAIL";
                    break;
                case (ulong)VixErr.VIX_E_HOST_CONNECTION_LOST:                   // 36
                    errMsg = "VIX_E_HOST_CONNECTION_LOST";
                    break;
                case (ulong)VixErr.VIX_E_DUPLICATE_NAME:                         // 41
                    errMsg = "ConVIX_E_DUPLICATE_NAME";
                    break;
                case (ulong)VixErr.VIX_E_ARGUMENT_TOO_BIG:                       // 44
                    errMsg = "VIX_E_ARGUMENT_TOO_BIG";
                    break;
#endif
                case (ulong)VixErr.VIX_E_TIMEOUT_WAITING_FOR_TOOLS:              // 3000
                    errMsg = "Power operation has not been able to be completed.";
                    break;
#if !DEBUG
                case (ulong)VixErr.VIX_E_UNRECOGNIZED_COMMAND:                   // 3001
                    errMsg = "VIX_E_UNRECOGNIZED_COMMAND";
                    break;
                case (ulong)VixErr.VIX_E_OP_NOT_SUPPORTED_ON_GUEST:              // 3003
                    errMsg = "VIX_E_OP_NOT_SUPPORTED_ON_GUEST";
                    break;
                case (ulong)VixErr.VIX_E_PROGRAM_NOT_STARTED:                    // 3004
                    errMsg = "VIX_E_PROGRAM_NOT_STARTED";
                    break;
                case (ulong)VixErr.VIX_E_CANNOT_START_READ_ONLY_VM:              // 3005
                    errMsg = "VIX_E_CANNOT_START_READ_ONLY_VM";
                    break;
                case (ulong)VixErr.VIX_E_VM_NOT_RUNNING:                         // 3006
                    errMsg = "VIX_E_VM_NOT_RUNNING";
                    break;
                case (ulong)VixErr.VIX_E_VM_IS_RUNNING:                          // 3007
                    errMsg = "VIX_E_VM_IS_RUNNING";
                    break;
                case (ulong)VixErr.VIX_E_CANNOT_CONNECT_TO_VM:                   // 3008
                    errMsg = "VIX_E_CANNOT_CONNECT_TO_VM";
                    break;
#endif
                case (ulong)VixErr.VIX_E_POWEROP_SCRIPTS_NOT_AVAILABLE:          // 3009
                    errMsg = "VMware Tools are NOT running or NOT installed properly.";
                    break;
#if !DEBUG
                case (ulong)VixErr.VIX_E_NO_GUEST_OS_INSTALLED:                  // 3010
                    errMsg = "VIX_E_NO_GUEST_OS_INSTALLED";
                    break;
                case (ulong)VixErr.VIX_E_VM_INSUFFICIENT_HOST_MEMORY:            // 3011
                    errMsg = "VIX_E_VM_INSUFFICIENT_HOST_MEMORY";
                    break;
                case (ulong)VixErr.VIX_E_SUSPEND_ERROR:                          // 3012
                    errMsg = "VIX_E_SUSPEND_ERROR";
                    break;
                case (ulong)VixErr.VIX_E_VM_NOT_ENOUGH_CPUS:                     // 3013
                    errMsg = "VIX_E_VM_NOT_ENOUGH_CPUS";
                    break;
                case (ulong)VixErr.VIX_E_HOST_USER_PERMISSIONS:                  // 3014
                    errMsg = "VIX_E_HOST_USER_PERMISSIONS";
                    break;
                case (ulong)VixErr.VIX_E_GUEST_USER_PERMISSIONS:                 // 3015
                    errMsg = "VIX_E_GUEST_USER_PERMISSIONS";
                    break;
                case (ulong)VixErr.VIX_E_TOOLS_NOT_RUNNING:                      // 3016
                    errMsg = "VIX_E_TOOLS_NOT_RUNNING";
                    break;
                case (ulong)VixErr.VIX_E_GUEST_OPERATIONS_PROHIBITED:            // 3017
                    errMsg = "VIX_E_GUEST_OPERATIONS_PROHIBITED";
                    break;
                case (ulong)VixErr.VIX_E_ANON_GUEST_OPERATIONS_PROHIBITED:       // 3018
                    errMsg = "VIX_E_ANON_GUEST_OPERATIONS_PROHIBITED";
                    break;
                case (ulong)VixErr.VIX_E_ROOT_GUEST_OPERATIONS_PROHIBITED:       // 3019
                    errMsg = "VIX_E_ROOT_GUEST_OPERATIONS_PROHIBITED";
                    break;
                case (ulong)VixErr.VIX_E_MISSING_ANON_GUEST_ACCOUNT:             // 3023
                    errMsg = "VIX_E_MISSING_ANON_GUEST_ACCOUNT";
                    break;
                case (ulong)VixErr.VIX_E_CANNOT_AUTHENTICATE_WITH_GUEST:         // 3024
                    errMsg = "VIX_E_CANNOT_AUTHENTICATE_WITH_GUEST";
                    break;
                case (ulong)VixErr.VIX_E_UNRECOGNIZED_COMMAND_IN_GUEST:          // 3025
                    errMsg = "VIX_E_UNRECOGNIZED_COMMAND_IN_GUEST";
                    break;
                case (ulong)VixErr.VIX_E_CONSOLE_GUEST_OPERATIONS_PROHIBITED:    // 3026
                    errMsg = "VIX_E_CONSOLE_GUEST_OPERATIONS_PROHIBITED";
                    break;
                case (ulong)VixErr.VIX_E_MUST_BE_CONSOLE_USER:                   // 3027
                    errMsg = "VIX_E_MUST_BE_CONSOLE_USER";
                    break;
                case (ulong)VixErr.VIX_E_VMX_MSG_DIALOG_AND_NO_UI:               // 3028
                    errMsg = "VIX_E_VMX_MSG_DIALOG_AND_NO_UI";
                    break;
                case (ulong)VixErr.VIX_E_OPERATION_NOT_ALLOWED_FOR_LOGIN_TYPE:   // 3031
                    errMsg = "VIX_E_OPERATION_NOT_ALLOWED_FOR_LOGIN_TYPE";
                    break;
                case (ulong)VixErr.VIX_E_LOGIN_TYPE_NOT_SUPPORTED:               // 3032
                    errMsg = "VIX_E_LOGIN_TYPE_NOT_SUPPORTED";
                    break;
                case (ulong)VixErr.VIX_E_EMPTY_PASSWORD_NOT_ALLOWED_IN_GUEST:    // 3033
                    errMsg = "VIX_E_EMPTY_PASSWORD_NOT_ALLOWED_IN_GUEST";
                    break;
                case (ulong)VixErr.VIX_E_INTERACTIVE_SESSION_NOT_PRESENT:        // 3034
                    errMsg = "VIX_E_INTERACTIVE_SESSION_NOT_PRESENT";
                    break;
                case (ulong)VixErr.VIX_E_INTERACTIVE_SESSION_USER_MISMATCH:      // 3035
                    errMsg = "VIX_E_INTERACTIVE_SESSION_USER_MISMATCH";
                    break;
                case (ulong)VixErr.VIX_E_CANNOT_POWER_ON_VM:                     // 3041
                    errMsg = "VIX_E_CANNOT_POWER_ON_VM";
                    break;
                case (ulong)VixErr.VIX_E_NO_DISPLAY_SERVER:                      // 3043
                    errMsg = "VIX_E_NO_DISPLAY_SERVER";
                    break;
                case (ulong)VixErr.VIX_E_TOO_MANY_LOGONS:                        // 3046
                    errMsg = "VIX_E_TOO_MANY_LOGONS";
                    break;
                case (ulong)VixErr.VIX_E_INVALID_AUTHENTICATION_SESSION:         // 3047
                    errMsg = "VIX_E_INVALID_AUTHENTICATION_SESSION";
                    break;
#endif
                case (ulong)VixErr.VIX_E_VM_NOT_FOUND:                           // 4000
#if DEBUG
                    Debug.WriteLine("VIX: VIX file not found");
#endif
                    break;      // vix file not found の時は無視する
                default:
#if !DEBUG
                    errMsg = "Inside VIX error.";
#endif
                    break;
            }
            MessageBox.Show("Inside VIX error: " + IVixLib.ErrorCode(err).ToString("D")
                                      + " : " + errMsg,
                                          "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
        }

        private bool CheckPowerOnByProcess(string vmxFilePath)
        {
            string vmxName = Path.GetFileNameWithoutExtension(vmxFilePath);  // VM 名を作成

            ManagementPath mngmntPath = new ManagementPath("Win32_Process");
            ManagementClass mngmntClass = new ManagementClass(mngmntPath);

            ManagementObjectCollection mngmntObjClctn = mngmntClass.GetInstances();
            foreach (ManagementObject obj in mngmntObjClctn)
            {
                object commandLineObj = obj["CommandLine"];
                if (commandLineObj != null)
                {
                    string commandLine = commandLineObj.ToString();
                    if (0 <= commandLine.IndexOf("vmware-vmx"))
                    {
                        if (0 <= commandLine.IndexOf(vmxName))
                        {
                            return true;  // "vmware-vmx + vmxName" runnning = Powered ON
                        }
                    }
                }
            }
            return false;    // "vmware-vmx + vmxName" not runnning = Powered OFF
        }

        public bool MovePlayerWindow(string vmxFilePath, int left, int top)
        {
            int pid;
            IntPtr hWnd = IntPtr.Zero;
            IntPtr hWndChildAfter = IntPtr.Zero;

            if ((pid = getVMProcessID(vmxFilePath)) == 0)
            {
                return false;
            }
            if ((hWnd = getHwndFromPid(pid, "VMPlayerFrame")) == IntPtr.Zero)
            {
                return false;
            }
            if (!moveWindow(hWnd, left, top))              // Windowを移動する
            {
                return false;
            }
            return true;
        }

        public bool GetPlayerWindowPos(string vmxFilePath, out int left, out int top)
        {
            int pid;
            IntPtr hWnd = IntPtr.Zero;
            IntPtr hWndChildAfter = IntPtr.Zero;

            left = 0; top = 0;

            if ((pid = getVMProcessID(vmxFilePath)) == 0)
            {
                return false;
            }
            if ((hWnd = getHwndFromPid(pid, "VMPlayerFrame")) == IntPtr.Zero)
            {
                return false;
            }
            if (!getWindowPos(hWnd, out left, out top))              // Windowの top, leftを取得する
            {
                return false;
            }

            return true;
        }

        private int getVMProcessID(string vmxFilePath)
        {
            string vmxName = Path.GetFileNameWithoutExtension(vmxFilePath);  // VM 名を作成

            ManagementPath mngmntPath = new ManagementPath("Win32_Process");
            ManagementClass mngmntClass = new ManagementClass(mngmntPath);

            ManagementObjectCollection mngmntObjClctn = mngmntClass.GetInstances();
            foreach (ManagementObject obj in mngmntObjClctn)
            {
                object commandLineObj = obj["CommandLine"];
                if (commandLineObj != null)
                {
                    string commandLine = commandLineObj.ToString();
                    if (0 <= commandLine.IndexOf("vmplayer"))
                    {
                        if (0 <= commandLine.IndexOf(vmxName))
                        {
                            return int.Parse(obj["ProcessId"].ToString());
                        }
                    }
                }
            }
            return 0;
        }

        [StructLayout(LayoutKind.Sequential, Pack = 4)]
        public struct RECT
        {
            public int left { get; set; }
            public int top { get; set; }
            public int right { get; set; }
            public int bottom { get; set; }
        }

        [StructLayout(LayoutKind.Sequential, Pack = 4)]
        public struct POINT
        {
            public int x { get; set; }
            public int y { get; set; }
        }

        [StructLayout(LayoutKind.Sequential, Pack = 4)]
        public struct WINDOWPLACEMENT
        {
            public UInt32 length { get; set; }
            public UInt32 flags { get; set; }
            public UInt32 showCmd { get; set; }
            public POINT ptMinPosition { get; set; }
            public POINT ptMaxPosition { get; set; }
            public RECT rcNormalPosition { get; set; }
        }

internal class NativeMethods
        {
            [DllImport("user32.dll", EntryPoint = "GetParent", SetLastError = true)]
            public static extern IntPtr GetParent(IntPtr hwnd);
            [DllImport("user32.dll", EntryPoint = "GetWindow", SetLastError = true)]
            public static extern IntPtr GetWindow(IntPtr hWnd, int wCmd);
            [DllImport("user32.dll", EntryPoint = "FindWindow", SetLastError = true, BestFitMapping = false, ThrowOnUnmappableChar = true)]
            public static extern IntPtr FindWindow(String lpClassName, String lpWindowName);
            [DllImport("user32.dll", EntryPoint = "GetWindowThreadProcessId", SetLastError = true)]
            public static extern Int32 GetWindowThreadProcessId(IntPtr hWnd, out int lpdwprocessid);
            [DllImport("user32.dll", EntryPoint = "IsWindowVisible", SetLastError = true)]
            public static extern int IsWindowVisible(IntPtr hWnd);
            //[DllImport("user32.dll", EntryPoint = "GetAncestor", SetLastError = true)]
            //public static extern IntPtr GetAncestor(IntPtr hWnd, uint gaFlags);
            //[DllImport("user32.dll", EntryPoint = "GetWindowRect", SetLastError = true)]
            //public static extern bool GetWindowRect(IntPtr hWnd, out RECT lpRect);
            [DllImport("user32.dll", EntryPoint = "GetWindowPlacement", SetLastError = true)]
            public static extern bool GetWindowPlacement(IntPtr hWnd, out WINDOWPLACEMENT lpwndpl);
            [DllImport("user32.dll", EntryPoint = "SetWindowPlacement", SetLastError = true)]
            public static extern bool SetWindowPlacement(IntPtr hWnd, ref WINDOWPLACEMENT lpwndpl);
            //[DllImport("user32.dll", EntryPoint = "MoveWindow", SetLastError = true)]
            //public static extern bool MoveWindow(IntPtr hWnd, int x, int y, int nWidth, int nHeight, int bRepaint);
            //[DllImport("user32.dll", EntryPoint = "SendMessage", SetLastError = true, BestFitMapping = false, ThrowOnUnmappableChar = true)]
            //public static extern IntPtr SendMessage(IntPtr hWnd, UInt32 Msg, IntPtr wParam, StringBuilder lParam);
            //[DllImport("user32.dll", EntryPoint = "GetWindowText", SetLastError = true, BestFitMapping = false, ThrowOnUnmappableChar = true)]
            //public static extern int GetWindowText(IntPtr hWnd, StringBuilder lpString, int nMaxCount);
            //[DllImport("user32.dll", EntryPoint = "GetWindowTextLength", SetLastError = true)]
            //public static extern int GetWindowTextLength(IntPtr hWnd);
            [DllImport("user32.dll", EntryPoint = "GetClassName", SetLastError = true, BestFitMapping = false, ThrowOnUnmappableChar = true)]
            public static extern int GetClassName(IntPtr hWnd, StringBuilder lpClassName, int nMaxCount);
            //[DllImport("user32.dll", EntryPoint = "FindWindowEx", SetLastError = true, BestFitMapping = false, ThrowOnUnmappableChar = true)]
            //public static extern IntPtr FindWindowEx(IntPtr hWndParent, IntPtr hWndChildAfter, String lpszClass, String lpszWindow);
        }

        // 指定したhWndのWindowを移動させる
        private bool moveWindow(IntPtr hWnd, int left, int top)
        {
            RECT rect = new RECT();
            const int SW_SHOW = 5;          // for ShowWindowAsync()
            WINDOWPLACEMENT lpwndpl = new WINDOWPLACEMENT();
            lpwndpl.length = (UInt32)Marshal.SizeOf(lpwndpl);

            if (!NativeMethods.GetWindowPlacement(hWnd, out lpwndpl))
            {
                return false;
            }
            rect.left = left;
            rect.top = top;
            rect.right = lpwndpl.rcNormalPosition.right - (lpwndpl.rcNormalPosition.left - left);
            rect.bottom = lpwndpl.rcNormalPosition.bottom - (lpwndpl.rcNormalPosition.top - top);
            lpwndpl.rcNormalPosition = rect;
            lpwndpl.showCmd = SW_SHOW;
            if (!NativeMethods.SetWindowPlacement(hWnd, ref lpwndpl))
            {
                return false;
            }
            System.Threading.Thread.Sleep(1000);
            if (!NativeMethods.GetWindowPlacement(hWnd, out lpwndpl))
            {
                return false;
            }
            if(lpwndpl.rcNormalPosition.left != left || lpwndpl.rcNormalPosition.top != top)
            {
                return false;
            }
            return true;
        }

        // 指定したPIDのWindowのleft topを取得する
        private bool getWindowPos(IntPtr hWnd, out int left, out int top)
        {
            WINDOWPLACEMENT lpwndpl = new WINDOWPLACEMENT();
            lpwndpl.length = (UInt32)Marshal.SizeOf(lpwndpl);

            left = 0; top = 0;

            if (!NativeMethods.GetWindowPlacement(hWnd, out lpwndpl))
            {
                return false;
            }
            left = lpwndpl.rcNormalPosition.left; top = lpwndpl.rcNormalPosition.top;
            return true;
        }

        // プロセスID(pid)をウィンドウハンドル(hWnd)に変換する
        private IntPtr getHwndFromPid(int pid, string classname)
        {
            IntPtr hWnd;
            const int GW_HWNDNEXT = 2;    // for GetWindow()
            // GetAncestor()
            // const uint GA_PARENT = 1;
            // const uint GA_ROOT = 2;
            // const uint GA_ROOTOWNER = 3;
            // IntPtr parent = IntPtr.Zero, root = IntPtr.Zero, rootowner=IntPtr.Zero;
            StringBuilder csb = new StringBuilder(256);

            hWnd = NativeMethods.FindWindow(null, null);
            while (hWnd != IntPtr.Zero)
            {
                if (NativeMethods.GetParent(hWnd) == IntPtr.Zero && NativeMethods.IsWindowVisible(hWnd) != 0)
                {
                    if (pid == getPidFromHwnd(hWnd))
                    {
                        NativeMethods.GetClassName(hWnd, csb, csb.Capacity);
                        if (0 <= csb.ToString().IndexOf(classname))
                        {
#if DEBUG

                            Debug.WriteLine("ClassName = " + csb.ToString());
#endif
                            return hWnd;
                        }
                        //return rootowner;    // 現在のではなくてひとつ前のroot ownerを返す
                    }
                }
                // parent = GetAncestor(hWnd, GA_PARENT);
                // root = GetAncestor(hWnd, GA_ROOT);
                // rootowner = GetAncestor(hWnd, GA_ROOTOWNER);
                hWnd = NativeMethods.GetWindow(hWnd, GW_HWNDNEXT);
            }
            return IntPtr.Zero;
        }

        // ウィンドウハンドル(hWnd)をプロセスID(pid)に変換する
        private int getPidFromHwnd(IntPtr hWnd)
        {
            int pid;

            NativeMethods.GetWindowThreadProcessId(hWnd, out pid);
            return pid;
        }
    }
}
