﻿/*@@@*
The MIT License (MIT)

Copyright (c) 2013 GMA0BN

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
**@@@*/
#include "stdafx.h"
#include "sockutil.h"
#include "sockutil_sv.h"

static SOCKET g_aListenSocket[MAXLISTEN];

static ServerCB g_aSessionCB[MAXACCEPT];
static int g_nLastIdxSessionCB;

static unsigned g_uSessionID;

/// サーバ機能を初期化します。
void init_sockutil_sv(void)
{
    int i;

    for (i = 0; i < MAXLISTEN; ++i) {
        g_aListenSocket[i] = INVALID_SOCKET;
    }

    memset(g_aSessionCB, 0, sizeof(g_aSessionCB));
    init_session_res(g_aSessionCB, sizeof(g_aSessionCB[0]), MAXACCEPT);
    g_nLastIdxSessionCB = 0;

    g_uSessionID = 0;
}

/// サーバ機能を後始末します。
void term_sockutil_sv(void)
{
    int i;

    release_session_res(g_aSessionCB, sizeof(g_aSessionCB[0]), MAXACCEPT, TRUE);

    for (i = 0; i < MAXLISTEN; ++i) {
        if (g_aListenSocket[i] != INVALID_SOCKET) {
            closesocket(g_aListenSocket[i]);
            g_aListenSocket[i] = INVALID_SOCKET;
        }
    }
}

/// 空いているCBを取得します。
static int get_new_cb(void)
{
    int nRet;

    nRet = get_new_session_cb(g_aSessionCB, sizeof(g_aSessionCB[0]), MAXACCEPT, g_nLastIdxSessionCB);
    if (nRet >= 0) g_nLastIdxSessionCB = nRet;
    return nRet;
}

/// ユーザー関数のlauncher。
static unsigned __stdcall SessionThreadLauncher(void* exinf)
{
    ServerCB* p;
    unsigned uRet;

    p = (ServerCB*)exinf;
    uRet = p->startAddress(exinf);
    closesocket(p->res.socket);
    p->res.socket = INVALID_SOCKET;
#if USE_BARRIER
    _WriteBarrier();
#endif
    /*= ソケットのcloseは、接続断を通信相手に知らせる意味もあるので
     *  ユーザー関数終了直後に速やかに行う必要がある。
     *  スレッドオブジェクトの開放は、release_cb()でスレッド終了確認後に行う。
     */

    return uRet;
}

/// CBが握っている資源を解放します。
static BOOL release_cb(BOOL bFull)
{
    return release_session_res(g_aSessionCB, sizeof(g_aSessionCB[0]), MAXACCEPT, bFull);
}

/// 指定したサービスのlistenを開始します。
int start_listen(PCSTR pServiceName, SOCKET s[], int n)
{
    int i;
    struct addrinfo hints;
    struct addrinfo* pRes0;
    struct addrinfo* pRes;
    int nErr, nRet;
    int scnt;
    DWORD optv_ON;

    // 資源開放
    if (!release_cb(TRUE)) {
        return -3;
    }
    scnt = 0;

    // Listenソケット列挙
    memset(&hints, 0, sizeof(hints));
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_flags = AI_PASSIVE;
#if USE_CANONNAME
    //hints.ai_flags |= AI_CANONNAME;
#endif
#if USE_NUMERICHOST
    hints.ai_flags |= AI_NUMERICHOST;
#endif
    hints.ai_family = AF_UNSPEC;
    nErr = getaddrinfo(NULL, pServiceName, &hints, &pRes0);
    if (nErr != 0) {
        fprintf(stderr, "\nERROR: " __FUNCTION__ ": %s\n", gai_strerrorA(nErr));
        pRes0 = NULL;
        goto on_error;
    }

    for (pRes = pRes0; pRes && scnt < MAXLISTEN; pRes = pRes->ai_next) {
#if VERBOSE
        // 情報表示
        printf("addrinfo[%d]:\n", scnt);
        dump_addrinfo(stdout, pRes, "  ");
#endif

        s[scnt] = socket(pRes->ai_family, pRes->ai_socktype, pRes->ai_protocol);
        if (s[scnt] == INVALID_SOCKET) {
            int err = WSAGetLastError();
            fprintf(stderr, "\nWarning: " __FUNCTION__ ": socket() failed. (err=%d)\n", err);
            continue;
        }
#ifdef IPV6_V6ONLY
        if (pRes->ai_family == AF_INET6) {
            nRet = setsockopt(s[scnt], IPPROTO_IPV6, IPV6_V6ONLY, (const char*)&optv_ON, sizeof(optv_ON));
            if (nRet == SOCKET_ERROR) {
                int err = WSAGetLastError();
                fprintf(stderr, "\nERROR: " __FUNCTION__ ": setsockopt() failed. (err=%d)\n", err);
                goto on_error;
            }
        }
#endif

        nRet = bind(s[scnt], pRes->ai_addr, pRes->ai_addrlen);
        if (nRet == SOCKET_ERROR) {
            int err = WSAGetLastError();
            fprintf(stderr, "\nERROR: " __FUNCTION__ ": bind() failed. (err=%d)\n", err);
            goto on_error;
        }

        nRet = listen(s[scnt], SOMAXCONN);
        if (nRet == SOCKET_ERROR) {
            int err = WSAGetLastError();
            fprintf(stderr, "\nERROR: " __FUNCTION__ ": listen() failed. (err=%d)\n", err);
            goto on_error;
        }

        ++scnt;
    }

    freeaddrinfo(pRes0);
    return scnt;

on_error:
    if (pRes0 != NULL) {
        freeaddrinfo(pRes0);
    }
    for (i = 0; i < scnt; ++i) {
        closesocket(s[i]);
        s[i] = INVALID_SOCKET;
    }
    return -1;
}

/// 接続を待ち受け、接続されればセッションを開始します。
int wait_connection(
    const SOCKET s[],
    int n,
    DWORD TCTmoIn_ms,
    unsigned (__stdcall *startAddress)(void* exinf)
    )
{
    int i, idx;
    fd_set rfd;
    struct timeval tmv;
    int nRet, nErr;
    SOCKET cnSock;
    SOCKET sockMax;
    struct sockaddr_storage from;
    socklen_t fromLen;
    ServerCB* p;
    uintptr_t upRet;

    if (n <= 0) {
        fprintf(stderr, "WARNING: " __FUNCTION__ ": No socket to listen to.\n");
        return -1;
    }

    // 完了済みセッションの資源を解放
    if (!release_cb(FALSE)) {
        return -3;
    }

    // 最大のfd番号を求める(一応)
    sockMax = s[0];
    for (i = 1; i < n; ++i) {
        if (s[i] > sockMax) {
            sockMax = s[i];
        }
    }

    // fd_set構成
    FD_ZERO(&rfd);
    for (i = 0; i < n; ++i) {
#pragma warning(push)
#pragma warning(disable: 4127)
        FD_SET(s[i], &rfd);
#pragma warning(pop)
    }

    // タイムアウト時定数構成
    tmv.tv_sec  = (TCTmoIn_ms / 1000);
    tmv.tv_usec = (TCTmoIn_ms % 1000) * 1000;

    // 受信待ち
    nRet = select(sockMax + 1, &rfd, NULL, NULL, &tmv);     // Winsockのselect()では第1引数は無意味
    if (nRet == SOCKET_ERROR) {
        int err = WSAGetLastError();
        fprintf(stderr, "\nERROR: " __FUNCTION__ ": select() failed. (err=%d)\n", err);
        return -2;
    } else if (nRet == 0) {
        return 0;              // タイムアウト
    }

    // 処理
    for (i = 0; i < n; ++i) {
        if (FD_ISSET(s[i], &rfd)) {
            // コネクション受理
            fromLen = sizeof(from);
            cnSock = accept(s[i], (struct sockaddr*)&from, &fromLen);
            if (cnSock == INVALID_SOCKET) {
                int err = WSAGetLastError();
                if (err == WSAECONNRESET) {
                    fprintf(stderr, "\nWARNING: " __FUNCTION__ ": accept() failed. (WSAECONNRESET)\n");
                    continue;
                } else {
                    fprintf(stderr, "\nWARNING: " __FUNCTION__ ": accept() failed. (err=%d)\n", err);
                    return -2;
                }
            }

            // CB決定
            idx = get_new_cb();
            if (idx < 0) {
                fprintf(stderr, "\nWARNING: " __FUNCTION__ ": Too many clients.\n");
                closesocket(cnSock);
                continue;
            }

            // CB構築
            p = &(g_aSessionCB[idx]);
            assert(p->res.socket == INVALID_SOCKET);
            assert(p->res.hThread == NULL);
            p->res.socket = cnSock;
            nErr = getnameinfo((struct sockaddr*)&from, fromLen, p->hostName, sizeof(p->hostName), NULL, 0, NI_NUMERICHOST);
            if (nErr != 0) {
                int err = WSAGetLastError();
                fprintf(stderr, "\nERROR: " __FUNCTION__ ": getnameinfo() failed. (err=%d)\n", err);
                goto on_error_after_accept;
            }
            memcpy(&(p->from), &from, fromLen);
            p->sessionID = g_uSessionID++;

            // セッションのスレッド起動
            p->startAddress = startAddress;
            upRet = _beginthreadex(NULL, THREAD_STKSZ, SessionThreadLauncher, p, 0, NULL);
            if (upRet == 0) {
                perror(__FUNCTION__);
                goto on_error_after_accept;
            } else {
                p->res.hThread = (HANDLE)upRet;     // hThread更新
            }

            ++g_nLastIdxSessionCB;      // 成功したらインクリメント
        }
    }

    return 1;

on_error_after_accept:
#if USE_BARRIER
    _ReadBarrier();
#endif
    if (p->res.socket != INVALID_SOCKET) {
        closesocket(p->res.socket);
        p->res.socket = INVALID_SOCKET;
    }
    assert(p->res.hThread == NULL);
    return -2;
}

/// 全セッションの完了を待ちます。
BOOL wait_completion_all()
{
    return release_session_res(g_aSessionCB, sizeof(g_aSessionCB[0]), MAXACCEPT, TRUE);
}
