package jp.operation.structure.trie;

import java.util.Arrays;
import java.util.Collection;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BasicTrieTree implements Trie {

	private static Logger log = LoggerFactory.getLogger(BasicTrieTree.class);

	private TrieNode root;

	private int wordnum = 0;

	public BasicTrieTree() {
		this.root = new TrieNode(' ');
	}

	@Override
	public void add(Word word) {
		addWord(this.root, word);
	}

	private synchronized void addWord(TrieNode root, Word w) {

		if (w == null)
			return;

		TrieNode current = root;
		String word = w.getWord();

		for (int i = 0; i < word.length(); i++) {
			TrieNode next = current.findChild(word.charAt(i));
			if (next == null) {
				TrieNode node = new TrieNode(word.charAt(i));
				node.setParent(current);
				current.insertChild(node);
				current = node;
			} else
				current = next;
		}

		if (current.getWord() == null)
			current.setWord(w);

		current.setTerminate(true);
		wordnum++;
	}

	@Override
	public void build(Words words) {
		TrieNode root = new TrieNode(' ');
		for (Word word : words) {
			if (word != null)
				addWord(root, word);
		}
		this.root = root;
	}

	@Override
	public void build(Collection<String> wordList) {
		TrieNode root = new TrieNode(' ');
		int i = 0;
		for (String word : wordList) {
			if (word != null) {
				Word _word = new Word(word, i++);
				addWord(root, _word);
			}
		}
		this.root = root;
	}

	@Override
	public boolean contains(String word) {
		TrieNode current = this.root;
		for (char c : word.toCharArray()) {
			TrieNode next = current.findChild(c);
			if (next == null) {
				return false;
			}
			current = next;
		}
		return current.hasTerminate();
	}

	@Override
	public void extractWords(Collection<Word> extracts, String document, boolean isDuplicatable) {

		if (document == null)
			return;

		TrieNode root = this.root;
		TrieNode current = root;
		int start = 0;
		Word extract = null;

		for (int pos = 0; pos < document.length(); pos++) {
			TrieNode next = current.findChild(document.charAt(pos));
			if (next == null) {
				if (extract == null)
					pos = start;
				else {
					pos = start + extract.getWord().length() - 1;
					if (isDuplicatable || !extracts.contains(extract)) {
						try {
							Word trieWord = extract.clone();
							extracts.add(trieWord);
						} catch (CloneNotSupportedException e) {
							log.warn(e.toString());
						}
					}
				}

				current = root;
				extract = null;
				start = pos + 1;
				continue;

			} else {
				current = next;
				if (current.getWord() != null) {
					try {
						extract = current.getWord().clone();
					} catch (Exception e) {
						log.warn(e.toString());
					}
				}
			}
		}
		if (current.hasTerminate()) {
			try {
				Word word = current.getWord().clone();
				extracts.add(word);
			} catch (CloneNotSupportedException e) {
				log.warn(e.toString());
			}

		}

	}

	@Override
	public void extractStringWords(Collection<String> extracts, String document, boolean isDuplicatable) {

		if (document == null)
			return;

		TrieNode root = this.root;
		TrieNode current = root;
		int start = 0;
		Word extract = null;

		for (int pos = 0; pos < document.length(); pos++) {
			TrieNode next = current.findChild(document.charAt(pos));
			if (next == null) {
				if (extract == null)
					pos = start;
				else {
					pos = start + extract.getWord().length() - 1;
					if (isDuplicatable || !extracts.contains(extract))
						extracts.add(extract.getWord().trim());
				}

				current = root;
				extract = null;
				start = pos + 1;
				continue;

			} else {
				current = next;
				if (current.getWord() != null) {
					try {
						extract = current.getWord().clone();
					} catch (Exception e) {
						log.warn(e.toString());
					}
				}
			}
		}
		if (current.hasTerminate()) {
			Word word = current.getWord();
			extracts.add(word.getWord().trim());
		}

	}

	@Override
	public void extractIntWords(Collection<Integer> extracts, String document,
			boolean isDuplicatable) {

		if (document == null)
			return;

		TrieNode root = this.root;
		TrieNode current = root;
		int start = 0;
		Word extract = null;

		for (int pos = 0; pos < document.length(); pos++) {
			TrieNode next = current.findChild(document.charAt(pos));
			if (next == null) {
				if (extract == null)
					pos = start;
				else {
					pos = start + extract.getWord().length() - 1;
					if (isDuplicatable || !extracts.contains(extract))
						extracts.add(extract.getId());
				}

				current = root;
				extract = null;
				start = pos + 1;
				continue;

			} else {
				current = next;
				if (current.getWord() != null) {
					try {
						extract = current.getWord().clone();
					} catch (Exception e) {
						log.warn(e.toString());
					}
				}
			}
		}
		if (current.hasTerminate()) {
			extracts.add(current.getWord().getId());
		}

	}

	@Override
	public void remove(String word) {
		throw new UnsupportedOperationException();
	}

	@Override
	public int size() {
		return wordnum;
	}

	/**
	 * TrieNode
	 */
	static class TrieNode implements Comparable<TrieNode> {

		/** 親ノード */
		private TrieNode _parent;
		/** 現在位置の文字 */
		private char _value;
		/** 子ノードの文字配列 */
		private char[] _children = new char[0];
		/** 子ノード配列 */
		private TrieNode[] _childNodes = new TrieNode[0];
		/** 単語 */
		private Word _word;
		/** このノードで文字の終端となる単語があるかどうかのフラグ */
		private boolean terminate = false;

		/**
		 * TrieNodeを初期化します。
		 */
		public TrieNode(char value) {
			_value = value;
		}

		/**
		 * 親ノードを設定します。
		 * 
		 * @param parent
		 */
		public void setParent(TrieNode parent) {
			_parent = parent;
		}

		public TrieNode getParent() {
			return _parent;
		}

		/**
		 * ルートノードであるかチェックします。
		 * 
		 * @return
		 */
		public boolean isRoot() {
			if (_parent == null) {
				return true;
			} else {
				return false;
			}
		}

		/**
		 * 子ノードを持っているかチェックします。
		 * 
		 * @return
		 */
		public boolean hasChildren() {
			if (_children.length == 0) {
				return false;
			} else {
				return true;
			}
		}

		/**
		 * 子ノードを挿入し、内部配列をソートしなおします。
		 * 
		 * @param node
		 */
		public synchronized void insertChild(TrieNode node) {
			/* 新配列を生成 */
			char[] children = new char[_children.length + 1];
			TrieNode[] childNodes = new TrieNode[_childNodes.length + 1];

			/* 新配列にコピー */
			System.arraycopy(_children, 0, children, 0, _children.length);
			System.arraycopy(_childNodes, 0, childNodes, 0, _childNodes.length);

			/* 配列にノードを追加 */
			children[_children.length] = node.getValue();
			childNodes[_childNodes.length] = node;

			/* 新配列をソート */
			Arrays.sort(children);
			Arrays.sort(childNodes);

			/* 差し替え */
			_children = children;
			_childNodes = childNodes;
		}

		/**
		 * 子ノードを全て取得します。
		 * 
		 * @return node
		 */
		public TrieNode[] getChildren() {
			return _childNodes;
		}

		/**
		 * 次に接続する文字を指定して、子ノードを検索します。
		 * 
		 * @param next
		 * @return
		 */
		public TrieNode findChild(char next) {
			char[] children = _children;
			TrieNode[] nodes = _childNodes;

			int index = Arrays.binarySearch(children, next);
			if (index > -1) {
				return nodes[index];
			} else {
				return null;
			}
		}

		/**
		 * このノードの文字を取得します。
		 * 
		 * @return
		 */
		public char getValue() {
			return _value;
		}

		/**
		 * このノードに単語を設定します。
		 * 
		 * @param word
		 */
		public void setWord(Word word) {
			_word = word;
		}

		/**
		 * このノードの単語を取得します。
		 * 
		 * @return
		 */
		public Word getWord() {
			return _word;
		}

		/**
		 * このノードで文字の終端となる単語があるかどうかの判定。trueの場合、文字の終端となる単語が存在します。
		 * 
		 * @return
		 */
		public boolean hasTerminate() {
			return this.terminate;
		}

		/**
		 * このノードで文字の終端となる単語があるかどうかのフラグを設定
		 * 
		 * @param terminate
		 */
		public void setTerminate(boolean terminate) {
			this.terminate = terminate;
		}

		/**
		 * 別のTrieNodeとの比較を行います。
		 * 
		 * @param another
		 * @return
		 */
		public int compareTo(TrieNode another) {
			return _value - another.getValue();
		}

		/**
	     *
	     */
		public String toString() {
			StringBuffer buffer = new StringBuffer(30);
			buffer.append("VALUE:[");
			buffer.append(_value);
			buffer.append("] WORD:[");
			buffer.append(_word);
			buffer.append("]");
			return buffer.toString();
		}

	}

}
