/*
 * JHPdf Free PDF Library : HPdfType1FontDef.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.pdfobject.font;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.jhpdf.ByteUtil;
import net.sf.jhpdf.HPdfConst;
import net.sf.jhpdf.HPdfErrorCode;
import net.sf.jhpdf.HPdfException;
import net.sf.jhpdf.HPdfUtil;
import net.sf.jhpdf.encoder.HPdfEncoder;
import net.sf.jhpdf.graphics.HPdfRect;
import net.sf.jhpdf.io.HPdfMemStream;
import net.sf.jhpdf.io.HPdfReadStream;

/**
 * Class represents PDF Type1 Font definition.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public class HPdfType1FontDef extends HPdfFontDef {

    private static final Logger logger = LoggerFactory.getLogger(HPdfType1FontDef.class);
    
    public HPdfType1FontDef() {
        super();
        logger.trace("HPdfType1FontDef#ctor");
        
        this.setFontDefType(HPdfFontDefType.TYPE1);
        
        this.setFlag(HPDF_FONT_STD_CHARSET);
        
    }
    
    public static HPdfType1FontDef createBase14FontDef(String fontName) {
        // MEMO original code is 'HPDF_Base14FontDef_New'
        // in hpdf_fontdef_base14.c
        
        HPdfBase14FontDefData data =
                HPdfBase14FontDefData.findBuiltinData(fontName);
        if (data == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_FONT_NAME, 0);
        }
        
        HPdfType1FontDef fontDef = new HPdfType1FontDef();
        fontDef.setBaseFont(data.fontName);
        fontDef.fontDefAttr.setBase14Font(true);
        
        if (data.isFontSpecific) {
            fontDef.fontDefAttr.setEncodingScheme(
                    HPdfEncoder.HPDF_ENCODING_FONT_SPECIFIC);
        }
        fontDef.setWidths(data.widthsTable);
        fontDef.setFontBbox(data.bBox);
        fontDef.setAscent(data.ascent);
        fontDef.setDescent(data.descent);
        fontDef.setXHeight(data.xHeight);
        fontDef.setCapHeight(data.capHeight);
        
        fontDef.setValid(true);
        
        return fontDef;
    }
    
    public static HPdfType1FontDef load(HPdfReadStream afm, HPdfReadStream fontData) {
        logger.trace("HPdfType1FontDef.load");
        
        if (afm == null) {
            return null;
        }
        
        HPdfType1FontDef fontDef = new HPdfType1FontDef();
        
        fontDef.loadAfm(afm);
        
        /* if font-data is specified, the font data is embeded */
        if (fontData != null) {
            fontDef.loadFontData(fontData);
        }
        
        return fontDef;
    }
    
    public HPdfType1FontDef duplicate() {
        HPdfType1FontDef fontDef = new HPdfType1FontDef();
        logger.trace("HPdfType1FontDef#duplicate");
        
        fontDef.setFontDefType(this.getFontDefType());
        fontDef.setValid(this.isValid());
        
        /* copy data of attr,widths
         attention to charset */
        
        // TODO original code is all over.
        // maybe unimplemented? (should throw Exception?)
        
        return null;
    }
    
    public void setWidths(HPdfCharData[] src) {
        logger.trace("HPdfType1FontDef#setWidths");
        int count = 0;
        for (final HPdfCharData d : src) {
            if (d.getUnicode() == 0xFFFF) {
                break;
            }
            ++count;
        }
        
        HPdfCharData[] dst = new HPdfCharData[count];
        fontDefAttr.setWidths(dst);
        for (int i = 0; i < count; ++i) {
            dst[i] = src[i].clone();
            if (dst[i].getUnicode() == 0x0020) {
                this.setMissingWidth(src[i].getWidth());
            }
        }
    }
    
    public int getWidthByName(String glyphName) {
        char unicode = HPdfEncoder.glyphNameToUnicode(glyphName);
        logger.trace("HPdfType1FontDef#getWidthByName");
        
        return this.getWidth(unicode);
    }

    @Override
    public void init() {
        // nothing to do.
    }

    @Override
    public void clean() {
        // nothing to do.
    }

    @Override
    public void dispose() {
        fontDefAttr.setCharSet(null);
        fontDefAttr.disposeFontData();
        fontDefAttr.setWidths(null);
    }

    @Override
    public int getWidth(char code) {
        logger.trace("HPdfType1FontDef#getWidthByName");
        for (final HPdfCharData d : fontDefAttr.getWidths()) {
            if (d.getCharCd() == code) {
                return d.getWidth();
            }
        }
        return this.getMissingWidth();
    }
    
    private void loadAfm(HPdfReadStream stream) {
        logger.trace("HPdfType1FontDef#loadAfm");
        
        Pattern p = Pattern.compile("[" + HPdfUtil.PDF_WHITE_SPACES + "]+");
        /* check AFM header */
        String buf = stream.readLn();
        Matcher m = p.matcher(buf);
        String keyword;
        if (m.find()) {
            keyword = buf.substring(0, m.start());
        } else {
            keyword = buf; // only a token
        }
        
        if (!keyword.equals("StartFontMetrics")) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_AFM_HEADER, 0);
        }
        
        /* Global Font Information */
        
        String remainder;
        for (;;) {
            buf = stream.readLn();
            m = p.matcher(buf);
            
            if (m.find()) {
                keyword = buf.substring(0, m.start());
                remainder = buf.substring(m.end(), buf.length());
            } else {
                keyword = buf; // only a token
                remainder = "";
            }
            if (keyword.equals("FontName")) {
                this.setBaseFont(remainder);
            } else if (keyword.equals("Weight")) {
                if (remainder.startsWith("Bold")) {
                    this.setFlag(HPDF_FONT_FOURCE_BOLD);
                }
            } else if (keyword.equals("IsFixedPitch")) {
                if (remainder.startsWith("true")) {
                    this.setFlag(HPDF_FONT_FIXED_WIDTH);
                }
            } else if (keyword.equals("ItalicAngle")) {
                this.setItalicAngle(Integer.parseInt(remainder));
                if (this.getItalicAngle() != 0) {
                    this.setFlag(HPDF_FONT_ITALIC);
                }
            } else if (keyword.equals("CharacterSet")) {
                if (remainder.length() > 0) {
                    fontDefAttr.setCharSet(remainder);
                }
            } else if (keyword.equals("FontBBox")) {
                String[] elems = remainder.split(p.pattern());
                this.setFontBbox(new HPdfRect(
                        Integer.parseInt(elems[0]),
                        Integer.parseInt(elems[1]),
                        Integer.parseInt(elems[2]),
                        Integer.parseInt(elems[3])));
            } else if (keyword.equals("EncodingScheme")) {
                fontDefAttr.setEncodingScheme(remainder);
            } else if (keyword.equals("CapHeight")) {
                this.setCapHeight(Integer.parseInt(remainder));
            } else if (keyword.equals("Ascender")) {
                this.setAscent(Integer.parseInt(remainder));
            } else if (keyword.equals("Descender")) {
                this.setDescent(Integer.parseInt(remainder));
            } else if (keyword.equals("STDHW")) {
                this.setStemh(Integer.parseInt(remainder));
            } else if (keyword.equals("STDHV")) {
                this.setStemv(Integer.parseInt(remainder));
            } else if (keyword.equals("StartCharMetrics")) {
                fontDefAttr.createWidths(Integer.parseInt(remainder));
                break;
            }
        }
        
        /* load CharMetrics */
        for (final HPdfCharData cdata : fontDefAttr.getWidths()) {
            buf = stream.readLn();
            String[] elems = remainder.split(p.pattern());
            
            /* C default character code. */
            if (elems[0].equals("CX")) {
                /* not suppoted yet. */
                throw new HPdfException(HPdfErrorCode.HPDF_INVALID_CHAR_MATRICS_DATA, 0);
            } else if (elems[0].equals("C")) {
                cdata.setCharCd(Integer.parseInt(elems[1]));
            } else {
                throw new HPdfException(HPdfErrorCode.HPDF_INVALID_CHAR_MATRICS_DATA, 0);
            }
            
            int i;
            /* WX Character width */
            for (i = 2; i < elems.length; ++i) {
                if (elems[i].equals("WX")) {
                    cdata.setWidth(Integer.parseInt(elems[i + 1]));
                    break;
                }
            }
            if (i >= elems.length) { // not found
                throw new HPdfException(HPdfErrorCode.HPDF_INVALID_WX_DATA, 0);
            }
            
            /* N PostScript language character name */
            for (; i < elems.length; ++i) {
                if (elems[i].equals("N")) {
                    cdata.setUnicode(
                            HPdfEncoder.glyphNameToUnicode(elems[i + 1])
                            );
                    break;
                }
            }
            if (i >= elems.length) { // not found
                throw new HPdfException(HPdfErrorCode.HPDF_INVALID_N_DATA, 0);
            }
        }
    }
    
    private static final String PFB_EEXEC_KEY = "eexec";
    
    private static final String PFB_CLEARTOMARK_KEY = "cleartomark";
    
    private final static int LOAD_FONTDATA_CACHE_SIZE = PFB_CLEARTOMARK_KEY.length();
    
    private void loadFontData(HPdfReadStream stream) {
        logger.trace("HPdfType1FontDef#loadFontData");
        
        fontDefAttr.createFontData(HPdfConst.HPDF_STREAM_BUF_SIZ);
        
        byte[] buf = new byte[HPdfConst.HPDF_STREAM_BUF_SIZ];
        
        stream.read(buf, LOAD_FONTDATA_CACHE_SIZE);
        
        boolean firstFlg = true;
        for (;;) {
            int len = stream.read(buf,
                    LOAD_FONTDATA_CACHE_SIZE, buf.length - LOAD_FONTDATA_CACHE_SIZE);
            if (len == -1) {
                // EOF
                break;
            } else if (len > 0) {
                if (fontDefAttr.getLength1() == 0) {
                    int pos = ByteUtil.indexOf(buf, PFB_EEXEC_KEY,
                            len + LOAD_FONTDATA_CACHE_SIZE);
                    
                    /* length1 indicate the size of ascii-data of font-file. */
                    if (pos >= 0) {
                        fontDefAttr.setLength1(
                                (int) fontDefAttr.getFontData().getSize() + pos
                                        + PFB_EEXEC_KEY.length() + 1);
                        // last +1 is 0x0A follows "eexec".
                    }
                }
                
                if (fontDefAttr.getLength1() > 0 && fontDefAttr.getLength2() == 0) {
                    int pos = ByteUtil.indexOf(buf, PFB_CLEARTOMARK_KEY,
                            len + LOAD_FONTDATA_CACHE_SIZE);
                    
                    if (pos >= 0) {
                        fontDefAttr.setLength2(
                                (int) fontDefAttr.getFontData().getSize() + -520
                                - fontDefAttr.getLength1() + pos);
                    }
                    // MEMO: original comments refers length1, but probably misspelled.
                    /*  length2 indicate the size of binary-data.
                     *  in most fonts, it is all right at 520 bytes . but it need
                     *  to modify because it does not fully satisfy the
                     *  specification of type-1 font.
                     */
                }
            }
            
            if (firstFlg) {
                fontDefAttr.getFontData().write(buf,
                        len + LOAD_FONTDATA_CACHE_SIZE);
                firstFlg = false;
            } else {
                fontDefAttr.getFontData().write(buf,
                        LOAD_FONTDATA_CACHE_SIZE, len);
            }
            // rotates cache.
            System.arraycopy(buf, len, buf, 0, LOAD_FONTDATA_CACHE_SIZE);
        }
        
        if (fontDefAttr.getLength1() == 0 || fontDefAttr.getLength2() == 0) {
            throw new HPdfException(HPdfErrorCode.HPDF_UNSUPPORTED_TYPE1_FONT, 0);
        }
        
        fontDefAttr.setLength3(
                (int) fontDefAttr.getFontData().getSize()
                - fontDefAttr.getLength1()
                - fontDefAttr.getLength2());
    }
    
    boolean isBase14Font() {
        return fontDefAttr.isBase14Font();
    }
    
    String getCharset() {
        return fontDefAttr.getCharSet();
    }
    
    HPdfMemStream getFontData() {
        return fontDefAttr.getFontData();
    }
    
    int getLength1() {
        return fontDefAttr.getLength1();
    }
    
    int getLength2() {
        return fontDefAttr.getLength2();
    }
    
    int getLength3() {
        return fontDefAttr.getLength3();
    }
    
    private HPdfType1FontDefAttr fontDefAttr = new HPdfType1FontDefAttr();
    
    private final class HPdfType1FontDefAttr {
        private char firstChar;
        
        private char getFirstChar() {
            return this.firstChar;
        }
        
        private void setFirstChar(char firstChar) {
            this.firstChar = firstChar;
        }
        
        private char lastChar;
        
        private char getLastChar() {
            return this.lastChar;
        }
        
        private void setLastChar(char lastChar) {
            this.lastChar = lastChar;
        }
        
        private HPdfCharData[] widths;
        
        private HPdfCharData[] getWidths() {
            return this.widths;
        }
        
        private void setWidths(HPdfCharData[] widths) {
            this.widths = widths;
        }
        
        private void createWidths(int size) {
            this.widths = new HPdfCharData[size];
            for (int i = 0; i < this.widths.length; ++i) {
                this.widths[i] = new HPdfCharData();
            }
        }
        
        private int getWidthsCount() {
            if (this.widths == null) {
                return 0;
            } else {
                return this.widths.length;
            }
        }
        
        private int leading;
        
        private int getLeading() {
            return this.leading;
        }
        
        private void setLeading(int leading) {
            this.leading = leading;
        }
        
        private String charSet;
        
        private String getCharSet() {
            return this.charSet;
        }
        
        private void setCharSet(String charSet) {
            this.charSet = charSet;
        }
        
        private String encodingScheme;
        
        private String getEncodingScheme() {
            return this.encodingScheme;
        }
        
        private void setEncodingScheme(String encodingScheme) {
            this.encodingScheme = encodingScheme;
        }
        
        private int length1;
        
        private int getLength1() {
            return this.length1;
        }
        
        private void setLength1(int length1) {
            this.length1 = length1;
        }
        
        private int length2;
        
        private int getLength2() {
            return this.length2;
        }
        
        private void setLength2(int length2) {
            this.length2 = length2;
        }
        
        private int length3;
        
        private int getLength3() {
            return this.length3;
        }
        
        private void setLength3(int length3) {
            this.length3 = length3;
        }
        
        private boolean flgBase14Font;
        
        private boolean isBase14Font() {
            return this.flgBase14Font;
        }
        
        private void setBase14Font(boolean flg) {
            this.flgBase14Font = flg;
        }
        
        private boolean flgFixedPitch;
        
        private boolean isFixedPitch() {
            return this.flgFixedPitch;
        }
        
        private void setFixedPitch(boolean flg) {
            this.flgFixedPitch = flg;
        }
        
        private HPdfMemStream fontData;
        
        private HPdfMemStream getFontData() {
            return this.fontData;
        }
        
        private void createFontData(int size) {
            this.fontData = new HPdfMemStream(size);
        }
        
        private void disposeFontData() {
            if (this.fontData != null) {
                this.fontData.close();
                this.fontData = null;
            }
        }
    }
    
    
}
