/*
 * JHPdf Free PDF Library : HPdfDict.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.pdfobject;

import java.util.Map;
import java.util.HashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.jhpdf.HPdfErrorCode;
import net.sf.jhpdf.HPdfException;
import net.sf.jhpdf.encrypt.HPdfEncrypt;
import net.sf.jhpdf.encrypt.HPdfEncryptDict;
import net.sf.jhpdf.io.HPdfWriteStream;

/**
 * Class represents PDF Dictionary.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public class HPdfDict extends HPdfObject {

    private static final Logger logger = LoggerFactory.getLogger(HPdfDict.class);
    
    private final Map<String, HPdfObject> dictBody = new HashMap<String, HPdfObject>();
    
    /**
     * ctor.
     */
    public HPdfDict() {
        super();
    }
    
    protected Map<String, HPdfObject> getDictBody() {
        return this.dictBody;
    }
    
    public void add(String key, HPdfObject obj) {
        
        if (obj == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_OBJECT, 0);
        }
        
        if (obj.isTypeDirect()) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_OBJECT, 0);
        }
        
        if (key == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_OBJECT, 0);
        }
        // MEMO: original code inspects internal list size here(HPDF_LIMIT_MAX_DICT_ELEMENT).
        // it should be class-private concerns, but may PDF specification?
        
        // MEMO: original code holds key-value entries in a list.
        // adding-order has to be holded?
        
        if (obj.isTypeIndirect()) {
            HPdfProxy proxy = new HPdfProxy(obj);
            this.dictBody.put(key, proxy);
            proxy.setObjType(HPdfObjectType.DIRECT);
        } else {
            this.dictBody.put(key, obj);
            obj.setObjType(HPdfObjectType.DIRECT);
        }
    }
    
    public void addName(String key, String value) {
        add(key, new HPdfName(value));
    }
    
    public void addNumber(String key, long value) {
        add(key, new HPdfNumber(value));
    }

    public void addReal(String key, float value) {
        add(key, new HPdfReal(value));
    }

    public void addBoolean(String key, boolean value) {
        add(key, new HPdfBoolean(value));
    }
    
    public <T extends HPdfObject> T getItem(String key, Class<T> clazz) {
        HPdfObject obj = getDictBody().get(key);
        if (obj != null) {
            if (obj instanceof HPdfProxy) {
                obj = ((HPdfProxy)obj).getContent();
            }
            
            if (!clazz.isInstance(obj)) {
                if (logger.isTraceEnabled()) {
                    logger.trace(String.format("HPdfDict#getItem dict=%X key=%s obj_class=%s",
                        this.hashCode(), key, clazz.getName()));
                }
                throw new HPdfException(HPdfErrorCode.HPDF_DICT_ITEM_UNEXPECTED_TYPE, 0);
            }
            
            return castItem(obj);
        } else {
            return null;
        }
    }
    
    @SuppressWarnings("unchecked")
    private <T extends HPdfObject> T castItem(HPdfObject obj) {
        return (T)obj;
    }
    
    public void removeElement(String key) {
        HPdfObject removed = this.dictBody.remove(key);
        if (removed == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_DICT_ITEM_NOT_FOUND, 0);
        }
    }
    
    @Override
    protected void writeValue(HPdfWriteStream stream, HPdfEncrypt e) {
        
        stream.writeStr("<<\012");
        
        this.beforeWrite(stream, e);
        
        /* encrypt-dict must not be encrypted. */
        if (this instanceof HPdfEncryptDict) {
            e = null;
        }
        
        // MEMO: if (dict->stream) {} proc written in HPdfStreamDict#beforeWrite
        
        for (final Map.Entry<String, HPdfObject> element : this.dictBody.entrySet()) {
            HPdfObject obj = element.getValue();
            if (obj == null) {
                throw new HPdfException(HPdfErrorCode.HPDF_INVALID_OBJECT, 0);
            }
            if (obj.isTypeHidden()) {
                if (logger.isTraceEnabled()) {
                    logger.trace(String.format("HPdfDict#write obj=%X skipped obj_id=0x%08X",
                        obj.hashCode(), obj.getFullObjId()));
                }
            } else {
                stream.writeEscapeName(element.getKey());
                stream.writeChar(' ');
                obj.write(stream, e);
                stream.writeStr("\012");
            }
        }
        
        this.onWrite(stream);
        
        stream.writeStr(">>");
        
        // MEMO: if (dict->stream) {} proc written in HPdfStreamDictr#afterWrite
        this.afterWrite(stream, e);
    }
    
    protected void beforeWrite(HPdfWriteStream stream, HPdfEncrypt e) {
        // default, nothing to do.
    }
    
    /**
     * performs class-specific writing operation.<br>
     * this method corresponds 'write_fn' member in original code.
     * @param stream stream for output this object.
     */
    protected void onWrite(HPdfWriteStream stream) {
        // default, nothing to do.
    }
    
    protected void afterWrite(HPdfWriteStream stream, HPdfEncrypt e) {
        // default, nothing to do.
    }
}
