/*
 * JHPdf Free PDF Library : HPdfFileReadStream.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.io;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;

import net.sf.jhpdf.HPdfErrorCode;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Stream class for reading from a file.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public class HPdfFileReadStream extends HPdfAbstractReadStream {
    // FIXME: buffering not implemented, so performance should be inefficient probably.

    private static final Logger logger = LoggerFactory.getLogger(HPdfFileReadStream.class);
    
    private Path sourcePath;
    
    private FileChannel sourceChannel;

    /**
     * constructor.
     * @param filePath path of file to be read
     */
    public HPdfFileReadStream(String filePath) {
        super();
        logger.trace("HPdfFileReadStream#ctor.");
        
        this.sourcePath = Paths.get(filePath);
        this.sourceChannel = null;
    }
    
    private FileChannel getSourceChannel() {
        logger.trace("HPdfFileReadStream#getSourceChannel.");
        if (this.sourceChannel == null) {
            try {
                logger.trace("  opens channel for {}", this.sourcePath);
                this.sourceChannel = FileChannel.open(this.sourcePath, StandardOpenOption.READ);
            } catch (IOException e) {
                throw new HPdfIoException(HPdfErrorCode.HPDF_FILE_OPEN_ERROR, 0, e);
            }
        }
        return this.sourceChannel;
    }
    
    private ByteBuffer bufForSingleRead = ByteBuffer.allocate(1);

    @Override
    protected int readByte() {
        logger.trace("HPdfFileReadStream#readByte.");
        
        int readBytes;
        try {
            bufForSingleRead.rewind();
            readBytes = this.getSourceChannel().read(bufForSingleRead);
            if (readBytes < 0) {
                return readBytes;
            } else {
                return 0xFF & bufForSingleRead.get(0);
            }
        } catch (IOException e) {
            throw new HPdfIoException(HPdfErrorCode.HPDF_FILE_IO_ERROR, 0, e);
        }
    }

    @Override
    public int read(byte[] buf, int bufOffset, int siz) {
        logger.trace("HPdfFileReadStream#read.");
        
        ByteBuffer chBuf = ByteBuffer.wrap(buf);
        int bufLimit = bufOffset + siz;
        if (bufLimit >= buf.length) {
            bufLimit = buf.length;
        }
        chBuf.limit(bufLimit);
        chBuf.position(bufOffset);
        try {
            return this.getSourceChannel().read(chBuf);
        } catch (IOException e) {
            throw new HPdfIoException(HPdfErrorCode.HPDF_FILE_IO_ERROR, 0, e);
        }
    }

    @Override
    public void seek(long offset, SeekMode mode) {
        logger.trace("HPdfFileReadStream#seek.");
        try {
            this.getSourceChannel().position(
                    this.offsetToAbsolutePos(offset, mode));
        } catch (IOException e) {
            throw new HPdfIoException(HPdfErrorCode.HPDF_FILE_IO_ERROR, 0, e);
        }
    }

    @Override
    public long tell() {
        logger.trace("HPdfFileReadStream#tell.");
        try {
            return this.getSourceChannel().position();
        } catch (IOException e) {
            throw new HPdfIoException(HPdfErrorCode.HPDF_FILE_IO_ERROR, 0, e);
        }
    }

    @Override
    public void close() {
        logger.trace("HPdfFileReadStream#close.");
        HPdfIOUtil.closeQuietly(this.sourceChannel);
        this.sourceChannel = null;
    }

    @Override
    public long getSize() {
        logger.trace("HPdfFileReadStream#getSize.");
        try {
            return this.getSourceChannel().size();
        } catch (IOException e) {
            throw new HPdfIoException(HPdfErrorCode.HPDF_FILE_IO_ERROR, 0, e);
        }
    }
}
