/*
 * JHPdf Free PDF Library : HPdfAbstractReadStream.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.io;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.EnumSet;
import java.util.zip.Deflater;

import net.sf.jhpdf.HPdfConst;
import net.sf.jhpdf.HPdfErrorCode;
import net.sf.jhpdf.encrypt.HPdfEncrypt;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Abstract stream class for reading.
 * It has some utility methods.<br>
 * This class defines charsetForReadLn as ISO-8859-1.
 * Subclasses properly execute setCharsetForReadLn() if needed.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public abstract class HPdfAbstractReadStream implements HPdfReadStream {
    
    private static final Logger logger = LoggerFactory.getLogger(HPdfAbstractReadStream.class);

    private HPdfStreamSignature signature = new HPdfStreamSignature();
    
    protected HPdfAbstractReadStream() {
        super();
        this.setCharsetForReadLn(StandardCharsets.ISO_8859_1);
    }
    
    private Charset charsetForReadLn;
    
    @Override
    public final Charset getCharsetForReadLn() {
        return this.charsetForReadLn;
    }
    
    protected final void setCharsetForReadLn(Charset cs) {
        this.charsetForReadLn = cs;
    }
    
    @Override
    public boolean validate() {
        return signature.validate();
    }

    @Override
    public final int read(byte[] buf, int siz) {
        return this.read(buf,  0, siz);
    }

    @Override
    public final String readLn() {
        logger.trace("HPdfAbstractReadStream#readLn");
        
        ByteArrayOutputStream buf = new ByteArrayOutputStream();
        for (;;) {
            int ch = this.readByte();
            if (ch < 0) {
                logger.trace(" readLn: encountered EOF.");
                break;
            } else if (ch == 0x0d) {
                ch = this.readByte();
                if (ch != 0x0a) {
                    this.seek(-1, SeekMode.CUR);
                }
                break;
            }
            buf.write(ch);
        }
        if (buf.size() == 0) {
            return "";
        } else {
            return new String(buf.toByteArray(), this.getCharsetForReadLn());
        }
    }
    
    /**
     * read a byte value from stream.<br>
     * <strong>the read value treated as 'unsigned' (normally returns 0 to 255).
     * if returns -1, the stream has reached at end.</strong>
     * @return read 'unsigned' byte value or -1 when EOF.
     */
    protected abstract int readByte();

    @Override
    public final void writeToStream(HPdfWriteStream dst, EnumSet<HPdfFilterFlag> filter, HPdfEncrypt e) {
        logger.trace("HPdfAbstractReadStream#writeToStream");
        
        if (dst == null) {
            throw new HPdfIoException(HPdfErrorCode.HPDF_INVALID_OBJECT, 0);
        }
        
        /* initialize input stream */
        if (this.getSize() == 0) {
            return;
        }
        
        if (filter.contains(HPdfFilterFlag.FLATE_DECODE)) {
            writeToStreamWithDeflate(dst, e);
            return;
        }
        
        this.seek(0, SeekMode.SET);
        
        byte[] buf = new byte[HPdfConst.HPDF_STREAM_BUF_SIZ];
        byte[] ebuf = new byte[HPdfConst.HPDF_STREAM_BUF_SIZ];
        for (;;) {
            int size = this.read(buf, buf.length);
            
            if (size <= 0) {
                break;
            }
            
            if (e != null) {
                e.cryptBuf(buf, ebuf, size);
                dst.write(ebuf, size);
            } else {
                dst.write(buf, size);
            }
        }
    }

    @Override
    public final void writeToStream(HPdfWriteStream dst, HPdfFilterFlag filter, HPdfEncrypt e) {
        if (filter == null) {
            this.writeToStream(dst, EnumSet.noneOf(HPdfFilterFlag.class), e);
        } else {
            this.writeToStream(dst, EnumSet.of(filter), e);
        }
    }
    
    private static final int DEFLATE_BUF_SIZ = (int) (HPdfConst.HPDF_STREAM_BUF_SIZ * 1.1 + 13);

    private final void writeToStreamWithDeflate(HPdfWriteStream dst, HPdfEncrypt e) {
        logger.trace("HPdfAbstractReadStream#writeToStreamWithDeflate");
        
        /* initialize input stream */
        this.seek(0, SeekMode.SET);
        
        byte[] inbuf = new byte[HPdfConst.HPDF_STREAM_BUF_SIZ];
        byte[] otbuf = new byte[DEFLATE_BUF_SIZ];
        int otbufPos = 0;
        byte[] ebuf = new byte[DEFLATE_BUF_SIZ];

        /* initialize decompression stream. */
        Deflater compressor = new Deflater(Deflater.DEFAULT_COMPRESSION);
        try {
            for (;;) {
                int size = this.read(inbuf, inbuf.length);
                if (size <= 0) {
                    compressor.finish();
                    break;
                }
                compressor.setInput(inbuf, 0, size);
                
                while (!compressor.needsInput()) {
                    otbufPos += compressor.deflate(otbuf, otbufPos, otbuf.length - otbufPos);
                    if (otbufPos >= otbuf.length) {
                        if (e != null) {
                            e.cryptBuf(otbuf, ebuf, ebuf.length);
                            dst.write(ebuf, ebuf.length);
                        } else {
                            dst.write(otbuf, otbuf.length);
                        }
                        otbufPos = 0;
                    }
                }
            }
            
            while (!compressor.finished()) {
                // MEMO: On first loop, otbuf may contain data as much as otbufPos.
                otbufPos += compressor.deflate(otbuf, otbufPos, otbuf.length - otbufPos);
                if (e != null) {
                    e.cryptBuf(otbuf, ebuf, otbufPos);
                    dst.write(ebuf, otbufPos);
                } else {
                    dst.write(otbuf, otbufPos);
                }
                otbufPos = 0;
            }
        } finally {
            compressor.end();
        }
    }

    protected final long offsetToAbsolutePos(long offset, SeekMode mode) {
        long absolutePos;
        switch (mode) {
        case SET:
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " offsetToAbsolutePos: Seek position %d"
                        , offset));
            }
            absolutePos = offset;
            break;
        case CUR:
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " offsetToAbsolutePos: Seek position %d from %d"
                        , this.tell(), offset));
            }
            absolutePos = this.tell() + offset;
            break;
        case END:
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " offsetToAbsolutePos: Seek position %d from EOF"
                        , offset));
            }
            absolutePos = this.getSize() + offset;
            break;
        default:
            if (logger.isTraceEnabled()) {
                logger.trace(
                        "offsetToAbsolutePos: unknown SeekMode " + mode.toString());
            }
            throw new HPdfIoException(HPdfErrorCode.HPDF_FILE_IO_ERROR, 0,
                    new IOException("Invalid SeekMode " + mode.toString()));
        }
        if (absolutePos < 0) {
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " offsetToAbsolutePos: backward failed pos %d"
                        , absolutePos));
            }
            throw new HPdfIoException(HPdfErrorCode.HPDF_FILE_IO_ERROR, 0,
                    new IOException("backward seek exceeded stream head."));
        } else if (absolutePos > this.getSize()) {
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " offsetToAbsolutePos: forward failed pos %d, size=%d"
                        , absolutePos, this.getSize()));
            }
            throw new HPdfIoException(HPdfErrorCode.HPDF_FILE_IO_ERROR, 0,
                    new IOException("forward seek exceeded stream end."));
        }
        
        return absolutePos;
    
    }
}
