/*
 * JHPdf Free PDF Library : ByteUtil.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf;

import java.nio.charset.StandardCharsets;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * byte data utility methods class.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public final class ByteUtil {

    private static final Logger logger = LoggerFactory.getLogger(ByteUtil.class);

    private ByteUtil() {
        // inhibits instantiation.
    }
    
    /**
     * method for convenience to find C-like String in byte-array.
     * @param target search target byte-array.
     * @param findStr find key. <strong>It's encoding is assumed as ISO-8859-1.</strong>
     * @param limit limit size of search.
     * @return when found, matched first index. not found, -1.
     */
    public static int indexOf(byte[] target, String findStr, int limit) {
        return indexOf(target,
                findStr.getBytes(StandardCharsets.ISO_8859_1), limit);
    }
    
    /**
     * find byte-array in byte-array.
     * @param target search target byte-array.
     * @param find find key byte-array.
     * @param limit limit size of search.
     * @return when found, matched first index. not found, -1.
     */
    public static int indexOf(byte[] target, byte[] find, int limit) {
        if (target == null || find == null) {
            throw new IllegalArgumentException("target or find is null.");
        }
        
        if (target.length == 0 || find.length == 0) {
            // nothing to search
            return -1;
        }
        
        if (limit > target.length) {
            limit = target.length;
        }
        
        for (int pos = 0; pos < limit; ++pos) {
            if (target[pos] == find[0]) {
                if (limit - pos < find.length) {
                    // remainder of target is too short. 
                    return -1;
                } else {
                    int unmatch = -1;
                    // first elem ([0]) is already checked.
                    for (int fpos = 1; fpos < find.length; ++fpos) {
                        if (target[pos + fpos] != find[fpos]) {
                            unmatch = fpos;
                            break;
                        }
                    }
                    if (unmatch == -1) {
                        // all elements matched.
                        return pos;
                    }
                }
            }
        }
        // unmatched
        return -1;
    }
    
    
}
