﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;
using Llh = Yubeshi.GeodeticCoordinate;
using Const = Yubeshi.Constants.Grs80;

namespace Yubeshi.Japan
{

    public class Jpc
    {
        // cf.
        // http://vldb.gsi.go.jp/sokuchi/surveycalc/algorithm/bl2xy/bl2xy.htm
        #region type definition
        private static class MeridianArcConstants
        {
            public static readonly double[] K = new double[]{
                    +1.00505250181314665196,
                    -5.06310862232687815719e-3,
                    +1.06275903276004425316e-5,
                    -2.08204071585121779847e-8,
                    +3.93323047003588273989e-11,
                    -7.26523616912507470164e-14,
                    +1.32165336892313669049e-16,
                    -2.37675735369947451391e-19,
                    +4.10941974026919833771e-22
                };
            /*
            private static long[] GenerateSquaredBinomial(int length)
            { 
                // cf. A002894 Binomial(2n,n)^2
                long k = 1;
                long[] b = new long[length];
                for(int i = 1; i < length; ++i)
                {
                    b[i - 1] = k * k;
                    long p = (i + i) * (i + i - 1);
                    long q = i * i;
                    k *= p;
                    k /= q;
                }
                b[length - 1] = k * k;
                return b;
            }
            */

        }
        public enum System
        {
            Unknown = 0,
            I = 1,
            II = 2,
            III = 3,
            IV = 4,
            V = 5,
            VI = 6,
            VII = 7,
            VIII = 8,
            IX = 9,
            X = 10,
            XI = 11,
            XII = 12,
            XIII = 13,
            XIV = 14,
            XV = 15,
            XVI = 16,
            XVII = 17,
            XVIII = 18,
            XIX = 19,
        }
        #endregion

        #region fields

        public static readonly GeodeticCoordinate[] Origins;
        private const double m0 = 0.9999;
        #endregion

        #region constructors
        public Jpc(GeodeticCoordinate llh, System number)
            : this(llh.Latitude, llh.Longitude, number)
        {
        }

        public Jpc(Degree latitude, Degree longitude, System number)
        {
            SystemId = number;
            GeodeticCoordinate origin = Origins[(int)number];
            double n = GetRadiusOfPrimeVertical(latitude);
            double cos = Math.Cos(latitude.Radian);

            double delta = longitude.Radian - origin.Longitude.Radian;

            double t = Math.Tan(latitude.Radian);
            double t2 = t * t;
            double t4 = t2 * t2;

            double eta2 = Const.SecondEccentricitySquared;
            eta2 *= cos * cos;
            double eta4 = eta2 * eta2;

            double cd2 = cos * cos * delta * delta;
            double cd = cd2 * t;

            X = cd / 2.0;

            cd *= cd2;
            X += cd / 24.0 * (5.0 - t2 + 9.0 * eta2 + 4.0 * eta4);

            cd *= cd2;
            X += cd / 720.0 *
                (61.0 - 58.0 * t2 - t4 + 270.0 * eta2 - 330.0 * t2 * eta2);

            cd *= cd2;
            X += cd / 40320.0 *
                    (1385.0 - 3111.0 * t2 + 543.0 * t4 - t4 * t2);

            double deltaS = GetLengthOfMeridianArc(latitude) -
                                    GetLengthOfMeridianArc(origin.Latitude);
            X = (X * n + deltaS) * m0;


            cd = cos * delta;
            Y = cd;

            cd *= cd2;
            Y -= cd / 6.0 * (-1.0 + t2 - eta2);

            cd *= cd2;
            Y -= cd / 120.0 *
                    (-5.0 + 18.0 * t2 - t4 - 14.0 * eta2 + 58.0 * t2 * eta2);

            cd *= cd2;
            Y -= cd / 5040.0 *
                    (-61.0 + 479.0 * t2 - 179.0 * t4 + t4 * t2);
            Y *= n * m0;
        }

        public Jpc(double x, double y)
        {
            X = x;
            Y = y;
            SystemId = System.Unknown;
        }

        static Jpc()
        {
            Origins = new GeodeticCoordinate[20]{
                null,
                new Llh(new Degree(33, 0, 0), new Degree(129, 30, 0)),
                new Llh(new Degree(33, 0, 0), new Degree(131, 0, 0)),
                new Llh(new Degree(36, 0, 0), new Degree(132, 10, 0)),
                new Llh(new Degree(33, 0, 0), new Degree(133, 30, 0)),
                new Llh(new Degree(36, 0, 0), new Degree(134, 20, 0)),
                new Llh(new Degree(36, 0, 0), new Degree(136, 0, 0)),
                new Llh(new Degree(36, 0, 0), new Degree(137, 10, 0)),
                new Llh(new Degree(36, 0, 0), new Degree(138, 30, 0)),
                new Llh(new Degree(36, 0, 0), new Degree(139, 50, 0)),
                new Llh(new Degree(40, 0, 0), new Degree(140, 50, 0)),
                new Llh(new Degree(44, 0, 0), new Degree(140, 15, 0)),
                new Llh(new Degree(44, 0, 0), new Degree(142, 15, 0)),
                new Llh(new Degree(44, 0, 0), new Degree(144, 15, 0)),
                new Llh(new Degree(26, 0, 0), new Degree(142, 0, 0)),
                new Llh(new Degree(26, 0, 0), new Degree(127, 30, 0)),
                new Llh(new Degree(26, 0, 0), new Degree(124, 0, 0)),
                new Llh(new Degree(26, 0, 0), new Degree(131, 0, 0)),
                new Llh(new Degree(20, 0, 0), new Degree(136, 0, 0)),
                new Llh(new Degree(26, 0, 0), new Degree(154, 0, 0)),
            };
        }

        #endregion

        #region properties
        public double X
        {
            get;
            private set;
        }

        public double Y
        {
            get;
            private set;
        }

        public System SystemId
        {
            get;
            private set;
        }


        #endregion

        #region public methods

        public static double GetLengthOfMeridianArc(Degree latitude)
        {
            double s = 0;
            double phi = latitude.Radian;
            double[] k = MeridianArcConstants.K;
            for (int i = k.Length - 1; i > 0; --i)
            {
                int i2 = i + i;
                s += k[i] * Math.Sin(i2 * phi) / i2;
            }
            return Const.SemiMajorAxisA * 
                    (1.0 - Const.FirstEccentricitySquared) * (s + k[0] * phi);
        }

        public static double GetRadiusOfPrimeVertical(Degree latitude)
        {
            double sin = Math.Sin(latitude.Radian);
            double r = 
                    Math.Sqrt(1 -  Const.FirstEccentricitySquared * sin * sin);
            return Const.SemiMajorAxisA / r;
        }

        public static System GetSystemWithNearestOriginTo(
                                                    GeodeticCoordinate coord)
        {
            System nearestSystem = System.Unknown;
            double min = Double.MaxValue;
            for (int i = 1; i <= (int)System.XIX; ++i)
            {
                System s = (System)i;
                Jpc c = new Jpc(coord, s);
                double d = c.X * c.X + c.Y * c.Y;
                if (d < min)
                {
                    min = d;
                    nearestSystem = s;
                }
            }
            return nearestSystem;
        }

        public EnuCoordinate ToEnuCoordinate(Height height)
        {
            GeodeticCoordinate origin = Origins[(int)SystemId];
            return new EnuCoordinate(Y, X, height, origin.ToEcefCoordinate());
        }

        #endregion
    }
}
