﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi.Ubx
{
    public class RxmRaw : Packet
    {
        #region type definitions
        public struct MeasurementData
        { 
            public byte[] Raw
            {
                get;
                private set;
            }

            public double CarrierPhase
            {
                get
                {
                    return BitConverter.ToDouble(Raw, 0);
                }
            }

            public double Pseudorange
            {
                get
                {
                    return BitConverter.ToDouble(Raw, 8);
                }
            }

            public double Doppler
            {
                get
                {
                    return BitConverter.ToSingle(Raw, 16);
                }
            }

            public uint SpaceVehicleNumber
            {
                get
                {
                    return Raw[20];
                }
            }

            public int QualityIndicator
            {
                get
                {
                    return (sbyte)Raw[21];
                }
            }

            public int SignalStrength
            {
                get
                {
                    return (sbyte)Raw[22];
                }
            }

            public int LossOfLockIndicator
            {
                get
                {
                    return Raw[23];
                }
            }


            public void Set(byte[] raw)
            {
                Raw = raw;
            }
        }
        #endregion

        #region constructors

        public RxmRaw(byte[] sentence, int length)
            : base(sentence, length)
        {
            Data = new MeasurementData[NumberOfSatelites];
            for (int i = 0; i < Data.Length; ++i)
            {
                Data[i].Set(OctetString.Substring(Raw, 8 + 24 * i, 24));
            }
        }

        #endregion

        #region properties

        public GpsTime GpsTime
        {
            get
            {
                return new GpsTime(Week, TimeOfWeek);
            }
        }

        public decimal TimeOfWeek
        {
            get
            {
                return BitConverter.ToUInt32(Raw, 6 + 0) * 1e-3m;
            }
        }

        public int Week
        {
            get
            {
                return BitConverter.ToInt16(Raw, 6 + 4);
            }
        }

        public int NumberOfSatelites
        {
            get
            {
                return Raw[6 + 6];
            }
        }

        public MeasurementData[] Data
        {
            get;
            private set;
        }

        #endregion

        #region public methods

        public static bool TryParse(byte[] sentence, out UnknownPacket packet)
        {
            packet = null;
            if (sentence.Length < 16)
            {
                return false;
            }
            int length = 8 + 24 * sentence[6 + 6];
            return TryParse(sentence, out packet, 
                                        MessageID.RxmRaw, length, Build);
        }

        #endregion

        #region private methods

        private static Packet Build(byte[] sentence, int length)
        {
            return new RxmRaw(sentence, length);
        }

        #endregion
    }
}
