﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi.Ubx
{
    public class Parser : Yubeshi.Parser
    {
        #region type definitions
        #endregion

        #region fields
        private static readonly 
            Dictionary<Protocol, ParserFunctions> functions;
        private static Dictionary<Packet.MessageID, TrialParser> parsers;
        private static readonly Encoding ascii = Encoding.ASCII;

        #endregion

        #region constructors
        static Parser()
        {
            functions = new Dictionary<Protocol, ParserFunctions>();
            functions.Add(Protocol.Ubx, 
                                new ParserFunctions(TryParse, SearchSyncFrom));

            parsers = new Dictionary<Packet.MessageID, TrialParser>();
            parsers[Packet.MessageID.CfgRate] = CfgRate.TryParse;

            parsers[Packet.MessageID.InfDebug] = InfDebug.TryParse;
            parsers[Packet.MessageID.InfError] = InfError.TryParse;
            parsers[Packet.MessageID.InfNotice] = InfNotice.TryParse;
            parsers[Packet.MessageID.InfTest] = InfTest.TryParse;
            parsers[Packet.MessageID.InfWarning] = InfWarning.TryParse;

            parsers[Packet.MessageID.MonHw2] = MonHw2.TryParse;

            parsers[Packet.MessageID.NavClock] = NavClock.TryParse;
            parsers[Packet.MessageID.NavDop] = NavDop.TryParse;
            parsers[Packet.MessageID.NavPosEcef] = NavPosEcef.TryParse;
            parsers[Packet.MessageID.NavPosLlh] = NavPosLlh.TryParse;
            parsers[Packet.MessageID.NavSol] = NavSol.TryParse;
            parsers[Packet.MessageID.NavStatus] = NavStatus.TryParse;
            parsers[Packet.MessageID.NavSvInfo] = NavSvInfo.TryParse;
            parsers[Packet.MessageID.NavTimeGps] = NavTimeGps.TryParse;
            parsers[Packet.MessageID.NavTimeUtc] = NavTimeUtc.TryParse;
            parsers[Packet.MessageID.NavVelEcef] = NavVelEcef.TryParse;
            parsers[Packet.MessageID.NavVelNed] = NavVelNed.TryParse;

            parsers[Packet.MessageID.RxmAlm] = RxmAlm.TryParse;
            parsers[Packet.MessageID.RxmRaw] = RxmRaw.TryParse;
            parsers[Packet.MessageID.RxmSfrB] = RxmSfrB.TryParse;
        }
        #endregion

        #region public method
        public new void Push(byte[] fragment)
        {
            Concat(fragment);
            ParseBuffer(functions);
        }

        public static void AddParser(Packet.MessageID id, TrialParser parser)
        {
            parsers[id] = parser;
        }

        public static void RemoveParser(Packet.MessageID id)
        {
            parsers.Remove(id);
        }

        public new static bool TryParse(
                                    byte[] sentence, out UnknownPacket packet)
        {
            packet = null;
            if (sentence.Length < 8)
            {
                return false;
            }
            if (sentence[0] != Packet.SyncCharacter1 ||
                sentence[1] != Packet.SyncCharacter2)
            {
                return false;
            }
            Packet.MessageID messageID = 
                            (Packet.MessageID)(sentence[2] << 8 | sentence[3]);
            if (parsers.ContainsKey(messageID))
            {
                return parsers[messageID](sentence, out packet);
            }
            return Unknown.TryParse(sentence, out packet);
        }

        public static int SearchSyncFrom(byte[] input)
        {
            return SearchSyncFrom(input, 0);
        }

        public static int SearchSyncFrom(byte[] input, int index)
        {
            return OctetString.IndexOf(input, Packet.SyncCharacters, index);
        }

        #endregion
    }
}
