﻿/*
 *	Yubeshi GPS Parser
 *
 *	This software is distributed under a zlib-style license.
 *	See license.txt for more information.
 */

using System;
using System.Collections.Generic;
using System.Text;

namespace Yubeshi.Ubx
{
    public class NavTimeUtc : Packet
    {
        #region type definitions

        [Flags]
        public enum Validity : byte
        { 
            TimeOfWeek = 0x01,
            WeekNumber = 0x02,
            Utc = 0x04,
        }

        #endregion

        #region constructors

        public NavTimeUtc(byte[] sentence, int length)
            : base(sentence, length)
        {
        }

        #endregion

        #region properties

        public decimal TimeOfWeek
        {
            get
            {
                uint tow = BitConverter.ToUInt32(Raw, 6 + 0);
                int ftow = BitConverter.ToInt32(Raw, 6 + 8);
                return tow * 1e-3m + ftow * 1e-9m;
            }
        }

        public decimal Accuracy
        {
            get 
            {
                return BitConverter.ToUInt32(Raw, 6 + 4) * 1e-9m;
            }
        }

        public DateTime Utc
        {
            get
            { 
                int y = BitConverter.ToUInt16(Raw, 6 + 12);
                int m = Raw[6 + 14];
                int d = Raw[6 + 15];
                int h = Raw[6 + 16];
                int min = Raw[6 + 17];
                int s = Raw[6 + 18];
                int ftow = BitConverter.ToInt32(Raw, 6 + 8);
                DateTime utc = new DateTime(y, m, d, h, min, s, 
                                                            DateTimeKind.Utc);
                return utc.AddTicks(ftow / 100);
            }
        }

        public Validity ValidityFlag
        {
            get
            {
                return (Validity)Raw[19];
            }
        }
        #endregion

        #region public methods

        public static bool TryParse(byte[] sentence, out UnknownPacket packet)
        {
            return TryParse(sentence, out packet, 
                                            MessageID.NavTimeUtc, 20, Build);
        }

        #endregion

        #region private methods

        private static Packet Build(byte[] sentence, int length)
        {
            return new NavTimeUtc(sentence, length);
        }

        #endregion
    }
}
