package hiro.yoshioka.csveditor;

import java.net.MalformedURLException;
import java.net.URL;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.text.DocumentEvent;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IDocumentListener;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.browser.IWorkbenchBrowserSupport;
import org.eclipse.ui.editors.text.TextEditor;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.editor.IFormPage;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.events.IHyperlinkListener;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.internal.browser.WorkbenchBrowserSupport;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.texteditor.IDocumentProvider;

/**
 * An example showing how to create a multi-page editor. This example has 3
 * pages:
 * <ul>
 * <li>page 0 contains a nested text editor.
 * <li>page 1 allows you to change the font used in page 2
 * <li>page 2 shows the words in page 0 in sorted order
 * </ul>
 */
public class CSVEditor extends FormEditor implements IResourceChangeListener {
	private CSVRDHView csv_table_editor;

	/** Page index of the text editor (always the last page) */
	private int mTextPageIndex;

	/** The text editor */
	private TextEditor mTextEditor;

	/** flag set during page creation */
	private boolean mIsCreatingPage = false;

	private static String BROWSER_ID = "yonsama"; // $NON-NLS-1$

	private IDocument mDocument;
	QualifiedName qname = new QualifiedName("yonsama", getClass()
			.getSimpleName()
			+ "_current_page");

	/**
	 * Creates a multi-page editor example.
	 */
	public CSVEditor() {
		super();
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
	}

	protected void onDocumentChanged(DocumentEvent event) {
		System.out.println("docChanged " + event);
	}

	@Override
	protected void addPages() {
		createAndroidPages();
		selectDefaultPage(null /* defaultPageId */);

	}

	protected void createAndroidPages() {
		mIsCreatingPage = true;
		// createFormPages();
		createPage0();
		createTextEditor();
		createUndoRedoActions();
		mIsCreatingPage = false;
	}

	/**
	 * Returns whether the editor is currently creating its pages.
	 */
	public boolean isCreatingPages() {
		return mIsCreatingPage;
	}

	@Override
	protected void setInput(IEditorInput input) {
		super.setInput(input);
		if (input instanceof FileEditorInput) {
			FileEditorInput fileInput = (FileEditorInput) input;
			IFile file = fileInput.getFile();
			setPartName(String.format("%1$s", file.getName()));
		}
	}

	/**
	 * Creates undo redo actions for the editor site (so that it works for any
	 * page of this multi-page editor) by re-using the actions defined by the
	 * {@link TextEditor} (aka the XML text editor.)
	 */
	private void createUndoRedoActions() {
		IActionBars bars = getEditorSite().getActionBars();
		if (bars != null) {
			IAction action = mTextEditor.getAction(ActionFactory.UNDO.getId());
			bars.setGlobalActionHandler(ActionFactory.UNDO.getId(), action);

			action = mTextEditor.getAction(ActionFactory.REDO.getId());
			bars.setGlobalActionHandler(ActionFactory.REDO.getId(), action);

			bars.updateActionBars();
		}
	}

	/**
	 * Selects the default active page.
	 * 
	 * @param defaultPageId
	 *            the id of the page to show. If <code>null</code> the editor
	 *            attempts to find the default page in the properties of the
	 *            {@link IResource} object being edited.
	 */
	protected void selectDefaultPage(String defaultPageId) {
		if (defaultPageId == null) {
			if (getEditorInput() instanceof IFileEditorInput) {
				IFile file = ((IFileEditorInput) getEditorInput()).getFile();

				String pageId;
				try {
					pageId = file.getPersistentProperty(qname);
					if (pageId != null) {
						defaultPageId = pageId;
					}
				} catch (CoreException e) {
					// ignored
				}
			}
		}

		if (defaultPageId != null) {
			try {
				setActivePage(Integer.parseInt(defaultPageId));
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * Removes all the pages from the editor.
	 */
	protected void removePages() {
		int count = getPageCount();
		for (int i = count - 1; i >= 0; i--) {
			removePage(i);
		}
	}

	@Override
	public IFormPage setActivePage(String pageId) {
		System.out.println("pageId=" + pageId);
		if (pageId.equals("editor_part")) {
			super.setActivePage(mTextPageIndex);
			return null;
		} else {
			return super.setActivePage(pageId);
		}
	}

	@Override
	protected void pageChange(int newPageIndex) {
		super.pageChange(newPageIndex);

		System.out.println("nowPage=" + newPageIndex + "/" + mIsCreatingPage);
		if (mIsCreatingPage) {
			return;
		}

		if (newPageIndex == 0) {
			csv_table_editor.refreshFromString(getDocument().get());
		} else {
			String newInput = csv_table_editor.getInputString();
			if (!newInput.equals(mDocument.get())) {
				mDocument.set(newInput);
			}
		}
		System.out.println("ei=" + getEditorInput());

		if (getEditorInput() instanceof IFileEditorInput) {
			IFile file = ((IFileEditorInput) getEditorInput()).getFile();

			try {
				file.setPersistentProperty(qname, Integer
						.toString(newPageIndex));
			} catch (CoreException e) {
				// ignore
			}
		}
	}

	/**
	 * Creates page 0 of the multi-page editor, which contains a text editor.
	 */
	void createPage0() {
		try {
			csv_table_editor = new CSVRDHView();
			int index = addPage(csv_table_editor, getEditorInput());
			setPageText(index, csv_table_editor.getTitle());
			csv_table_editor.refreshFromEditorInput();
		} catch (PartInitException e) {
			ErrorDialog.openError(getSite().getShell(),
					"Error creating nested text editor", null, e.getStatus());
		}
	}

	public IFile getFile() {
		if (getEditorInput() instanceof IFileEditorInput) {
			return ((IFileEditorInput) getEditorInput()).getFile();
		}
		return null;
	}

	/**
	 * The <code>MultiPageEditorPart</code> implementation of this
	 * <code>IWorkbenchPart</code> method disposes all nested editors.
	 * Subclasses may extend.
	 */
	public void dispose() {
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		super.dispose();
	}

	/**
	 * Saves the multi-page editor's document.
	 */
	public void doSave(IProgressMonitor monitor) {
		commitPages(true /* onSave */);

		// The actual "save" operation is done by the Structured XML Editor
		getEditor(mTextPageIndex).doSave(monitor);

	}

	/**
	 * Saves the multi-page editor's document as another file. Also updates the
	 * text for page 0's tab, and updates this multi-page editor's input to
	 * correspond to the nested editor's.
	 */
	public void doSaveAs() {
		commitPages(true /* onSave */);

		IEditorPart editor = getEditor(mTextPageIndex);
		editor.doSaveAs();
		setPageText(mTextPageIndex, editor.getTitle());
		setInput(editor.getEditorInput());

	}

	@Override
	public void commitPages(boolean onSave) {
		if (pages != null) {
			for (int i = 0; i < pages.size(); i++) {
				Object page = pages.get(i);
				if (page != null && page instanceof IFormPage) {
					IFormPage form_page = (IFormPage) page;
					IManagedForm managed_form = form_page.getManagedForm();
					if (managed_form != null && managed_form.isDirty()) {
						managed_form.commit(onSave);
					}
				}
			}
		}
	}

	public final IHyperlinkListener createHyperlinkListener() {
		return new HyperlinkAdapter() {
			/**
			 * Switch to the page corresponding to the link that has just been
			 * clicked. For this purpose, the HREF of the &lt;a&gt; tags above
			 * is the page ID to switch to.
			 */
			@Override
			public void linkActivated(HyperlinkEvent e) {
				super.linkActivated(e);
				String link = e.data.toString();
				if (link.startsWith("http") || //$NON-NLS-1$
						link.startsWith("file:/")) { //$NON-NLS-1$
					openLinkInBrowser(link);
				} else if (link.startsWith("page:")) { //$NON-NLS-1$
					// Switch to an internal page
					setActivePage(link.substring(5 /* strlen("page:") */));
				}
			}
		};
	}

	private void openLinkInBrowser(String link) {
		try {
			IWorkbenchBrowserSupport wbs = WorkbenchBrowserSupport
					.getInstance();
			wbs.createBrowser(BROWSER_ID).openURL(new URL(link));
		} catch (PartInitException e1) {
			// pass
		} catch (MalformedURLException e1) {
			// pass
		}
	}

	/*
	 * (non-Javadoc) Method declared on IEditorPart
	 */
	public void gotoMarker(IMarker marker) {
		setActivePage(0);
		IDE.gotoMarker(getEditor(0), marker);
	}

	/**
	 * The <code>MultiPageEditorExample</code> implementation of this method
	 * checks that the input is an instance of <code>IFileEditorInput</code>.
	 */
	public void init(IEditorSite site, IEditorInput editorInput)
			throws PartInitException {
		if (!(editorInput instanceof IFileEditorInput))
			throw new PartInitException(
					"Invalid Input: Must be IFileEditorInput");
		super.init(site, editorInput);
	}

	private void createTextEditor() {
		try {
			mTextEditor = new TextEditor();
			int index = addPage(mTextEditor, getEditorInput());
			mTextPageIndex = index;
			setPageText(index, mTextEditor.getTitle());

			IDocumentProvider provider = mTextEditor.getDocumentProvider();
			mDocument = provider.getDocument(getEditorInput());

			mDocument.addDocumentListener(new IDocumentListener() {
				public void documentChanged(DocumentEvent event) {
					onDocumentChanged(event);
				}

				public void documentAboutToBeChanged(DocumentEvent event) {
					// ignore
				}
			});

		} catch (PartInitException e) {
			ErrorDialog.openError(getSite().getShell(),
					"Android Text Editor Error", null, e.getStatus());
		}
	}

	public IDocument getDocument() {
		return mDocument;
	}

	/**
	 * Returns the {@link IProject} for the edited file.
	 */
	public IProject getProject() {
		if (mTextEditor != null) {
			IEditorInput input = mTextEditor.getEditorInput();
			if (input instanceof FileEditorInput) {
				FileEditorInput fileInput = (FileEditorInput) input;
				IFile inputFile = fileInput.getFile();

				if (inputFile != null) {
					return inputFile.getProject();
				}
			}
		}

		return null;
	}

	/*
	 * (non-Javadoc) Method declared on IEditorPart.
	 */
	public boolean isSaveAsAllowed() {
		return true;
	}

	public void resourceChanged(final IResourceChangeEvent event) {
		if (event.getType() == IResourceChangeEvent.PRE_CLOSE) {
			Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					IWorkbenchPage[] pages = getSite().getWorkbenchWindow()
							.getPages();
					for (int i = 0; i < pages.length; i++) {
						if (((FileEditorInput) mTextEditor.getEditorInput())
								.getFile().getProject().equals(
										event.getResource())) {
							IEditorPart editorPart = pages[i]
									.findEditor(mTextEditor.getEditorInput());
							pages[i].closeEditor(editorPart, true);
						}
					}
				}
			});
		}
	}

}
