/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-10 by Raw Material Software Ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the GNU General
   Public License (Version 2), as published by the Free Software Foundation.
   A copy of the license is included in the JUCE distribution, or can be found
   online at www.gnu.org/licenses.

   JUCE is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  ------------------------------------------------------------------------------

   To release a closed-source product which uses JUCE, commercial licenses are
   available: visit www.rawmaterialsoftware.com/juce for more information.

  ==============================================================================
*/

#include "includes.h"
#include "sf_UndoManager.h"

using namespace juce;

namespace sf 
{
	struct ScopedSetTrue {
		ScopedSetTrue(bool& p) : flag(p) { flag = true;}
		~ScopedSetTrue(){flag = false;}
	private:
		bool& flag;
	};

	//==============================================================================
UndoManager::UndoManager (/*const int maxNumberOfUnitsToKeep,
                          const int minimumTransactions*/)
   : /*totalUnitsStored (0),*/
     nextIndex (0),
     newTransaction (true),
     reentrancyCheck (false),redoing(false),performing(false)
{
    //setMaxNumberOfStoredUnits (maxNumberOfUnitsToKeep,
    //                           minimumTransactions);
}

UndoManager::~UndoManager()
{
    clearUndoHistory();
}

//==============================================================================
void UndoManager::clearUndoHistory()
{
    transactions.clear();
    transactionNames.clear();
    //totalUnitsStored = 0;
    nextIndex = 0;
    sendChangeMessage (this);
}

//int UndoManager::getNumberOfUnitsTakenUpByStoredCommands() const
//{
//    return totalUnitsStored;
//}

//void UndoManager::setMaxNumberOfStoredUnits (const int maxNumberOfUnitsToKeep,
//                                             const int minimumTransactions)
//{
//    //maxNumUnitsToKeep          = jmax (1, maxNumberOfUnitsToKeep);
//    //minimumTransactionsToKeep  = jmax (1, minimumTransactions);
//}

//==============================================================================
bool UndoManager::perform (UndoableAction* const command, const String& actionName)
{
    if (command != 0)
    {
        if (actionName.isNotEmpty())
            currentTransactionName = actionName;

        if (reentrancyCheck)
        {
            jassertfalse    // don't call perform() recursively from the UndoableAction::perform() or
                            // undo() methods, or else these actions won't actually get done.

            return false;
        }
        else
        {
		    //reentrancyCheck = true;
            bool success = false;
//            if(!redoing){

//			}

			{
				std::auto_ptr<UndoableNode> node(new UndoableNode(command));
				currentNodeStack.push(node.get());

				JUCE_TRY
				{
					success = node->perform();
				}
				JUCE_CATCH_EXCEPTION

				currentNodeStack.pop();

				if(success){
					if(!currentNodeStack.empty())
					{
						currentNodeStack.top()->subCommands().push_back(node.release());

					} else if (!redoing)
					{ 
						if (nextIndex > 0 && ! newTransaction)
						{
							OwnedArray<UndoableNode>* commandSet = transactions [nextIndex - 1];

							jassert (commandSet != 0);
							if (commandSet == 0)
								return false;
							commandSet->add (node.release());
						}
						else
						{
					
							jassert(currentNodeStack.empty());

							OwnedArray<UndoableNode>* commandSet = new OwnedArray<UndoableNode>();
							commandSet->add (node.release());
							transactions.insert (nextIndex, commandSet);
							transactionNames.insert (nextIndex, currentTransactionName);
							++nextIndex;
						}

						//totalUnitsStored += command->getSizeInUnits();
						//bool transactionBackup = newTransaction;

						while (nextIndex < transactions.size())
						{
							const OwnedArray <UndoableNode>* const lastSet = transactions.getLast();

							//for (int i = lastSet->size(); --i >= 0;)
							//	totalUnitsStored -= lastSet->getUnchecked (i)->getSizeInUnits();

							transactions.removeLast();
							transactionNames.remove (transactionNames.size() - 1);
						}

						//while (nextIndex > 0
						//		&& totalUnitsStored > maxNumUnitsToKeep
						//		&& transactions.size() > minimumTransactionsToKeep)
						//{
						//	const OwnedArray <UndoableNode>* const firstSet = transactions.getFirst();

						//	for (int i = firstSet->size(); --i >= 0;)
						//		totalUnitsStored -= firstSet->getUnchecked (i)->getSizeInUnits();

						//	jassert (totalUnitsStored >= 0); // something fishy going on if this fails!

						//	transactions.remove (0);
						//	transactionNames.remove (0);
						//	--nextIndex;
						//}
						jassert (success);
						sendChangeMessage (this);
					}
				}
			} 
		    //reentrancyCheck = false;
            return success;
        }
    }
	//reentrancyCheck = false;
    return false;
}

void UndoManager::beginNewTransaction (const String& actionName)
{
    newTransaction = true;
    currentTransactionName = actionName;
}

void UndoManager::setCurrentTransactionName (const String& newName)
{
    currentTransactionName = newName;
}

//==============================================================================
bool UndoManager::canUndo() const
{
    return nextIndex > 0;
}

bool UndoManager::canRedo() const
{
    return nextIndex < transactions.size();
}

const String UndoManager::getUndoDescription() const
{
    return transactionNames [nextIndex - 1];
}

const String UndoManager::getRedoDescription() const
{
    return transactionNames [nextIndex];
}

bool UndoManager::undo()
{
    OwnedArray<UndoableNode>* const commandSet = transactions [nextIndex - 1];

    if (commandSet == 0)
        return false;

	bool failed = false;
	{
		ScopedSetTrue reentrancyCheckStart(reentrancyCheck);


		for (int i = commandSet->size(); --i >= 0;)
		{
			if (! commandSet->getUnchecked(i)->undo())
			{
				jassertfalse
				failed = true;
				break;
			}
		}

		//for(int i = 0;i < commandSet->size();)
		//{
		//	//if(commandSet->getUnchecked(i)->isNestedAction())
		//	//{
		//		commandSet->remove(i);
		//	//} else {
		//	//	++i;
		//	//}
		//}
		//commandSet->clear();

	}

    if (failed)
    {
        clearUndoHistory();
    }
    else
    {
        --nextIndex;
    }

    beginNewTransaction();

    sendChangeMessage (this);
    return true;
}

bool UndoManager::redo()
{
    const OwnedArray<UndoableNode>* const commandSet = transactions [nextIndex];

    if (commandSet == 0)
        return false;

    //reentrancyCheck = true;
	bool failed = false;

	{
		ScopedSetTrue redoStart(redoing);

		for (int i = 0; i < commandSet->size(); ++i)
		{
			currentNodeStack.push(commandSet->getUnchecked(i));
			bool failed =  !commandSet->getUnchecked(i)->perform();
			currentNodeStack.pop();
			if (failed)
			{
				jassertfalse
				failed = true;
				break;
			}
		}
	}


    if (failed)
    {
        clearUndoHistory();
    }
    else
    {
        ++nextIndex;
    }

    beginNewTransaction();

    sendChangeMessage (this);
    return true;
}

bool UndoManager::undoCurrentTransactionOnly()
{
    return newTransaction ? false
                          : undo();
}

void UndoManager::getActionsInCurrentTransaction (Array <const UndoableAction*>& actionsFound) const
{
    const OwnedArray <UndoableNode>* const commandSet = transactions [nextIndex - 1];

    if (commandSet != 0 && ! newTransaction)
    {
        for (int i = 0; i < commandSet->size(); ++i)
            actionsFound.add (commandSet->getUnchecked(i)->get());
    }
}

int UndoManager::getNumActionsInCurrentTransaction() const
{
    const OwnedArray <UndoableNode>* const commandSet = transactions [nextIndex - 1];

    if (commandSet != 0 && ! newTransaction)
        return commandSet->size();

    return 0;
}

bool UndoManager::UndoableNode::undo() 
{
	bool  result = action->undo();
	if(result){
		SubCommands::iterator it = subCommands_.end();
		while(it != subCommands_.begin())
		{
			--it;
			it->undo();
		}
	}
	subCommands_.clear();
	return result;
}

}
