/*
  ==============================================================================

   This file is part of the JUCE library - "Jules' Utility Class Extensions"
   Copyright 2004-7 by Raw Material Software ltd.

  ------------------------------------------------------------------------------

   JUCE can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   JUCE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with JUCE; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307 USA

  ------------------------------------------------------------------------------

   If you'd like to release a closed-source product which uses JUCE, commercial
   licenses are also available: visit www.rawmaterialsoftware.com/juce for
   more information.

  ==============================================================================
*/

#pragma once

namespace sf{
class MidiInputSelectorComponentListBox;
//==============================================================================
/**
    A component containing controls to let the user change the audio settings of
    an AudioDeviceManager object.

    Very easy to use - just create one of these and show it to the user.

    @see AudioDeviceManager
*/
	class JUCE_API  AudioDeviceSelectorComponent  : public juce::Component,
                                                public juce::ComboBoxListener,
                                                public juce::ButtonListener,
                                                public juce::ChangeListener
{
public:
    //==============================================================================
    /** Creates the component.

        If your app needs only output channels, you might ask for a maximum of 0 input
        channels, and the component won't display any options for choosing the input
        channels. And likewise if you're doing an input-only app.

        @param deviceManager            the device manager that this component should control
        @param minAudioInputChannels    the minimum number of audio input channels that the application needs
        @param maxAudioInputChannels    the maximum number of audio input channels that the application needs
        @param minAudioOutputChannels   the minimum number of audio output channels that the application needs
        @param maxAudioOutputChannels   the maximum number of audio output channels that the application needs
        @param showMidiInputOptions     if true, the component will allow the user to select which midi inputs are enabled
        @param showMidiOutputSelector   if true, the component will let the user choose a default midi output device
        @param showChannelsAsStereoPairs    if true, channels will be treated as pairs; if false, channels will be
                                        treated as a set of separate mono channels.
        @param hideAdvancedOptionsWithButton    if true, only the minimum amount of UI components
                                        are shown, with an "advanced" button that shows the rest of them
    */
	AudioDeviceSelectorComponent (juce::AudioDeviceManager& deviceManager,
                                  const int minAudioInputChannels,
                                  const int maxAudioInputChannels,
                                  const int minAudioOutputChannels,
                                  const int maxAudioOutputChannels,
                                  const bool showMidiInputOptions,
                                  const bool showMidiOutputSelector,
                                  const bool showChannelsAsStereoPairs,
                                  const bool hideAdvancedOptionsWithButton);

    /** Destructor */
    ~AudioDeviceSelectorComponent();


    //==============================================================================
    /** @internal */
    void resized();
    /** @internal */
	void comboBoxChanged (juce::ComboBox*);
	/** @internal */
	void buttonClicked (juce::Button*);
    /** @internal */
    void changeListenerCallback (void*);

    //==============================================================================
    juce_UseDebuggingNewOperator

private:
    juce::AudioDeviceManager& deviceManager;
    juce::ComboBox* deviceTypeDropDown;
    juce::Label* deviceTypeDropDownLabel;
    juce::Component* audioDeviceSettingsComp;
    juce::String audioDeviceSettingsCompType;
    const int minOutputChannels, maxOutputChannels, minInputChannels, maxInputChannels;
    const bool showChannelsAsStereoPairs;
    const bool hideAdvancedOptionsWithButton;

	MidiInputSelectorComponentListBox* midiInputsList;
    juce::Label* midiInputsLabel;
    juce::ComboBox* midiOutputSelector;
	juce::Label* midiOutputLabel;

    AudioDeviceSelectorComponent (const AudioDeviceSelectorComponent&);
    const AudioDeviceSelectorComponent& operator= (const AudioDeviceSelectorComponent&);
};

}
