#pragma once
/*
  ==============================================================================

   This file is part of the WinAudioJuce
   Copyright 2005-10 by Satoshi Fujiwara.

   S.F.Tracker can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   S.F.Tracker is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with S.F.Tracker; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ==============================================================================
*/

#include "singleton.h"
namespace sf{

class UndoManager :
	public juce::ChangeBroadcaster,public Singleton<UndoManager>,boost::noncopyable
{
public:
	class UndoableNode  
	{
	public:
		typedef boost::ptr_vector<UndoableNode> SubCommands;
		UndoableNode(juce::UndoableAction* action_)
		{
			action.reset(action_);
		}

		bool perform() { return action->perform();}
		bool undo();
		//int getSizeInUnits()    { return action->getSizeInUnits(); }
//		bool isNestedAction() {return nestedAction;}

		juce::UndoableAction* get(){return action.get();}
		SubCommands& subCommands(){ return subCommands_;}

		//==============================================================================
	    juce_UseDebuggingNewOperator
	private:
		boost::shared_ptr<juce::UndoableAction> action;
		SubCommands subCommands_;
	};

    //==============================================================================
    /** Creates an UndoManager.

        @param maxNumberOfUnitsToKeep       each UndoableAction object returns a value
                                            to indicate how much storage it takes up
                                            (UndoableAction::getSizeInUnits()), so this
                                            lets you specify the maximum total number of
                                            units that the undomanager is allowed to
                                            keep in memory before letting the older actions
                                            drop off the end of the list.
        @param minimumTransactionsToKeep    this specifies the minimum number of transactions
                                            that will be kept, even if this involves exceeding
                                            the amount of space specified in maxNumberOfUnitsToKeep
    */

	UndoManager(/*const int maxNumberOfUnitsToKeep = 30000,
                 const int minimumTransactionsToKeep = 30*/);

	~UndoManager();


    //==============================================================================
    /** Deletes all stored actions in the list. */
    void clearUndoHistory();

    ///** Returns the current amount of space to use for storing UndoableAction objects.

    //    @see setMaxNumberOfStoredUnits
    //*/
    //int getNumberOfUnitsTakenUpByStoredCommands() const;

    ///** Sets the amount of space that can be used for storing UndoableAction objects.

    //    @param maxNumberOfUnitsToKeep       each UndoableAction object returns a value
    //                                        to indicate how much storage it takes up
    //                                        (UndoableAction::getSizeInUnits()), so this
    //                                        lets you specify the maximum total number of
    //                                        units that the undomanager is allowed to
    //                                        keep in memory before letting the older actions
    //                                        drop off the end of the list.
    //    @param minimumTransactionsToKeep    this specifies the minimum number of transactions
    //                                        that will be kept, even if this involves exceeding
    //                                        the amount of space specified in maxNumberOfUnitsToKeep
    //    @see getNumberOfUnitsTakenUpByStoredCommands
    //*/
    //void setMaxNumberOfStoredUnits (int maxNumberOfUnitsToKeep,
    //                                int minimumTransactionsToKeep);

    //==============================================================================
    /** Performs an action and adds it to the undo history list.

        @param action   the action to perform - this will be deleted by the UndoManager
                        when no longer needed
        @param actionName   if this string is non-empty, the current transaction will be
                            given this name; if it's empty, the current transaction name will
                            be left unchanged. See setCurrentTransactionName()
        @returns true if the command succeeds - see UndoableAction::perform
        @see beginNewTransaction
    */
    bool perform (juce::UndoableAction* action,
                  const juce::String& actionName = juce::String::empty);

    /** Starts a new group of actions that together will be treated as a single transaction.

        All actions that are passed to the perform() method between calls to this
        method are grouped together and undone/redone together by a single call to
        undo() or redo().

        @param actionName   a description of the transaction that is about to be
                            performed
    */
    void beginNewTransaction (const juce::String& actionName = juce::String::empty);

    /** Changes the name stored for the current transaction.

        Each transaction is given a name when the beginNewTransaction() method is
        called, but this can be used to change that name without starting a new
        transaction.
    */
    void setCurrentTransactionName (const juce::String& newName);

    //==============================================================================
    /** Returns true if there's at least one action in the list to undo.

        @see getUndoDescription, undo, canRedo
    */
    bool canUndo() const;

    /** Returns the description of the transaction that would be next to get undone.

        The description returned is the one that was passed into beginNewTransaction
        before the set of actions was performed.

        @see undo
    */
    const juce::String getUndoDescription() const;

    /** Tries to roll-back the last transaction.

        @returns    true if the transaction can be undone, and false if it fails, or
                    if there aren't any transactions to undo
    */
    bool undo();

    /** Tries to roll-back any actions that were added to the current transaction.

        This will perform an undo() only if there are some actions in the undo list
        that were added after the last call to beginNewTransaction().

        This is useful because it lets you call beginNewTransaction(), then
        perform an operation which may or may not actually perform some actions, and
        then call this method to get rid of any actions that might have been done
        without it rolling back the previous transaction if nothing was actually
        done.

        @returns true if any actions were undone.
    */
    bool undoCurrentTransactionOnly();

    /** Returns a list of the UndoableAction objects that have been performed during the
        transaction that is currently open.

        Effectively, this is the list of actions that would be undone if undoCurrentTransactionOnly()
        were to be called now.

        The first item in the list is the earliest action performed.
    */
    void getActionsInCurrentTransaction (juce::Array <const juce::UndoableAction*>& actionsFound) const;

    /** Returns the number of UndoableAction objects that have been performed during the
        transaction that is currently open.
        @see getActionsInCurrentTransaction
    */
    int getNumActionsInCurrentTransaction() const;

    //==============================================================================
    /** Returns true if there's at least one action in the list to redo.

        @see getRedoDescription, redo, canUndo
    */
    bool canRedo() const;

    /** Returns the description of the transaction that would be next to get redone.

        The description returned is the one that was passed into beginNewTransaction
        before the set of actions was performed.

        @see redo
    */
    const juce::String getRedoDescription() const;

    /** Tries to redo the last transaction that was undone.

        @returns    true if the transaction can be redone, and false if it fails, or
                    if there aren't any transactions to redo
    */
    bool redo();


    //==============================================================================
    juce_UseDebuggingNewOperator
private:
    //==============================================================================
    juce::OwnedArray <juce::OwnedArray <UndoableNode> > transactions;
    juce::StringArray transactionNames;
    juce::String currentTransactionName;
    int /*totalUnitsStored, maxNumUnitsToKeep, minimumTransactionsToKeep,*/ nextIndex;
    bool newTransaction, reentrancyCheck,redoing,performing;
	std::stack<UndoableNode*> currentNodeStack;


};
}

