/*
  ==============================================================================

   This file is part of the S.F.Tracker
   Copyright 2005-7 by Satoshi Fujiwara.

   S.F.Tracker can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   S.F.Tracker is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with S.F.Tracker; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ==============================================================================
*/
#include "includes.h"
#include "sequenceFilter.h"
#include "SequenceEditor.h"
#define NOTE_CHANNEL       1
#define NOTE_VELOCITY      0.8f
#define NOTE_PREFRAMES     0.001
using namespace juce;
using namespace boost;

namespace sf {

	SequenceFilter::SequenceFilter() :     
		allNotesOffMessage (MidiMessage::allNotesOff (1)),
		samplePositionCounter (0.0),
//		endTick (0.0),
		framesPerBeat (0.0),
		leftLocator (0.0),
		rightLocator (INIT_RIGHTLOCATOR),
		bpmTempo (INIT_TEMPO),
		numBars (INIT_NUMBERS),
		divDenominator (INIT_DIVDENOMINATOR),
		numerator(INIT_NUMERATOR),
		playing (false),
		looping (true),
		recording (false),
		doStopRecord (false),
		doRewind (false),
		doAllNotesOff (false),
		tickCounter(0.0),lastTick(0.0),timeBase(INIT_TIMEBASE)
	{
		numMidiOutputs = INIT_TRACK,
		setPlayConfigDetails(0,0,INIT_SAMPLERATE,0);
		startTimer(250);
//		loadSequenceData(juce::FileInputStream(File(L"c:\\temp\\Satie_gimun_k.mid")));

	};

	bool SequenceFilter::readSequenceDataFromFile(const juce::File& inputFile)
	{
		juce::ScopedLock s(graph->getGraph().getCallbackLock());
		juce::ScopedPointer<MidiFile>  f(new juce::MidiFile());
		if(f->readFrom(juce::FileInputStream(inputFile)))
		{
			file.swapWith(f);
			stop();

			samplePositionCounter = 0;
//			endTick = 0;
			tickCounter = 0;

			// e|єqf[^̃NA
			tempoSig.clear();
			// gbNƂ̃V[PXf[^CfbNX̃NA
			currentSequenceIndexs.clear();

			readFile = inputFile;

			//file->convertTimestampTicksToSeconds();

			numMidiOutputs = numOfTracks = file->getNumTracks();// gbN
			//numOfTracks = file->getNumTracks();// gbN
			/*if(numOfTracks > numMidiOutputs)
			{
				numMidiOutputs = numOfTracks;

			}*/
			DBG((wformat(L"Number Of Tracks: %d \n") % file->getNumTracks()).str().c_str());

			timeBase = file->getTimeFormat();// ^Cx[X
			//DBG((wformat(L"last time stamp: %d \n") % file->getLastTimestamp()).str().c_str());

			// tempo & sig initialize
			juce::MidiMessageSequence tempo,sig;
			file->findAllTempoEvents(tempo);
			file->findAllTimeSigEvents(sig);
			

            double newNumerator, newDenominator,newTempo;

			// qݒ
			if(sig.getNumEvents() > 0){
				juce::int32 n,d;
	            sig.getEventPointer(0)->message.getTimeSignatureInfo (n, d);
				numerator = newNumerator = (double)n;
				newDenominator = (double)d;

			}

			// e|ݒ
			if(tempo.getNumEvents() > 0)
			{
				newTempo = (60000000.0) 
					/ (double)(
						(((int)(tempo.getEventPointer(0)->message.getMetaEventData()[0])) << 16) 
						+ (((int)(tempo.getEventPointer(0)->message.getMetaEventData()[1])) << 8) 
						+ ((int)(tempo.getEventPointer(0)->message.getMetaEventData()[2]))
					);
			}

			// e|єqf[^̒o
			tempoSig.addSequence(tempo,0,0,0);
			tempoSig.addSequence(sig,0,0,0);

			// 1߂tick߂
			double bar = newNumerator * 4.0 / newDenominator * timeBase ;
			// ߐ߂
			double bars = ceil(file->getLastTimestamp() / bar);
			lastTick = file->getLastTimestamp();

			setTimeSignature (newTempo,// e|
                              bars,// ߐ
                              newDenominator,timeBase);			


			//@gbNƂ̃V[PXf[^CfbNX̐
			for(int i = 0;i < numOfTracks;++i)
			{
				currentSequenceIndexs.add(0);
			}

			//DBG((wformat(L"tempo event: %d \n") % tempo.getNumEvents()).str().c_str());
			//for(int i = 0;i < tempo.getNumEvents();++i)
			//{
			//	MidiMessageSequence::MidiEventHolder* evt(tempo.getEventPointer(i));
			//	int t = (60000000L) / ((((int)(evt->message.getMetaEventData()[0])) << 16) + (((int)(evt->message.getMetaEventData()[1])) << 8) + ((int)(evt->message.getMetaEventData()[2]))); 
			//	DBG((wformat(L"event : %d %d \n") % evt->message.getTimeStamp() % t).str().c_str());
			//}
			//DBG((wformat(L"sig event: %d \n") % sig.getNumEvents()).str().c_str());
			graph->removeIllegalConnections();
			graph->changed();
			updateHostDisplay();

//			graph->getGraph().removeIllegalConnections();
//			graph->getGraph().triggerAsyncUpdate();
			return true;
		} else {
			return false;
		}
	}

	void SequenceFilter::fillInPluginDescription (juce::PluginDescription& d) const
	{
		d.name = getName();
		d.uid = d.name.hashCode();
		d.category = "I/O devices";
		d.pluginFormatName = "Internal";
		d.manufacturerName = "SFPGMR";
		d.version = "1.0";
		d.isInstrument = false;

		d.numInputChannels = getNumInputChannels();
		d.numOutputChannels = getNumOutputChannels();

	};

	void SequenceFilter::prepareToPlay (double sampleRate, int estimatedSamplesPerBlock)
	{
		setPlayConfigDetails(getNumInputChannels(),getNumOutputChannels(),sampleRate,estimatedSamplesPerBlock);
		sampleRateDelta = 1.0 / sampleRate;
	    startTimer (1000 / 25);

		setTimeSignature (bpmTempo,    
                      numBars,         
                      divDenominator,timeBase);  

		#if JUCE_PLUGINHOST_VST
			// prepare vsttimeinfo
			timeInfo.samplePos = 0.0;
			timeInfo.sampleRate = sampleRate;
			timeInfo.nanoSeconds = 0.0;
			timeInfo.ppqPos = 0.0;
			timeInfo.tempo = bpmTempo;                        // in bpm
			timeInfo.barStartPos = 0;                         // last bar start, in 1 ppq
			timeInfo.cycleStartPos = 0;                       // 1 ppq
			timeInfo.cycleEndPos = 0;                         // 1 ppq
			timeInfo.barStartPos = 0;                         // last bar start, in 1 ppq
			timeInfo.barStartPos = 0;                         // last bar start, in 1 ppq
			timeInfo.timeSigNumerator = divDenominator;       // time signature numerator
			timeInfo.timeSigDenominator = divDenominator;     // time signature denominator
			timeInfo.smpteOffset = 0;
			timeInfo.smpteFrameRate = 1;
			timeInfo.samplesToNextClock = 0;
			timeInfo.flags = 0 | kVstTransportPlaying
							   | kVstNanosValid
							   | kVstTransportChanged
							   | kVstPpqPosValid
							   | kVstTempoValid
							   | kVstTimeSigValid
							   | kVstClockValid;
							   // kVstBarsValid | kVstCyclePosValid | kVstTimeSigValid | kVstSmpteValid | kVstClockValid
		#endif

		//play();
	};

	void SequenceFilter::releaseResources()
	{
		stop();
		stopTimer();
	};

	void SequenceFilter::processBlock (juce::AudioSampleBuffer& buffer, juce::Array<MidiBuffer*>& midiMessages)
	{
		
		const int blockSize = buffer.getNumSamples ();

		//processAudioPlayHead (getPlayHead());
		//keyboardState.processNextMidiBuffer (*midiBuffer,
		//                                     0, blockSize,
		//                                     true);

		if (isPlaying ())
		{
			const double frameCounter = getSamplePosition ();
			const double sampleRate = getSampleRate();//getsampleRate ();

			const double currentTick = getTickCounter();
			const double nextTick = currentTick + (double)blockSize / getSamplesPerTick();

			// prepare record incoming midi messages
			if (isRecording ())
			{
				int samplePos = 0;
				MidiMessage msg (0xf4, 0.0);

				MidiBuffer::Iterator eventIterator (*midiMessages.getUnchecked(0));
				while (eventIterator.getNextEvent (msg, samplePos))
				{
					msg.setTimeStamp (floor((frameCounter + (double)samplePos) / getSamplesPerTick()));
					recordingSequence.addEvent (msg);
				}
			}

			// fill midi buffers with the events !
			for (int j = 0,nt = file->getNumTracks();j < nt;j++)
			{
				const juce::MidiMessageSequence* seq(file->getTrack(j));
				int i,numEvents;
				if(j != 0)
				{
					midiMessages.getUnchecked(j)->clear();
				}
				for (i = currentSequenceIndexs.getUnchecked(j),numEvents = seq->getNumEvents ();
							i < numEvents;
							i++)
				{

					double timeStamp = seq->getEventTime (i);

					if ( timeStamp >= nextTick)
					{
						break;
					}

					MidiMessage* midiMessage = &seq->getEventPointer (i)->message;
					midiMessages.getUnchecked(j)->addEvent (*midiMessage, (timeStamp - currentTick) * getSamplesPerTick() /*- frameCounter*/);
//					DBG ("Playing event @ " + String (frameCounter) + " : " + String ((timeStamp - currentTick) * getSamplesPerTick()));
				}
//				DBG((boost::wformat(L"midi event count %d %d %f" ) % j % midiMessages.getUnchecked(j)->getNumEvents() %  midiMessages.getUnchecked(j)->getLastEventTime()).str().c_str());
				currentSequenceIndexs.setUnchecked(j,i);
			}

		}

	#if JUCE_PLUGINHOST_VST
		timeInfo.tempo = bpmTempo;
		timeInfo.nanoSeconds = (double) Time::getMillisecondCounterHiRes () * 1000000.0;
	#endif

		if (playing)
		{
	#if JUCE_PLUGINHOST_VST
			static double smpteDiv[] = { 24.f, 25.f, 24.f, 30.f, 29.97f, 30.f };

			const double ppqPos = samplePositionCounter / timeInfo.sampleRate;
			timeInfo.ppqPos = ppqPos * timeInfo.tempo / 60.0;
			timeInfo.smpteOffset = (long) ((ppqPos - floor (ppqPos))
										   * smpteDiv [timeInfo.smpteFrameRate] * 80.0);
			timeInfo.flags |= kVstTransportPlaying;
			timeInfo.flags &= ~kVstTransportChanged;
	#endif

			// we should update rewind button
			//if (samplePositionCounter == leftLocator)
			//{
			//	//updateHostDisplay();
			//}

			// increase counter
			samplePositionCounter += (double)blockSize;
			tickCounter += (double)blockSize / samplesPerTick;

			// we should update play button
			if (tickCounter >= lastTick /* endTick */) // rightLocator
			{
				if (looping)
				{
					setPosition(leftLocator);
				}
				else
				{
					playing = false;
					//updateHostDisplay();

	#if JUCE_PLUGINHOST_VST
					timeInfo.flags &= ~kVstTransportPlaying;
					timeInfo.flags |= kVstTransportChanged;
	#endif
				}
			}
		}



		processIncomingMidi (*midiMessages.getUnchecked(0));

		if (doAllNotesOff || willSendAllNotesOff ())
		{
			for(int i = 0;i < midiMessages.size();++i){
				midiMessages.getUnchecked(i)->addEvent (allNotesOffMessage, blockSize - 1);
			}
			doAllNotesOff = false;
		}

		if (doAllNotesOff)
		{
			doAllNotesOff = false;
		}

		if (doRewind)
		{
			setPosition(leftLocator);
//			samplePositionCounter = leftLocator;
			doRewind = false;
		}

	#if JUCE_PLUGINHOST_VST
		timeInfo.samplePos = (float) samplePositionCounter;
	#endif

	};

	bool SequenceFilter::timeSignatureChanged (const double barsCount,const double timeDenominator)
	{
		setTimeSignature (getTempo (),
									 barsCount,
									 timeDenominator,getTimeBase());

		//setRightLocator (barsCount * timeDenominator);
		return true;
	}

	bool SequenceFilter::playingPositionChanged (const double absolutePosition)
	{
		setPosition (absolutePosition);
		return true;
	}

	bool SequenceFilter::noteAdded (const int noteNumber,
										const double beatNumber,
										const double noteLength)
	{
		MidiMessageSequence * seq(const_cast<MidiMessageSequence *>(file->getTrack(currentTrackNo)));
		MidiMessage msgOn = MidiMessage::noteOn (NOTE_CHANNEL, noteNumber, NOTE_VELOCITY);
		msgOn.setTimeStamp (beatNumber);
		MidiMessage msgOff = MidiMessage::noteOff (NOTE_CHANNEL, noteNumber);
		msgOff.setTimeStamp ((beatNumber + noteLength) - NOTE_PREFRAMES);

		DBG ("Adding:" + String (noteNumber) + " " + String (beatNumber));

		{
			const ScopedLock sl (graph->getGraph().getCallbackLock());
			seq->addEvent (msgOn);
			seq->addEvent (msgOff);
			seq->updateMatchedPairs ();
		}

		DBG ("Added:" + String (noteNumber) + " " + String (beatNumber));

		return true;
	}

	bool SequenceFilter::noteRemoved (const int noteNumber,
										  const double beatNumber,
										  const double noteLength)
	{

		DBG ("Removing:" + String (noteNumber) + " " + String (beatNumber));
		MidiMessageSequence * seq(const_cast<MidiMessageSequence *>(file->getTrack(currentTrackNo)));

		double noteOnBeats = beatNumber - NOTE_PREFRAMES;

		int eventIndex = seq->getNextIndexAtTime (noteOnBeats);
		while (eventIndex < seq->getNumEvents ())
		{
			MidiMessage* eventOn = &seq->getEventPointer (eventIndex)->message;

			if (eventOn->isNoteOn () && eventOn->getNoteNumber () == noteNumber)
			{
				// TODO - check note off distance == noteLength
				{
					const ScopedLock sl (graph->getGraph().getCallbackLock());

					seq->deleteEvent (eventIndex, true);
					seq->updateMatchedPairs ();
				}

				DBG ("Removed:" + String (eventIndex) + " > " + String (noteNumber) + " @ " + String (beatNumber));

				if (isPlaying ())
					doAllNotesOff = true;

				return true;
			}

			eventIndex++;
		}

		DBG (">>> Remove failed:" + String (noteNumber) + " @ " + String (beatNumber));

		return false;
	}

	bool SequenceFilter::noteMoved (const int oldNote,
										const double oldBeat,
										const int noteNumber,
										const double beatNumber,
										const double noteLength)
	{
		DBG ("Moving:" + String (oldNote) + " " + String (oldBeat));
		MidiMessageSequence * seq(const_cast<MidiMessageSequence *>(file->getTrack(currentTrackNo)));

		double noteOnBeats = oldBeat - NOTE_PREFRAMES;

		int eventIndex = seq->getNextIndexAtTime (noteOnBeats);
		while (eventIndex < seq->getNumEvents ())
		{
			MidiMessage* eventOn = &seq->getEventPointer (eventIndex)->message;

			if (eventOn->isNoteOn () && eventOn->getNoteNumber () == oldNote)
			{
				// TODO - check old note off distance == oldNoteLength

				MidiMessage msgOn = MidiMessage::noteOn (NOTE_CHANNEL, noteNumber, NOTE_VELOCITY);
				msgOn.setTimeStamp (beatNumber);
				MidiMessage msgOff = MidiMessage::noteOff (NOTE_CHANNEL, noteNumber);
				msgOff.setTimeStamp ((beatNumber + noteLength) - NOTE_PREFRAMES);

				{
	            const ScopedLock sl (graph->getGraph().getCallbackLock());

				// remove old events
				seq->deleteEvent (eventIndex, true);
				seq->updateMatchedPairs ();
				// add new events
				seq->addEvent (msgOn);
				seq->addEvent (msgOff);
				seq->updateMatchedPairs ();
				}

				DBG ("Moved:" + String (eventIndex) + " > "
							  + String (oldNote) + " @ " + String (oldBeat) + " to "
							  + String (noteNumber) + " @ " + String (beatNumber));

				if (isPlaying ())
					doAllNotesOff = true;

				return true;
			}

			eventIndex++;
		}

		DBG (">>> Move failed:" + String (oldNote) + " @ " + String (oldBeat));

		return false;
	}

	bool SequenceFilter::noteResized (const int noteNumber,
										  const double beatNumber,
										  const double noteLength)
	{

		DBG ("Resizing:" + String (noteNumber) + " " + String (beatNumber));
		MidiMessageSequence * seq(const_cast<MidiMessageSequence *>(file->getTrack(currentTrackNo)));

		double noteOnBeats = beatNumber - NOTE_PREFRAMES;

		int eventIndex = seq->getNextIndexAtTime (noteOnBeats);
		while (eventIndex < seq->getNumEvents ())
		{
			MidiMessage* eventOn = &seq->getEventPointer (eventIndex)->message;

			if (eventOn->isNoteOn () && eventOn->getNoteNumber () == noteNumber)
			{
				// TODO - check old note off distance == oldNoteLength

				MidiMessage msgOn = MidiMessage::noteOn (NOTE_CHANNEL, noteNumber, NOTE_VELOCITY);
				msgOn.setTimeStamp (beatNumber);
				MidiMessage msgOff = MidiMessage::noteOff (NOTE_CHANNEL, noteNumber);
				msgOff.setTimeStamp ((beatNumber + noteLength) - NOTE_PREFRAMES);

				{
	            const ScopedLock sl (graph->getGraph().getCallbackLock());

				// delete old events
				seq->deleteEvent (eventIndex, true);
				seq->updateMatchedPairs ();
				// add new events
				seq->addEvent (msgOn);
				seq->addEvent (msgOff);
				seq->updateMatchedPairs ();
				}

				DBG ("Resized:" + String (eventIndex) + " > "
								 + String (noteNumber) + " @ " + String (beatNumber) + " to " + String (noteLength));

				if (isPlaying ())
					doAllNotesOff = true;

				return true;
			}

			eventIndex++;
		}

		DBG (">>> Resize failed:" + String (noteNumber) + " @ " + String (beatNumber));

		return false;
	}

		bool SequenceFilter::allNotesRemoved ()
		{
			{
			const ScopedLock sl (graph->getGraph().getCallbackLock());
			MidiMessageSequence * seq(const_cast<MidiMessageSequence *>(file->getTrack(currentTrackNo)));
			seq->clear ();

			if (isPlaying ())
				doAllNotesOff = true;

			}

			return true;
		}

	//==============================================================================
	void SequenceFilter::getNoteOnIndexed (const int index, int& note, double& beat, double& length)
	{
		MidiMessageSequence * seq(const_cast<MidiMessageSequence *>(file->getTrack(currentTrackNo)));

		int numNoteOn = 0;
		for (int i = 0; i < seq->getNumEvents (); i++)
		{
			MidiMessageSequence::MidiEventHolder* eventOn = seq->getEventPointer (i);
			MidiMessageSequence::MidiEventHolder* eventOff = seq->getEventPointer (i)->noteOffObject;

			MidiMessage* msgOn = & eventOn->message;
			if (eventOn->message.isNoteOn () && eventOff)
			{
				MidiMessage* msgOff = & eventOff->message;
				if (index == numNoteOn)
				{
					note = msgOn->getNoteNumber ();
					beat = msgOn->getTimeStamp ();
					length = ((msgOff->getTimeStamp () + NOTE_PREFRAMES) - msgOn->getTimeStamp ());
					break;
				}
				numNoteOn++;
			}
		}
	}

	int SequenceFilter::getNumNoteOn () const
	{
		const MidiMessageSequence * seq(file->getTrack(currentTrackNo));

		int numNoteOn = 0;

		for (int i = 0; i < seq->getNumEvents (); i++)
			if (seq->getEventPointer (i)->message.isNoteOn ()) numNoteOn++;

		return numNoteOn;
	}

	////==============================================================================
	//static void SequenceFilter::dumpMidiMessageSequence (MidiMessageSequence* seq)
	//{
	//#if JUCE_DEBUG
	//	MidiMessageSequence * seq(const_cast<MidiMessageSequence *>(file->getTrack(currentTrackNo)));
	//
	//    for (int i = 0; i < seq->getNumEvents (); i++)
	//    {
	//        MidiMessage* midiMessage = &seq->getEventPointer (i)->message;
	//
	//        DBG (String (midiMessage->getNoteNumber()) + " "
	//             + (midiMessage->isNoteOn() ? "ON " : "OFF ")
	//             + String (midiMessage->getTimeStamp()));
	//
	//        // DBG ("Playing event @ " + String (frameCounter) + " : " + String (timeStamp));
	//    }
	//#endif
	//}

	//==============================================================================
	void SequenceFilter::getChunk (MemoryBlock& mb)
	{
		MidiMessageSequence * seq(const_cast<MidiMessageSequence *>(file->getTrack(currentTrackNo)));

		MidiFile midiFile;

	#if 0
		MidiMessageSequence timeTrack;
		timeTrack.addEvent (MidiMessage::timeSignatureMetaEvent (4, divDenominator));

		// add timing informations sequence
		midiFile.addTrack (timeTrack);
	#endif

		// add real sequence scaled down !
		MidiMessageSequence scaledSequence;
		for (int i = 0; i < seq->getNumEvents (); i++)
		{
			MidiMessage midiMessage = seq->getEventPointer (i)->message;
			midiMessage.setTimeStamp (midiMessage.getTimeStamp() * 100000.0f);
			scaledSequence.addEvent (midiMessage);
		}
		midiFile.addTrack (scaledSequence);

		MemoryOutputStream os (4096, 2048, &mb);
		midiFile.writeTo (os);

	//    dumpMidiMessageSequence (seq);
	}

	void SequenceFilter::setChunk (const MemoryBlock& mb)
	{
		MidiMessageSequence * seq(const_cast<MidiMessageSequence *>(file->getTrack(currentTrackNo)));

		MemoryInputStream is (mb.getData(), mb.getSize(), false);

		MidiFile midiFile;
		midiFile.readFrom (is);

	#if 0
		// use the first track as signature time track !
		const MidiMessageSequence* timeSequence = midiFile.getTrack (0);
		if (timeSequence)
		{
			for (int i = 0; i < timeSequence->getNumEvents (); i++)
			{
				MidiMessage* msg = & timeSequence->getEventPointer (i)->message;
				if (msg->isTimeSignatureMetaEvent ())
				{
					int numerator, denominator;
					msg->getTimeSignatureInfo (numerator, denominator);
					setTimeSignature (bpmTempo, numBars, denominator);
				}
			}
		}

		// add the note track !
		const MidiMessageSequence* notesSequence = midiFile.getTrack (1);
	#endif

		const MidiMessageSequence* notesSequence = midiFile.getTrack (0);
		if (notesSequence)
		{
			// add real sequence scaled down !
			seq->clear ();
			for (int i = 0; i < notesSequence->getNumEvents (); i++)
			{
				MidiMessage midiMessage = notesSequence->getEventPointer (i)->message;
				if (midiMessage.isNoteOnOrOff())
				{
					midiMessage.setTimeStamp (midiMessage.getTimeStamp() / 100000.0f);
					seq->addEvent (midiMessage);
				}
			}
			seq->updateMatchedPairs ();

			if (isPlaying ())
				doAllNotesOff = true;
		}

	//    dumpMidiMessageSequence (seq);
	}

	//==============================================================================
	void SequenceFilter::savePropertiesToXml (XmlElement* xml)
	{
		//BasePlugin::savePropertiesToXml (xml);

		//xml->setAttribute (PROP_SEQROWOFFSET,            getIntValue (PROP_SEQROWOFFSET, 0));
		//xml->setAttribute (PROP_SEQCOLSIZE,              getIntValue (PROP_SEQCOLSIZE, 100));
		//xml->setAttribute (PROP_SEQNOTELENGTH,           getIntValue (PROP_SEQNOTELENGTH, 4));
		//xml->setAttribute (PROP_SEQNOTESNAP,             getIntValue (PROP_SEQNOTESNAP, 4));
		//xml->setAttribute (PROP_SEQBAR,                  getIntValue (PROP_SEQBAR, 4));
	}

	void SequenceFilter::loadPropertiesFromXml (XmlElement* xml)
	{
		//BasePlugin::loadPropertiesFromXml (xml);

		//setValue (PROP_SEQROWOFFSET,                     xml->getIntAttribute (PROP_SEQROWOFFSET, 0));
		//setValue (PROP_SEQCOLSIZE,                       xml->getIntAttribute (PROP_SEQCOLSIZE, 100));
		//setValue (PROP_SEQNOTELENGTH,                    xml->getIntAttribute (PROP_SEQNOTELENGTH, 4));
		//setValue (PROP_SEQNOTESNAP,                      xml->getIntAttribute (PROP_SEQNOTESNAP, 4));
		//setValue (PROP_SEQBAR,                           xml->getIntAttribute (PROP_SEQBAR, 4));
	}

	//==============================================================================
void SequenceFilter::play ()
{
    DBG ("SequenceFilter::play");

    if (! playing)
    {
        playing = true;

        updateHostDisplay ();

    }
}

void SequenceFilter::stop ()
{
    DBG ("SequenceFilter::stop");

    if (playing)
    {
        doAllNotesOff = true;
        playing = false;

        if (recording)
            doStopRecord = true;
        else
            updateHostDisplay();
    }
}

void SequenceFilter::record ()
{
    DBG ("SequenceFilter::record");

    if (playing)
    {
        if (recording && ! doStopRecord)
            doStopRecord = true;
    }
    else
    {
        if (recording)
        {
            recording = false;
            doStopRecord = false;
        }
        else
        {
            recording = true;
            doStopRecord = false;
        }
    }
}

void SequenceFilter::rewind ()
{
    DBG ("SequenceFilter::rewind");

    if (playing)
    {
        if (recording)
            doStopRecord = true;

        doAllNotesOff = true;
        doRewind = true;
    }
    else
    {
		setPosition(leftLocator);
//        samplePositionCounter = leftLocator;
        doRewind = false;
        updateHostDisplay();
    }

}

void SequenceFilter::allNotesOff ()
{
    DBG ("SequenceFilter::allNotesOff");

    doAllNotesOff = true;
}

void SequenceFilter::resetRecording ()
{
    DBG ("SequenceFilter::resetRecording");

    doStopRecord = false;
    recording = false;
}

////==============================================================================
//void SequenceFilter::processAudioPlayHead (AudioPlayHead* head)
//{
//    AudioPlayHead::CurrentPositionInfo info;
//    if (head->getCurrentPosition (info))
//    {
///*
//        std::cout << "bpm " << info.bpm << std::endl;
//        std::cout << "timeSigNumerator " << info.timeSigNumerator << std::endl;
//        std::cout << "timeSigDenominator " << info.timeSigDenominator << std::endl;
//        std::cout << "timeInSeconds " << info.timeInSeconds << std::endl;
//        std::cout << "ppqPosition " << info.ppqPosition << std::endl;
//        std::cout << "ppqPositionOfLastBarStart " << info.ppqPositionOfLastBarStart << std::endl;
//        std::cout << "frameRate " << info.frameRate << std::endl;
//        std::cout << "isPlaying" << info.isPlaying << std::endl;
//        std::cout << std::endl;
//*/
//    }
//}

void SequenceFilter::processIncomingMidi (MidiBuffer& midiMessages)
{
    int hours, minutes, seconds, frames;

    int samplePosition;
    MidiMessage msg (0xf4, 0.0);
    MidiBuffer::Iterator it (midiMessages);

    while (it.getNextEvent (msg, samplePosition))
    {
        if (msg.isNoteOnOrOff ())
            continue;

        if (msg.isTimeSignatureMetaEvent ())
        {
            int newNumerator, newDenominator;

            msg.getTimeSignatureInfo (newNumerator, newDenominator);
            setTimeSignature (bpmTempo,
                              numBars,
                              newDenominator,timeBase);
        }
        else if (msg.isTempoMetaEvent ())
        {
            // double  getTempoMetaEventTickLength (const short timeFormat) const throw ()
            DBG ("isTempoMetaEvent " + String (msg.getTempoSecondsPerQuarterNote ()));
			setTempo(60000000.0 / (double)(((int) msg.getMetaEventData()[0] << 16)
	             | ((int) msg.getMetaEventData()[1] << 8)
		         | msg.getMetaEventData()[2]));
//            leftLocator = jmax (0, roundFloatToInt (beatNumber * framesPerBeat));
        }
        else if (msg.isMidiStart ())
        {
            rewind ();
            play ();
        }
        else if (msg.isMidiContinue ())
        {
            play ();
        }
        else if (msg.isMidiStop ())
        {
            stop ();
        }
        else if (msg.isSongPositionPointer ())
        {
            DBG ("isSongPositionPointer " + String (msg.getSongPositionPointerMidiBeat ()));
        }
        else if (msg.isQuarterFrame ())
        {
            DBG ("isQuarterFrame " + String (msg.getQuarterFrameSequenceNumber ())
                             + " " + String (msg.getQuarterFrameValue ()));
        }
        else if (msg.isFullFrame ())
        {
            // void    getFullFrameParameters (int &hours, int &minutes, int &seconds, int &frames, SmpteTimecodeType &timecodeType) const throw ()
        }
        else if (msg.isMidiMachineControlGoto (hours, minutes, seconds, frames))
        {
        }
    }
}

//==============================================================================
void SequenceFilter::setTimeSignature (const double bpmTempo_,
                                  const double barsCount_,
                                  const double timeDenominator_,
								  const double timeBase_
								  )
{
    bpmTempo = bpmTempo_;
    numBars = barsCount_;
    divDenominator = timeDenominator_;
	timeBase = timeBase_;
	samplesPerTick = (getSampleRate() / (bpmTempo * timeBase / 60.0));

    framesPerBeat = roundDoubleToInt (getSampleRate() * 60.0 / (float) bpmTempo);
	lastTick = ceil(numBars * timeBase * 4.0 / divDenominator * numerator);
    //endTick = timeBase * (numBars * divDenominator);

//    setRightLocator (rightLocator / (float) framesPerBeat);

//    if (playing)
//        doAllNotesOff = true;

    if (tickCounter >= lastTick)
        doRewind = true;

    //updateHostDisplay();

    // TODO - we should make this as change listener !
    //if (externalSequenceFilter)
    //    externalSequenceFilter->setTempo (bpmTempo);
}

void SequenceFilter::setLeftLocator (const double beatNumber)
{
    leftLocator = jmax (0, roundDoubleToInt (beatNumber * framesPerBeat));

    //updateHostDisplay();
}

void SequenceFilter::setRightLocator (const double beatNumber)
{
//    rightLocator = jmin (endTick, roundFloatToInt (beatNumber * framesPerBeat));
    rightLocator = beatNumber;

    //updateHostDisplay();
}

//==============================================================================
void SequenceFilter::setTempo (const double newTempo)
{
    setTimeSignature (newTempo, numBars, divDenominator,timeBase);
}


juce::AudioProcessorEditor* SequenceFilter::createEditor()
{
	return new SequenceEditor(this);
};

////==============================================================================
//void SequenceFilter::saveToXml (XmlElement* xml)
//{
//    xml->setAttribute (T("tempo"), bpmTempo);
//    xml->setAttribute (T("bars"), numBars);
//    xml->setAttribute (T("numerator"), divDenominator);
//    xml->setAttribute (T("denominator"), divDenominator);
//    xml->setAttribute (T("timebase"), timeBase);
//	xml->setAttribute (T("ppq"), 960 /* ppq */);
//    xml->setAttribute (T("position"), getPositionAbsolute());
//    xml->setAttribute (T("llocator"), getLeftLocator());
//    xml->setAttribute (T("rlocator"), getRightLocator());
//    xml->setAttribute (T("looping"), looping);
//}
//
//void SequenceFilter::loadFromXml (XmlElement* xml)
//{
////    divNumerator = xml->getIntAttribute (T("numerator"), 4);
////    ppq = xml->getIntAttribute (T("ppq"), 960);
//
//    setTimeSignature (xml->getDoubleAttribute (T("tempo"), 120.0),
//                      xml->getDoubleAttribute (T("bars"), 4.0),
//                      xml->getDoubleAttribute (T("denominator"), 4.0), xml->getDoubleAttribute (T("timebase"), 4.0));
//
//    setPositionAbsolute (xml->getDoubleAttribute (T("position"), 0.0));
//    setLeftLocator (xml->getDoubleAttribute (T("llocator"), 0.0));
//    setRightLocator (xml->getDoubleAttribute (T("rlocator"), 1.0));
//    looping = xml->getIntAttribute (T("looping"), 1) == 1;
//
//    sendChangeMessage (this);
//}

void SequenceFilter::convertTickToSamplePosition()
{
	double tempo = INIT_TEMPO;
	double sampleCount = 0.0;
	double ticks = 0.0;
	if(!isEmptyData())
	{
		if(tempoSig.getNumEvents() > 0)
		{
			for(int i = 0;i < tempoSig.getNumEvents();++i)
			{
				MidiMessageSequence::MidiEventHolder* f = tempoSig.getEventPointer(i);
				if(f->message.isTempoMetaEvent())
				{
					MidiMessage& m(f->message);
					if(tickCounter > m.getTimeStamp())
					{
						double t = m.getTimeStamp() - ticks;
						
						sampleCount += (t * getSampleRate() / (tempo * timeBase / 60.0));

						tempo = (60000000.0 / (double)(((int) m.getMetaEventData()[0] << 16)
						 | ((int) m.getMetaEventData()[1] << 8)
						 | m.getMetaEventData()[2]));
						ticks = m.getTimeStamp();
					} else {
						double t = tickCounter - ticks;
						sampleCount += (t * getSampleRate() / (tempo * timeBase / 60.0));
						ticks = tickCounter;
						break;
					}
				}
			}

			if(ticks != tickCounter)
			{
				double t = tickCounter - ticks;
				sampleCount += (t * getSampleRate() / (tempo * timeBase / 60.0));
			}
		} else {
			sampleCount += (tickCounter * getSampleRate() / (tempo * timeBase / 60.0));
		}
	} else {
			sampleCount += (tickCounter * getSampleRate() / (tempo * timeBase / 60.0));
	}

	samplePositionCounter = sampleCount; 
}

void SequenceFilter::seekMidiEvent()
{
	currentSequenceIndexs.clear();
	for(int i = 0;i < numOfTracks;++i)
	{
		currentSequenceIndexs.add(file->getTrack(i)->getNextIndexAtTime(tickCounter));
	}
}


void SequenceFilter::getStateInformation (JUCE_NAMESPACE::MemoryBlock& destData)
{
	// ɕۑ
	std::wstringstream s;
	boost::archive::xml_woarchive oa(s);
	oa & serialization::make_nvp("SequenceFilter",*this);
	DBG(s.str().c_str());
	destData.setSize(sizeof(wchar_t) * (s.str().size() + 1));
	destData.copyFrom(s.str().c_str(),0,sizeof(wchar_t) * (s.str().size() + 1));
}

void SequenceFilter::setStateInformation (const void* data, int sizeInBytes)
{
	std::wstringstream st;
	st << (wchar_t*)data;
	boost::archive::xml_wiarchive ia(st);
	ia & serialization::make_nvp("SequenceFilter",*this);
//			juce::DBG(L"setStateInformation " + juce::String(testState));
};

template<class Archive>
void SequenceFilter::save(Archive & ar, const unsigned int version) const
{
	ar & serialization::make_nvp("FullPathName",readFile.getFullPathName());
};

template<class Archive>
void SequenceFilter::load(Archive & ar, const unsigned int version)
{
	juce::String s;
	ar &  serialization::make_nvp("FullPathName",s);
	readSequenceDataFromFile(juce::File(s));
};


}

