#include "includes.h"
#include "WASAPI.h"
using namespace JUCE_NAMESPACE;
#include "WASAPIControlPanel.h"
#define INITGUID
#undef T
#include <wtypes.h>
#include <winerror.h>
#include <avrt.h>
#include <strsafe.h>
#include <audioclient.h>
#include <audiopolicy.h>
#include <endpointvolume.h>
#include <objbase.h>
#include <comdef.h>
#include <mmreg.h>
#include <ks.h>
#include <ksmedia.h>
#include <ksguid.h>
#include <propsys.h>
#include <initguid.h>
#include <functiondiscoverykeys.h>
#include <mmdeviceapi.h>
#undef INITGUID
#include <assert.h>
#include <propidl.h>
#include <functiondiscoverykeys.h>
#include "sf_com.h"
#include "singleton.h"
//#include "CComptr.h"
#define BOOST_ASSIGN_MAX_PARAMS 7
#include <boost/assign.hpp>
#include <boost/assign/ptr_list_of.hpp>
#include <boost/assign/ptr_list_inserter.hpp>
#include <boost/foreach.hpp>
#define MAX_CHARS  256

// REFERENCE_TIME time units per millisecond
#define REFTIMES_PER_MILLISEC  10000

// Milliseconds per second
#define MILLISECS_PER_SEC  1000
// Calculate number of elements in array.
#define ARRAY_LENGTH(x)  (sizeof(x)/sizeof((x)[0]))

#define EXIT_ON_ERROR(hres)  \
              if (FAILED(hres)) { goto Exit; }
#define EXCEPTION_ON_ERROR(hres) \
	if (FAILED(hres)) { throw sf::Win32ErrorException(hres); }
			

#define SAFE_RELEASE(punk)  \
              if ((punk) != NULL)  \
                { (punk)->Release(); (punk) = NULL; }

inline void intrusive_ptr_add_ref(IUnknown * p){p->AddRef();};
inline void intrusive_ptr_release(IUnknown * p){p->Release();};

using namespace boost::assign;
using namespace boost;

namespace sf {

#if _DEBUG
#define DEBUGPRINTF(x) ::OutputDebugStringW((x).str().c_str())
#else
	#define DEBUGPRINTF(x) 
#endif


//
// Convert a WAVEFORMATEX structure to a WAVEFORMATEXTENSIBLE
// structure. The caller-supplied buffer must be at least
// sizeof(WAVEFORMATEXTENSIBLE).
//
HRESULT ConvertToExtensible(WAVEFORMATEX *pWfx)
{
    WAVEFORMATEXTENSIBLE *pWfxexten;

    if (pWfx == NULL)
    {
        return E_INVALIDARG;
    }
    if (pWfx->wFormatTag == WAVE_FORMAT_EXTENSIBLE)
    {
        // Format is already a WAVEFORMATEXTENSIBLE structure.
        return S_OK;
    }
    if (pWfx->nChannels != 1 && pWfx->nChannels != 2)
    {
        // Not a valid WAVEFORMATEX structure.
        return E_INVALIDARG;
    }
    if (pWfx->wFormatTag == WAVE_FORMAT_PCM)
    {
        // Convert a fixed-point PCM format.
        pWfx->wFormatTag = WAVE_FORMAT_EXTENSIBLE;
        pWfx->cbSize = sizeof(WAVEFORMATEXTENSIBLE) - sizeof(WAVEFORMATEX);
        pWfxexten = reinterpret_cast<WAVEFORMATEXTENSIBLE*>(pWfx);
        pWfxexten->Samples.wValidBitsPerSample = pWfx->wBitsPerSample;
        pWfxexten->dwChannelMask = (pWfx->nChannels == 2) ? 3 : 1;
        pWfxexten->SubFormat = KSDATAFORMAT_SUBTYPE_PCM;

        return S_OK;
    }

    if (pWfx->wFormatTag == WAVE_FORMAT_IEEE_FLOAT)
    {
        // Convert a floating-point PCM format.
        pWfx->wFormatTag = WAVE_FORMAT_EXTENSIBLE;
        pWfx->cbSize = sizeof(WAVEFORMATEXTENSIBLE) - sizeof(WAVEFORMATEX);
        pWfxexten = reinterpret_cast<WAVEFORMATEXTENSIBLE*>(pWfx);
        pWfxexten->Samples.wValidBitsPerSample = pWfx->wBitsPerSample;
        pWfxexten->dwChannelMask = (pWfx->nChannels == 2) ? 3 : 1;
        pWfxexten->SubFormat = KSDATAFORMAT_SUBTYPE_IEEE_FLOAT;

        return S_OK;
    }

    return E_INVALIDARG;
}

}

namespace {

//
// Generic audio source.
//
class AudioSource
{
public:
    virtual HRESULT GetStreamStatus() = 0;
    virtual WAVEFORMATEX *GetWaveFormat() = 0;
    virtual BOOL MoreDataAvailable() = 0;
    virtual HRESULT LoadDataBytes(BYTE *pDataBuffer, ULONG NumBytes) = 0;
    virtual HRESULT ResetDataPosition() = 0;
};

//
// Sine wave generator.
//
class SineWaveSource : public AudioSource
{
    HRESULT m_hrStreamStatus;
    WAVEFORMATEXTENSIBLE m_wfx;
    BYTE *m_pBuf;
    BYTE *m_pBufEnd;
    BYTE *m_pBufPos;

public:
    SineWaveSource(WAVEFORMATEX *pFormat);
    ~SineWaveSource() {};
    HRESULT GetStreamStatus() { return m_hrStreamStatus; };
    WAVEFORMATEX *GetWaveFormat() { return &m_wfx.Format; };
    BOOL MoreDataAvailable() { return (m_hrStreamStatus == S_OK); };
    HRESULT LoadDataBytes(BYTE *pDataBuffer, ULONG NumBytes);
    HRESULT ResetDataPosition();
};


//
// Wave file reader.
//
class WaveFileReader : public AudioSource
{
    FILE *m_pFile;
    HRESULT m_hrStreamStatus;
    fpos_t m_dataChunkPosition;
    ULONG m_totalDataBytes;
    ULONG m_dataBytesRemaining;
    WAVEFORMATEXTENSIBLE m_wfx;
    BOOL m_repeatMode;

public:
    WaveFileReader(LPCWSTR pszFileName, BOOL RepeatMode);
    ~WaveFileReader();
    HRESULT GetStreamStatus() { return m_hrStreamStatus; };
    WAVEFORMATEX *GetWaveFormat()
    {
        return (m_hrStreamStatus==S_OK) ? &m_wfx.Format : NULL;
    };
    BOOL MoreDataAvailable()
    {
        return (m_hrStreamStatus == S_OK &&
                (m_repeatMode == TRUE || m_dataBytesRemaining > 0));
    };
    HRESULT LoadDataBytes(BYTE *pDataBuffer, ULONG NumBytes);
    HRESULT ResetDataPosition();
};

// To parse a wave file, it is helpful to have two user-defined types.
// The first will receive all the information in the file header and
// in the header of the format chunk. The second will receive the
// information in the headers for the data chunks, and, if the file
// contains additional chunks, the headers for those chunks as well.

typedef struct
{
    ULONG Riff4CC;      // "RIFF" 4-character code
    ULONG FileSize;     // total file size in bytes
    ULONG Wave4CC;      // "WAVE" 4-character code
    ULONG Fmt4CC;       // "fmt " 4-character code
    ULONG FormatSize;   // wave format size in bytes
} FileHeader;

typedef struct
{
    ULONG ChunkType;
    ULONG ChunkSize;
} ChunkHeader;

// Any file smaller than this cannot possibly contain wave data.
#define MIN_WAVE_FILE_SIZE (sizeof(FileHeader)+sizeof(PCMWAVEFORMAT)+sizeof(ChunkHeader)+1)

// Macro to build FOURCC from first four characters in ASCII string
#define FOURCC(s)  ((ULONG)(s[0] | (s[1]<<8) | (s[2]<<16) | (s[3]<<24)))

//
// Constructor -- Open wave file and parse file header.
//
WaveFileReader::WaveFileReader(LPCWSTR pszFileName, BOOL repeat)
{
    m_pFile = NULL;
    m_hrStreamStatus = S_OK;
    m_dataChunkPosition = 0;
    m_totalDataBytes = 0;
    m_dataBytesRemaining = 0;
    m_repeatMode = repeat;
    ZeroMemory(&m_wfx, sizeof(m_wfx));

    // Try to open the wave file.
    if (_wfopen_s(&m_pFile, pszFileName, L"rb") != 0)
    {
        m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;  // failed to open wave file
        return;
    }

    // Copy header from wave file.
    FileHeader fileHdr;

    if (fread(&fileHdr, sizeof(fileHdr), 1, m_pFile) != 1)
    {
        m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;  // not a valid wave file
        return;
    }

    // Verify that wave file header is valid.
    if (fileHdr.Riff4CC != FOURCC("RIFF") ||
        fileHdr.FileSize < MIN_WAVE_FILE_SIZE ||
        fileHdr.Wave4CC != FOURCC("WAVE") ||
        fileHdr.Fmt4CC != FOURCC("fmt ") ||
        fileHdr.FormatSize < sizeof(PCMWAVEFORMAT))
    {
        m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;  // not a valid wave file
        return;
    }

    // Copy wave format descriptor from file.
    if (fread(&m_wfx, min(fileHdr.FormatSize,sizeof(m_wfx)), 1, m_pFile) != 1)
    {
        m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;  // not a valid wave file
        return;
    }

    // Skip over any padding at the end of the format in the format chunk.
    if (fileHdr.FormatSize > sizeof(m_wfx))
    {
        if (fseek(m_pFile, fileHdr.FormatSize-sizeof(m_wfx), SEEK_CUR) != 0)
        {
            // m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;
            return;  // not a valid wave file
        }
    }

    // If format type is PCMWAVEFORMAT, convert to valid WAVEFORMATEX structure.
    if (m_wfx.Format.wFormatTag == WAVE_FORMAT_PCM)
    {
        m_wfx.Format.cbSize = 0;
    }

    // If format type is WAVEFORMATEX, convert to WAVEFORMATEXTENSIBLE.
    if (m_wfx.Format.wFormatTag == WAVE_FORMAT_PCM ||
        m_wfx.Format.wFormatTag == WAVE_FORMAT_IEEE_FLOAT)
    {
        if (m_wfx.Format.wFormatTag == WAVE_FORMAT_PCM)
        {
            m_wfx.SubFormat = KSDATAFORMAT_SUBTYPE_PCM;
        }
        else
        {
            m_wfx.SubFormat = KSDATAFORMAT_SUBTYPE_IEEE_FLOAT;
        }
        m_wfx.Format.wFormatTag = WAVE_FORMAT_EXTENSIBLE;

        // Note that the WAVEFORMATEX structure is valid for
        // representing wave formats with only 1 or 2 channels.
        if (m_wfx.Format.nChannels == 1)
        {
            m_wfx.dwChannelMask = SPEAKER_FRONT_CENTER;
        }
        else if (m_wfx.Format.nChannels == 2)
        {
            m_wfx.dwChannelMask = SPEAKER_FRONT_LEFT | SPEAKER_FRONT_RIGHT;
        }
        else
        {
            m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;  // not a valid wave format
            return;
        }
        m_wfx.Format.cbSize = sizeof(WAVEFORMATEXTENSIBLE) - sizeof(WAVEFORMATEX);
        m_wfx.Samples.wValidBitsPerSample = m_wfx.Format.wBitsPerSample;
    }

    // This wave file reader understands only PCM and IEEE float formats.
    if (m_wfx.Format.wFormatTag != WAVE_FORMAT_EXTENSIBLE ||
        m_wfx.SubFormat != KSDATAFORMAT_SUBTYPE_PCM &&
        m_wfx.SubFormat != KSDATAFORMAT_SUBTYPE_IEEE_FLOAT)
    {
        m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;  // not a format we can handle
        return;
    }

    // Find chunk header for wave data. Skip past any other kinds of chunks.
    ChunkHeader chunkHdr;   // buffer for chunk header
    for (;;)
    {
        // Remember the file position of the data chunk (which this might be
        // -- we don't know yet). That way, if we need to play the file more
        // than once, we won't have to parse the header again each time.
        if (fgetpos(m_pFile, &m_dataChunkPosition) != 0)
        {
            m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;  // not a valid wave format
            return;
        }

        // Read header at start of next chunk of file.
        if (fread(&chunkHdr, sizeof(ChunkHeader), 1, m_pFile) != 1)
        {
            m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;  // not a valid wave file
            return;
        }
        if (chunkHdr.ChunkType == FOURCC("data"))
        {
            break;  // found start of data chunk
        }
        // This is *not* a data chunk. Skip this chunk and go to the next chunk.
        if (fseek(m_pFile, chunkHdr.ChunkSize, SEEK_CUR) != 0)
        {
            m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;  // not a valid wave file
            return;
        }
    }

    // We've found the start of the data chunk. We're ready to start
    // playing wave data...
    m_totalDataBytes = chunkHdr.ChunkSize;
    m_dataBytesRemaining = m_totalDataBytes;
    if (m_totalDataBytes == 0)
    {
        m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;
    }
}

//
// Destructor
//
WaveFileReader::~WaveFileReader()
{
    if (m_pFile)
    {
        fclose(m_pFile);
    }
}

//
// Reset the file pointer to the start of the wave data.
//
HRESULT WaveFileReader::ResetDataPosition()
{
    if (m_hrStreamStatus != S_OK)
    {
        return m_hrStreamStatus;  // oops -- can't read from this file
    }

    // Move to the header info at the start of the data chunk.
    if (fsetpos(m_pFile, &m_dataChunkPosition) != 0)
    {
        m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;  // not a valid wave file
        return AUDCLNT_E_UNSUPPORTED_FORMAT;
    }

    // Read the header for the data chunk.
    ChunkHeader chunkHdr;
    if (fread(&chunkHdr, sizeof(chunkHdr), 1, m_pFile) != 1)
    {
        m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;  // possible file corruption
        return AUDCLNT_E_UNSUPPORTED_FORMAT;
    }

    // Sanity check: The chunk header shouldn't have changed.
    if (chunkHdr.ChunkType != FOURCC("data") ||
        chunkHdr.ChunkSize != m_totalDataBytes)
    {
        m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;  // possible file corruption
        return AUDCLNT_E_UNSUPPORTED_FORMAT;
    }
    m_dataBytesRemaining = m_totalDataBytes;

    // At this point, the file pointer is positioned at
    // the beginning of the wave data.
    return S_OK;
}

//
// Load next block of wave data from file into playback buffer.
// In repeat mode, when we reach the end of the wave data in the
// file, we just reset the file pointer back to the start of the
// data and continue filling the caller's buffer until it is full.
// In single-play mode, once we reach the end of the wave data in
// the file, we just fill the buffer with silence instead of with
// real data.
//
HRESULT WaveFileReader::LoadDataBytes(BYTE *pBuffer, ULONG NumBytes)
{
    if (m_hrStreamStatus != S_OK)
    {
        return m_hrStreamStatus;
    }
    if (pBuffer == NULL)
    {
        return E_POINTER;
    }
    if (NumBytes == 0)
    {
        return E_INVALIDARG;
    }

    BYTE *pCurrent = (BYTE*)pBuffer;
    ULONG numBytesToCopy = NumBytes;

    while (numBytesToCopy > m_dataBytesRemaining)
    {
        if (fread(pCurrent, 1, m_dataBytesRemaining, m_pFile) != m_dataBytesRemaining)
        {
            m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;  // not a valid wave file
            return AUDCLNT_E_UNSUPPORTED_FORMAT;
        }
        pCurrent += m_dataBytesRemaining;
        numBytesToCopy -= m_dataBytesRemaining;
        m_dataBytesRemaining = 0;

        // The file pointer now sits at the end of the data chunk.
        // Are we operating in repeat mode?
        if (m_repeatMode == FALSE)
        {
            // Nope, we're operating in single-play mode. Fill
            // the rest of the buffer with silence and return.
            BYTE silence = (m_wfx.Format.wBitsPerSample==8) ? 0x80 : 0;
            memset(pCurrent, silence, numBytesToCopy);
            return S_OK;  // yup, we're done
        }
        // Yes, we're operating in repeat mode, so loop back to
        // the start of the wave data in the file's data chunk
        // and continue loading data into the caller's buffer.
        if (ResetDataPosition() != S_OK)
        {
            return m_hrStreamStatus;  // not a valid wave file
        }
    }

    assert(numBytesToCopy > 0);
    assert(numBytesToCopy <= m_dataBytesRemaining);

    // The remainder of the data chunk is big enough to
    // completely fill the remainder of the caller's buffer.
    if (fread(pBuffer, 1, numBytesToCopy, m_pFile) != numBytesToCopy)
    {
        m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;  // not a valid wave file
        return AUDCLNT_E_UNSUPPORTED_FORMAT;
    }
    m_dataBytesRemaining -= numBytesToCopy;
    pCurrent += numBytesToCopy;

    return S_OK;
}

// Audio sample representation
typedef enum
{
    eUnknownType = 0,
    eIeeeFloat,  // IEEE float
    e8BitPcm,    // 8-bit PCM
    e16BitPcm,   // 16-bit PCM
    e32BitPcm    // 32-bit PCM

} ESampleType;

// Celebrity constant
#define PI  3.14159265358979323846

// Amplitude of generated sine wave
#define SINE_WAVE_AMPLITUDE  0.7L

// Middle C frequency (in Hz)
#define MIDDLE_C_FREQ  261.63L

//
// Constructor -- Open wave file and parse file header.
//
SineWaveSource::SineWaveSource(WAVEFORMATEX *pFormat) :
                  m_hrStreamStatus(E_FAIL),
                  m_pBuf(NULL),
                  m_pBufEnd(NULL),
                  m_pBufPos(NULL)
{
    ZeroMemory(&m_wfx, sizeof(WAVEFORMATEXTENSIBLE));
    if (pFormat == NULL)
    {
        m_hrStreamStatus = E_INVALIDARG;
        return;
    }
    if (pFormat->wFormatTag == WAVE_FORMAT_EXTENSIBLE)
    {
        memcpy(&m_wfx, pFormat, sizeof(WAVEFORMATEXTENSIBLE));
    }
    else
    {
        // Convert WAVEFORMATEX format to WAVEFORMATEXTENSIBLE.
        memcpy(&m_wfx, pFormat, sizeof(WAVEFORMATEX)-sizeof(m_wfx.Format.cbSize));
		m_hrStreamStatus = sf::ConvertToExtensible(&m_wfx.Format);
        if (m_hrStreamStatus != S_OK)
        {
            return;
        }
    }

    if (m_wfx.SubFormat != KSDATAFORMAT_SUBTYPE_IEEE_FLOAT &&
        m_wfx.SubFormat != KSDATAFORMAT_SUBTYPE_PCM)
    {
        // Can't handle the specified format.
        m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;
        return;
    }

    int nFrameSize = 0;
    ESampleType sampleType = eUnknownType;

    if (m_wfx.SubFormat == KSDATAFORMAT_SUBTYPE_IEEE_FLOAT)
    {
        nFrameSize = m_wfx.Format.nChannels * sizeof(float);
        sampleType = eIeeeFloat;
    }
    else
    {
        switch (m_wfx.Format.wBitsPerSample)
        {
        case 8:
            nFrameSize = m_wfx.Format.nChannels * sizeof(BYTE);
            sampleType = e8BitPcm;
            break;
        case 16:
            nFrameSize = m_wfx.Format.nChannels * sizeof(INT16);
            sampleType = e16BitPcm;
            break;
        case 32:
            nFrameSize = m_wfx.Format.nChannels * sizeof(INT32);
            sampleType = e32BitPcm;
            break;
        default:
            // Can't handle the specified format.
            m_hrStreamStatus = AUDCLNT_E_UNSUPPORTED_FORMAT;
            return;
        }
    }

    // This object generates a 523-Hz sine wave (one octave above
    // middle C) regardless of the sample rate of the stream.
    // Calculate the number of samples to generate for each
    // period of the sine wave.
    int nSamplesPerPeriod = (int)(m_wfx.Format.nSamplesPerSec
                                  / (2 * MIDDLE_C_FREQ));
    int nBytesPerPeriod = nFrameSize * nSamplesPerPeriod;

    // Allocate an internal buffer to hold an entire waveform.
    // Assume the caller has already called CoInitialize.
    m_pBuf = (BYTE*)CoTaskMemAlloc(nBytesPerPeriod);
    if (m_pBuf == NULL)
    {
        m_hrStreamStatus = E_OUTOFMEMORY;
        return;
    }
    m_pBufPos = m_pBuf;  // initial position = start of buffer
    m_pBufEnd = m_pBuf + nBytesPerPeriod;  // 1st byte past end of buffer

    // Calculate the number of radians between successive samples.
    const double dt = 2 * PI / nSamplesPerPeriod;
    const double sindt = sin(dt);  // sine of angular increment
    const double cosdt = cos(dt);  // cosine of angular increment

    // Use the following formulas to incrementally calculate successive
    // sine wave samples:
    //           x[n] = x[n-1] * cos(dt) - y[n-1] * sin(dt)
    //           y[n] = x[n-1] * sin(dt) + y[n-1] * cos(dt)
    //
    // Use successive values of y as the generated sine wave.
    // Generate enough samples for one complete waveform.
    double x = SINE_WAVE_AMPLITUDE;  // initial value
    double y = 0;

    switch (sampleType)
    {
    case eIeeeFloat:
        {
            float *pData = (float*)m_pBuf;
            for (int i = 0; i < nSamplesPerPeriod; i++)
            {
                for (int j = 0; j < m_wfx.Format.nChannels; j++)
                {
                    *pData++ = (float)y;
                }
                double tmp = x * cosdt - y * sindt;
                y = x * sindt + y * cosdt;
                x = tmp;
            }
        }
        break;
    case e8BitPcm:
        {
            BYTE *pData = (BYTE*)m_pBuf;
            for (int i = 0; i < nSamplesPerPeriod; i++)
            {
                for (int j = 0; j < m_wfx.Format.nChannels; j++)
                {
                    *pData++ = BYTE(0x7F * y + 0x80);  // unsigned 8-bit
                }
                double tmp = x * cosdt - y * sindt;
                y = x * sindt + y * cosdt;
                x = tmp;
            }
        }
        break;
    case e16BitPcm:
        {
            INT16 *pData = (INT16*)m_pBuf;
            for (int i = 0; i < nSamplesPerPeriod; i++)
            {
                for (int j = 0; j < m_wfx.Format.nChannels; j++)
                {
                    *pData++ = INT16(0x7FFF * y);  // signed 16-bit
                }
                double tmp = x * cosdt - y * sindt;
                y = x * sindt + y * cosdt;
                x = tmp;
            }
        }
        break;
    case e32BitPcm:
        {
            INT32 *pData = (INT32*)m_pBuf;
            for (int i = 0; i < nSamplesPerPeriod; i++)
            {
                for (int j = 0; j < m_wfx.Format.nChannels; j++)
                {
                    *pData++ = INT32(0x7FFFFFFF * y);  // signed 32-bit
                }
                double tmp = x * cosdt - y * sindt;
                y = x * sindt + y * cosdt;
                x = tmp;
            }
        }
        break;
    default:
        assert(0);
        m_hrStreamStatus = E_FAIL;
        return;
    }
    m_hrStreamStatus = S_OK;
}

//
// Reset the stream.
//
HRESULT SineWaveSource::ResetDataPosition()
{
    if (m_hrStreamStatus != S_OK)
    {
        return m_hrStreamStatus;  // oops -- can't handle this format
    }

    // Reset position to start of waveform.
    m_pBufPos = m_pBuf;

    return S_OK;
}

///
// Load next block of wave data from file.
//
HRESULT SineWaveSource::LoadDataBytes(BYTE *pBuffer, ULONG NumBytes)
{
    if (m_hrStreamStatus != S_OK)
    {
        return m_hrStreamStatus;  // oops -- can't handle this format
    }
    if (pBuffer == NULL)
    {
        return E_POINTER;
    }
    if (NumBytes == 0)
    {
        return E_INVALIDARG;
    }
    while (NumBytes != 0)
    {
        ULONG count = (ULONG)(m_pBufEnd - m_pBufPos);
        if (NumBytes < count)
        {
            count = NumBytes;
        }
        memcpy(pBuffer, m_pBufPos, count);
        NumBytes -= count;
        pBuffer += count;
        m_pBufPos += count;
        if (m_pBufPos == m_pBufEnd)
        {
            m_pBufPos = m_pBuf;
        }
    }
    return S_OK;
}

}


namespace sf {
// Event context GUID -- call CoCreateGuid to initialize.
GUID g_EventContext = GUID_NULL;


// Audio endpoint device ID strings
//LPWSTR g_pszRenderDeviceID = NULL;
//LPWSTR g_pszCaptureDeviceID = NULL;

struct WAVEFORMATEX : public ::WAVEFORMATEX {
	WAVEFORMATEX(
		WORD        FormatTag = 0,         /* format type */
		WORD        Channels = 0,          /* number of channels (i.e. mono, stereo...) */
		DWORD       SamplesPerSec = 0,     /* sample rate */
		DWORD       AvgBytesPerSec = 0,    /* for buffer estimation */
		WORD        BlockAlign = 0,        /* block size of data */
		WORD        BitsPerSample = 0,     /* number of bits per sample of mono data */
		WORD        Size = sizeof(::WAVEFORMATEX)            /* the count in bytes of the size of */
		)  
		{
			wFormatTag = FormatTag;
			nChannels = Channels;
			nSamplesPerSec = SamplesPerSec;
			nAvgBytesPerSec = AvgBytesPerSec;
			nBlockAlign = BlockAlign;
			wBitsPerSample = BitsPerSample;
			cbSize = Size;			
		};
};


template <typename COMInterface> 
struct IUnknownImpl : public COMInterface 
{
	IUnknownImpl() : ref_(1) {}; 
	virtual ~IUnknownImpl() {};
	ULONG __stdcall AddRef()
	{
		return InterlockedIncrement(&ref_);
	}

	ULONG __stdcall Release()
	{
		ULONG ref = InterlockedDecrement(&ref_);
		if (0 == ref)
		{
			delete this;
		}
		return ref;
	}

	HRESULT __stdcall QueryInterface(REFIID riid, VOID **ppObj)
	{
		if (IID_IUnknown == riid)
		{
			AddRef();
			*ppObj = (IUnknown*)this;
		}
		else if (__uuidof(COMInterface) == riid)
		{
			AddRef();
			*ppObj = (COMInterface*)this;
		}
		else
		{
			*ppObj = NULL;
			return E_NOINTERFACE;
		}
		return S_OK;
	}
private:
	LONG ref_;
};


// policy class
struct HeapMemoryFreePolicy
{
template< typename T >
void operator()( const T* AMemory ) const
{
if( NULL != AMemory )
::HeapFree( ::GetProcessHeap(), 0, AMemory );
}
};
// policy class
struct LocalMemoryFreePolicy
{
template< typename T >
void operator()( const T* AMemory ) const
{
if( NULL != AMemory )
::LocalFree( AMemory );
}
};
// policy class
struct CoTaskMemoryFreePolicy
{
template< typename T >
void operator()( const T* AMemory ) const
{
if( NULL != AMemory )
::CoTaskMemFree( AMemory );
}
};
// base guard class
template< typename T,class TFreePolicy >
class BaseMemory
{
private:
T *FMemory;

public:
BaseMemory( T* AMemory = NULL )
: FMemory( AMemory ) {}

virtual ~BaseMemory( void )
{ Reset(); }

T* Release( void )
{
T *tmp = FMemory;
FMemory = NULL;
return tmp;
}

void Reset( T* AMemory = NULL )
{
if( AMemory != FMemory )
{
if( NULL != FMemory )
TFreePolicy( FMemory );

FMemory = AMemory;
}
}

T* Get( void )
{ return FMemory; }

T** operator&( void )
{ return &FMemory; }

};
template< typename T >
class HeapMemory : public BaseMemory< T, 
HeapMemoryFreePolicy >
{
public:
HeapMemory( T* AMemory = NULL )
: BaseMemory< T, HeapMemoryFreePolicy >( AMemory )
{ }
};
template< typename T >
class LocalMemory : public BaseMemory< T, 
LocalMemoryFreePolicy >
{
public:
LocalMemory( T* AMemory = NULL )
: BaseMemory< T, LocalMemoryFreePolicy >( AMemory )
{ }
};
template< typename T >
class CoTaskMemory : public BaseMemory< T, 
CoTaskMemoryFreePolicy >
{
public:
CoTaskMemory( T* AMemory = NULL )
: BaseMemory< T, CoTaskMemoryFreePolicy >( AMemory )
{ }
};

struct PROPVARIANT : public ::PROPVARIANT
{
	PROPVARIANT() {PropVariantInit(this);}
	~PROPVARIANT() {PropVariantClear(this);} 
};




_COM_SMARTPTR_TYPEDEF(IMMDeviceEnumerator,__uuidof(IMMDeviceEnumerator));
_COM_SMARTPTR_TYPEDEF(IMMDeviceCollection,__uuidof(IMMDeviceCollection));
_COM_SMARTPTR_TYPEDEF(IMMDevice,__uuidof(IMMDevice));
_COM_SMARTPTR_TYPEDEF(IAudioClient,__uuidof(IAudioClient));
_COM_SMARTPTR_TYPEDEF(IAudioRenderClient,__uuidof(IAudioRenderClient));
_COM_SMARTPTR_TYPEDEF(IAudioCaptureClient,__uuidof(IAudioCaptureClient));

_COM_SMARTPTR_TYPEDEF(IAudioSessionControl,__uuidof(IAudioSessionControl));
_COM_SMARTPTR_TYPEDEF(IAudioSessionEvents,__uuidof(IAudioSessionEvents));
_COM_SMARTPTR_TYPEDEF(IAudioEndpointVolume,__uuidof(IAudioEndpointVolume));
_COM_SMARTPTR_TYPEDEF(IAudioEndpointVolumeCallback,__uuidof(IAudioEndpointVolumeCallback));
_COM_SMARTPTR_TYPEDEF(IAudioSessionManager,__uuidof(IAudioSessionManager));
_COM_SMARTPTR_TYPEDEF(ISimpleAudioVolume,__uuidof(ISimpleAudioVolume));
_COM_SMARTPTR_TYPEDEF(IPropertyStore,__uuidof(IPropertyStore));


using namespace boost;
using namespace boost::assign;

//typedef CComPtr<IAudioClient> IAudioClientPtr;
std::map<HRESULT,std::wstring> com_error_  = boost::assign::list_of<std::pair<HRESULT,std::wstring> >
    (E_POINTER,L"E_POINTER")
    (E_INVALIDARG,L"E_INVALIDARG")
	(AUDCLNT_E_NOT_INITIALIZED,L"AUDCLNT_E_NOT_INITIALIZED")
	(AUDCLNT_E_ALREADY_INITIALIZED,L"AUDCLNT_E_ALREADY_INITIALIZED")
    (AUDCLNT_E_WRONG_ENDPOINT_TYPE,L"AUDCLNT_E_WRONG_ENDPOINT_TYPE")
	(AUDCLNT_E_DEVICE_INVALIDATED,L"AUDCLNT_E_DEVICE_INVALIDATED")
	(AUDCLNT_E_NOT_STOPPED,L"AUDCLNT_E_NOT_STOPPED")
	(AUDCLNT_E_BUFFER_TOO_LARGE,L"AUDCLNT_E_BUFFER_TOO_LARGE")
	(AUDCLNT_E_OUT_OF_ORDER,L"AUDCLNT_E_OUT_OF_ORDER")
	(AUDCLNT_E_UNSUPPORTED_FORMAT,L"AUDCLNT_E_UNSUPPORTED_FORMAT")
	(AUDCLNT_E_INVALID_SIZE,L"AUDCLNT_E_INVALID_SIZE")
	(AUDCLNT_E_DEVICE_IN_USE,L"AUDCLNT_E_DEVICE_IN_USE")
	(AUDCLNT_E_BUFFER_OPERATION_PENDING,L"AUDCLNT_E_BUFFER_OPERATION_PENDING")
	(AUDCLNT_E_THREAD_NOT_REGISTERED,L"AUDCLNT_E_THREAD_NOT_REGISTERED")
	(AUDCLNT_E_EXCLUSIVE_MODE_NOT_ALLOWED,L"AUDCLNT_E_EXCLUSIVE_MODE_NOT_ALLOWED")
	(AUDCLNT_E_ENDPOINT_CREATE_FAILED,L"AUDCLNT_E_ENDPOINT_CREATE_FAILED")
	(AUDCLNT_E_SERVICE_NOT_RUNNING,L"AUDCLNT_E_SERVICE_NOT_RUNNING")
	(AUDCLNT_E_EVENTHANDLE_NOT_EXPECTED,L"AUDCLNT_E_EVENTHANDLE_NOT_EXPECTED")
	(AUDCLNT_E_EXCLUSIVE_MODE_ONLY,L"AUDCLNT_E_EXCLUSIVE_MODE_ONLY")
	(AUDCLNT_E_BUFDURATION_PERIOD_NOT_EQUAL,L"AUDCLNT_E_BUFDURATION_PERIOD_NOT_EQUAL")
	(AUDCLNT_E_EVENTHANDLE_NOT_SET,L"AUDCLNT_E_EVENTHANDLE_NOT_SET")
	(AUDCLNT_E_INCORRECT_BUFFER_SIZE,L"AUDCLNT_E_INCORRECT_BUFFER_SIZE")
	(AUDCLNT_E_BUFFER_SIZE_ERROR,L"AUDCLNT_E_BUFFER_SIZE_ERROR")
	(AUDCLNT_S_BUFFER_EMPTY,L"AUDCLNT_S_BUFFER_EMPTY")
	(AUDCLNT_S_THREAD_ALREADY_REGISTERED,L"AUDCLNT_S_THREAD_ALREADY_REGISTERED");
	

Win32ErrorException::Win32ErrorException(boost::uint32_t hr)
: std::exception("HRESULT ERROR"),hresult_(hr)
{
	LocalMemory<wchar_t> mem;
	DWORD result = FormatMessageW(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,0,hr,0,(LPWSTR)&mem,0,0);
	if(result != 0){
		error_ = mem.Get();
		
		DEBUGPRINTF(boost::wformat(L"COM Error %x %s") % hr % mem.Get() );
	} else {
		std::map<HRESULT,std::wstring>::iterator it = com_error_.find(hr);
		if(it != com_error_.end())
		{
			error_ = it->second;
			DEBUGPRINTF(boost::wformat(L"COM Error %x %s") % hr % error_.c_str());
		} else {
			error_ = (boost::wformat(L"0x%x sCOMG[") % hr).str();
			DEBUGPRINTF(boost::wformat(L"%s") % error_);
		}

	}
};

Win32ErrorException::Win32ErrorException()
{
	hresult_ = ::GetLastError();
	LocalMemory<wchar_t> mem;
	DWORD rv =  FormatMessageW(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,0,hresult_,0,(LPWSTR)&mem,0,0);
	error_ = mem.Get();
	DEBUGPRINTF(boost::wformat(L"Win32 Error %x %s") % hresult_ % mem.Get() );
};


struct WASAPIDeviceInfos : public sf::Singleton<WASAPIDeviceInfos>,public IUnknownImpl<IMMNotificationClient>
{

	boost::signal<void (juce::String&)>  onDeviceAdded;
	boost::signal<void (juce::String&)>  onDeviceRemoved;

	WASAPIDeviceInfos():  hasScanned(false),IUnknownImpl<IMMNotificationClient>()
	{
		CoCreateGuid(&g_EventContext);
		EXCEPTION_ON_ERROR(enumerator_.CreateInstance(__uuidof(MMDeviceEnumerator)));

		//CoCreateInstance( __uuidof(MMDeviceEnumerator), NULL,
		//		 CLSCTX_ALL, __uuidof(IMMDeviceEnumerator),
		//		 (void**)&enumerator_);
		EXCEPTION_ON_ERROR(enumerator_->RegisterEndpointNotificationCallback(this));
		scanForDevices();
	}

    void scanForDevices()
    {
		HRESULT hr = enumerator_->EnumAudioEndpoints(::eRender, DEVICE_STATE_ACTIVE,&renderCollection_);
		assert(hr == S_OK);


		hr = enumerator_->EnumAudioEndpoints(::eCapture, DEVICE_STATE_ACTIVE,&captureCollection_);
		assert(hr == S_OK);
        hasScanned = true;
		
    }

	~WASAPIDeviceInfos()
	{
		enumerator_->UnregisterEndpointNotificationCallback(this);
		if(renderCollection_)
		{	renderCollection_.Release();
		}

		if(captureCollection_){
			captureCollection_.Release();
		}

		if(enumerator_){
			enumerator_.Release();
		}
	}
	//
	// Local implementation of IMMNotificationClient interface.
	// When the status of an audio endpoint device changes, the
	// MMDevice API calls one of these methods to notify the
	// client.
	//
	HRESULT __stdcall OnDefaultDeviceChanged(::EDataFlow flow, ::ERole role, LPCWSTR pwstrDeviceId)
	{
		return S_OK;
	}

	HRESULT __stdcall OnDeviceAdded(LPCWSTR pwstrDeviceId)
	{
		onDeviceAdded(juce::String(pwstrDeviceId));
		return S_OK;
	};

	HRESULT __stdcall OnDeviceRemoved(LPCWSTR pwstrDeviceId)
	{
		onDeviceRemoved(juce::String(pwstrDeviceId));
		//if (NULL != pPlayer_->pPlayerCallbacks_)
		//{
		//	if (0 == wcsncmp(pwstrDeviceId, g_pszRenderDeviceID, MAX_CHARS))
		//	{
		//		pPlayer_->pPlayerCallbacks_->RenderDisconnectCallback();
		//	}
		//	else if (0 == wcsncmp(pwstrDeviceId, g_pszCaptureDeviceID, MAX_CHARS))
		//	{
		//		pPlayer_->pPlayerCallbacks_->CaptureDisconnectCallback();
		//	}
		//}
		return S_OK;
	}

	HRESULT __stdcall OnDeviceStateChanged(LPCWSTR pwstrDeviceId, DWORD dwNewState)
	{
		//if (DEVICE_STATE_ACTIVE != dwNewState && NULL != pPlayer_->pPlayerCallbacks_)
		//{
		//	if (0 == wcsncmp(pwstrDeviceId, g_pszRenderDeviceID, MAX_CHARS))
		//	{
		//		pPlayer_->pPlayerCallbacks_->RenderDisconnectCallback();
		//	}
		//	else if (0 == wcsncmp(pwstrDeviceId, g_pszCaptureDeviceID, MAX_CHARS))
		//	{
		//		pPlayer_->pPlayerCallbacks_->CaptureDisconnectCallback();
		//	}
		//}
		return S_OK;
	}

	HRESULT __stdcall OnPropertyValueChanged(LPCWSTR pwstrDeviceId, const PROPERTYKEY key)
	{
		return S_OK;
	}

	IMMDeviceEnumeratorPtr& getEnumerator() {return enumerator_;};
	IMMDeviceCollectionPtr& getRenderCollection() { return renderCollection_;}
	IMMDeviceCollectionPtr& getCaptureCollection() { return captureCollection_;}

private:
	bool hasScanned;
	IMMDeviceEnumeratorPtr enumerator_;
    IMMDeviceCollectionPtr renderCollection_;
    IMMDeviceCollectionPtr captureCollection_;
	juce::String currentRenderDeviceID_;
	juce::String currentCaptureDeviceID_;
};


struct WASAPI::impl : public WASAPIInterface
{
	impl(int outputDeviceIndex,const String& outputDeviceId,int inputDeviceIndex ,const String& inputDeviceId,juce::AudioIODevice *parent) : 
		outputDeviceIndex_(outputDeviceIndex),
			inputDeviceIndex_(inputDeviceIndex),inputDeviceId_(inputDeviceId),outputDeviceId_(outputDeviceId),hDlg_(::GetDesktopWindow()),
			exclusiveMode_(false),
			eventMode_(false),
			enableMMCSS_(true),
			hwVolumeSupported_(false),mmThreadPriority_(::AVRT_PRIORITY_CRITICAL),currentBufferSize_(576),isOpen_(false),
			inputBufferSize_(0),outputBufferSize_(0),isStart_(false),isPlaying_(false),hStopPlaying_(0),hThreadIn_(0),hThreadOut_(0),callback_(0),parent_(parent),inputChannels_(0),outputChannels_(0),actualInputChannels_(0),actualOutputChannels_(0)
	{
		bitInChannels_.clear();
		bitOutChannels_.clear();
		hStopPlaying_ = NULL;
		initialize();
	};

	~impl()
	{
		close();
		if ((NULL != renderEndpointVolume_) && (hwVolumeSupported_))
		{
			renderEndpointVolume_->UnregisterControlChangeNotify(renderEndpointVolumeCallback_);
			renderEndpointVolume_.Release();
		}

		if (NULL != renderSessionControl_)
		{
			renderSessionControl_->UnregisterAudioSessionNotification(renderSessionEvents_);
			renderSessionControl_.Release();
		}

		if (NULL != captureSessionControl_)
		{
			captureSessionControl_->UnregisterAudioSessionNotification(captureSessionEvents_);
			captureSessionControl_.Release();

		}


		if (NULL != hStopPlaying_)
		{
			CloseHandle(hStopPlaying_);
		}

		if (NULL != hThreadIn_)
		{
			CloseHandle(hThreadIn_);
		}

		if (NULL != hThreadOut_)
		{
			CloseHandle(hThreadOut_);
		}

		if(renderSessionEvents_){
			renderSessionEvents_.Release();
		}
		if(captureSessionEvents_)
			captureSessionEvents_.Release();
		if(renderEndpointVolumeCallback_)
			renderEndpointVolumeCallback_.Release();
		if(renderClient_)
			renderClient_.Release();
		if(captureClient_)
			captureClient_.Release();
		if(outputClient_)
			outputClient_.Release();
		if(inputClient_)
			inputClient_.Release();
		if(outputDevice_)
			outputDevice_.Release();
		if(inputDevice_)
			inputDevice_.Release();
	};

	const juce::StringArray getOutputChannelNames(){return outChannels_;};

    /** Returns the names of all the available input channels on this device.
        To find out which of these are currently in use, call getActiveInputChannels().
    */
	const juce::StringArray getInputChannelNames(){return inChannels_;};

    //==============================================================================
    /** Returns the number of sample-rates this device supports.

        To find out which rates are available on this device, use this method to
        find out how many there are, and getSampleRate() to get the rates.

        @see getSampleRate
    */
	int getNumSampleRates(){return support_formats_.size();};

    /** Returns one of the sample-rates this device supports.

        To find out which rates are available on this device, use getNumSampleRates() to
        find out how many there are, and getSampleRate() to get the individual rates.

        The sample rate is set by the open() method.

        (Note that for DirectSound some rates might not work, depending on combinations
        of i/o channels that are being opened).

        @see getNumSampleRates
    */
	double getSampleRate (int index){return (double)support_formats_[index].nSamplesPerSec;};

    /** Returns the number of sizes of buffer that are available.

        @see getBufferSizeSamples, getDefaultBufferSize
    */
	int getNumBufferSizesAvailable(){return latencyMap_[currentSampleRate_].size(); 0;};

    /** Returns one of the possible buffer-sizes.

        @param index    the index of the buffer-size to use, from 0 to getNumBufferSizesAvailable() - 1
        @returns a number of samples
        @see getNumBufferSizesAvailable, getDefaultBufferSize
    */
	int getBufferSizeSamples (int index){return latencyMap_[currentSampleRate_][index];}

    /** Returns the default buffer-size to use.

        @returns a number of samples
        @see getNumBufferSizesAvailable, getBufferSizeSamples
    */

	int getDefaultBufferSize(){return 1024;};

    //==============================================================================
    /** Tries to open the device ready to play.

        @param inputChannels        a BitArray in which a set bit indicates that the corresponding
                                    input channel should be enabled
        @param outputChannels       a BitArray in which a set bit indicates that the corresponding
                                    output channel should be enabled
        @param sampleRate           the sample rate to try to use - to find out which rates are
                                    available, see getNumSampleRates() and getSampleRate()
        @param bufferSizeSamples    the size of i/o buffer to use - to find out the available buffer
                                    sizes, see getNumBufferSizesAvailable() and getBufferSizeSamples()
        @returns    an error description if there's a problem, or an empty string if it succeeds in
                    opening the device
        @see close
    */
     const juce::String open (const juce::BitArray& inputChannels,
                               const juce::BitArray& outputChannels,
                               double sampleRate,
							   int bufferSizeSamples)
	 {
		 String error_msg(L"");
		 if(bufferSizeSamples > 0)
		 {

			 currentSampleRate_  = sampleRate;
			 currentBufferSize_ = bufferSizeSamples;
			 currentLatency_ = (bufferSizeSamples * 1000 / sampleRate) * 10000;

			 bitInChannels_ = inputChannels;
			 bitInChannels_.setRange(inputChannels_,bitInChannels_.getHighestBit() + 1 - inputChannels_,false);
			 actualInputChannels_ = bitInChannels_.getHighestBit() + 1;
			 
			 bitOutChannels_ = outputChannels;
			 bitOutChannels_.setRange(outputChannels_,bitOutChannels_.getHighestBit() + 1 - outputChannels_,false);
			 actualOutputChannels_ = bitOutChannels_.getHighestBit() + 1;

			 close();
		
			 try {
				 BOOST_FOREACH(WAVEFORMATEX& wave_format,wave_formats_)
				 {
					if(wave_format.nSamplesPerSec == (DWORD)sampleRate)
					{
						if(outputDevice_ && actualOutputChannels_ > 0){
							try {
								
								EXCEPTION_ON_ERROR(outputDevice_->Activate(__uuidof(IAudioClient),CLSCTX_ALL,0,reinterpret_cast<void**>(&outputClient_)));
							
								WAVEFORMATEXTENSIBLE wave_ext_out;
								wave_ext_out.Format = wave_format;
								wave_ext_out.Format.nChannels = outputChannels_;
								ConvertToExtensible(reinterpret_cast<WAVEFORMATEX*>(&wave_ext_out));
								EXCEPTION_ON_ERROR(outputClient_->Initialize(AUDCLNT_SHAREMODE_EXCLUSIVE,AUDCLNT_STREAMFLAGS_EVENTCALLBACK ,
													currentLatency_,currentLatency_,reinterpret_cast<WAVEFORMATEX*>(&wave_ext_out),NULL));
								EXCEPTION_ON_ERROR(outputClient_->GetService(__uuidof(IAudioRenderClient),reinterpret_cast<void**>(&renderClient_)));
								outputClient_->GetBufferSize((UINT32*)&outputBufferSize_);
								outputClient_->SetEventHandle(hThreadOut_);
							} catch  (sf::Win32ErrorException& e) 
							{
								error_msg = e.error().c_str();
								if(outputClient_)
								{
									outputClient_.Release();
								}
								if(renderClient_)
								{
									renderClient_.Release();
								}
							} 
						}

						if(inputDevice_ && actualInputChannels_ > 0){

							WAVEFORMATEXTENSIBLE wave_ext_in;
							wave_ext_in.Format = wave_format;
							wave_ext_in.Format.nChannels = inputChannels_;
							ConvertToExtensible(reinterpret_cast<WAVEFORMATEX*>(&wave_ext_in));

							try {
								EXCEPTION_ON_ERROR(inputDevice_->Activate(__uuidof(IAudioClient),CLSCTX_ALL,0,reinterpret_cast<void**>(&inputClient_)));
								EXCEPTION_ON_ERROR(inputClient_->Initialize(AUDCLNT_SHAREMODE_EXCLUSIVE,AUDCLNT_STREAMFLAGS_EVENTCALLBACK ,
														currentLatency_,currentLatency_,reinterpret_cast<WAVEFORMATEX*>(&wave_ext_in),NULL));
								EXCEPTION_ON_ERROR(inputClient_->GetService(__uuidof(IAudioCaptureClient),reinterpret_cast<void**>(&captureClient_)));
								inputClient_->GetBufferSize((UINT32*)&inputBufferSize_);
								inputClient_->SetEventHandle(hThreadIn_);
							} catch (sf::Win32ErrorException& e) 
							{
								error_msg = e.error().c_str();
								if(inputClient_)
								{
									inputClient_.Release();
								}
								if(captureClient_)
								{
									captureClient_.Release();
								}
							}
						}
						break;
					}
				 }
			 } catch (sf::Win32ErrorException& e)
			 {
	  			boost::mutex::scoped_lock lk(mutex_);
				isOpen_ = false;
				error_msg = e.error().c_str();
				return error_msg;
			 }
			{
				boost::mutex::scoped_lock lk(mutex_);
				isOpen_ = true;
			}
			captureAndRenderThread_ = boost::thread(boost::bind(&sf::WASAPI::impl::captureAndRender,this));
		 }
		return error_msg;
	 };

    /** Closes and releases the device if it's open. */
	 void close()
	 {
		 stop();
		 if(isOpen_)
		 {
			 ::SetEvent(hStopPlaying_);
			 captureAndRenderThread_.join();
	      	 boost::mutex::scoped_lock lk(mutex_);
			 isOpen_ = false;

			 if(renderClient_)
			 {
				 renderClient_.Release();
			 }

			 if(outputClient_)
			 {
				 outputClient_.Release();
			 }

			 if(captureClient_)
			 {
				 captureClient_.Release();
			 }

			 if(inputClient_)
			 {
				 inputClient_.Release();
			 }


		 }

		 juce::Logger::outputDebugString(L"close()܂B");
	 };

    /** Returns true if the device is still open.

        A device might spontaneously close itself if something goes wrong, so this checks if
        it's still open.
    */

	 bool isOpen(){return isOpen_;};

    /** Starts the device actually playing.

        This must be called after the device has been opened.

        @param callback     the callback to use for streaming the data.
        @see AudioIODeviceCallback, open
    */
	 void start (juce::AudioIODeviceCallback* callback)
	 {
        if (isOpen_ && callback != 0 && ! isStart_)
        {
            callback->audioDeviceAboutToStart (parent_);
			boost::mutex::scoped_lock lk(mutex_);
			callback_ = callback;
			isStart_ = true;
		}
	 };

    /** Stops the device playing.

        Once a device has been started, this will stop it. Any pending calls to the
        callback class will be flushed before this method returns.
    */
	 void stop()
	 {
        if (isStart_)
        {
            AudioIODeviceCallback* const callbackLocal = callback_;
            {
				boost::mutex::scoped_lock lk(mutex_);
                isStart_ = false;
            }
            if (callbackLocal != 0)
                callbackLocal->audioDeviceStopped();
        }
	 };

    /** Returns true if the device is still calling back.

        The device might mysteriously stop, so this checks whether it's
        still playing.
    */
	 bool isPlaying()
	 {
		 mutex::scoped_lock lk(mutex_);
		 return isPlaying_;
	 };

    /** Returns the last error that happened if anything went wrong. */
	 const juce::String getLastError(){return String();} ;

    //==============================================================================
    /** Returns the buffer size that the device is currently using.

        If the device isn't actually open, this value doesn't really mean much.
    */
     int getCurrentBufferSizeSamples(){return jmax(outputBufferSize_,inputBufferSize_);} ;

    /** Returns the sample rate that the device is currently using.

        If the device isn't actually open, this value doesn't really mean much.
    */
     double getCurrentSampleRate() {return currentSampleRate_;};

    /** Returns the device's current physical bit-depth.

        If the device isn't actually open, this value doesn't really mean much.
    */
     int getCurrentBitDepth(){return currentBitDepth_;} ;

    /** Returns a mask showing which of the available output channels are currently
        enabled.
        @see getOutputChannelNames
    */
	 const juce::BitArray getActiveOutputChannels() const {return bitOutChannels_;};

    /** Returns a mask showing which of the available input channels are currently
        enabled.
        @see getInputChannelNames
    */
     const juce::BitArray getActiveInputChannels() const {return bitInChannels_;};

    /** Returns the device's output latency.

        This is the delay in samples between a callback getting a block of data, and
        that data actually getting played.
    */
     int getOutputLatencyInSamples() {return outputBufferSize_;};

    /** Returns the device's input latency.

        This is the delay in samples between some audio actually arriving at the soundcard,
        and the callback getting passed this block of data.
    */
     int getInputLatencyInSamples() {return inputBufferSize_;};


    //==============================================================================
    /** True if this device can show a pop-up control panel for editing its settings.

        This is generally just true of ASIO devices. If true, you can call showControlPanel()
        to display it.
    */
	 bool hasControlPanel() const {return true;};

    /** Shows a device-specific control panel if there is one.

        This should only be called for devices which return true from hasControlPanel().
    */
    bool showControlPanel()
	{
		boost::scoped_ptr<WASAPIControlPanel> ptr(new WASAPIControlPanel(this));
		juce::DialogWindow::showModalDialog
			(L"WASPAI",ptr.get(),0,juce::Colour(0xff8080ff),true);
		return false;
	};

	int getOutputDeviceIndex(){return outputDeviceIndex_;};
	int getInputDeviceIndex(){return inputDeviceIndex_;};

	virtual void setExclusiveMode(bool enable){exclusiveMode_ = enable;};
	virtual const bool getExclusiveMode() const {return exclusiveMode_;};
	virtual const int getDefaultPeriod() const
	{
		return (int)(jmax(outputDefaultPeriod_,inputDefaultPeriod_) / 10000);
	};

	virtual const int getMinPeriod() const 
	{
		return (int)(jmax(outputMinPeriod_,inputMinPeriod_) / 10000);
	}

    virtual void setBufferLatency(int bufferLatency)
	{
		currentLatency_ = ((REFERENCE_TIME)bufferLatency) * 10000;
	}

	virtual const int getBufferLatency() const { return (int)(currentLatency_ / 10000);}

	virtual void enableMMCSS(bool enable) {enableMMCSS_ = enable;}
	virtual const bool isMMCSSEnabled() const {return enableMMCSS_;};
	virtual const bool isHwVolumeSupported() const {return hwVolumeSupported_;}
	virtual void setMmThreadPriority(AVRT_PRIORITY priority) {mmThreadPriority_ = (::AVRT_PRIORITY)priority;}
	virtual const AVRT_PRIORITY getMmThreadPriority() const {return (sf::AVRT_PRIORITY)mmThreadPriority_;}
	virtual void setVolume(float volume) {currentVolume_ = volume;};
	virtual const float getVolume() const {return currentVolume_;};

	void captureAndRender()
	{

		com_initialize com_init(0,sf::multi_threaded);

		HANDLE events[3] =
		{
				hStopPlaying_,	// ~Cxg𔭍sCxg
			    hThreadOut_,	// o̓Cxg
				hThreadIn_	// ̓Cxg
		};

        int numEvents = 3;
		DWORD hr(E_FAIL);

		// The playback thread will exit when Stop() sets this event.
		ResetEvent(hStopPlaying_);

		HANDLE hMmTask(0);
		DWORD taskIndex(0);
		if(enableMMCSS_)
		{
	        // Register this thread with MMCSS.
		    hMmTask = AvSetMmThreadCharacteristics(L"Pro Audio", &taskIndex);
		   // Set MMCSS thread priority.
		    AvSetMmThreadPriority(hMmTask, mmThreadPriority_);
		}

        const int numInputBuffers = getActiveInputChannels().countNumberOfSetBits();
        const int numOutputBuffers = getActiveOutputChannels().countNumberOfSetBits();

        AudioSampleBuffer ins (jmax (1, numInputBuffers), inputBufferSize_  );
        AudioSampleBuffer outs (jmax (1, numOutputBuffers), outputBufferSize_ );
        float** const inputBuffers = ins.getArrayOfChannels();
        float** const outputBuffers = outs.getArrayOfChannels();
        ins.clear();
		outs.clear();


		//WaveFileReader reader(L"f:\\data\\behind2.wav",true);

		BYTE* pRender(0);
		DWORD numBuffer(0);
		
		if(renderClient_){
			renderClient_->GetBuffer(outputBufferSize_,&pRender);
			memset(pRender,0,outputBufferSize_*2*numOutputBuffers);
			renderClient_->ReleaseBuffer(outputBufferSize_,AUDCLNT_BUFFERFLAGS_SILENT);
		}

		{
			boost::mutex::scoped_lock lk_(mutex_);
			isPlaying_ = true;
		}
		
		//captureClient_->Reset();

		if(inputClient_){
			EXCEPTION_ON_ERROR(inputClient_->Start());
		}

		if(outputClient_){
			EXCEPTION_ON_ERROR(outputClient_->Start());
		}

		while(true)
		{
			const DWORD result = WaitForMultipleObjects (numEvents, events, false, 5000);
			switch (result)
			{
			case WAIT_OBJECT_0:  // Exit loop, stop playing, and exit thread.
				{
					boost::mutex::scoped_lock lk_(mutex_);
					isPlaying_ = false;
					if(hMmTask)
						AvRevertMmThreadCharacteristics(hMmTask);
				}
				return;

			case WAIT_OBJECT_0+1:  // Render notification

				// Copy data from the transfer buffer into the output if needed.
				//	UINT32 packetSize = bufferLengthOut * frameSize;
				// Make the callback..
				{
					boost::mutex::scoped_lock lk (mutex_);

					if (isStart_)
					{
						JUCE_TRY
						{
							callback_->audioDeviceIOCallback ((const float**) inputBuffers,
															 numInputBuffers,
															 outputBuffers,
															 numOutputBuffers,
															 outputBufferSize_ );
						}
						JUCE_CATCH_EXCEPTION
					}
					else
					{
						outs.clear();
					}
				
					if(renderClient_){
						// Get pointer to next space in render buffer.
						hr = renderClient_->GetBuffer(outputBufferSize_, &pRender);
						if (S_OK != hr)
						{
							if(inputClient_)
								inputClient_->Stop();
							outputClient_->Stop();
							return;
						}

						for (int i = 0; i < numOutputBuffers; ++i)
						{
							const float* const source = outputBuffers[i];
							AudioDataConverters::convertFloatToInt16LE (source, pRender + 2 * i, outputBufferSize_, 2 * numOutputBuffers);
						}

						EXCEPTION_ON_ERROR(renderClient_->ReleaseBuffer(outputBufferSize_, 0));
					}
				}
				break;

			case WAIT_OBJECT_0+2: // Capture notification
				{
					UINT32 packetLength = 0;
					DWORD flags = 0;
					BYTE* pCapture = 0;

					if(captureClient_){

						//captureClient_->GetNextPacketSize(&packetLength);

						// Get pointer to next data packet in capture buffer.
						hr = captureClient_->GetBuffer(&pCapture, &packetLength, &flags, NULL, NULL);
						if ((hr != S_OK) && (hr != AUDCLNT_E_BUFFER_OPERATION_PENDING) && (hr != AUDCLNT_S_BUFFER_EMPTY))
						{
							outputClient_->Stop();
							inputClient_->Stop();
							throw Win32ErrorException(hr);
						}

						if(pCapture && !((flags & AUDCLNT_BUFFERFLAGS_SILENT) > 0) ){
							for (int i = 0; i < numInputBuffers; ++i)
							{
								float* dest = inputBuffers[i];
								AudioDataConverters::convertInt16LEToFloat((void*)pCapture, dest, inputBufferSize_, 2 * numInputBuffers);
							}
						} else {
							ins.clear();
						}
						// Release the capture buffer.
						if(pCapture){
							EXCEPTION_ON_ERROR(captureClient_->ReleaseBuffer(packetLength));
						} else {
							EXCEPTION_ON_ERROR(captureClient_->ReleaseBuffer(0));
						}
					}
				}
				break;

			case WAIT_TIMEOUT:  // Timeout notification
				continue;

			default:  // Unexpected error

				if(inputClient_)
				{
					inputClient_->Stop();
				}

				if(outputClient_)
				{
					outputClient_->Stop();
				}
				//throw sf::Win32ErrorException();
				if(hMmTask){
					AvRevertMmThreadCharacteristics(hMmTask);
				}
				break;
			}
		}
	}

private:
	void initialize()
	{
		inputDefaultPeriod_ = inputMinPeriod_ = outputDefaultPeriod_ = outputMinPeriod_ =0;
		currentSampleRate_ = 48000.0;
		currentBitDepth_ = 16;
		currentVolume_ = 0.5f;
		currentBufferSize_ = 0;


//		wave_formats_ += { WAVE_FORMAT_PCM, 2, 192000,768000, 4, 16, 0 },{ WAVE_FORMAT_PCM, 2, 192000,768000, 4, 16, 0 };
		//// Create IAudioSessionEvents interface instances to notify us of status changes
		//// in the rendering and capture sessions. We'll register these interfaces later,
		//// after we create the sessions.

		renderSessionEvents_.Attach(new AudioSessionEvents(eRender));
		captureSessionEvents_.Attach(new AudioSessionEvents(eCapture));

		//// Create an IAudioEndpointVolumeCallback interface instance to notify us of
		//// status changes in the volume of the rendering endpoint device. We'll register
		//// this interface later, after the user selects a rendering endpoint device.
		//renderEndpointVolumeCallback_.Attach(new AudioEndpointVolumeCallback(eRender));

		
		sf::WASAPIDeviceInfos::getInstance()->getEnumerator()->GetDevice(outputDeviceId_,&outputDevice_);
		sf::WASAPIDeviceInfos::getInstance()->getEnumerator()->GetDevice(inputDeviceId_,&inputDevice_);

		CoTaskMemory<::WAVEFORMATEXTENSIBLE> outputMixFormat,inputMixFormat;

		if(outputDevice_){
			RegisterNotificationCallbacks(outputDevice_,eRender);
			EXCEPTION_ON_ERROR(outputDevice_->Activate(__uuidof(IAudioClient),CLSCTX_ALL,0,reinterpret_cast<void**>(&outputClient_)));
			if(outputClient_){
				EXCEPTION_ON_ERROR(outputClient_->GetDevicePeriod(&outputDefaultPeriod_,&outputMinPeriod_));
				//EXCEPTION_ON_ERROR(outputClient_->GetStreamLatency(&outputLatency_));
				EXCEPTION_ON_ERROR(outputClient_->GetMixFormat(reinterpret_cast<::WAVEFORMATEX**>(&outputMixFormat)));
				//outputClient_->
			}
			outputChannels_ = outputMixFormat.Get()->Format.nChannels;
		}

		if(inputDevice_){
			RegisterNotificationCallbacks(inputDevice_,eCapture);
			EXCEPTION_ON_ERROR(inputDevice_->Activate(__uuidof(IAudioClient),CLSCTX_ALL,0,reinterpret_cast<void**>(&inputClient_)));
			if(inputClient_)
			{
				EXCEPTION_ON_ERROR(inputClient_->GetDevicePeriod(&inputDefaultPeriod_,&inputMinPeriod_));
				//EXCEPTION_ON_ERROR(inputClient_->GetStreamLatency(&inputLatency_));
				EXCEPTION_ON_ERROR(inputClient_->GetMixFormat(reinterpret_cast<::WAVEFORMATEX**>(&inputMixFormat)));
			}
			inputChannels_ = inputMixFormat.Get()->Format.nChannels;
		}

		// `l(in,outłꂼŃ`FbNj
		// TvO[g̃`FbNiin,outňv́j
		// CeṼ`FbN(in,outňv́j

		BOOST_FOREACH(WAVEFORMATEX wave_format,wave_formats_)
		{
			HRESULT hr(S_OK);
			WAVEFORMATEXTENSIBLE wave_ext;
			if(outputClient_) {
				wave_ext.Format = wave_format;
				wave_ext.Format.nChannels = outputMixFormat.Get()->Format.nChannels;
				ConvertToExtensible(reinterpret_cast<WAVEFORMATEX*>(&wave_ext));
				hr = outputClient_->IsFormatSupported(AUDCLNT_SHAREMODE_EXCLUSIVE,reinterpret_cast<WAVEFORMATEX*>(&wave_ext),0);
			} 

			if(SUCCEEDED(hr))
			{
				if(inputClient_){ 
					try {
						WAVEFORMATEXTENSIBLE wave_ext_in;
						wave_ext_in.Format = wave_format;
						wave_ext_in.Format.nChannels = inputMixFormat.Get()->Format.nChannels;
						ConvertToExtensible(reinterpret_cast<WAVEFORMATEX*>(&wave_ext_in));
						EXCEPTION_ON_ERROR(inputClient_->IsFormatSupported(AUDCLNT_SHAREMODE_EXCLUSIVE,reinterpret_cast<WAVEFORMATEX*>(&wave_ext_in),NULL));
						support_formats_.push_back(new sf::WAVEFORMATEX(wave_format));
						std::vector<int> * vec = new std::vector<int>();
						int& sampleRate((int&)wave_format.nSamplesPerSec); 
						latencyMap_.insert(sampleRate,vec);
						REFERENCE_TIME begin,end;
						if(outputClient_)
						{
							begin = outputMinPeriod_;
							end = outputDefaultPeriod_;
						} else {
							begin = inputMinPeriod_ ;
							end = inputDefaultPeriod_; 
						}

						for(REFERENCE_TIME i = begin;i <= end;i += 10000)
						{

							hr = S_OK;
							if(outputClient_){
								hr = outputClient_->Initialize(AUDCLNT_SHAREMODE_EXCLUSIVE,AUDCLNT_STREAMFLAGS_EVENTCALLBACK ,
									i,i,reinterpret_cast<WAVEFORMATEX*>(&wave_ext),NULL);
							}

							if(SUCCEEDED(hr)){
								int bufferSize(0);
								if(SUCCEEDED(inputClient_->Initialize(AUDCLNT_SHAREMODE_EXCLUSIVE,AUDCLNT_STREAMFLAGS_EVENTCALLBACK ,
									i,i,reinterpret_cast<WAVEFORMATEX*>(&wave_ext_in),NULL)))
								{
									UINT32 bufsize(0);
									inputClient_->GetBufferSize(&bufsize);
									vec->push_back((int)bufsize);
								}
								inputClient_.Release();
								EXCEPTION_ON_ERROR(inputDevice_->Activate(__uuidof(IAudioClient),CLSCTX_ALL,0,reinterpret_cast<void**>(&inputClient_)));
							}
							outputClient_.Release();
							EXCEPTION_ON_ERROR(outputDevice_->Activate(__uuidof(IAudioClient),CLSCTX_ALL,0,reinterpret_cast<void**>(&outputClient_)));
						}
				
						DEBUGPRINTF(boost::wformat(L"%dchannel %d Sample/Sec %d AvgBytes/Sec %d BlockAlign %d Bits/Sample support \n")
							%wave_format.nChannels%wave_format.nSamplesPerSec%wave_format.nAvgBytesPerSec%wave_format.nAvgBytesPerSec%wave_format.wBitsPerSample);
						
					} catch(Win32ErrorException& e) 
					{
						DEBUGPRINTF(boost::wformat(L"Error %s\n") % e.error().c_str());
					}
				} else {
					if(outputClient_) {

						support_formats_.push_back(new sf::WAVEFORMATEX(wave_format));
						std::vector<int> * vec = new std::vector<int>();
						int& sampleRate((int&)wave_format.nSamplesPerSec); 
						latencyMap_.insert(sampleRate,vec);
						for(REFERENCE_TIME i = outputMinPeriod_;i <= outputDefaultPeriod_;i += 10000)
						{								
							if(SUCCEEDED(outputClient_->Initialize(AUDCLNT_SHAREMODE_EXCLUSIVE,AUDCLNT_STREAMFLAGS_EVENTCALLBACK ,
									i,i,reinterpret_cast<WAVEFORMATEX*>(&wave_ext),NULL)))
							{
									UINT32 bufsize(0);
									outputClient_->GetBufferSize(&bufsize);
									vec->push_back((int)bufsize);
							};
							outputClient_.Release();
							EXCEPTION_ON_ERROR(outputDevice_->Activate(__uuidof(IAudioClient),CLSCTX_ALL,0,reinterpret_cast<void**>(&outputClient_)));
						}
						DEBUGPRINTF(boost::wformat(L"%dchannel %d Sample/Sec %d AvgBytes/Sec %d BlockAlign %d Bits/Sample support \n")
						%wave_format.nChannels%wave_format.nSamplesPerSec%wave_format.nAvgBytesPerSec%wave_format.nAvgBytesPerSec%wave_format.wBitsPerSample);
					} 
				}
			}
		}


		if(!support_formats_.empty())
		{
			if(outputClient_)
			{
				for(int i = 0;i < outputChannels_;++i){
					outChannels_.add((boost::wformat(L"Output Channel%d") % (i + 1)).str().c_str());
				}
			}

			if(inputClient_)
			{
				for(int i = 0;i < inputChannels_;++i){
					inChannels_.add((boost::wformat(L"Input Channel%d") % (i + 1)).str().c_str());
				}
			}
		}


		//// Event handle to signal the buffer-processing thread to stop playing.
		//// This event will require a "manual" reset (explicit ResetEvent call).
		hStopPlaying_ = CreateEvent(NULL, TRUE, FALSE, NULL);
		if (NULL == hStopPlaying_)
		{
			EXCEPTION_ON_ERROR(E_FAIL);
		}

		hThreadIn_ = CreateEvent(NULL,FALSE,FALSE,NULL);
		if(!hThreadIn_)
		{
			EXCEPTION_ON_ERROR(::GetLastError());
		}

		hThreadOut_ = CreateEvent(NULL,FALSE,FALSE,NULL);
		if(!hThreadOut_)
		{
			EXCEPTION_ON_ERROR(::GetLastError());
		}
	};

	boost::ptr_vector<sf::WAVEFORMATEX> support_formats_;
	static boost::ptr_vector<sf::WAVEFORMATEX> wave_formats_;
	std::vector<int> sample_buffer_sizes_;

	HWND hDlg_;
    bool exclusiveMode_;
    bool eventMode_;
    bool enableMMCSS_;
    int bufferLatency_;
    bool hwVolumeSupported_;
	sf::EAudioSourceType audioSourceType_;
    HANDLE hStopPlaying_;
    HANDLE hThreadIn_,hThreadOut_;
	::AVRT_PRIORITY mmThreadPriority_;
	juce::AudioIODeviceCallback *callback_;
	juce::AudioIODevice *parent_;

	int outputDeviceIndex_;
	int inputDeviceIndex_;
	int currentBitDepth_;
	String inputDeviceId_,outputDeviceId_;
    StringArray inChannels_, outChannels_;
	juce::BitArray bitInChannels_,bitOutChannels_;

	REFERENCE_TIME inputDefaultPeriod_,inputMinPeriod_;
	REFERENCE_TIME outputDefaultPeriod_,outputMinPeriod_;
	REFERENCE_TIME currentLatency_,inputLatency_,outputLatency_;
	float currentVolume_;
	int currentBufferSize_,inputBufferSize_,outputBufferSize_;
	double currentSampleRate_;
	int inputChannels_,outputChannels_,actualInputChannels_,actualOutputChannels_;
	bool isOpen_,isStart_,isPlaying_;
	boost::thread captureAndRenderThread_;
	boost::mutex mutex_;

	
	IMMDevicePtr outputDevice_;
	IMMDevicePtr inputDevice_;
	IAudioClientPtr outputClient_;
	IAudioClientPtr inputClient_;
	IAudioRenderClientPtr renderClient_;
	IAudioCaptureClientPtr captureClient_;
	
	IAudioSessionEventsPtr renderSessionEvents_;
	IAudioSessionEventsPtr captureSessionEvents_;

	IAudioSessionControlPtr renderSessionControl_;
	IAudioSessionControlPtr captureSessionControl_;
	IAudioEndpointVolumePtr renderEndpointVolume_;
	IAudioEndpointVolumeCallbackPtr renderEndpointVolumeCallback_;

	typedef boost::ptr_map<int ,std::vector<int> > LatencyMapType;
	LatencyMapType latencyMap_;



    // Monitored by audio stream threads
    bool keepPlaying_;

    // Audio playback threads and audio event notifications

	//
	// Local implementation of IAudioSessionEvents interface.
	// WASAPI calls these methods to notify the Player when
	// an audio session control or parameter changes.
	//
	class AudioSessionEvents : public IUnknownImpl<IAudioSessionEvents>
	{
		//impl *pPlayer_;
		EDataFlow dataFlow_;
	public:
		AudioSessionEvents(/**impl pPlayer,*/ EDataFlow dir) :
			IUnknownImpl<IAudioSessionEvents>(),dataFlow_(dir)
		{
			assert(eRender == dir || eCapture == dir);
		}

		~AudioSessionEvents() {};

		HRESULT __stdcall OnDisplayNameChanged(LPCWSTR NewDisplayName, LPCGUID EventContext)
		{
			return S_OK;
		}

		HRESULT __stdcall OnIconPathChanged(LPCWSTR NewIconPath, LPCGUID EventContext)
		{
			return S_OK;
		}

		HRESULT __stdcall OnSimpleVolumeChanged(float NewVolume, BOOL NewMute,
												LPCGUID EventContext)
		{
			//if (FALSE == pPlayer_->exclusiveMode_ &&
			//	eRender == dataFlow_ &&
			//	NULL != pPlayer_->pPlayerCallbacks_ &&
			//	*EventContext != g_EventContext)
			//{
			//	pPlayer_->pPlayerCallbacks_->VolumeChangeCallback(NewVolume, NewMute);
			//}
			return S_OK;
		}

		HRESULT __stdcall OnChannelVolumeChanged(DWORD ChannelCount,
												 float NewChannelVolumeArray[],
												 DWORD ChangedChannel,
												 LPCGUID EventContext)
		{
			return S_OK;
		}

		HRESULT __stdcall OnGroupingParamChanged(LPCGUID NewGroupingParam,
												 LPCGUID EventContext)
		{
			return S_OK;
		}

		HRESULT __stdcall OnStateChanged(AudioSessionState NewState)
		{
			return S_OK;
		}

		HRESULT __stdcall OnSessionDisconnected(AudioSessionDisconnectReason DisconnectReason)
		{
			//if (TRUE != pPlayer_->exclusiveMode_ && NULL != pPlayer_->pPlayerCallbacks_)
			//{
			//	if (eRender == dataFlow_)
			//	{
			//		pPlayer_->getPlayerCallbacks()->RenderDisconnectCallback();
			//	}
			//	else
			//	{
			//		pPlayer_->getPlayerCallbacks()->CaptureDisconnectCallback();
			//	}
			//}
			return S_OK;
		}
	};

	//
	// Local implementation of IAudioEndpointVolume interface.
	// WASAPI calls these methods to notify the Player when
	// endpoint volume/mute changes occur.
	//
	class AudioEndpointVolumeCallback : public IUnknownImpl<IAudioEndpointVolumeCallback>
	{
		EDataFlow dataFlow_;

	public:
		AudioEndpointVolumeCallback(EDataFlow dir) :
			 dataFlow_(dir), IUnknownImpl<IAudioEndpointVolumeCallback>()

		{
			assert(eRender == dir || eCapture == dir);
		}

		~AudioEndpointVolumeCallback() {}

		HRESULT __stdcall OnNotify(PAUDIO_VOLUME_NOTIFICATION_DATA pNotifyData)
		{
			//if (TRUE == pPlayer_->exclusiveMode_ &&
			//	eRender == dataFlow_ &&
			//	NULL != pPlayer_->pPlayerCallbacks_ &&
			//	NULL != pNotifyData &&
			//	pNotifyData->guidEventContext != g_EventContext)
			//{
			//	pPlayer_->pPlayerCallbacks_->VolumeChangeCallback(pNotifyData->fMasterVolume, pNotifyData->bMuted);
			//}
			return S_OK;
		}
	};

 //   friend class MMNotificationClient;
 //   friend class AudioSessionEvents;
 //   friend class AudioEndpointVolumeCallback;
 //   // Notification callbacks from Player to client
	//sf::PlayerCallbacks *pPlayerCallbacks_;

    void _UpdateVolumeControl()
	{
		HRESULT hr = S_OK;
		float volume = 0.0f;
		BOOL mute = FALSE;
		IAudioSessionManagerPtr pManager;
		ISimpleAudioVolumePtr pVolume;

		try {
			if (NULL != outputDevice_)
			{
				if (exclusiveMode_)
				{
					// Exclusive mode: Get the endpoint volume and mute settings.
					if (hwVolumeSupported_ && NULL != renderEndpointVolume_)
					{
						EXCEPTION_ON_ERROR(renderEndpointVolume_->GetMasterVolumeLevelScalar(&volume));
						EXCEPTION_ON_ERROR(renderEndpointVolume_->GetMute(&mute));
					}
				}
				else
				{
					// Shared mode: Get the session volume and mute settings.
					hr = outputDevice_->Activate(__uuidof(IAudioSessionManager), CLSCTX_ALL,
												NULL, (void**)&pManager);
					EXCEPTION_ON_ERROR(hr)
					EXCEPTION_ON_ERROR(pManager->GetSimpleAudioVolume(NULL, FALSE, &pVolume));
					EXCEPTION_ON_ERROR(pVolume->GetMasterVolume(&volume));
					EXCEPTION_ON_ERROR(pVolume->GetMute(&mute));
				}
			}
			//pPlayerCallbacks_->VolumeChangeCallback(volume, mute);

		}catch (Win32ErrorException& ) 
		{
			;
		}
	};

    void RegisterNotificationCallbacks(IMMDevice *pDevice, EDataFlow dir)
	{
		HRESULT hr = S_OK;

		IAudioSessionManagerPtr pManager;
		ISimpleAudioVolumePtr pVolume;
		CoTaskMemory<wchar_t> pszNewDeviceID;
		CoTaskMemory<wchar_t> pszOldDeviceID;
		float volume(0.0f);
		bool mute(false);

		assert(eRender == dir || eCapture == dir);

		if (NULL == pDevice)
		{
			return;
		}
		try {
			EXCEPTION_ON_ERROR( pDevice->GetId(&pszNewDeviceID) );
			EXCEPTION_ON_ERROR( pDevice->Activate(__uuidof(IAudioSessionManager), CLSCTX_ALL,
								   NULL, (void**)&pManager));
			if (eRender == dir)
			{
				DWORD dwHwVolumeSupportMask = 0;

				if (NULL != renderSessionControl_)
				{
					// Ignore return error if notification interface not previously registered.
					renderSessionControl_->UnregisterAudioSessionNotification(renderSessionEvents_);
					renderSessionControl_.Release();
				}

				EXCEPTION_ON_ERROR( pManager->GetAudioSessionControl(NULL, FALSE, &renderSessionControl_));

				EXCEPTION_ON_ERROR(renderSessionControl_->RegisterAudioSessionNotification(renderSessionEvents_));

				// Check the hardware volume capabilities.
				EXCEPTION_ON_ERROR(pDevice->Activate(__uuidof(IAudioEndpointVolume), CLSCTX_ALL,
									   NULL, (void**)&renderEndpointVolume_));
				
				EXCEPTION_ON_ERROR(renderEndpointVolume_->QueryHardwareSupport(&dwHwVolumeSupportMask));

				if (dwHwVolumeSupportMask & ENDPOINT_HARDWARE_SUPPORT_VOLUME)
				{
					hwVolumeSupported_ = true;
					EXCEPTION_ON_ERROR(renderEndpointVolume_->RegisterControlChangeNotify(renderEndpointVolumeCallback_));
				}
				else
				{
					hwVolumeSupported_ = false;
				}

				outputDeviceId_ = pszNewDeviceID.Get();

				_UpdateVolumeControl();
			}
			else  // eCapture == dir
			{

				if (NULL != captureSessionControl_)
				{
					// Ignore return error if notification interface not previously registered.
					captureSessionControl_->UnregisterAudioSessionNotification(captureSessionEvents_);
					captureSessionControl_.Release();
				}

				EXCEPTION_ON_ERROR(pManager->GetAudioSessionControl(NULL, FALSE, &captureSessionControl_));
				EXCEPTION_ON_ERROR(captureSessionControl_->RegisterAudioSessionNotification(captureSessionEvents_));

				inputDeviceId_ = pszNewDeviceID.Get();
			}

		} catch (Win32ErrorException& )
		{
			;
		}
	};

	/** Get Shared Mode Device Format */
	WAVEFORMATEX* GetDeviceFormat(IMMDevice *pDevice)
	{
		HRESULT hr = E_FAIL;
		WAVEFORMATEX* pWfx;
		IPropertyStorePtr pProps;
		sf::PROPVARIANT varWfx;

		assert(pDevice != NULL);

		pWfx = (WAVEFORMATEX*)CoTaskMemAlloc(sizeof(WAVEFORMATEXTENSIBLE));
		if (pWfx == NULL)
		{
			throw Win32ErrorException(E_OUTOFMEMORY);
		}
		ZeroMemory(pWfx, sizeof(WAVEFORMATEXTENSIBLE));
		hr = pDevice->OpenPropertyStore(STGM_READ, &pProps);
		EXCEPTION_ON_ERROR(hr);

		hr = pProps->GetValue(PKEY_AudioEngine_DeviceFormat, &varWfx);
		EXCEPTION_ON_ERROR(hr)

		if (varWfx.vt != VT_BLOB || varWfx.blob.cbSize < sizeof(WAVEFORMATEX) ||
			varWfx.blob.cbSize > sizeof(WAVEFORMATEXTENSIBLE))
		{
			EXCEPTION_ON_ERROR(hr = E_FAIL);
		}

		assert(SUCCEEDED(hr));
		hr = S_OK;
		memcpy(pWfx, varWfx.blob.pBlobData, varWfx.blob.cbSize);
		
		return pWfx;
	};
}; // WASAP::impl


boost::ptr_vector<sf::WAVEFORMATEX> WASAPI::impl::wave_formats_ = 
		ptr_list_of<sf::WAVEFORMATEX>
		( WAVE_FORMAT_PCM, 2, 192000,768000, 4, 16, 0 )    
		( WAVE_FORMAT_PCM, 2, 96000, 384000, 4, 16, 0 )    // 96-kHz, 16-bit stereo
		( WAVE_FORMAT_PCM, 2, 48000, 192000, 4, 16, 0 )    // 48-kHz, 16-bit stereo
		( WAVE_FORMAT_PCM, 2, 44100, 176400, 4, 16, 0 );    // 44.1-kHz, 16-bit stereo


WASAPI::WASAPI(const juce::String & deviceName,int outputDeviceIndex,const juce::String& outputDeviceId,int inputDeviceIndex,const juce::String& inputDeviceId)
: juce::AudioIODevice(deviceName,L"WASAPI"),impl_(new impl(outputDeviceIndex,outputDeviceId,inputDeviceIndex,inputDeviceId,this))
{
	//
	
};


WASAPI::~WASAPI()
{
	
};

const juce::StringArray WASAPI::getOutputChannelNames()
{
	return impl_->getOutputChannelNames();
};

const juce::StringArray WASAPI::getInputChannelNames()
{
	return impl_->getInputChannelNames();
}

int WASAPI::getNumSampleRates()
{
	return impl_->getNumSampleRates();
};

double WASAPI::getSampleRate (int index)
{
	return impl_->getSampleRate(index);
};

int WASAPI::getNumBufferSizesAvailable()
{
	return impl_->getNumBufferSizesAvailable();
};

int WASAPI::getBufferSizeSamples (int index)
{
	return impl_->getBufferSizeSamples(index);
}

int WASAPI::getDefaultBufferSize()
{
	return impl_->getDefaultBufferSize();
};

const juce::String WASAPI::open (const juce::BitArray& inputChannels,
                           const juce::BitArray& outputChannels,
                           double sampleRate,
                           int bufferSizeSamples)
{
	return impl_->open(inputChannels,outputChannels,sampleRate,bufferSizeSamples);
};

void WASAPI::close()
{
	impl_->close();
};

bool WASAPI::isOpen()
{
	return impl_->isOpen();
}

void WASAPI::start (juce::AudioIODeviceCallback* callback)
{
	impl_->start(callback);
};

void WASAPI::stop()
{
	impl_->stop();
};

bool WASAPI::isPlaying()
{
	return impl_->isPlaying();
};

const juce::String WASAPI::getLastError()
{
	return impl_->getLastError();
};

int WASAPI::getCurrentBufferSizeSamples() 
{
	return impl_->getCurrentBufferSizeSamples();
};

double WASAPI::getCurrentSampleRate()
{
	return impl_->getCurrentSampleRate();
};

int WASAPI::getCurrentBitDepth()
{
	return impl_->getCurrentBitDepth();
};

const juce::BitArray WASAPI::getActiveOutputChannels() const {
	return impl_->getActiveOutputChannels();
}

const juce::BitArray WASAPI::getActiveInputChannels() const
{
	return impl_->getActiveInputChannels();
}

int WASAPI::getOutputLatencyInSamples()
{
	return impl_->getOutputLatencyInSamples();
};

int WASAPI::getInputLatencyInSamples()
{
	return impl_->getInputLatencyInSamples();
} ;

bool WASAPI::hasControlPanel() const
{
	return impl_->hasControlPanel();
};

bool WASAPI::showControlPanel()
{
	return impl_->showControlPanel();
};

int WASAPI::getInputDeviceIndex(){return impl_->getInputDeviceIndex();};
int WASAPI::getOutputDeviceIndex(){return impl_->getOutputDeviceIndex();};


class WASAPIAudioIODeviceType  : public AudioIODeviceType
{
public:
    WASAPIAudioIODeviceType()
        : AudioIODeviceType (L"WASAPI"),
          hasScanned (false)
    {
		scanForDevices();
    }

    ~WASAPIAudioIODeviceType()
    {
    }

    //==============================================================================
    void scanForDevices()
    {
        hasScanned = true;
        outputDeviceNames.clear();
        outputIds.clear();
        inputDeviceNames.clear();
        inputIds.clear();
		scanForDevices(WASAPIDeviceInfos::getInstance()->getRenderCollection(),outputDeviceNames,outputIds);
		scanForDevices(WASAPIDeviceInfos::getInstance()->getCaptureCollection(),inputDeviceNames,inputIds);
		
    }

	void scanForDevices(IMMDeviceCollectionPtr& deviceCollection,juce::StringArray &deviceNames, StringArray& ids)
	{
		unsigned int  deviceCount(0);
		deviceCollection->GetCount(&deviceCount);
		for(unsigned int i = 0;i < deviceCount;++i)
		{
			IMMDevice* pDevice = 0;
			EXCEPTION_ON_ERROR(deviceCollection->Item(i,&pDevice));
			boost::intrusive_ptr<IMMDevice> device(pDevice,false);
			IPropertyStore *pProp = 0;
			EXCEPTION_ON_ERROR(device->OpenPropertyStore(STGM_READ,&pProp));
			boost::intrusive_ptr<IPropertyStore> props(pProp,false);

			{
				sf::PROPVARIANT varName;
				EXCEPTION_ON_ERROR(props->GetValue(PKEY_Device_FriendlyName,&varName));
				wchar_t deviceName[1024];
				memset(deviceName,0,sizeof(wchar_t)*1024);
				wcscpy_s(deviceName,1024,varName.pwszVal);
				deviceNames.add(String(deviceName));
				DEBUGPRINTF(boost::wformat(L"device name %s \n") % deviceName);
				CoTaskMemory<wchar_t> deviceId;
				device->GetId(&deviceId);
				ids.add(deviceId.Get());
			}

			{
				//PROPVARIANT varName;
				//PropVariantInit(&varName);
				//EXCEPTION_ON_ERROR(props->GetValue(PKEY_Device_ClassGuid,&varName));
				//guids.add(new GUID(*varName.puuid));
				//PropVariantClear(&varName);
			}
		}

	}

	void outputDeviceInfo()
	{

	}



    const StringArray getDeviceNames (const bool wantInputNames) const
    {
        jassert (hasScanned); // need to call scanForDevices() before doing this

        return wantInputNames ? inputDeviceNames
                               : outputDeviceNames;
    }

    int getDefaultDeviceIndex (const bool /*forInput*/) const
    {
        jassert (hasScanned); // need to call scanForDevices() before doing this
        return 0;
    }

    int getIndexOfDevice (AudioIODevice* device, const bool asInput) const
    {
        jassert (hasScanned); // need to call scanForDevices() before doing this

        WASAPI* const d = dynamic_cast <WASAPI*> (device);
        if (d == 0)
            return -1;

        return asInput ? d->getInputDeviceIndex()
                       : d->getOutputDeviceIndex();
    }

    bool hasSeparateInputsAndOutputs() const    { return true; }

    AudioIODevice* createDevice (const String& outputDeviceName,
                                 const String& inputDeviceName)
    {
        jassert (hasScanned); // need to call scanForDevices() before doing this

        const int outputIndex = outputDeviceNames.indexOf (outputDeviceName);
        const int inputIndex = inputDeviceNames.indexOf (inputDeviceName);

        if (outputIndex >= 0 || inputIndex >= 0)
            return new WASAPI(outputDeviceName,outputIndex,outputIds[outputIndex],inputIndex,inputIds[inputIndex]);
        return 0;
    }

    //==============================================================================
    juce_UseDebuggingNewOperator


private:
    StringArray outputDeviceNames;
    StringArray outputIds;

    StringArray inputDeviceNames;
    StringArray inputIds;
    bool hasScanned;

     //==============================================================================
    WASAPIAudioIODeviceType (const WASAPIAudioIODeviceType&);
    const WASAPIAudioIODeviceType& operator= (const WASAPIAudioIODeviceType&);

};

//==============================================================================
AudioIODeviceType* juce_createWASAPIAudioIODeviceType()
{
    return new WASAPIAudioIODeviceType();
}


void AudioDeviceManager::createAudioDeviceTypes (juce::OwnedArray <juce::AudioIODeviceType>& list)
{
	juce::AudioDeviceManager::createAudioDeviceTypes (list);
	list.add(sf::juce_createWASAPIAudioIODeviceType());

}

}
