/*
  win32_graphics.cpp
  base class of screen driver

  Made by Studio Breeze. 2002

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
 */

#include "sted_screen_win32.h"

// graphics

// private

BOOL
CSTedScreenWin32::toWindowPos(int in_x, int in_y, int *out_x, int *out_y)
{
  if (out_x) *out_x = fWindowWidth * in_x / fX68WindowWidth;
  if (out_y) *out_y = fWindowHeight * (in_y%fX68WindowHeight) / fX68WindowHeight;
  return TRUE;
}

// apis

void
CSTedScreenWin32::TextXBox(int in_x0, int in_y0, int in_x1, int in_y1, int in_page)
{
  int x0, y0, x1, y1;
  HBRUSH br, mbr;
  COLORREF c;
  RECT r;

  c = PageToColor(in_page);
  if (!toWindowPos(in_x0, in_y0, &x0, &y0)) return;
  if (!toWindowPos(in_x1, in_y1, &x1, &y1)) return;

  ::SetRect(&r, x0, y0, x1, y1);

  if (in_page!=0) {
	  br = ::CreateSolidBrush(c);
	  mbr = MASK_ON;
  } else {
	  br = fBrushTextClear;
	  mbr = MASK_OFF;
  }
  if (!br) return;

  ::FrameRect(fTextDC, &r, br);
  ::FrameRect(fTextMaskDC, &r, MASK_ON);

  if (in_page!=0)
	  ::DeleteObject(br);
  InvalidateRect(&r);
}

void 
CSTedScreenWin32::TextXXLine(int in_page, int in_x, int in_y, int in_w, int in_linestyle)
{
  int x0, x1;
  int y;
  HPEN p,mp;
  HPEN orig, morig;

  if (!toWindowPos(in_x, in_y, &x0, &y)) return;
  if (!toWindowPos(in_x+in_w, in_y, &x1, &y)) return;

  if (in_linestyle!=0 /*&& in_page!=0*/) {
	  p = ::CreatePen(PS_SOLID, 1, PageToColor(in_page));
	  mp = ::CreatePen(PS_SOLID, 1, RGB(0xff, 0xff, 0xff));
  } else {
	  p = ::CreatePen(PS_SOLID, 1, RGB(0xff, 0xff, 0xff));
	  mp = ::CreatePen(PS_SOLID, 1, RGB(0x00, 0x00, 0x00));
  }
  if (!p || !mp) {
	  if (p) ::DeleteObject(p);
	  if (mp) ::DeleteObject(mp);
	  return;
  }

  orig = (HPEN)::SelectObject(fTextDC, p);
  morig = (HPEN)::SelectObject(fTextMaskDC, mp);

  ::MoveToEx(fTextDC, x0,y, NULL);
  ::LineTo(fTextDC, x1,y);
  ::MoveToEx(fTextMaskDC, x0,y, NULL);
  ::LineTo(fTextMaskDC, x1,y);

  ::SelectObject(fTextDC, orig);
  ::SelectObject(fTextMaskDC, morig);
  if (p) ::DeleteObject(p);
  if (mp) ::DeleteObject(mp);

  InvalidateRect(x0,y,x1-x0,1);
}

void
CSTedScreenWin32::TextXYLine(int in_page, int in_x, int in_y, int in_h, int in_linestyle)
{
  int x;
  int y0, y1;
  HPEN p,mp;
  HPEN orig, morig;

  if (!toWindowPos(in_x, in_y, &x, &y0)) return;
  if (!toWindowPos(in_x, in_y+in_h, &x, &y1)) return;

  if (in_linestyle!=0 /*&& in_page!=0*/) {
	  p = ::CreatePen(PS_SOLID, 1, PageToColor(in_page));
	  mp = ::CreatePen(PS_SOLID, 1, RGB(0xff, 0xff, 0xff));
  } else {
	  p = ::CreatePen(PS_SOLID, 1, RGB(0xff, 0xff, 0xff));
	  mp = ::CreatePen(PS_SOLID, 1, RGB(0x00, 0x00, 0x00));
  }
  if (!p || !mp) {
	  if (p) ::DeleteObject(p);
	  if (mp) ::DeleteObject(mp);
	  return;
  }

  orig = (HPEN)::SelectObject(fTextDC, p);
  morig = (HPEN)::SelectObject(fTextMaskDC, mp);

  ::MoveToEx(fTextDC, x,y0, NULL);
  ::LineTo(fTextDC, x,y1);
  ::MoveToEx(fTextMaskDC, x,y0, NULL);
  ::LineTo(fTextMaskDC, x,y1);

  ::SelectObject(fTextDC, orig);
  ::SelectObject(fTextMaskDC, morig);
  if (p) ::DeleteObject(p);
  if (mp) ::DeleteObject(mp);

  InvalidateRect(x,y0,1,y1-y0);
}

void
CSTedScreenWin32::TextReverse(int in_x, int in_y, int in_width, int in_page)
{
  int x,y,w,h;
  COLORREF c,b;
  int page;


  if (!TextToWindowPos(in_x, in_y, &x, &y)) return;
  if (!TextToWindowPos(in_width, 0, &w, &h)) return;
  h = fTextLineHeight;

  switch (in_page) {
	case 0: page=2;break;
	case 1: page=0;break;
	case 2: page=0;break;
	case 3: page=2;break;
  }

  c = ::SetTextColor(fTextDC, PageToColor(page));
  b = ::SetBkColor(fTextDC, RGB(0x00, 0x00, 0x00));
  ::BitBlt(fTextDC, x,y,w,h, fTextMaskDC, x, y, SRCCOPY);
  ::SetTextColor(fTextDC, c);
  ::SetBkColor(fTextDC, b);

  ::BitBlt(fTextMaskDC, x,y,w,h, fTextMaskDC, x, y, DSTINVERT);

  InvalidateRect(x,y,w,h);
}

void
CSTedScreenWin32::TextReverseArea(int in_r_ad, int in_r_ln, int in_editscr)
{
  int x0,y0;
  int w0,h0;
  int x,y;
  int w,h;
  COLORREF b,c;

  x0 = (in_editscr==0) ? 2 : 56;
  y0 = 6+in_r_ad;
  w0 = 36;
  h0 = in_r_ln;
  if (!TextToWindowPos(x0, y0, &x, &y)) return;
  if (!TextToWindowPos(w0, h0, &w, &h)) return;
    
  c = ::SetTextColor(fTextDC, PageToColor(3));
  b = ::SetBkColor(fTextDC, RGB(0x00, 0x00, 0x00));
  ::BitBlt(fTextDC, x,y,w,h, fTextMaskDC, x, y, SRCCOPY);
  ::SetTextColor(fTextDC, c);
  ::SetBkColor(fTextDC, b);

  ::BitBlt(fTextMaskDC, x,y,w,h, fTextMaskDC, x, y, DSTINVERT);

  InvalidateRect(x,y,w,h);  
}

void
CSTedScreenWin32::TextFill(int in_page, int in_x, int in_y, int in_w, int in_h, int in_linestyle)
{
  int x,y,w,h;
  COLORREF c;
  RECT r;
  HBRUSH br, mbr;

  if (!toWindowPos(in_x, in_y, &x, &y)) return;
  if (!toWindowPos(in_w, in_h, &w, &h)) return;

  ::SetRect(&r, x, y, x+w, y+h);

  if (in_linestyle!=0 && in_page!=0) {
	  c = PageToColor(in_page);
	  br = ::CreateSolidBrush(c);
	  mbr = MASK_ON;
  } else {
	  br = fBrushTextClear;
	  mbr = MASK_OFF;
  }

  ::FillRect(fTextDC, &r, br);
  ::FillRect(fTextMaskDC, &r, mbr);

  if (in_linestyle!=0 && in_page!=0)
	::DeleteObject(br);

  InvalidateRect(&r);
}

extern "C" int edit_scr;

void
CSTedScreenWin32::TextRasterCopy(int in_dst, int in_src, int in_line, int in_mode)
{
  int sx, lx;
  int sy, dy, ly;
  RECT r;

  ly = fWindowHeight * (in_line/4 - 1) / fX68TextHeight;
  if ((in_src==63*4) || (in_src==30*4)) {
    sx = 0;
    lx = fWindowWidth;
  } else {
    if (edit_scr==0) {
      sx = fWindowWidth * 2  / fX68TextWidth;
      lx = fWindowWidth * 36 / fX68TextWidth;
    } else {
      sx = fWindowWidth * 58 / fX68TextWidth;
      lx = fWindowWidth * 36 / fX68TextWidth;
    }
  }

  if (in_mode >= 0x8000) {
    sy = fWindowHeight * ((in_src+1)/4) / fX68TextHeight;
    dy = fWindowHeight * ((in_dst+1)/4) / fX68TextHeight;
    ::BitBlt(fTextDC, sx, dy-ly, lx, ly, fTextDC, sx, sy-ly, SRCCOPY);
    ::BitBlt(fTextMaskDC, sx, dy-ly, lx, ly, fTextMaskDC, sx, sy-ly, SRCCOPY);
    InvalidateRect(sx, dy-ly, lx, ly);
  } else {
    if (in_src==63*4) {
      sy = fWindowHeight * in_dst/4 / fX68TextHeight;
      dy = sy+fTextLineHeight;
      ::SetRect(&r, sx, sy, sx+lx, dy);
      ::FillRect(fTextDC, &r, fBrushTextClear);
      ::FillRect(fTextMaskDC, &r, MASK_OFF);
      InvalidateRect(&r);
    } else {
      sy = fWindowHeight * (in_src/4) / fX68TextHeight;
      dy = fWindowHeight * (in_dst/4) / fX68TextHeight;
      ::BitBlt(fTextDC, sx, dy, lx, ly, fTextDC, sx, sy, SRCCOPY);
      ::BitBlt(fTextMaskDC, sx, dy, lx, ly, fTextMaskDC, sx, sy, SRCCOPY);
      InvalidateRect(sx, dy, lx, ly);
    }
  }

}

void
CSTedScreenWin32::TextScroll(int in_x0, int in_y0, int in_xs, int in_ys, int in_x1, int in_y1)
{
  int x0, y0, x1, y1;
  int w,h;
  int d;

  if (!TextToWindowPos(in_x0, in_y0, &x0, &d)) return;
  if (!toWindowPos(in_x0, in_y0, &d, &y0)) return;
  if (!TextToWindowPos(in_x1, in_y1, &x1, &d)) return;
  if (!toWindowPos(in_x1, in_y1, &d, &y1)) return;
  if (!TextToWindowPos(in_xs, in_ys, &w, &d)) return;
  if (!toWindowPos(in_xs, in_ys, &d, &h)) return;

  ::BitBlt(fTextDC, x1, y1, w, h, fTextDC, x0, y0, SRCCOPY);
  ::BitBlt(fTextMaskDC, x1, y1, w, h, fTextMaskDC, x0, y0, SRCCOPY);

  InvalidateRect(x1, y1, w, h);
}

void 
CSTedScreenWin32::GraphicsBox(int in_x0, int in_y0, int in_x1, int in_y1, unsigned int in_color, unsigned int in_linestyle)
{
  int sx, sy, ex,ey;
  int x,y;
  int d;
  int c;
  RECT r;
  HBRUSH br;
  HPEN pen, orig;

  if (!toWindowPos(in_x0, in_y0, &sx, &sy)) return;
  if (!toWindowPos(in_x1, in_y1, &ex, &ey)) return;
  d = (in_y1 >= fX68WindowHeight) ? 1:0;
  if (sx>ex) { x=ex; ex=sx; sx=x;}
  if (sy>ey) { y=ey; ey=sy; sy=y;}

  c = (in_linestyle!=0) ? SetGraphicsColor(in_color) : 0;

  ::SetRect(&r, sx, sy, ex, ey);
  if (sx==ex || sy==ey) {
	  pen = ::CreatePen(PS_SOLID, 1, c);
	  if (pen) {
		  orig = (HPEN)::SelectObject(fGraphicsWindowDC[d], pen);
		  ::MoveToEx(fGraphicsWindowDC[d], sx, sy, NULL);
		  ::LineTo(fGraphicsWindowDC[d], ex, ey);
		  ::SelectObject(fGraphicsWindowDC[d], orig);
		  ::DeleteObject(pen);
	  }
  } else {
	  br = ::CreateSolidBrush(c);
	  if (br) {
		::FrameRect(fGraphicsWindowDC[d], &r, br);
		::DeleteObject(br);
	  }
  }

  InvalidateRect(&r);
}

int
CSTedScreenWin32::GraphicsPoint(int in_x, int in_y)
{
  // just ignore
  return 0;
}

void 
CSTedScreenWin32::GraphicsLine(int in_x0, int in_y0, int in_x1, int in_y1, int in_color, int in_linestyle)
{
  int sx, sy, ex,ey;
  int x,y;
  int d;
  int c;
  HPEN pen, orig;

  if (!toWindowPos(in_x0, in_y0, &sx, &sy)) return;
  if (!toWindowPos(in_x1, in_y1, &ex, &ey)) return;
  d = (in_y1 >= fX68WindowHeight) ? 1:0;
  if (sx>ex) { x=ex; ex=sx; sx=x;}
  if (sy>ey) { y=ey; ey=sy; sy=y;}

  c = (in_linestyle!=0) ? SetGraphicsColor(in_color) : 0;

  pen = ::CreatePen(PS_SOLID, 1, c);
  if (pen) {
	  orig = (HPEN)::SelectObject(fGraphicsWindowDC[d], pen);
	  ::MoveToEx(fGraphicsWindowDC[d], sx, sy, NULL);
	  ::LineTo(fGraphicsWindowDC[d], ex, ey);
	  ::SelectObject(fGraphicsWindowDC[d], orig);
	  ::DeleteObject(pen);
  }
  InvalidateRect(sx, sy, ex-sx, ey-sy);
}

void
CSTedScreenWin32::GraphicsFill(int in_x0, int in_y0, int in_x1, int in_y1, int in_color)
{
  int sx, sy, ex,ey;
  int x,y;
  int d;
  int c;
  RECT r;
  HBRUSH br;

  if (!toWindowPos(in_x0, in_y0, &sx, &sy)) return;
  if (!toWindowPos(in_x1, in_y1, &ex, &ey)) return;
  d = (in_y1 >= fX68WindowHeight) ? 1:0;
  if (sx>ex) { x=ex; ex=sx; sx=x;}
  if (sy>ey) { y=ey; ey=sy; sy=y;}

  c = SetGraphicsColor(in_color);

  ::SetRect(&r, sx, sy, ex, ey);
  br = ::CreateSolidBrush(c);
  if (br) {
    ::FillRect(fGraphicsWindowDC[d], &r, br);
	::DeleteObject(br);
  }

  InvalidateRect(&r);
}

void
CSTedScreenWin32::CopyTextToGraphics(int in_editscreen)
{
  int x0, y0, x1, y1;
  int xt, xl, yt, yl;
  COLORREF b,c;

  xl = 36;
  yl = 24;
  if (in_editscreen==0) {
    xt = 2;
    yt = 6;
  } else {
    xt = 58;
    yt = 6;
  }
  if (!TextToWindowPos(xt, yt, &x0, &y0)) return;
  if (!TextToWindowPos(xl, (yt+yl), &x1, &y1)) return;

  c = ::SetTextColor(fGraphicsWindowDC[1], RGB(0x00, 0x00, 0x00));  
  b = ::SetBkColor(fGraphicsWindowDC[1], RGB(0xff, 0xff, 0xff));  

  ::BitBlt(fGraphicsWindowDC[1], x0, y0, x1, y1-y0, fTextMaskDC, x0, y0, SRCPAINT);
  ::BitBlt(fGraphicsWindowDC[1], x0, y0, x1, y1-y0, fTextDC, x0, y0, SRCAND);

  ::SetTextColor(fGraphicsWindowDC[1], c);  
  ::SetBkColor(fGraphicsWindowDC[1], b);  

  InvalidateRect(x0, y0, x1, y1-y0);
}

void
CSTedScreenWin32::CopyTextToGraphics2(int in_editscreen)
{
  int x0, y0, x1, y1;
  int xt, xl, yt, yl;
  COLORREF b,c;

  xl = 56;
  yl = 24;
  if (in_editscreen==0) {
    xt = 0;
    yt = 6;
  } else {
    xt = 38;
    yt = 6;
  }
  if (!TextToWindowPos(xt, yt, &x0, &y0)) return;
  if (!TextToWindowPos(xl, (yt+yl), &x1, &y1)) return;

  c = ::SetTextColor(fGraphicsWindowDC[1], RGB(0x00, 0x00, 0x00));  
  b = ::SetBkColor(fGraphicsWindowDC[1], RGB(0xff, 0xff, 0xff));  

  ::BitBlt(fGraphicsWindowDC[1], x0, y0, x1, y1-y0, fTextMaskDC, x0, y0, SRCPAINT);
  ::BitBlt(fGraphicsWindowDC[1], x0, y0, x1, y1-y0, fTextDC, x0, y0, SRCAND);

  ::SetTextColor(fGraphicsWindowDC[1], c);  
  ::SetBkColor(fGraphicsWindowDC[1], b);  

  InvalidateRect(x0, y0, x1, y1-y0);
}

void
CSTedScreenWin32::GraphicsHome(int in_home)
{
  if (in_home!=0 && in_home!=1) in_home = 0;
  fCurrentGraphics = in_home;
  InvalidateRect(NULL);
}

void
CSTedScreenWin32::ClsAll(void)
{
  RECT r;
  ::SetRect(&r, 0, 0, fWindowWidth, fWindowHeight);
  ::FillRect(fTextDC, &r, fBrushTextClear);
  ::FillRect(fTextMaskDC, &r, MASK_OFF);
  InvalidateRect(&r);
}

void
CSTedScreenWin32::GraphicsClear(void)
{
  RECT r;
  ::SetRect(&r, 0, 0, fWindowWidth, fWindowHeight);
  ::FillRect(fGraphicsWindowDC[fCurrentGraphics], &r, fBrushClear);
  InvalidateRect(&r);
}
