﻿/*
  win32_midi.cpp
  midi related drivers

  Made by Studio Breeze. 2002

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
 */
#include "stdafx.h"
#include "sted_screen.h"
#include "sted_screen_win32.h"
#include "rcp.h"
#include "assert.h"

int rcpplay_start(RCP_DATA* in_rcp);
int rcpplay_stop(void);
int rcpplay_close(void);

// HWND STedGetWindow(void);

// midi devices

void
CSTedScreenWin32::AppendMidiData(int in_c)
{
	fMidiBuffer[fMidiBufferEndPtr] = (unsigned char)in_c;

	fMidiBufferEndPtr = (fMidiBufferEndPtr+1)%fMidiBufferLen;

	if (fMidiBufferEndPtr==fMidiBufferStartPtr)
		fMidiBufferStartPtr = (fMidiBufferStartPtr+1)%fMidiBufferLen;

	fMidiBufferAvailable++;
	if (fMidiBufferAvailable>=fMidiBufferLen) fMidiBufferAvailable = fMidiBufferLen;
}

void
CSTedScreenWin32::NotifyMidiIn(unsigned long in_data)
{
	int c1, c2, c3;
	int l=0;

	c1 = in_data&0x000000ff;
	c2 = (in_data&0x0000ff00)>>8;
	c3 = (in_data&0x00ff0000)>>16;

	switch (c1&0xf0) {
		case 0x80: // note off
		case 0x90: // note on
		case 0xa0: // poly key-pressure
		case 0xb0: // control change
		case 0xe0: // pitch-bend
			l = 2;
			break;
		case 0xc0: // program change
		case 0xd0: // channel pressure
			l = 1;
			break;
		case 0xf0:
			switch (c1) {
				case 0xf0: // exclusive: not happen
				case 0xf4: // not defined
					break;
				case 0xf1: // midi time code
				case 0xf3: // song select
				case 0xf5: // port change
					l = 1;
					break;
				case 0xf2: // song posision
					l = 2;
					break;
				case 0xf6:
				case 0xf7:
				case 0xf8:
				case 0xf9:
				case 0xfa:
				case 0xfb:
				case 0xfc:
				case 0xfd:
				case 0xfe:
				case 0xff:
					l = 0;
					break;
			}
			break;
	}

	switch (l) {
		case 0:
			AppendMidiData(c1);
			break;
		case 1:
			AppendMidiData(c1);
			AppendMidiData(c2);
			break;
		case 2:
			AppendMidiData(c1);
			AppendMidiData(c2);
			AppendMidiData(c3);
			break;
	}

	return;
}

int
CSTedScreenWin32::OpenMidiDevice(void)
{
	HMIDIIN din;
	HMIDIOUT dout;
	MMRESULT result;
	int indevs;
	int outdevs;
	int inid, outid;
	int i;

	fMidiInDevice = NULL;
	fMidiBufferAvailable = 0;
	fMidiBufferStartPtr = 0;
	fMidiBufferEndPtr = 0;
	for (i=0; i<fMidiOutPorts; i++) {
		fMidiOutDevice[i] = NULL;
	}
	fMidiOutCurrentPort = 0;

	indevs = ::midiInGetNumDevs();
	outdevs = ::midiOutGetNumDevs();

	for (i=0; i<fMidiOutPorts; i++) {
		if (fMidiOutPort[i]==-1) continue;
		outid = (fMidiOutPort[i]>=0) ? fMidiOutPort[i] : MIDI_MAPPER;
		if (outid>=outdevs) outid = MIDI_MAPPER;
		result = ::midiOutOpen(&dout, outid, (DWORD_PTR)raw_handle(), NULL, CALLBACK_WINDOW);
		if (result==MMSYSERR_NOERROR) {
			fMidiOutDevice[i] = dout;
		}
	}

	if (indevs>0 && fMidiInPort>=0) {
		inid = (fMidiInPort>=0) ? fMidiInPort : MIDI_MAPPER;
		if (inid>=indevs) inid = indevs-1;
		result = ::midiInOpen(&din, 0, (DWORD_PTR)raw_handle(), NULL, CALLBACK_WINDOW);
		if (result==MMSYSERR_NOERROR) {
			fMidiInDevice = din;
			result = ::midiInStart(din);
			assert(result == MMSYSERR_NOERROR );

		}
	}
	
    return 0;
}

void
CSTedScreenWin32::CloseMidiDevice(void)
{
	MMRESULT result;
	if (fMidiInDevice) {
		result = ::midiInStop(fMidiInDevice);
		assert(result == MMSYSERR_NOERROR );
		result = ::midiInClose(fMidiInDevice);
		assert(result == MMSYSERR_NOERROR );
		fMidiInDevice = NULL;
	}

	for (int i=0; i<fMidiOutPorts; i++) {
		if (fMidiOutDevice[i]) {
			result = ::midiOutClose(fMidiOutDevice[i]);
			assert(result == MMSYSERR_NOERROR );
			fMidiOutDevice[i] = NULL;
		}
	}
}

BOOL
CSTedScreenWin32::SetMidiPorts(int in_min, int* in_mout)
{
	int i;

	fMidiInPort = in_min;

	if (in_mout) {
		for (i=0; i<fMidiOutPorts; i++) {
			fMidiOutPort[i] = in_mout[i];
		}
	}

	return TRUE;
}

int
CSTedScreenWin32::GetMidiData(void)
{
	int c;

	if (fMidiBufferAvailable>0) {
		c = fMidiBuffer[fMidiBufferStartPtr];
		fMidiBufferAvailable--;
		fMidiBufferStartPtr = (fMidiBufferStartPtr+1)%fMidiBufferLen;
		return c;
	}
    return -1;
}

int
CSTedScreenWin32::CanMidiRead(void)
{
	if (fMidiBufferAvailable>0) return 1;
    return 0;
}

void
CSTedScreenWin32::PutMidiData(int in_data)
{
	int i;
	MIDIHDR hdr;
	MMRESULT result;
	int p;

	p = fMidiOutCurrentPort;
	if (p<0) p=0;
	if (p>fMidiOutPorts) p=fMidiOutPorts-1;

	if (in_data>=0x80) {
		if (in_data==0xf7) { // flush exclusive data
			for (i=0 ;i<fMidiExclusiveBufLen; i++) {if (fMidiExclusiveBuf[i]<0) break;}
			if (i==fMidiExclusiveBufLen) return;
			fMidiExclusiveBuf[i] = 0xf7;
			hdr.lpData = (LPSTR)fMidiExclusiveBuf;
			hdr.dwBufferLength = i;
			hdr.dwBytesRecorded = i;
			result = ::midiOutPrepareHeader(fMidiOutDevice[p], &hdr, sizeof(MIDIHDR));
			assert(result == MMSYSERR_NOERROR );
			result = ::midiOutLongMsg(fMidiOutDevice[p], &hdr, sizeof(MIDIHDR));
			assert(result == MMSYSERR_NOERROR );
			result = ::midiOutUnprepareHeader(fMidiOutDevice[p], &hdr, sizeof(MIDIHDR));
			assert(result == MMSYSERR_NOERROR );

		}
		fMidiOutPacket[0] = in_data;
		fMidiOutPacket[1] = -1;
		fMidiOutPacket[2] = -1;
		fMidiOutPacket[3] = -1;
		fMidiExclusiveBuf[0] = 0xf0;
		for (i=1; i<fMidiExclusiveBufLen; i++) fMidiExclusiveBuf[i] = -1;
		if (in_data>=0xf6 && in_data<=0xff) {
			result = ::midiOutShortMsg(fMidiOutDevice[p], in_data);
			assert(result == MMSYSERR_NOERROR );
		}
	} else {
		if (fMidiOutPacket[0] == 0xf0) {
			for (i=0 ;i<fMidiExclusiveBufLen; i++) {if (fMidiExclusiveBuf[i]<0) break;}
			if (i==fMidiExclusiveBufLen) return;
			fMidiExclusiveBuf[i] = in_data;
			return;
		}
		for (i=0; i<4; i++) {if (fMidiOutPacket[i]<0) break;}
		if (i==4) return;
		fMidiOutPacket[i] = in_data;

		switch(fMidiOutPacket[0]&0xf0) {
			case 0x80:
			case 0x90:
			case 0xa0:
			case 0xb0:
			case 0xe0:
				if (i==2) {
					result = ::midiOutShortMsg(fMidiOutDevice[p], (fMidiOutPacket[2]<<16 | fMidiOutPacket[1]<<8 | fMidiOutPacket[0]));
					assert(result == MMSYSERR_NOERROR );
				}
				break;
			case 0xc0:
			case 0xd0:
				if (i==1) {
					result =::midiOutShortMsg(fMidiOutDevice[p], (fMidiOutPacket[1]<<8 | fMidiOutPacket[0]));
					assert(result == MMSYSERR_NOERROR );
				}
				break;
			case 0xf0:
				switch(fMidiOutPacket[0]) {
					case 0xf1:
					case 0xf3:
						if (i==2) {
							result =::midiOutShortMsg(fMidiOutDevice[p], (fMidiOutPacket[2]<<16 |  fMidiOutPacket[1]<<8 | fMidiOutPacket[0]));
							assert(result == MMSYSERR_NOERROR );
						}
						break;
					case 0xf2:
						if (i==1) {
							result =::midiOutShortMsg(fMidiOutDevice[p], (fMidiOutPacket[1]<<8 | fMidiOutPacket[0]));
							assert(result == MMSYSERR_NOERROR );
						}
						break;
					case 0xf5:
						fMidiOutCurrentPort = in_data-1;
						break;
					default:
						break;
				}
				break;
		}

	}
}

void
CSTedScreenWin32::MidiWait(void)
{
	while(fKeyEventNotified==0 && fKeyBufferAvailable==0 && fMidiBufferAvailable==0) {
		DoMessageLoop();
	}
	fKeyEventNotified = FALSE;
}

// player
static RCP_DATA* rcp = NULL;

int
CSTedScreenWin32::StartMidiPlayer(unsigned char* in_data, int in_len)
{
	if (rcp) {
		CloseMidiPlayer(0);
	}
	rcp = (RCP_DATA *)::malloc(sizeof(RCP_DATA));
	if (!rcp) return -1;
	::memset((void *)rcp, 0, sizeof(RCP_DATA));

	rcp->command_name = (unsigned char *)"STed2";
	rcp->is_player = FLAG_TRUE;
	rcp->is_send_rtm = FLAG_TRUE;

	rcp->data = in_data;
	rcp->length = in_len;

	rcpplay_start(rcp);

	return -1;
}

int
CSTedScreenWin32::StopMidiPlayer(void)
{
	return rcpplay_stop();
}

int
CSTedScreenWin32::CloseMidiPlayer(int in_sig)
{
	if (rcp) {
		rcpplay_stop();
		rcpplay_close();
		::free((void *)rcp);
		rcp = NULL;
	}
	return 0;
}

void
CSTedScreenWin32::MidiPlayerWait(void)
{
	// 5msのタイマセット
	::SetTimer((HWND)raw_handle(), IDT_STED2_WAIT_TIMER, 5, NULL);
	// 何かイベントが発生するまでメッセージループを回す。
	while(fKeyEventNotified == 0 && fKeyBufferAvailable == 0 && !fTimerEventNotified /* && fMidiBufferAvailable == 0 */)
	{
		DoMessageLoop();
	}

	fKeyEventNotified = FALSE;
	fTimerEventNotified = FALSE;
}

void
CSTedScreenWin32::NotifyTimer(UINT in_timerid)
{
	fTimerEventNotified = TRUE;
}
