/*
 * Copyright (C) 2012 infodb.org. All rights reserved.
 * This program is made available under the terms of
 * the Common Public License v1.0
 */
package org.infodb.wax.server;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import org.apache.commons.dbcp.BasicDataSource;
import org.infodb.wax.core.HeaderMenu;
import org.infodb.wax.core.SystemEnv;
import org.infodb.wax.core.db.Definition;
import org.infodb.wax.core.db.TableDef;
import org.infodb.wax.core.db.XmlStorage;

public class WaxEnv implements SystemEnv {
    private File currentDirectory;
    
    private Map<String, XmlStorage> storages;
    private Properties serverProperties;
    private Properties resourceProperties;
    
    private List<HeaderMenu> menuList;
    
    public WaxEnv(String path) throws IOException {
        currentDirectory = new File(path);
        storages = new HashMap<>();
        
        serverProperties = new Properties();
        serverProperties.load(new FileInputStream(new File(currentDirectory, "conf/server.properties")));
        
        menuList = new ArrayList<>();
    }
    public int getPort() {
        return Integer.parseInt(serverProperties.getProperty("port", "8080"));
    }
    public int getStopPort() {
        return Integer.parseInt(serverProperties.getProperty("stop.port", "8089"));
    }
    @Override
    public File getCurrentDirectory() {
        return currentDirectory;
    }
    @Override
    public Locale getLocale() {
        Locale loc = new Locale(serverProperties.getProperty("lang", "ja"), serverProperties.getProperty("country", "JP"));
        return loc;
    }
    @Override
    public void addMenu(HeaderMenu menu) {
        menuList.add(menu);
    }
    @Override
    public List<HeaderMenu> getMenuList() {
        Collections.sort(menuList, new Comparator<HeaderMenu>() {
            @Override
            public int compare(HeaderMenu o1, HeaderMenu o2) {
                return o1.getOrderNo() - o2.getOrderNo();
            }
        });
        return menuList;
    }
    @Override
    public XmlStorage getStorage(String name) {
        String propName = getDSPropName(name);
        if(propName == null) {
            throw new RuntimeException("not found property key: ds." + name);
        }
        return storages.get(propName);
    }
    @Override
    public XmlStorage newStorage(String name, TableDef tableDef) {
        String propName = getDSPropName(name);
        if(propName == null) {
            throw new RuntimeException("not found property key: ds." + name);
        }
        
        Properties prop = new Properties();
        try (FileInputStream fis = new FileInputStream(new File(currentDirectory, "conf/" + propName))) {
            prop.load(fis);
        }
        catch(IOException e) {
            throw new RuntimeException("not found properties file: conf/" + propName);
        }
        String url = prop.getProperty("url");

        BasicDataSource ds = new BasicDataSource();
        ds.setDriverClassName(prop.getProperty("driverName"));
        ds.setUrl(url);
        ds.setUsername(prop.getProperty("userName"));
//            Encrypt encrypt = new Encrypt(prop.getProperty("password"), false);
//            ds.setPassword(encrypt.getDecrypt());
        ds.setPassword(prop.getProperty("password"));
        ds.setMaxActive(Integer.parseInt(prop.getProperty("maxActive")));
        ds.setMaxIdle(Integer.parseInt(prop.getProperty("maxIdle")));
        ds.setMaxWait(Integer.parseInt(prop.getProperty("maxWait")));
        ds.setMinIdle(Integer.parseInt(prop.getProperty("minIdle")));
        ds.setInitialSize(Integer.parseInt(prop.getProperty("minIdle")));
        ds.setTimeBetweenEvictionRunsMillis(Long.parseLong(prop.getProperty("timeBetweenEvictionRunsMillis")));
        ds.setValidationQuery(prop.getProperty("validationQuery"));
        ds.setTestWhileIdle(Boolean.parseBoolean(prop.getProperty("testWhileIdle")));
        ds.setDefaultAutoCommit(false);

        Definition def;
        try {
            def = (Definition)Class.forName(prop.getProperty("definition")).newInstance();
        }
        catch(ClassNotFoundException | IllegalAccessException | InstantiationException e) {
            throw new RuntimeException(e);
        }
        XmlStorage storage = new XmlStorage(ds, def, tableDef);
        storages.put(propName, storage);
        
        return storage;
    }
    private String getDSPropName(String name) {
        String key = "ds." + name;
        return serverProperties.getProperty(key);
    }
}
