/*
 * Copyright (C) 2011-2012 infodb.org. All rights reserved.
 * This program is made available under the terms of
 * the Common Public License v1.0
 */
package org.infodb.wax.core.html;

import java.io.IOException;
import java.io.Writer;

public abstract class AbstractBody {
    protected Writer writer;
    protected String elementName;
    
    protected AbstractBody(Writer writer, String elementName, Attr... attr) throws IOException {
        this.writer = writer;
        this.elementName = elementName;
        writer.write("<");
        writer.write(elementName);
        Attr.output(writer, attr);
        writer.write(">");
    }
    public final void end() throws IOException {
        writer.write("</");
        writer.write(elementName);
        writer.write(">");
    }
    public AbstractBody text(String text) throws IOException {
        if(text != null) {
            writer.write(text);
        }
        return this;
    }
    public AbstractBody text(String text, int from, int to) throws IOException {
        if(text == null) {
            return this;
        }
        if(from == to) { // 同じなら処理しない。
            return this;
        }
        writer.write(text.substring(from, to));
        return this;
    }
    public A a( Attr... attr) throws IOException {
        return new A(writer, attr);
    }
    public AbstractBody br() throws IOException {
        writer.write("<br/>");
        return this;
    }
    public AbstractBody hr() throws IOException {
        writer.write("<hr/>");
        return this;
    }
    public HEADER header() throws IOException {
        return new HEADER(writer);
    }
    public ARTICLE article(Attr... attr) throws IOException {
        return new ARTICLE(writer, attr);
    }
    public FOOTER footer() throws IOException {
        return new FOOTER(writer);
    }
    public H h(int count, Attr... attr) throws IOException {
        return new H(writer, "h" + count, attr);
    }
    public H h1(Attr... attr) throws IOException {
        return new H(writer, "h1", attr);
    }
    public H h2(Attr... attr) throws IOException {
        return new H(writer, "h2", attr);
    }
    public H h3(Attr... attr) throws IOException {
        return new H(writer, "h3", attr);
    }
    public H h4(Attr... attr) throws IOException {
        return new H(writer, "h4", attr);
    }
    public H h5(Attr... attr) throws IOException {
        return new H(writer, "h5", attr);
    }
    public H h6(Attr... attr) throws IOException {
        return new H(writer, "h6", attr);
    }
    public DIV div(Attr... attr) throws IOException {
        return new DIV(writer, attr);
    }
    public SPAN span(Attr... attr) throws IOException {
        return new SPAN(writer, attr);
    }
    public OL ol(Attr... attr) throws IOException {
        return new OL(writer, attr);
    }
    public UL ul(Attr... attr) throws IOException {
        return new UL(writer, attr);
    }
    public NAV nav(Attr... attr) throws IOException {
        return new NAV(writer, attr);
    }
    public P p(Attr... attr) throws IOException {
        return new P(writer, attr);
    }
    public SECTION section(Attr... attr) throws IOException {
        return new SECTION(writer, attr);
    }
    public AbstractBody img(String src, Attr... attr) throws IOException {
        writer.write("<img src=\"");
        writer.write(src);
        writer.write("\"");
        Attr.output(writer, attr);
        writer.write("/>");
        return this;
    }
    public void imgWithAnchor(String cls, String name) throws IOException {
        String path = "../img/" + cls + "/" + name;
        A a = a(Attr.href(path), Attr.target("_blank"));
        a.img(path, Attr.alt(name), Attr.clazz(cls));
        a.end();
    }
    public TABLE table(Attr... attr) throws IOException {
        return new TABLE(writer, attr);
    }
    public STRONG strong(Attr... attr) throws IOException {
        return new STRONG(writer, attr);
    }
    public I i(Attr... attr) throws IOException {
        return new I(writer, attr);
    }
    public S s(Attr... attr) throws IOException {
        return new S(writer, attr);
    }
    public FORM form(Attr... attr) throws IOException {
        return new FORM(writer, attr);
    }
}
