/*
 * Copyright (C) 2006-2012 infodb.org. All rights reserved.
 * This program is made available under the terms of
 * the Common Public License v1.0
 */
package org.infodb.wax.core.db;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.Blob;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import javax.inject.Inject;
import org.infodb.wax.core.utils.StreamHelper;

/**
 * SQLを直接実行するクラス 
 */
public class SQLDirect implements AutoCloseable {
    private Definition def;
    private Connection conn;
    private String sessionID;
    /**
     *
     */
    @Inject
    public SQLDirect(Definition def, Connection conn, String sessionID) {
        this.def = def;
        this.conn = conn;
        this.sessionID = sessionID;
    }
    @Override
    public void close() throws SQLException {
        conn.close();
    }
    public void commit() throws SQLException {
        conn.commit();
    }
    public void rollback() throws SQLException {
        conn.rollback();
    }
    /**
     * SQLの直接実行
     * @param sql
     * @return
     * @throws SQLException
     */
    public boolean executeSQL(String sql) throws SQLException {
        SQLLog log = new SQLLog(sessionID);
        log.begin(sql);
        try (Statement s = conn.createStatement()) {
            return s.execute(sql);
        } finally {
            log.end();
        }
    }
    /**
     *
     * @param sql
     * @param args
     * @return
     * @throws SQLException
     */
    public boolean executeSQL(String sql, Object... args) throws SQLException {
        SQLLog log = new SQLLog(sessionID);
        log.begin(sql);
        try (PreparedStatement s = conn.prepareStatement(sql)) {
            int index = 1;
            for (Object obj : args) {
                if (obj == null) {
                    log.paramlog(index, "NULL");
                    s.setNull(index++, Types.NULL); // NULLで良いの？
                } else if(obj instanceof XmlStore) {
                    XmlStore store = (XmlStore)obj;
                    BindResolver adapter = def.getAdapter();
                    adapter.bind(conn, s, store, index++);
                } else if(obj instanceof InputStream) {
                    InputStream is = (InputStream)obj;
                    Blob blob = conn.createBlob();
                    try (OutputStream os = blob.setBinaryStream(1)) {
                        StreamHelper.bufferdCopy(is, os);
                        s.setBlob(index++, blob);
                    }
                    catch(IOException e) {
                        throw new SQLException(e.getMessage(), e);
                    }
                } else {
                    log.paramlog(index, obj.toString());
                    s.setObject(index++, obj);
                }
            }
            return s.execute();
        } finally {
            log.end();
        }
    }
}
