/*!
******************************************************************************

	@file	x86.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "printf.h"
#include "atomic.h"
#include "cpu.h"
#include "msr.h"

static DESCRIPTOR	gdt[GDT_ENTRY_MAX];
static DESCRIPTOR	idt[IDT_ENTRY_MAX];
static TSS  		vmm_tss;
static TSS			x86_tss[CPU_CONTEXT_MAX];
static u32			x86_tss_used;

static void init_mtrr( void );
static void init_gdt( void );
static void init_idt( void );
static void init_page_table( void );
static void init_v86( void );
static void init_tss( void );

#ifndef _VSUN86_PCSIM
static void excep_handler( EXCEPTION_HANDLER_ARGS * );
static void unknown_exception( EXCEPTION_HANDLER_ARGS * );

EXCEPTION_HANDLER excep_handler_table[32] = {
	// 00h～07h
	excep_handler,		// #DE
	excep_handler,		// #DB
	unknown_exception,	// NMI
	excep_handler,		// #BP
	excep_handler,		// #OF
	excep_handler,		// #BR
	excep_handler,		// #UD
	excep_handler,		// #NM
	// 08h～0Fh
	excep_handler,		// #DF
	unknown_exception,	// (rsvd)
	excep_handler,		// #TS
	excep_handler,		// #NP
	excep_handler,		// #SS
	excep_handler,		// #GP
	excep_handler,		// #PF
	unknown_exception,	// (rsvd)
	// 10h～17h
	excep_handler,		// #MF
	excep_handler,		// #AC
	excep_handler,		// #MC
	excep_handler,		// #XF
	unknown_exception,	// (rsvd)
	unknown_exception,	// (rsvd)
	unknown_exception,	// (rsvd)
	unknown_exception,	// (rsvd)
	// 18h～1Fh
	unknown_exception,	// (rsvd)
	unknown_exception,	// (rsvd)
	unknown_exception,	// (rsvd)
	unknown_exception,	// (rsvd)
	unknown_exception,	// (rsvd)
	unknown_exception,	// (rsvd)
	unknown_exception,	// (rsvd)
	unknown_exception	// (rsvd)
};

ALIGN(4096) static u32	vmm_page_dir[1024];
ALIGN(4096) static u32	vmm_page_tbl[1024*1024];

ALIGN(4096) static u32	v86_page_dir[1024];
ALIGN(4096) static u32	v86_page_tbl[1024];

#define V86_TSS_SIZE	(sizeof(TSS)+8192)
ALIGN(4096) static u8	_v86_tss_buf[V86_TSS_SIZE];
#define V86_TSS			((TSS *)_v86_tss_buf)

#define V86_RAM_SIZE	0x00080000
ALIGN(4096) static u8	v86_ram[V86_RAM_SIZE];

#define V86_STACK		(v86_ram + (V86_SEG_SS<<4))
#define V86_DATA_ES		(v86_ram + (V86_SEG_ES<<4))
#define V86_DATA_DS		(v86_ram + (V86_SEG_DS<<4))
#define V86_CODE		(v86_ram + (V86_SEG_CS<<4))

#endif //!_VSUN86_PCSIM

bool x86_init( void )
{
	// MTRRを初期化する
	init_mtrr();

	// GDT/IDTを初期化する
	init_gdt();
	init_idt();

	// ページテーブルを初期化する
	init_page_table();

	// 仮想86モードを使えるようにする
	init_v86();

	// CR0/CR3/CR4をセットする
	// ※以降、ページ保護と仮想86モード拡張が有効になる
	u32 cr0;
	X86_SET_CR( 3, vmm_page_dir );
	X86_GET_CR( 0, cr0 );
	X86_SET_CR( 0, (cr0 | CR0_PG | CR0_WP) & ~(CR0_CD | CR0_NW) );
	X86_SET_CR( 4, CR4_VME );

	// VMMのTSSを初期化する
	init_tss();

	return true;
}

static const char * get_mtrr_type_name( u8 type )
{
	switch ( type )
	{
	case 0x00:	return "UC";
	case 0x01:	return "WC";
	case 0x04:	return "WT";
	case 0x05:	return "WP";
	case 0x06:	return "WB";
	default:	return "??";
	}
}

static void init_mtrr( void )
{
	vmm_printf( VMM_DEBUG, "[MTRR Configuration]\n" );

	for ( int i=0; i<8; i++ ) {
		const u64 mtrr_mask = rdmsr( MSR_MTRRphysMask(i) );
		if ( (mtrr_mask & MTRRphysMask_V) == 0 )
			continue;
		const u64 mtrr_base = rdmsr( MSR_MTRRphysBase(i) );
		const u64 phys_base = mtrr_base & 0x000FFFFFFFFFF000ULL;
		const u64 phys_mask = mtrr_mask & 0x000FFFFFFFFFF000ULL;
		vmm_printf( VMM_DEBUG, "base=%016llx, mask=%016llx ... %02x (%s)\n",
					phys_base, phys_mask,
					(u8)mtrr_base, get_mtrr_type_name( (u8)mtrr_base ) );
	}

	u64 mtrr;
	mtrr = rdmsr( MSR_MTRRfix64K_00000 );
	vmm_printf( VMM_DEBUG, "0000000000000000-000000000007ffff ... %016llx (%s",
				mtrr, get_mtrr_type_name( (u8)mtrr ) );
	for ( int i=1; i<8; i++ ) {
		mtrr >>= 8;
		vmm_printf( VMM_DEBUG, "/%s", get_mtrr_type_name( (u8)mtrr ) );
	}
	vmm_printf( VMM_DEBUG, ")\n" );

	for ( int i=0; i<2; i++ ) {
		const u8 index = 8+i*2;
		mtrr = rdmsr( MSR_MTRRfix16K_80000 + i );
		vmm_printf( VMM_DEBUG, "00000000000%x0000-00000000000%xffff ... %016llx (%s",
					index, index+1, mtrr, get_mtrr_type_name( (u8)mtrr ) );
		for ( int j=1; j<8; j++ ) {
			mtrr >>= 8;
			vmm_printf( VMM_DEBUG, "/%s", get_mtrr_type_name( (u8)mtrr ) );
		}
		vmm_printf( VMM_DEBUG, ")\n" );
	}

	for ( int i=0; i<8; i++ ) {
		const u8 index = 0xC0+(i<<3);
		mtrr = rdmsr( MSR_MTRRfix4K_C0000 + i );
		vmm_printf( VMM_DEBUG, "00000000000%02x000-00000000000%02xfff ... %016llx (%s",
					index, index+7, mtrr, get_mtrr_type_name( (u8)mtrr ) );
		for ( int j=1; j<8; j++ ) {
			mtrr >>= 8;
			vmm_printf( VMM_DEBUG, "/%s", get_mtrr_type_name( (u8)mtrr ) );
		}
		vmm_printf( VMM_DEBUG, ")\n" );
	}
}

static void init_gdt( void )
{
	BASELIMIT gdtr = { sizeof(gdt)-1, (u32)gdt };

	gdt[0].low32  = 0x00000000;
	gdt[0].high32 = 0x00000000;

	x86_set_gdt_desc( SEG_VMM_CODE, 0x00000000, 0xFFFFFFFF, DESC_TYPE_CODE32 );
	x86_set_gdt_desc( SEG_VMM_DATA, 0x00000000, 0xFFFFFFFF, DESC_TYPE_DATA32 );

	X86_LGDT( gdtr );
}

static void init_idt( void )
{
	BASELIMIT idtr = { sizeof(idt)-1, (u32)idt };
	extern u32 int_ptr_table[IDT_ENTRY_MAX];	// @ entry.asm

	for ( int i=0; i<IDT_ENTRY_MAX; i++ )
		x86_set_idt_desc( (u8)i, SEG_VMM_CODE, int_ptr_table[i], DESC_TYPE_INT32 );

	X86_LIDT( idtr );
}

static void init_page_table( void )
{
	u32 i;

	i = 0;
	for ( ; i<256; i++ )
		vmm_page_dir[i] = (u32)(&vmm_page_tbl[i<<10]) | PDE_KERNEL | PDE_READWRITE | PDE_PRESENT;
	for ( ; i<512; i++ )
		vmm_page_dir[i] = 0;
	for ( ; i<1024; i++ )
		vmm_page_dir[i] = (u32)(&vmm_page_tbl[i<<10]) | PDE_KERNEL | PDE_READWRITE | PDE_PRESENT;

	vmm_page_tbl[0] = 0;
	i = 1;
	for ( ; i<0x000A0; i++ )	// 0x00001000-0x0009FFFF <- 0x00001000-0x0009FFFF (read-only)
		vmm_page_tbl[i] = (u32)(i << 12) | PTE_KERNEL | PTE_READONLY | PTE_PRESENT;
	for ( ; i<0x40000; i++ )	// 0x000A0000-0x3FFFFFFF <- 0x000A0000-0x3FFFFFFF
		vmm_page_tbl[i] = (u32)(i << 12) | PTE_KERNEL | PTE_READWRITE | PTE_PRESENT;
	for ( ; i<0x80000; i++ )	// 0x40000000-0x7FFFFFFF <- (invalid)
		vmm_page_tbl[i] = 0;
	for ( ; i<0xC0000; i++ )	// 0x80000000-0xBFFFFFFF <- 0x00000000-0x3FFFFFFF (キャッシュ無効)
		vmm_page_tbl[i] = (u32)((i & 0x7FFFF) << 12) | PTE_KERNEL | PTE_READWRITE | PTE_PRESENT | PTE_PWT | PTE_PCD;
	for ( ; i<0x100000; i++ )	// 0xC0000000-0xFFFFFFFF <- 0xC0000000-0xFFFFFFFF (キャッシュ無効)
		vmm_page_tbl[i] = (u32)(i << 12) | PTE_KERNEL | PTE_READWRITE | PTE_PRESENT | PTE_PWT | PTE_PCD;

	// ページテーブルがあるページはキャッシュを無効にする
	for ( i=0; i<1024; i++ )
		x86_set_pte( &vmm_page_tbl[i<<10], x86_get_pte( &vmm_page_tbl[i<<10] ) | PTE_PWT | PTE_PCD );
}

static void init_v86( void )
{
	u32 i;

	u32 *src = (u32 *)0x00000000;
	u32 *dst = (u32 *)v86_ram;
	for ( int i=0; i<V86_RAM_SIZE>>2; i++ )
		dst[i] = src[i];

	TSS *tss = V86_TSS;
	memset( tss, 0, V86_TSS_SIZE );

	tss->ss0		= SEG_VMM_DATA;
	tss->esp0		= (u32)&vmm_int_stack[VMM_INT_STACK_SIZE - 4];
	tss->cr3		= (u32)v86_page_dir;
	tss->eflags		= 0x00000002 | F_VM;
	tss->iomap_off	= (u16)offsetof( TSS, iomap );
	tss->fpu.cw		= 0x037F;
	tss->fpu.sw		= 0x0000;
	tss->fpu.tw		= 0xFFFF;
	tss->iomap[2048]= 0xFFFFFFFF;

	x86_set_gdt_desc( SEG_V86_TSS, (u32)V86_TSS, V86_TSS_SIZE-1, DESC_TYPE_TSS32 );

	v86_page_dir[0] = (u32)v86_page_tbl | PDE_USER | PDE_READWRITE | PDE_PRESENT;
	for ( i=1; i<1024; i++ )
		v86_page_dir[i] = vmm_page_dir[i];

	i=0;
	for ( ; i<0x080; i++ )	// 0x00000000-0x0007FFFF <- v86_ram
		v86_page_tbl[i] = (u32)&v86_ram[i << 12] | PTE_USER | PTE_READWRITE | PTE_PRESENT;
	for ( ; i<0x100; i++ )	// 0x00080000-0x000FFFFF <- 0x00080000-0x000FFFFF
		v86_page_tbl[i] = (i << 12) | PTE_USER | PTE_READWRITE | PTE_PRESENT;
	for ( ; i<0x400; i++ )	// 0x00100000-0x003FFFFF <- (invalid)
		v86_page_tbl[i] = 0;

	u8 *v86_code = V86_CODE;
	for ( i=0; i<256; i++ ) {
		v86_code[(i<<2)+0] = 0xCD;		// int
		v86_code[(i<<2)+1] = (u8)i;		// n (=0x00～0xFF)
		v86_code[(i<<2)+2] = 0x0F;		//
		v86_code[(i<<2)+3] = 0x0B;		// ud2
	}
}

static void init_tss( void )
{
	TSS *tss = &vmm_tss;

	memset( tss, 0, sizeof(TSS) );
	tss->ss0		= SEG_VMM_DATA;
	tss->esp0		= (u32)&vmm_int_stack[VMM_INT_STACK_SIZE - 4];
	tss->eflags		= 0x00000002;
	tss->iomap_off	= (u16)offsetof( TSS, iomap );
	tss->fpu.cw		= 0x037F;
	tss->fpu.sw		= 0x0000;
	tss->fpu.tw		= 0xFFFF;
	tss->intmap[0]	= 0xFFFFFFFF;
	tss->intmap[1]	= 0xFFFFFFFF;
	tss->intmap[2]	= 0xFFFFFFFF;
	tss->intmap[3]	= 0xFFFFFFFF;
	tss->intmap[4]	= 0xFFFFFFFF;
	tss->intmap[5]	= 0xFFFFFFFF;
	tss->intmap[6]	= 0xFFFFFFFF;
	tss->intmap[7]	= 0xFFFFFFFF;
	tss->iomap[0]	= 0xFFFFFFFF;
	X86_GET_CR( 3, tss->cr3 );
	x86_set_gdt_desc( SEG_VMM_TSS, (u32)tss, sizeof(TSS)-1, DESC_TYPE_TSS32 );
	X86_LTR( SEG_VMM_TSS );

	x86_tss_used = 0;
}

TSS * x86_alloc_tss( const void *code, const void *stack, size_t stack_size )
{
	if ( x86_tss_used >= CPU_CONTEXT_MAX )
		return NULL;

	TSS *tss = &x86_tss[x86_tss_used++];
	memset( tss, 0, sizeof(TSS) );
	tss->ss0		= SEG_VMM_DATA;
	tss->esp0		= (u32)&vmm_int_stack[VMM_INT_STACK_SIZE - 4];
	tss->cs			= SEG_VMM_CODE;
	tss->eip		= (u32)code;
	tss->eflags		= 0x00000002 | F_IF;
	tss->ss			= SEG_VMM_DATA;
	tss->esp		= (u32)((const u8 *)stack + stack_size - 4);
	tss->ds			= SEG_VMM_DATA;
	tss->es			= SEG_VMM_DATA;
//	tss->fs			= SEG_FS_GS( id );
//	tss->gs			= SEG_FS_GS( id );
	tss->iomap_off	= (u16)offsetof( TSS, iomap );
	tss->fpu.cw		= 0x037F;
	tss->fpu.sw		= 0x0000;
	tss->fpu.tw		= 0xFFFF;
	tss->intmap[0]	= 0xFFFFFFFF;
	tss->intmap[1]	= 0xFFFFFFFF;
	tss->intmap[2]	= 0xFFFFFFFF;
	tss->intmap[3]	= 0xFFFFFFFF;
	tss->intmap[4]	= 0xFFFFFFFF;
	tss->intmap[5]	= 0xFFFFFFFF;
	tss->intmap[6]	= 0xFFFFFFFF;
	tss->intmap[7]	= 0xFFFFFFFF;
	tss->iomap[0]	= 0xFFFFFFFF;
	X86_GET_CR( 3, tss->cr3);

	return tss;
}

void x86_switch_task( register TSS *cur_tss, register TSS *new_tss )
{
	__ASM__(
		"movl	$task_switch_return,  %[cur_eip]	\n\t"
		"pushfl										\n\t"
		"popl	%[cur_eflags]						\n\t"
		"movl	%%ecx, %[cur_ecx]					\n\t"
		"movl	%%edx, %[cur_edx]					\n\t"
		"movl	%%ebx, %[cur_ebx]					\n\t"
		"movl	%%esp, %[cur_esp]					\n\t"
		"movl	%%ebp, %[cur_ebp]					\n\t"
		"movl	%%esi, %[cur_esi]					\n\t"
		"movl	%%edi, %[cur_edi]					\n\t"
		"pushl	%[new_eflags]						\n\t"
		"popfl										\n\t"
		"movl	%[new_esp], %%esp					\n\t"
		"pushl	%[new_eip]							\n\t"
		"pushl	%[new_ecx]							\n\t"
		"pushl	%[new_edx]							\n\t"
		"pushl	%[new_ebx]							\n\t"
		"pushl	%[new_esp]							\n\t"
		"pushl	%[new_ebp]							\n\t"
		"pushl	%[new_esi]							\n\t"
		"pushl	%[new_edi]							\n\t"
		"popal										\n\t"
		"jmp	*%%eax								\n\t"
"task_switch_return:								\n\t"
		:	  [cur_eip]		"=o"(cur_tss->eip)
			, [cur_eflags]	"=o"(cur_tss->eflags)
			, [cur_ecx]		"=o"(cur_tss->ecx)
			, [cur_edx]		"=o"(cur_tss->edx)
			, [cur_ebx]		"=o"(cur_tss->ebx)
			, [cur_esp]		"=o"(cur_tss->esp)
			, [cur_ebp]		"=o"(cur_tss->ebp)
			, [cur_esi]		"=o"(cur_tss->esi)
			, [cur_edi]		"=o"(cur_tss->edi)
		:	  [new_eip]		"a" (new_tss->eip)
			, [new_eflags]	"o" (new_tss->eflags)
			, [new_ecx]		"o" (new_tss->ecx)
			, [new_edx]		"o" (new_tss->edx)
			, [new_ebx]		"o" (new_tss->ebx)
			, [new_esp]		"o" (new_tss->esp)
			, [new_ebp]		"o" (new_tss->ebp)
			, [new_esi]		"o" (new_tss->esi)
			, [new_edi]		"o" (new_tss->edi)
		:	"memory", "cc"
	);
}

static bool x86_switch_fpu_regs( void )
{
	static u16 tr_fpu = SEG_VMM_TSS;
	static TSS *tss_fpu = &vmm_tss;

	u16 tr_cur;
	X86_STR( tr_cur );
	if ( tr_cur == SEG_V86_TSS )
		return false;
	__ASM__( "clts" );
	if ( tr_cur == tr_fpu )
		return true;

	TSS *tss_cur;
	if ( tr_cur == SEG_VMM_TSS )
		tss_cur = &vmm_tss;
	else
//		tss_cur = task_info[(tr_fpu - SEG_TSS(0)) / 0x20].tss;
		return false;

	FPU_REGS *old_regs = &tss_fpu->fpu;
	FPU_REGS *new_regs = &tss_cur->fpu;
	__ASM__(
		"fnsave		(%0)	\n\t"
		"frstor		(%1)	\n\t"
		:: "r"(old_regs), "r"(new_regs)
	);

	/*
	vmm_printf( VMM_DEBUG, "x86_switch_fpu_regs(): tss=%04x/%04x\n", tr_fpu, tr_cur );
	vmm_printf( VMM_DEBUG, "fpu(old): cw=%04x, sw=%04x, tw=%04x\n",
				old_regs->cw, old_regs->sw, old_regs->tw );
	vmm_printf( VMM_DEBUG, "fpu(new): cw=%04x, sw=%04x, tw=%04x\n",
				new_regs->cw, new_regs->sw, new_regs->tw );
	*/

	tr_fpu  = tr_cur;
	tss_fpu = tss_cur;

	return true;
}

void x86_set_gdt_desc( u16 seg, u32 base, u32 limit, u16 flags )
{
	DESCRIPTOR *desc = &gdt[seg >> 3];

	desc->seg.base_b15_0  = (u16)base;
	desc->seg.base_b23_16 = (u8)(base >> 16);
	desc->seg.base_b31_24 = (u8)(base >> 24);

	desc->seg.flags = flags;
	desc->seg.present = 1;

	if ( limit <= 0xFFFFF ) {
		desc->seg.limit_b15_0  = (u16)limit;
		desc->seg.limit_b23_16 = (u8)(limit >> 16);
	}
	else {
		desc->seg.granularity  = 1;
		desc->seg.limit_b15_0  = (u16)(limit >> 12);
		desc->seg.limit_b23_16 = (u8)((limit >> 28) & 0x0F);
	}
}

void x86_set_idt_desc( u8 n, u16 seg, u32 off, u8 type )
{
	DESCRIPTOR *desc = &idt[n];

	desc->gate.seg	 = seg;
	desc->gate.off_L = (u16)off;
	desc->gate.off_H = (u16)(off >> 16);
	desc->gate.count = 0;
	desc->gate.type  = type | DESC_PRESENT;
}

static void excep_handler( EXCEPTION_HANDLER_ARGS *p )
{
	u32 cr0;
	X86_GET_CR( 0, cr0 );
	if ( (p->n == 7) && (cr0 & CR0_TS) )
	{	// #NM && CR0.TS==1
		if ( x86_switch_fpu_regs() )
			return;
	}

	if ( p->eflags & F_VM ) {
		TSS *tss = V86_TSS;
		tss->v86.eip	= p->eip;
		tss->v86.eflags	= p->eflags;
		tss->v86.eax	= p->eax;
		tss->v86.ecx	= p->ecx;
		tss->v86.edx	= p->edx;
		tss->v86.ebx	= p->ebx;
		tss->v86.esp	= p->r3esp;
		tss->v86.ebp	= p->ebp;
		tss->v86.esi	= p->esi;
		tss->v86.edi	= p->edi;
		tss->v86.es		= p->r3es;
		tss->v86.cs		= p->cs;
		tss->v86.ss		= p->r3ss;
		tss->v86.ds		= p->r3ds;
		tss->v86.fs		= p->r3fs;
		tss->v86.gs		= p->r3gs;
		if ( ((p->cs<<4) + p->eip) == tss->v86.ret_addr )
		{	// VMMに戻る
			__ASM__( "push		%eax			\n\t"
					 "pushfl					\n\t"
					 "pop		%eax			\n\t"
					 "or		$0x4000, %eax	\n\t"
					 "push		%eax			\n\t"
					 "popfl						\n\t"
					 "pop		%eax			\n\t"
					 "iret						\n\t"
			);
		}
		vmm_printf( VMM_ERROR, "Exception %02x!!! (err=%08x)\n", p->n, p->err );
		vmm_printf( VMM_ERROR, "cs:eip=%04x:%08x eflags=%08x\n", p->cs, p->eip, p->eflags );
		vmm_printf( VMM_ERROR, "eax=%08x ecx=%08x edx=%08x ebx=%08x\n", p->eax, p->ecx, p->edx, p->ebx );
		vmm_printf( VMM_ERROR, "esp=%08x ebp=%08x esi=%08x edi=%08x\n", p->r3esp, p->ebp, p->esi, p->edi );
		vmm_printf( VMM_ERROR, "ss=%04x es=%04x ds=%04x fs=%04x gs=%04x\n",
					p->r3ss, p->r3es, p->r3ds, p->r3fs, p->r3gs );
	}
	else {
		if ( p->n == EXCEPTION_DB )
		{	// #DB
			u32 dr6;
			X86_GET_DR( 6, dr6 );
			if ( dr6 & DR6_BS ) {
				dr6 &= ~DR6_BS;
				X86_SET_DR( 6, dr6 );
				vmm_printf( VMM_DEBUG, "cs:eip=%04x:%08x eflags=%08x\n", p->cs, p->eip, p->eflags );
				return;
			}
		}
		u32 cr2, cr3, cr4;
		X86_GET_CR( 2, cr2 );
		X86_GET_CR( 3, cr3 );
		X86_GET_CR( 4, cr4 );
		vmm_printf( VMM_ERROR, "Exception %02x!!! (err=%08x)\n", p->n, p->err );
		vmm_printf( VMM_ERROR, "cs:eip=%04x:%08x eflags=%08x\n", p->cs, p->eip, p->eflags );
		vmm_printf( VMM_ERROR, "eax=%08x ecx=%08x edx=%08x ebx=%08x\n", p->eax, p->ecx, p->edx, p->ebx );
		vmm_printf( VMM_ERROR, "esp=%08x ebp=%08x esi=%08x edi=%08x\n", p->esp, p->ebp, p->esi, p->edi );
		vmm_printf( VMM_ERROR, "cr0=%08x cr2=%08x cr3=%08x cr4=%08x\n", cr0, cr2, cr3, cr4 );
	}

	while ( 1 )
		halt();
}

static void unknown_exception( EXCEPTION_HANDLER_ARGS *p )
{
	u32 cr0, cr2, cr3, cr4;

	X86_GET_CR( 0, cr0 );
	X86_GET_CR( 2, cr2 );
	X86_GET_CR( 3, cr3 );
	X86_GET_CR( 4, cr4 );

	vmm_printf( VMM_ERROR, "Unexpected Exception %02x!!!\n", p->n );
	vmm_printf( VMM_ERROR, "cs:eip=%04x:%08x eflags=%08x\n", p->cs, p->eip, p->eflags );
	vmm_printf( VMM_ERROR, "eax=%08x ecx=%08x edx=%08x ebx=%08x\n", p->eax, p->ecx, p->edx, p->ebx );
	vmm_printf( VMM_ERROR, "esp=%08x ebp=%08x esi=%08x edi=%08x\n", p->esp, p->ebp, p->esi, p->edi );
	vmm_printf( VMM_ERROR, "cr0=%08x cr2=%08x cr3=%08x cr4=%08x\n", cr0, cr2, cr3, cr4 );

	while ( 1 )
		halt();
}

void x86_v86int( u8 vector, V86INT_REGS *regs )
{
	static volatile int v86run = 0;

	lock( &v86run );

	TSS *tss = V86_TSS;

	tss->eax	= regs->eax;
	tss->ecx	= regs->ecx;
	tss->edx	= regs->edx;
	tss->ebx	= regs->ebx;
	tss->ebp	= regs->ebp;
	tss->esi	= regs->esi;
	tss->edi	= regs->edi;
	tss->es		= V86_SEG_ES;
	tss->ds		= V86_SEG_DS;
	tss->cs		= V86_SEG_CS;
	tss->eip	= vector << 2;
	tss->eflags = F_VM | (regs->eflags & (F_OF|F_DF|F_SF|F_ZF|F_AF|F_PF|F_CF));
	tss->ss		= V86_SEG_SS;
	tss->esp	= 0xFFFE;
	tss->v86.ret_addr = (V86_SEG_CS << 4) + (vector << 2) + 2;

	__ASM__( "lcall	%[v86_tss],$0" :: [v86_tss] "i"(SEG_V86_TSS) );

	regs->eax	 = tss->v86.eax;
	regs->ecx	 = tss->v86.ecx;
	regs->edx	 = tss->v86.edx;
	regs->ebx	 = tss->v86.ebx;
	regs->ebp	 = tss->v86.ebp;
	regs->esi	 = tss->v86.esi;
	regs->edi	 = tss->v86.edi;
	regs->eflags = tss->v86.eflags & (F_OF|F_DF|F_SF|F_ZF|F_AF|F_PF|F_CF);
	regs->ret_es = tss->v86.es;
	regs->ret_ds = tss->v86.ds;
	if ( regs->es_base != NULL )
		memcpy( regs->es_base, V86_DATA_ES, regs->es_size );
	if ( regs->ds_base != NULL )
		memcpy( regs->ds_base, V86_DATA_DS, regs->ds_size );

	unlock( &v86run );
}

u32 x86_get_pte( void *p )
{
	const u32 va = (u32)p;

	__ASM__( "invlpg %0" :"=m"(p) );
	return vmm_page_tbl[va >> 12];
}

void x86_set_pte( void *p, u32 pte )
{
	const u32 va = (u32)p;

	vmm_page_tbl[va >> 12] = pte;
}

void x86_map_page( u32 pa )
{
	u32 *pde = &vmm_page_dir[pa >> 22];
	u32 *pte = &vmm_page_tbl[pa >> 12];

	if ( *pte & PTE_PRESENT )
		return;

	if ( !(*pde & PDE_PRESENT) ) {
		for ( int i=0; i<1024; i++ )
			vmm_page_tbl[((pa & 0xFFC00000)>>12)+i] = 0;
		*pde = (u32)&vmm_page_tbl[(pa & 0xFFC00000)>>12] | PDE_KERNEL | PDE_READWRITE | PDE_PRESENT;
	}
	*pte = (pa & 0xFFFFF000) | PTE_KERNEL | PTE_READWRITE | PTE_PRESENT | PTE_PWT | PTE_PCD;
}

void x86_enable_single_step_exec( void )
{
	__ASM__(	"pushfl						\n\t"
				"popl		%%eax			\n\t"
				"orl		$0x100, %%eax	\n\t"
				"pushl		%%eax			\n\t"
				"popfl						\n\t"
			:
			:
			:	"memory", "cc" );
}

void x86_disable_single_step_exec( void )
{
	__ASM__(	"pushfl							\n\t"
				"popl		%%eax				\n\t"
				"andl		$0xFFFFFEFF, %%eax	\n\t"
				"pushl		%%eax				\n\t"
				"popfl							\n\t"
			:
			:
			:	"memory", "cc" );
}
