/*!
******************************************************************************

	@file	vbe.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "cpu.h"
#include "vbe.h"
#include "printf.h"
#include "vbe_ddc.h"
#include "io.h"

VBE_CTRL_INFO vbe_ctrl_info;
VBE_MODE_INFO vbe_mode_info;

static void * vbe_pmi_func05;
static void * vbe_pmi_func07;
static void * vbe_pmi_func09;

static bool vbe_get_controller_info( VBE_CTRL_INFO * );
static bool vbe_get_mode_info( u16, VBE_MODE_INFO * );
static bool vbe_set_mode( u16 );
static bool vbe_get_pm_interface( u16 **, u16 * );

static void vbe_print_ctrl_info( const VBE_CTRL_INFO * );
static void vbe_print_mode_info( u16, const VBE_MODE_INFO * );

bool vbe_probe( u16 width, u16 height, u8 bpp )
{
	// コントローラ情報を取得する
	memcpy( vbe_ctrl_info.id, "VBE2", 4 );
	if ( !vbe_get_controller_info( &vbe_ctrl_info ) )
		return false;	// 取得に失敗
	if ( strncmp( vbe_ctrl_info.id, "VESA", 4 ) != 0 )
		return false;	// 識別子が「VESA」じゃない
	vbe_print_ctrl_info( &vbe_ctrl_info );
	if ( vbe_ctrl_info.version < 0x0200 )
		return false;	// バージョンが2.0より古い

	int i = 0;
	u16 mode;
	while ( i < 10000 )
	{	// width/height/bppが一致する画面モードを探す
		mode = vbe_ctrl_info.video_mode[i++];
		if ( mode == 0xFFFF )
			return false;	// 不正なモード
		if ( !vbe_get_mode_info( mode, &vbe_mode_info ) )
			return false;	// 取得に失敗
		if ( !vbe_mode_info.mode_attr.supported )
			continue;		// モード非対応
		if ( !vbe_mode_info.mode_attr.linear_fb )
			continue;		// リニアフレームバッファ非対応
		if ( (vbe_mode_info.x_res == width ) &&
			 (vbe_mode_info.y_res == height) &&
			 (vbe_mode_info.bpp   == bpp   ) )
			break;	// 発見
	}
	vbe_print_mode_info( vbe_ctrl_info.version, &vbe_mode_info );

	// 32-bitプロテクトモードI/Fの取得を試みる
	u16 *pmi_table = NULL;
	u16 pmi_table_len;
	if ( vbe_get_pm_interface( &pmi_table, &pmi_table_len ) )
	{	// 取得に成功
		vmm_printf( VMM_DEBUG, "vbe: PM I/F found @ %08x (%u bytes)\n", pmi_table, pmi_table_len );
		for ( i=0; i<pmi_table_len; i++ ) {
			const u8 *addr = (u8 *)pmi_table + i;
			switch ( (u32)addr & 0x0F )
			{
			case 0x00:
				vmm_printf( VMM_DEBUG, "%08x: %02x", addr, *addr );
				break;
			case 0x0F:
				vmm_printf( VMM_DEBUG, " %02x\n", *addr );
				break;
			default:
				vmm_printf( VMM_DEBUG, " %02x", *addr );
				break;
			}
		}
		vbe_pmi_func05 = (u8 *)pmi_table + pmi_table[0];
		vbe_pmi_func07 = (u8 *)pmi_table + pmi_table[1];
		vbe_pmi_func09 = (u8 *)pmi_table + pmi_table[2];
		vmm_printf( VMM_DEBUG, " Function 05h ... %08x\n", vbe_pmi_func05 );
		vmm_printf( VMM_DEBUG, " Function 07h ... %08x\n", vbe_pmi_func07 );
		vmm_printf( VMM_DEBUG, " Function 09h ... %08x\n", vbe_pmi_func09 );
	}
	else
	{	// 取得に失敗
		vmm_printf( VMM_DEBUG, "vbe: PM I/F not found.\n" );
		vbe_pmi_func05 = NULL;
		vbe_pmi_func07 = NULL;
		vbe_pmi_func09 = NULL;
	}

	// VBEを有効にする
	if ( !vbe_set_mode( mode ) )
		return false;

	// VBE/DDCを初期化する
	// ※初期化に失敗しても気にしない
	vbe_ddc_init();

	return true;
}

bool vbe_check_status( u16 status )
{
	if ( status != 0x004F )
		return false;
	return true;
}

static void * vbe_segoff_to_linear( void *data_ptr, u32 segoff )
{
	const u16 seg = segoff >> 16;
	const u16 off = segoff & 0xFFFF;

	if ( seg == V86_SEG_ES )
		return (u8 *)data_ptr + off;
	else if ( seg >= 0xA000 )
		return (u8 *)(seg << 4) + off;
	else
		return NULL;
}

static bool vbe_get_controller_info( VBE_CTRL_INFO *dst )
{
	V86INT_REGS regs;
	memset( &regs, 0, sizeof(regs) );

	regs.eax	 = 0x4F00;	// Return VBE Controller Information
	regs.es_base = dst;
	regs.es_size = sizeof(VBE_CTRL_INFO);
	regs.edi	 = 0;

	x86_v86int( 0x10, &regs );
	const u16 status = (u16)regs.eax;

	if ( !vbe_check_status( status ) ) {
		vmm_printf( VMM_DEBUG, "vbe: Function 00h failed. (status=%04x)\n", status );
		return false;
	}

	dst->oem_string   = (char *)vbe_segoff_to_linear( dst, (u32)dst->oem_string   );
	dst->video_mode   = (u16  *)vbe_segoff_to_linear( dst, (u32)dst->video_mode   );
	dst->vendor_name  = (char *)vbe_segoff_to_linear( dst, (u32)dst->vendor_name  );
	dst->product_name = (char *)vbe_segoff_to_linear( dst, (u32)dst->product_name );
	dst->product_rev  = (char *)vbe_segoff_to_linear( dst, (u32)dst->product_rev  );

	return true;
}

static bool vbe_get_mode_info( u16 mode, VBE_MODE_INFO *dst )
{
	V86INT_REGS regs;
	memset( &regs, 0, sizeof(regs) );

	regs.eax	 = 0x4F01;	// Return VBE Mode Information
	regs.ecx	 = mode;
	regs.es_base = dst;
	regs.es_size = sizeof(VBE_MODE_INFO);
	regs.edi	 = 0;

	x86_v86int( 0x10, &regs );
	const u16 status = (u16)regs.eax;

	if ( !vbe_check_status( status ) ) {
		vmm_printf( VMM_DEBUG, "vbe: Function 01h failed. (status=%04x)\n", status );
		return false;
	}
	dst->win_func = vbe_segoff_to_linear( dst, (u32)dst->win_func );

	return true;
}

static bool vbe_set_mode( u16 mode )
{
	V86INT_REGS regs;
	memset( &regs, 0, sizeof(regs) );

	mode |= 0x4000;		// リニアフレームバッファを利用

	regs.eax = 0x4F02;	// Set VBE Mode
	regs.ebx = mode;

	x86_v86int( 0x10, &regs );
	const u16 status = (u16)regs.eax;

	if ( !vbe_check_status( status ) ) {
		vmm_printf( VMM_DEBUG, "vbe: Function 02h failed. (status=%04x)\n", status );
		return false;
	}
	return true;
}

static bool vbe_get_pm_interface( u16 **table, u16 *table_len )
{
	V86INT_REGS regs;
	memset( &regs, 0, sizeof(regs) );

	regs.eax = 0x4F0A;	// VBE 2.0 Protected Mode Interface
	regs.ebx = 0x0000;	// Return protected mode table

	x86_v86int( 0x10, &regs );
	const u16 status = (u16)regs.eax;

	if ( !vbe_check_status( status ) ) {
		vmm_printf( VMM_DEBUG, "vbe: Function 0Ah failed. (status=%04x)\n", status );
		return false;
	}
	*table = (u16 *)vbe_segoff_to_linear( NULL, (regs.ret_es << 16) | (regs.edi & 0x0000FFFF) );
	*table_len = (u16)regs.ecx;

	return true;
}

bool vbe_set_disp_start( u32 offset )
{
	if ( vbe_pmi_func07 != NULL ) {
		if ( vbe_ctrl_info.version < 0x0300 )
			offset = offset >> 2;
		else
			offset = (offset >> 2) | (offset << 30);
		__ASM__( "call *(%%eax)"
				 :: "a"(vbe_pmi_func07), "b"(0x0080), "c"(offset & 0xFFFF), "d"(offset >> 16) );
		return true;
	}

	if ( vbe_ctrl_info.version < 0x0300 )
		return false;

	V86INT_REGS regs;
	memset( &regs, 0, sizeof(regs) );

	regs.eax = 0x4F07;	// Set/Get Display Start
	regs.ebx = 0x0002;	// Schedule Display Start (Alternate)
	regs.ecx = offset;

	x86_v86int( 0x10, &regs );
	const u16 status = (u16)regs.eax;

	if ( !vbe_check_status( status ) ) {
		vmm_printf( VMM_DEBUG, "vbe: Function 07h failed. (status=%04x)\n", status );
		return false;
	}
	return true;
}

static void vbe_print_ctrl_info( const VBE_CTRL_INFO *info )
{
	vmm_printf( VMM_DEBUG, "[VbeInfoBlock]\n" );
	vmm_printf( VMM_DEBUG, "VbeSignature ..... \'%c%c%c%c\'\n",
				info->id[0], info->id[1], info->id[2], info->id[3] );
	vmm_printf( VMM_DEBUG, "VbeVersion ....... %04x\n", info->version );
	vmm_printf( VMM_DEBUG, "OemString ........ \"%s\"\n", info->oem_string );
	vmm_printf( VMM_DEBUG, "Capabilities ..... %02x:%02x:%02x:%02x\n",
				info->capabilities[0], info->capabilities[1], info->capabilities[2], info->capabilities[3] );
	vmm_printf( VMM_DEBUG, "VideoMode ........" );
	for ( int i=0; info->video_mode[i] != 0xFFFF; i++ )
		vmm_printf( VMM_DEBUG, " %04x", info->video_mode[i] );
	vmm_printf( VMM_DEBUG, "\n" );
	vmm_printf( VMM_DEBUG, "TotalMemory ...... %04x (%d bytes)\n",
				info->total_memory, info->total_memory << 16 );
	if ( info->version >= 0x0200 ) {
		vmm_printf( VMM_DEBUG, "OemSoftwareRev ... %04x\n", info->revision );
		vmm_printf( VMM_DEBUG, "OemVendorName .... \"%s\"\n", info->vendor_name );
		vmm_printf( VMM_DEBUG, "OemProductName ... \"%s\"\n", info->product_name );
		vmm_printf( VMM_DEBUG, "OemProductRev .... \"%s\"\n", info->product_rev );
	}
	vmm_printf( VMM_DEBUG, "\n" );
}

static void vbe_print_mode_info( u16 version, const VBE_MODE_INFO *info )
{
	static const char *mem_model[] =
	{
		"Text mode",
		"CGA graphics",
		"Hercules graphics",
		"Planar",
		"Packed pixel",
		"Non-chain 4, 256 color",
		"Direct Color",
		"YUV"
	};

	vmm_printf( VMM_DEBUG, "[VbeModeBlock]\n" );
	vmm_printf( VMM_DEBUG, "ModeAttributes .......... %04x\n", info->mode_attr.w );
	vmm_printf( VMM_DEBUG, "WinAAttributes .......... %02x\n", info->win_a_attr );
	vmm_printf( VMM_DEBUG, "WinBAttributes .......... %02x\n", info->win_b_attr );
	vmm_printf( VMM_DEBUG, "WinGranularity .......... %04x\n", info->win_granularity );
	vmm_printf( VMM_DEBUG, "WinSize ................. %04x\n", info->win_size );
	vmm_printf( VMM_DEBUG, "WinASegment ............. %04x\n", info->win_a_seg );
	vmm_printf( VMM_DEBUG, "WinBSegment ............. %04x\n", info->win_b_seg );
	vmm_printf( VMM_DEBUG, "WinFuncPtr .............. %08x\n", info->win_func );
	vmm_printf( VMM_DEBUG, "BytesPerScanLine ........ %04x\n", info->scan_line_bytes );
	if ( version >= 0x0102 ) {
		vmm_printf( VMM_DEBUG, "XResolution ............. %04x (%d)\n", info->x_res, info->x_res );
		vmm_printf( VMM_DEBUG, "YResolution ............. %04x (%d)\n", info->y_res, info->y_res );
		vmm_printf( VMM_DEBUG, "XCharSize ............... %04x (%d)\n", info->x_char_size, info->x_char_size );
		vmm_printf( VMM_DEBUG, "YCharSize ............... %04x (%d)\n", info->y_char_size, info->y_char_size );
		vmm_printf( VMM_DEBUG, "NumberOfPlanes .......... %02x\n", info->num_planes );
		vmm_printf( VMM_DEBUG, "BitsPerPixel ............ %02x (%d)\n", info->bpp, info->bpp );
		vmm_printf( VMM_DEBUG, "NumberOfBanks ........... %02x\n", info->num_banks );
		vmm_printf( VMM_DEBUG, "MemoryModel ............. %02x (%s)\n", info->mem_model,
					(info->mem_model <= VBE_MEM_MODEL_YUV)? mem_model[info->mem_model] : "Reserved" );
		vmm_printf( VMM_DEBUG, "BankSize ................ %02x\n", info->bank_size );
		vmm_printf( VMM_DEBUG, "NumberOfImagePages ...... %02x\n", info->num_img_pages );
		if ( (info->mem_model == VBE_MEM_MODEL_DIRECT) || (info->mem_model == VBE_MEM_MODEL_YUV) ) {
			vmm_printf( VMM_DEBUG, "RedMaskSize ............. %02x\n", info->r_mask_size );
			vmm_printf( VMM_DEBUG, "RedFieldPosition ........ %02x\n", info->r_field_pos );
			vmm_printf( VMM_DEBUG, "GreenMaskSize ........... %02x\n", info->g_mask_size );
			vmm_printf( VMM_DEBUG, "GreenFieldPosition ...... %02x\n", info->g_field_pos );
			vmm_printf( VMM_DEBUG, "BlueMaskSize ............ %02x\n", info->b_mask_size );
			vmm_printf( VMM_DEBUG, "BlueFieldPosition ....... %02x\n", info->b_field_pos );
			vmm_printf( VMM_DEBUG, "RsvdMaskSize ............ %02x\n", info->rsvd_mask_size );
			vmm_printf( VMM_DEBUG, "RsvdFieldPosition ....... %02x\n", info->rsvd_field_pos );
			vmm_printf( VMM_DEBUG, "DirectColorModeInfo ..... %02x\n", info->direct_mode_info );
		}
	}
	if ( version >= 0x0200 ) {
		vmm_printf( VMM_DEBUG, "PhysBasePtr ............. %08x\n", info->phys_base );
	}
	if ( version == 0x0200 ) {
		vmm_printf( VMM_DEBUG, "OffScreenMemOffset ...... %04x\n", info->off_scr_mem_off );
		vmm_printf( VMM_DEBUG, "OffScreenMemSize ........ %02x\n", info->off_scr_mem_size );
	}
	if ( version >= 0x0300 ) {
		vmm_printf( VMM_DEBUG, "LinBytesPerScanLine ..... %04x\n", info->lin_line_bytes );
		vmm_printf( VMM_DEBUG, "BnkNumberOfImagePages ... %02x\n", info->bnk_img_page_num );
		vmm_printf( VMM_DEBUG, "LinNumberOfImagePages ... %02x\n", info->lin_img_page_num );
		vmm_printf( VMM_DEBUG, "LinRedMaskSize .......... %02x\n", info->lin_r_mask_size );
		vmm_printf( VMM_DEBUG, "LinRedFieldPosition ..... %02x\n", info->lin_r_field_pos );
		vmm_printf( VMM_DEBUG, "LinGreenMaskSize ........ %02x\n", info->lin_g_mask_size );
		vmm_printf( VMM_DEBUG, "LinGreenFieldPosition ... %02x\n", info->lin_g_field_pos );
		vmm_printf( VMM_DEBUG, "LinBlueMaskSize ......... %02x\n", info->lin_b_mask_size );
		vmm_printf( VMM_DEBUG, "LinBlueFieldPosition .... %02x\n", info->lin_b_field_pos );
		vmm_printf( VMM_DEBUG, "LinRsvdMaskSize ......... %02x\n", info->lin_rsvd_mask_size );
		vmm_printf( VMM_DEBUG, "LinRsvdFieldPosition .... %02x\n", info->lin_rsvd_field_pos );
		vmm_printf( VMM_DEBUG, "MaxPixelClock ........... %08x (%d Hz)\n",
					info->max_pixel_clock, info->max_pixel_clock );
	}
	vmm_printf( VMM_DEBUG, "\n" );
}
