/*!
******************************************************************************

	@file	fat.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "disk.h"
#include "fs.h"
#include "fat.h"
#include "printf.h"

static bool fat_probe( FS_DRIVE * );
static bool fat_open ( FS_DRIVE *, FS_FILE * );
static bool fat_read ( FS_DRIVE *, FS_FILE *, void *, size_t );
static bool fat_seek ( FS_DRIVE *, FS_FILE *, long, int );

static bool fat_get_next_clus12( FS_DRIVE *, u32 * );
static bool fat_get_next_clus16( FS_DRIVE *, u32 * );

bool fat_init( FS_OPS *ops )
{
	ops->fs_id = FS_OPS_FAT;
	ops->probe = fat_probe;
	ops->open  = fat_open;
	ops->read  = fat_read;
	ops->seek  = fat_seek;

	return true;
}

static bool fat_probe( FS_DRIVE *drive )
{
	FAT_CACHE *cache	= (FAT_CACHE *)drive->cache;
	const u32 disk_id   = drive->disk_id;
	const u32 first_sec = drive->first_sec;
	const u32 block_len = 512;

	if ( drive->cache_size < sizeof(FAT_CACHE) )
	{	// キャッシュが足りない
		vmm_printf( VMM_ERROR, "FAT: probe() failed. (cache_size:%08x < %08x)\n",
					drive->cache_size, sizeof(FAT_CACHE) );
		return false;
	}

	// BPBを読む
	FAT_BPB *bpb = &cache->bpb;
	if ( !disk_read( disk_id, first_sec, 1, bpb, block_len ) )
		return false;
	cache->fat_page = 0;
	const u32 fat_sec = first_sec + bpb->rsvd_sec;
	if ( memcmp( bpb->fs_type, "FAT12   ", sizeof(bpb->fs_type) ) == 0 ) {
		drive->fs_type = FS_FAT12;
		// FATを読む
		if ( !disk_read( disk_id, fat_sec, 3, cache->fat12, sizeof(cache->fat12) ) )
			return false;
	}
	else if ( memcmp( bpb->fs_type, "FAT16   ", sizeof(bpb->fs_type) ) == 0 ) {
		drive->fs_type = FS_FAT16;
		// FATを読む
		if ( !disk_read( disk_id, fat_sec, 1, cache->fat16, sizeof(cache->fat16) ) )
			return false;
	}
	else {
		drive->fs_type = FS_UNKNOWN;
		vmm_printf( VMM_ERROR, "FAT: probe() failed. (unsupported fs type)\n" );
		return false;
	}
	/*
	vmm_printf( VMM_DEBUG, "[BPB]\n" );
	vmm_printf( VMM_DEBUG, "BS_jmpBoot ....... %02x %02x %02x\n",
				bpb->jmp[0], bpb->jmp[1], bpb->jmp[2] );
	vmm_printf( VMM_DEBUG, "BS_OEMName ....... \"" );
	for ( int i=0; i<8; i++ )
		vmm_printf( VMM_DEBUG, "%c", bpb->oem_name[i] );
	vmm_printf( VMM_DEBUG, "\"\n" );
	vmm_printf( VMM_DEBUG, "BPB_BytsPerSec ... %04x\n", bpb->bysec );
	vmm_printf( VMM_DEBUG, "BPB_SecPerClus ... %02x\n", bpb->sclus );
	vmm_printf( VMM_DEBUG, "BPB_RsvdSecCnt ... %04x\n", bpb->rsvd_sec );
	vmm_printf( VMM_DEBUG, "BPB_NumFATs ...... %02x\n", bpb->numFAT );
	vmm_printf( VMM_DEBUG, "BPB_RootEntCnt ... %04x\n", bpb->numDIR );
	vmm_printf( VMM_DEBUG, "BPB_TotSec16 ..... %04x\n", bpb->totalsec16 );
	vmm_printf( VMM_DEBUG, "BPB_Media ........ %02x\n", bpb->media );
	vmm_printf( VMM_DEBUG, "BPB_FATSz16 ...... %04x\n", bpb->sFAT );
	vmm_printf( VMM_DEBUG, "BPB_SecPerTrk .... %04x\n", bpb->strack );
	vmm_printf( VMM_DEBUG, "BPB_NumHeads ..... %04x\n", bpb->numhead );
	vmm_printf( VMM_DEBUG, "BPB_HiddSec ...... %08x\n", bpb->hiddensec );
	vmm_printf( VMM_DEBUG, "BPB_TotSec32 ..... %08x\n", bpb->totalsec32 );
	vmm_printf( VMM_DEBUG, "BS_DrvNum ........ %02x\n", bpb->pdrv );
	vmm_printf( VMM_DEBUG, "BS_Reserved1 ..... %02x\n", bpb->rsvd );
	vmm_printf( VMM_DEBUG, "BS_BootSig ....... %02x\n", bpb->boot_sig );
	vmm_printf( VMM_DEBUG, "BS_VolID ......... %08x\n", bpb->vol_serial );
	vmm_printf( VMM_DEBUG, "BS_VolLab ........ \"" );
	for ( int i=0; i<11; i++ )
		vmm_printf( VMM_DEBUG, "%c", bpb->vol_label[i] );
	vmm_printf( VMM_DEBUG, "\"\n" );
	vmm_printf( VMM_DEBUG, "BS_FilSysType .... \"" );
	for ( int i=0; i<8; i++ )
		vmm_printf( VMM_DEBUG, "%c", bpb->fs_type[i] );
	vmm_printf( VMM_DEBUG, "\"\n" );
	vmm_printf( VMM_DEBUG, "\n" );
	*/

	/*
	vmm_printf( VMM_DEBUG, "[FAT]\n" );
	u32 start_sec = fat_sec;
	u8 *buf = (u8 *)cache->fat;
	for ( u32 i=0; i<block_len>>4; i++ ) {
		vmm_printf( VMM_DEBUG, "%08x:", start_sec * block_len + (i<<4) );
		for ( u8 j=0; j<16; j++ )
			vmm_printf( VMM_DEBUG, " %02x", buf[(i<<4) + j] );
		vmm_printf( VMM_DEBUG, " | " );
		for ( u8 j=0; j<16; j++ ) {
			if ( (0x20 <= buf[(i<<4) + j]) && (buf[(i<<4) + j] <= 0x7E) )
				vmm_printf( VMM_DEBUG, "%c", buf[(i<<4) + j] );
			else
				vmm_printf( VMM_DEBUG, "." );
		}
		vmm_printf( VMM_DEBUG, "\n" );
	}
	vmm_printf( VMM_DEBUG, "\n" );
	*/

	// RDEを読む
	const u32 rde_sec = fat_sec + bpb->sFAT * bpb->numFAT;
	const u32 rde_sec_num = sizeof(FAT_RDE) * bpb->numDIR / block_len;
//	const u32 read_sectors = rde_sec_num / (8192 / block_len);
	const u32 read_sectors = 1;
	for ( u32 n=0; n<rde_sec_num; n+=read_sectors ) {
		if ( !disk_read( disk_id, rde_sec + n, read_sectors,
						 ((u8 *)cache->rde) + block_len * n, block_len * read_sectors ) ) {
			return false;
		}
	}
	vmm_printf( VMM_DEBUG, "[RDE]\n" );
	FAT_RDE *rde = cache->rde;
	for ( u32 i=0; i<bpb->numDIR; i++ ) {
		if ( rde[i].attr == FAT_ATTR_LONG_NAME )
			continue;
		switch ( (u8)rde[i].name[0] )
		{
		case 0x00:	// 空エントリ
		case 0x05:	// 削除エントリ (使用可能)
		case 0xE5:	// 削除エントリ (使用可能)
			break;
		default:
			{	// 使用済みエントリ
				for ( int n=0; n<8; n++ )
					vmm_printf( VMM_DEBUG, "%c", rde[i].name[n] );
				vmm_printf( VMM_DEBUG, " " );
				for ( int n=8; n<11; n++ )
					vmm_printf( VMM_DEBUG, "%c", rde[i].name[n] );
				if ( rde[i].attr & FAT_ATTR_DIRECTORY ) {
					vmm_printf( VMM_DEBUG, " <DIR>     ");
				}
				else {
					vmm_printf( VMM_DEBUG, " %10d",
								rde[i].file_size );
				}
				vmm_printf( VMM_DEBUG, " %04d/%02d/%02d %02d:%02d:%02d %c%c%c%c",
							(rde[i].update_date >>  9) + 1980,
							(rde[i].update_date >>  5) & 0x0F,
							(rde[i].update_date      ) & 0x1F,
							(rde[i].update_time >> 11),
							(rde[i].update_time >>  5) & 0x3F,
							(rde[i].update_time <<  1) & 0x3E,
							(rde[i].attr & FAT_ATTR_READ_ONLY)? 'R':'-',
							(rde[i].attr & FAT_ATTR_HIDDEN   )? 'H':'-',
							(rde[i].attr & FAT_ATTR_SYSTEM   )? 'S':'-',
							(rde[i].attr & FAT_ATTR_ARCHIVE  )? 'A':'-' );
				vmm_printf( VMM_DEBUG, "\n" );
			}
			break;
		}
	}
	vmm_printf( VMM_DEBUG, "\n" );
	cache->first_data_sec = rde_sec + rde_sec_num;
	vmm_printf( VMM_DEBUG, "FAT: 1st Data Sector = %08x\n", cache->first_data_sec );

	return true;
}

static bool fat_open( FS_DRIVE *drive, FS_FILE *fd )
{
	char name[12];

	if ( strchr( fd->name, '/' ) != NULL )
	{	// サブディレクトリは未対応
		vmm_printf( VMM_ERROR, "FAT: sub directory unsupported.\n" );
		return false;
	}

	// 「abcdefg.hij」→「ABCDEFG HIJ」に変換
	u32 n = 0;
	memset( name, ' ', 11 );
	for ( int i=0; i<8; i++ ) {
		if ( fd->name[n] == '\0' )
			break;
		if ( fd->name[n] == '.' ) {
			n++;
			break;
		}
		if ( ('a' <= fd->name[n]) && (fd->name[n] <= 'z') )
			name[i] = fd->name[n] - 'a' + 'A';
		else
			name[i] = fd->name[n];
		n++;
	}
	for ( int i=8; i<11; i++ ) {
		if ( (fd->name[n] == '\0') || (fd->name[n] == '.') )
			break;
		if ( ('a' <= fd->name[n]) && (fd->name[n] <= 'z') )
			name[i] = fd->name[n] - 'a' + 'A';
		else
			name[i] = fd->name[n];
		n++;
	}
	name[11] = '\0';
//	vmm_printf( VMM_DEBUG, "\"%s\"->\"%s\"\n", fd->filename, name );

	// ファイルを探す
	FAT_CACHE *cache = (FAT_CACHE *)drive->cache;
	FAT_BPB *bpb = &cache->bpb;
	FAT_RDE *rde = cache->rde;
	for ( n=0; n<bpb->numDIR; n++ ) {
		if ( memcmp( rde[n].name, name, 11 ) == 0 ) {
			if ( !(rde[n].attr & FAT_ATTR_DIRECTORY) &&
				 !(rde[n].attr & FAT_ATTR_VOLUME_ID) )
			break;	// ファイル発見！
		}
	}
	if ( n >= bpb->numDIR )
	{	// ファイルが見つからない
		vmm_printf( VMM_ERROR, "FAT: \"%s\" not found.\n", fd->name );
		return false;
	}

	fd->size = rde[n].file_size;
	fd->rp   = 0;
	fd->clus_1st = (rde[n].clus_high_word << 16) | rde[n].clus_low_word;
	fd->clus_cur = fd->clus_1st;
	vmm_printf( VMM_DEBUG, "FAT: \"%s\" found. (size=%d, clus_1st=%08x)\n",
				fd->name, fd->size, fd->clus_1st );

	return true;
}

static bool fat_read( FS_DRIVE *drive, FS_FILE *fd, void *buf, size_t size )
{
	bool (*fat_get_next_clus)( FS_DRIVE *, u32 * );
	switch ( drive->fs_type )
	{
	case FS_FAT12:	fat_get_next_clus = fat_get_next_clus12;	break;
	case FS_FAT16:	fat_get_next_clus = fat_get_next_clus16;	break;
	default:		return false;
	}

	FAT_CACHE *cache = (FAT_CACHE *)drive->cache;
	FAT_BPB *bpb = &cache->bpb;
	const u32 disk_id = drive->disk_id;

	u32 clus = fd->clus_cur;
	if ( (clus < 0x0002) && (0xFFF6 < clus) )
		return false;	// 終端までリード済み(シークが必要)

	const u32 clus_bytes = bpb->sclus * bpb->bysec;
	if ( clus_bytes > sizeof(cache->clus_data) )
		return false;	// クラスタサイズが想定外

	u32 buf_off = 0;
	u32 clus_off = fd->rp % clus_bytes;
	u32 clus_mod = clus_bytes - clus_off;
//	vmm_printf( VMM_DEBUG, "FAT: disk_read(1) clus_off=%d, clus_mod=%d\n", clus_off, clus_mod );
	if ( clus_off > 0 )
	{	// クラスタ全体をキャッシュに入れて一部をバッファにコピーする
		if ( cache->clus != clus )
		{	// キャッシュを更新する
			const u32 lba = cache->first_data_sec + (clus - 2) * bpb->sclus;
			if ( !disk_read( disk_id, lba, bpb->sclus, cache->clus_data, clus_bytes ) )
			{	// 読めない…
				vmm_printf( VMM_ERROR, "FAT: disk_read(1) failed. (sector=%08x-%08x)\n",
							fd->clus_cur * bpb->sclus,
							(fd->clus_cur + 1 ) * bpb->sclus - 1 );
				return false;
			}
			cache->clus = clus;
		}
		if ( clus_mod >= size )
		{	// バッファがいっぱいになったので正常終了
			memcpy( buf, &cache->clus_data[clus_off], size );
			return true;
		}
		memcpy( buf, &cache->clus_data[clus_off], clus_mod );
		buf_off += clus_mod;
		fd->rp  += clus_mod;
		if ( !fat_get_next_clus( drive, &clus ) ) {
			fd->clus_cur = clus;
			return true;
		}
	}

	u32 clus_remain = (size - buf_off) / clus_bytes;
//	vmm_printf( VMM_DEBUG, "FAT: disk_read(2) buf_off=%d, size=%d, clus_remain=%d\n",
//				buf_off, size, clus_remain );
	while ( clus_remain > 0 )
	{	// クラスタ全体を読み出してバッファに入れる(キャッシュは使わない)
		const u32 lba = cache->first_data_sec + (clus - 2) * bpb->sclus;
		if ( !disk_read( disk_id, lba, bpb->sclus, (u8 *)buf + buf_off, clus_bytes ) )
		{	// 読めない…
			vmm_printf( VMM_ERROR, "FAT: disk_read(2) failed. (sector=%08x-%08x)\n",
						fd->clus_cur * bpb->sclus,
						(fd->clus_cur + 1 ) * bpb->sclus - 1 );
			return false;
		}
		buf_off += clus_bytes;
		fd->rp  += clus_bytes;
		if ( !fat_get_next_clus( drive, &clus ) ) {
			fd->clus_cur = clus;
			return true;
		}
		clus_remain--;
	}

	clus_mod = size - buf_off;
//	vmm_printf( VMM_DEBUG, "FAT: disk_read(3) clus_mod=%d\n", clus_mod );
	if ( clus_mod > 0 )
	{	// クラスタ全体をキャッシュに入れて一部をバッファにコピーする
		const u32 lba = cache->first_data_sec + (clus - 2) * bpb->sclus;
		if ( !disk_read( disk_id, lba, bpb->sclus, cache->clus_data, clus_bytes ) )
		{	// 読めない…
			vmm_printf( VMM_ERROR, "FAT: disk_read(3) failed. (sector=%08x-%08x)\n",
						fd->clus_cur * bpb->sclus,
						(fd->clus_cur + 1 ) * bpb->sclus - 1 );
			return false;
		}
		cache->clus = clus;
		memcpy( (u8 *)buf + buf_off, cache->clus_data, clus_mod );
		fd->rp += clus_mod;
		fd->clus_cur = clus;
	}

	return true;
}

static bool fat_get_next_clus12( FS_DRIVE *drive, u32 *clus )
{
	FAT_CACHE *cache = (FAT_CACHE *)drive->cache;

	const u32 next_fat_page = *clus / 1024;
	if ( next_fat_page != cache->fat_page )
	{	// FATキャッシュを更新
		const u32 fat_sec = drive->first_sec + cache->bpb.rsvd_sec + next_fat_page * 3;
		if ( !disk_read( drive->disk_id, fat_sec, 3, cache->fat12, sizeof(cache->fat12) ) )
		{	// 読めない…
			vmm_printf( VMM_ERROR, "FAT: read FAT error. (sector=%08x-%08x)\n", fat_sec, fat_sec+2 );
			return false;
		}
		cache->fat_page = next_fat_page;
	}

	const u32 off = ((*clus % 1024) >> 1) * 3;
	if ( *clus & 0x01 )	// 奇数クラスタ
		*clus = (cache->fat12[off+2] << 4) | (cache->fat12[off+1] >> 4);
	else				// 偶数クラスタ
		*clus = ((cache->fat12[off+1] << 8) & 0xF00) | cache->fat12[off];

	if ( (*clus < 0x002) || (0xFF6 < *clus) )
		return false;

	return true;
}

static bool fat_get_next_clus16( FS_DRIVE *drive, u32 *clus )
{
	FAT_CACHE *cache = (FAT_CACHE *)drive->cache;

	const u32 next_fat_page = *clus / 256;
	if ( next_fat_page != cache->fat_page )
	{	// FATキャッシュを更新
		const u32 fat_sec = drive->first_sec + cache->bpb.rsvd_sec + next_fat_page;
		if ( !disk_read( drive->disk_id, fat_sec, 1, cache->fat16, sizeof(cache->fat16) ) )
		{	// 読めない…
			vmm_printf( VMM_ERROR, "FAT: read FAT error. (sector=%08x)\n", fat_sec );
			return false;
		}
		cache->fat_page = next_fat_page;
	}

	*clus = cache->fat16[*clus % 256];

	if ( (*clus < 0x0002) || (0xFFF6 < *clus) )
		return false;

	return true;
}

static bool fat_seek( FS_DRIVE *drive, FS_FILE *fd, long ptr, int method )
{
	u32 new_ptr;
	u32 clus;
	u32 clus_num;

	if ( ptr == 0 )
		return true;

	FAT_BPB *bpb = &((FAT_CACHE *)drive->cache)->bpb;
	const u32 clus_bytes = bpb->sclus * bpb->bysec;

	switch ( method )
	{
	case FS_SEEK_SET:
		if ( ptr < 0 )
			return false;
		new_ptr	 = ptr;
		clus	 = fd->clus_1st;
		clus_num = new_ptr / clus_bytes;
		break;

	case FS_SEEK_CUR:
		new_ptr = fd->rp + ptr;
		if ( new_ptr >= 0x80000000 )
			return false;
		// ★とりあえず常に先頭からシークする
		clus	 = fd->clus_1st;
		clus_num = new_ptr / clus_bytes;
		break;

	case FS_SEEK_END:
		// ★未実装
		vmm_printf( VMM_DEBUG, "FAT: seek error. (not impremented\n)" );
		return false;

	default:
		vmm_printf( VMM_DEBUG, "FAT: seek error. (unknown method=%d\n)", method );
		return false;
	}

	bool (*fat_get_next_clus)( FS_DRIVE *, u32 * );
	switch ( drive->fs_type )
	{
	case FS_FAT12:	fat_get_next_clus = fat_get_next_clus12;	break;
	case FS_FAT16:	fat_get_next_clus = fat_get_next_clus16;	break;
	default:		return false;
	}

	for ( u32 i=0; i<clus_num; i++ ) {
		if ( !fat_get_next_clus( drive, &clus ) )
			break;
	}
	fd->clus_cur = clus;
	fd->rp = new_ptr;

	return true;
}
