/*!
******************************************************************************

	@file	acpi.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/
/*-*- mode: C++; Encoding: utf8n -*-*/

#include "vsun86.h"
#include "acpi.h"
#include "irq.h"
#include "io.h"
#include "cpu.h"
#include "heap.h"
#include "printf.h"

#include <string.h>

static ACPI_RSDT *acpi_rsdt = NULL;
static ACPI_FADT *acpi_fadt = NULL;
static u64  acpi_tick_freq  = 0;
static u64  acpi_tick_count = 0;

static bool acpi_call_bios_e820( u32, u32 *, void *, size_t * );
static bool acpi_scan_rsdt( ACPI_RSDT *rsdt );
static bool acpi_enable( void );
static bool acpi_irq_handler( u8 irq, void *args );
static u32  acpi_read_pm1_sts_reg( void );
static void acpi_write_pm1_sts_reg( u32 data );
static u32  acpi_read_pm1_en_reg( void );
static void acpi_write_pm1_en_reg( u32 data );

bool acpi_init( void )
{
	ACPIBIOS_SMAP smap;
	u32 continuation;

	vmm_printf( VMM_DEBUG, "[Physical Memory Map]\n" );
	continuation = 0;
	u64 ram_size = 0;
	const char *smap_type = NULL;
	do {
		size_t buf_size = sizeof(smap);
		if ( !acpi_call_bios_e820( ACPIBIOS_SIG_SMAP, &continuation, &smap, &buf_size ) )
			return false;
		switch ( smap.type ) {
		case ACPIBIOS_SMAP_TYPE_RAM:
			smap_type = "RAM";
			ram_size += smap.len;
			break;
		case ACPIBIOS_SMAP_TYPE_RSVD:
			smap_type = "Reserved";
			break;
		case ACPIBIOS_SMAP_TYPE_ACPI:
			smap_type = "ACPI";
			for ( u32 i=0; i<(u32)smap.len>>12; i++ )
				x86_map_page( (u32)smap.addr + (i<<12) );
			break;
		case ACPIBIOS_SMAP_TYPE_NVS:
			smap_type = "ACPI NVS";
			for ( u32 i=0; i<(u32)smap.len>>12; i++ )
				x86_map_page( (u32)smap.addr + (i<<12) );
			break;
		case ACPIBIOS_SMAP_TYPE_UNUSABLE:
			smap_type = "Unusable";
			break;
		default:
			smap_type = "Undefined";
			break;
		}
		vmm_printf( VMM_DEBUG, "%016llx-%016llx (%s)\n",
					smap.addr, smap.addr + smap.len - 1, smap_type );
	} while ( continuation != 0 );
	vmm_printf( VMM_DEBUG, "%d MB RAM available.\n", (u32)(ram_size / 0x100000LL) );
	heap_init();

	ACPI_RSDP *rsdp = NULL;
	char *p = (char *)0xE0000;
	u32 off = 0;
	while ( off < 0x20000 ) {
		if ( strncmp( &p[off], "RSD PTR ", 8 ) == 0 ) {
			vmm_printf( VMM_DEBUG, "acpi: RSDP found @ %08x\n", &p[off] );
			rsdp = (ACPI_RSDP *)&p[off];
			break;
		}
		off += 16;
	}
	if ( rsdp == NULL ) {
		vmm_printf( VMM_DEBUG, "acpi: RSDP not found.\n" );
		return false;
	}
	vmm_printf( VMM_DEBUG, "acpi: [RSDP] Revision=%02x, OEMID=\"", rsdp->revision );
	for ( u32 i=0; i<6; i++ ) {
		if ( rsdp->oem_id[i] != '\0' )
			vmm_printf( VMM_DEBUG, "%c", rsdp->oem_id[i] );
		else
			vmm_printf( VMM_DEBUG, " " );
	}
	vmm_printf( VMM_DEBUG, "\", RSDT->%08x, XSDT->%016llx\n", rsdp->rsdt_addr, rsdp->xsdt_addr );

	acpi_rsdt = (ACPI_RSDT *)rsdp->rsdt_addr;
	if ( !acpi_scan_rsdt( acpi_rsdt ) )
		return false;

	if ( !acpi_enable() ) {
		vmm_printf( VMM_DEBUG, "acpi: acpi_enable() failed.\n" );
		return false;
	}

	if ( !irq_register( (u8)acpi_fadt->sci_int, IRQ_TRIGGER_LEVEL, acpi_irq_handler, NULL ) ) {
		vmm_printf( VMM_DEBUG, "acpi: intr_register() failed.\n" );
		return false;
	}

	acpi_tick_freq  = 3579545;
	acpi_tick_count = ind( acpi_fadt->pm_tmr_blk );
	acpi_write_pm1_en_reg( acpi_read_pm1_en_reg() | ACPI_PM1_TMR_EN );

	return true;
}

static bool acpi_call_bios_e820( u32 signature, u32 *continuation, void *buf, size_t *buf_size )
{
	V86INT_REGS regs;
	memset( &regs, 0, sizeof(regs) );

	regs.eax	 = 0xE820;
	regs.ebx	 = *continuation;
	regs.es_base = buf;
	regs.es_size = *buf_size;
	regs.edi	 = 0;
	regs.ecx	 = *buf_size;
	regs.edx	 = signature;

	x86_v86int( 0x15, &regs );

	if ( regs.eflags & F_CF ) {
		vmm_printf( VMM_ERROR, "ACPI: INT15h - e820h failed.\n" );
		return false;
	}

	*continuation = regs.ebx;
	*buf_size	  = regs.ecx;

	return true;
}

static bool acpi_scan_rsdt( ACPI_RSDT *rsdt )
{
	if ( rsdt == NULL ) {
		vmm_printf( VMM_DEBUG, "acpi: RSDT->(NULL)\n" );
		return false;
	}
	if ( strncmp( rsdt->hdr.sig, "RSDT", 4 ) != 0 ) {
		vmm_printf( VMM_DEBUG, "acpi: RSDT.Signature != \"RSDT\"\n" );
		return false;
	}

	vmm_printf( VMM_DEBUG, "acpi: [RSDT] Revision=%02x:%08x, OEMID=\"",
				rsdt->hdr.revision, rsdt->hdr.oem_revision );
	for ( u32 i=0; i<6; i++ ) {
		if ( rsdt->hdr.oem_id[i] != '\0' )
			vmm_printf( VMM_DEBUG, "%c", rsdt->hdr.oem_id[i] );
		else
			vmm_printf( VMM_DEBUG, " " );
	}
	vmm_printf( VMM_DEBUG, " " );
	for ( u32 i=0; i<8; i++ ) {
		if ( rsdt->hdr.oem_table_id[i] != '\0' )
			vmm_printf( VMM_DEBUG, "%c", rsdt->hdr.oem_table_id[i] );
		else
			vmm_printf( VMM_DEBUG, " " );
	}
	vmm_printf( VMM_DEBUG, "\"\n" );

	ACPI_FADT *fadt = NULL;
	const u32 rsdt_entry_num = (rsdt->hdr.len - sizeof(ACPI_DESC_HEADER)) >> 2;
	for ( u32 i=0; i<rsdt_entry_num; i++ ) {
		ACPI_DESC_HEADER *entry = (ACPI_DESC_HEADER *)rsdt->entry[i];
		vmm_printf( VMM_DEBUG, "acpi: [%c%c%c%c] Revision=%02x:%08x, OEMID=\"",
					entry->sig[0], entry->sig[1], entry->sig[2], entry->sig[3],
					entry->revision, entry->oem_revision );
		for ( u32 j=0; j<6; j++ ) {
			if ( entry->oem_id[j] != '\0' )
				vmm_printf( VMM_DEBUG, "%c", entry->oem_id[j] );
			else
				vmm_printf( VMM_DEBUG, " " );
		}
		vmm_printf( VMM_DEBUG, " " );
		for ( u32 j=0; j<8; j++ ) {
			if ( entry->oem_table_id[j] != '\0' )
				vmm_printf( VMM_DEBUG, "%c", entry->oem_table_id[j] );
			else
				vmm_printf( VMM_DEBUG, " " );
		}
		vmm_printf( VMM_DEBUG, "\"\n" );
		if ( strncmp( entry->sig, "FACP" , 4 ) == 0 )
			fadt = (ACPI_FADT *)entry;
	}

	if ( fadt == NULL ) {
		vmm_printf( VMM_DEBUG, "acpi: FADT not found.\n" );
		return false;
	}
	if ( fadt->hdr.len < 132 ) {
		vmm_printf( VMM_DEBUG, "acpi: FADT length < 132 (%d)\n", fadt->hdr.len );
		return false;
	}

	vmm_printf( VMM_DEBUG, "[Fixed ACPI Description Table]\n" );
	vmm_printf( VMM_DEBUG, "FIRMWARE_CTRL ......... %08x\n", fadt->facs_addr );
	vmm_printf( VMM_DEBUG, "DSDT .................. %08x\n", fadt->dsdt_addr );
	vmm_printf( VMM_DEBUG, "Prefered_PM_Profile ... %02x\n", fadt->pref_pm_prof );
	vmm_printf( VMM_DEBUG, "SCI_INT ............... %04x\n", fadt->sci_int );
	vmm_printf( VMM_DEBUG, "SMI_CMD ............... %08x\n", fadt->smi_cmd );
	vmm_printf( VMM_DEBUG, "ACPI_ENABLE ........... %02x\n", fadt->acpi_enable );
	vmm_printf( VMM_DEBUG, "ACPI_DISABLE .......... %02x\n", fadt->acpi_disable );
	vmm_printf( VMM_DEBUG, "S4BIOS_REQ ............ %02x\n", fadt->s4bios_req );
	vmm_printf( VMM_DEBUG, "PSTATE_CNT ............ %02x\n", fadt->pstate_cnt );
	vmm_printf( VMM_DEBUG, "PM1a_EVT_BLK .......... %08x\n", fadt->pm1a_evt_blk );
	vmm_printf( VMM_DEBUG, "PM1b_EVT_BLK .......... %08x\n", fadt->pm1b_evt_blk );
	vmm_printf( VMM_DEBUG, "PM1a_CNT_BLK .......... %08x\n", fadt->pm1a_cnt_blk );
	vmm_printf( VMM_DEBUG, "PM1b_CNT_BLK .......... %08x\n", fadt->pm1b_cnt_blk );
	vmm_printf( VMM_DEBUG, "PM2_CNT_BLK ........... %08x\n", fadt->pm2_cnt_blk );
	vmm_printf( VMM_DEBUG, "PM_TMR_BLK ............ %08x\n", fadt->pm_tmr_blk );
	vmm_printf( VMM_DEBUG, "GPE0_BLK .............. %08x\n", fadt->gpe0_blk );
	vmm_printf( VMM_DEBUG, "GPE1_BLK .............. %08x\n", fadt->gpe1_blk );
	vmm_printf( VMM_DEBUG, "PM1_EVT_LEN ........... %02x\n", fadt->pm1_evt_len );
	vmm_printf( VMM_DEBUG, "PM1_CNT_LEN ........... %02x\n", fadt->pm1_cnt_len );
	vmm_printf( VMM_DEBUG, "PM2_CNT_LEN ........... %02x\n", fadt->pm2_cnt_len );
	vmm_printf( VMM_DEBUG, "PM_TMR_LEN ............ %02x\n", fadt->pm_tmr_len );
	vmm_printf( VMM_DEBUG, "GPE0_BLK_LEN .......... %02x\n", fadt->gpe0_blk_len );
	vmm_printf( VMM_DEBUG, "GPE1_BLK_LEN .......... %02x\n", fadt->gpe1_blk_len );
	vmm_printf( VMM_DEBUG, "GPE1_BASE ............. %02x\n", fadt->gpe1_base );
	vmm_printf( VMM_DEBUG, "CST_CNT ............... %02x\n", fadt->cst_cnt );
	vmm_printf( VMM_DEBUG, "P_LVL2_LAT ............ %04x\n", fadt->p_lvl2_lat );
	vmm_printf( VMM_DEBUG, "P_LVL3_LAT ............ %04x\n", fadt->p_lvl3_lat );
	vmm_printf( VMM_DEBUG, "FLUSH_SIZE ............ %04x\n", fadt->flush_size );
	vmm_printf( VMM_DEBUG, "FLUSH_STRIDE .......... %04x\n", fadt->flush_stride );
	vmm_printf( VMM_DEBUG, "DUTY_OFFSET ........... %02x\n", fadt->duty_off );
	vmm_printf( VMM_DEBUG, "DUTY_WIDTH ............ %02x\n", fadt->duty_width );
	vmm_printf( VMM_DEBUG, "DAY_ALRM .............. %02x\n", fadt->day_alrm );
	vmm_printf( VMM_DEBUG, "MON_ALRM .............. %02x\n", fadt->mon_alrm );
	vmm_printf( VMM_DEBUG, "CENTURY ............... %02x\n", fadt->century );
	vmm_printf( VMM_DEBUG, "IAPC_BOOT_ARCH ........ %04x\n", fadt->iapc_boot_arch );
	vmm_printf( VMM_DEBUG, "Flags ................. %08x\n", fadt->flags );
	vmm_printf( VMM_DEBUG, "RESET_REG ............. [%016llx,%02x,%02x,%02x,%02x]\n",
				fadt->reset_reg.addr, fadt->reset_reg.access_size,
				fadt->reset_reg.bit_off, fadt->reset_reg.bit_width, fadt->reset_reg.addr_space_id );
	vmm_printf( VMM_DEBUG, "RESET_VALUE ........... %02x\n", fadt->reset_val );
	if ( fadt->hdr.len > 132 ) {
		vmm_printf( VMM_DEBUG, "X_FIRMWARE_CTRL ....... %016llx\n", fadt->x_facs_addr );
		vmm_printf( VMM_DEBUG, "X_DSDT ................ %016llx\n", fadt->x_dsdt_addr );
		vmm_printf( VMM_DEBUG, "X_PM1a_EVT_BLK ........ [%016llx,%02x,%02x,%02x,%02x]\n",
					fadt->x_pm1a_evt_blk.addr, fadt->x_pm1a_evt_blk.access_size,
					fadt->x_pm1a_evt_blk.bit_off, fadt->x_pm1a_evt_blk.bit_width, fadt->x_pm1a_evt_blk.addr_space_id );
		vmm_printf( VMM_DEBUG, "X_PM1b_EVT_BLK ........ [%016llx,%02x,%02x,%02x,%02x]\n",
					fadt->x_pm1b_evt_blk.addr, fadt->x_pm1b_evt_blk.access_size,
					fadt->x_pm1b_evt_blk.bit_off, fadt->x_pm1b_evt_blk.bit_width, fadt->x_pm1b_evt_blk.addr_space_id );
		vmm_printf( VMM_DEBUG, "X_PM1a_CNT_BLK ........ [%016llx,%02x,%02x,%02x,%02x]\n",
					fadt->x_pm1a_cnt_blk.addr, fadt->x_pm1a_cnt_blk.access_size,
					fadt->x_pm1a_cnt_blk.bit_off, fadt->x_pm1a_cnt_blk.bit_width, fadt->x_pm1a_cnt_blk.addr_space_id );
		vmm_printf( VMM_DEBUG, "X_PM1b_CNT_BLK ........ [%016llx,%02x,%02x,%02x,%02x]\n",
					fadt->x_pm1b_cnt_blk.addr, fadt->x_pm1b_cnt_blk.access_size,
					fadt->x_pm1b_cnt_blk.bit_off, fadt->x_pm1b_cnt_blk.bit_width, fadt->x_pm1b_cnt_blk.addr_space_id );
		vmm_printf( VMM_DEBUG, "X_PM2_CNT_BLK ......... [%016llx,%02x,%02x,%02x,%02x]\n",
					fadt->x_pm2_cnt_blk.addr, fadt->x_pm2_cnt_blk.access_size,
					fadt->x_pm2_cnt_blk.bit_off, fadt->x_pm2_cnt_blk.bit_width, fadt->x_pm2_cnt_blk.addr_space_id );
		vmm_printf( VMM_DEBUG, "X_PM_TMR_BLK .......... [%016llx,%02x,%02x,%02x,%02x]\n",
					fadt->x_pm_tmr_blk.addr, fadt->x_pm_tmr_blk.access_size,
					fadt->x_pm_tmr_blk.bit_off, fadt->x_pm_tmr_blk.bit_width, fadt->x_pm_tmr_blk.addr_space_id );
		vmm_printf( VMM_DEBUG, "X_GPE0_BLK ............ [%016llx,%02x,%02x,%02x,%02x]\n",
					fadt->x_gpe0_blk.addr, fadt->x_gpe0_blk.access_size,
					fadt->x_gpe0_blk.bit_off, fadt->x_gpe0_blk.bit_width, fadt->x_gpe0_blk.addr_space_id );
		vmm_printf( VMM_DEBUG, "X_GPE1_BLK ............ [%016llx,%02x,%02x,%02x,%02x]\n",
					fadt->x_gpe1_blk.addr, fadt->x_gpe1_blk.access_size,
					fadt->x_gpe1_blk.bit_off, fadt->x_gpe1_blk.bit_width, fadt->x_gpe1_blk.addr_space_id );
	}

	acpi_fadt = fadt;

	return true;
}

static bool acpi_enable( void )
{
	if ( acpi_fadt == NULL )
		return false;

	outb( acpi_fadt->smi_cmd, acpi_fadt->acpi_enable );
	return true;
}

static bool acpi_irq_handler( u8 irq, void *args )
{
	(void)args;

	if ( irq != acpi_fadt->sci_int )
		return false;

	u32 pm1_sts = acpi_read_pm1_sts_reg();
//	vmm_printf( VMM_DEBUG, "acpi: interrupt occured. (pm1_status=%08x)\n", pm1_sts );

	const u32 timer_val = ind( acpi_fadt->pm_tmr_blk );
	if ( pm1_sts & ACPI_PM1_TMR_STS ) {
		if ( acpi_fadt->f.tmr_val_ext ) {
			acpi_tick_count &= 0xFFFFFFFF00000000ULL;
			if ( !(timer_val & 0x80000000) )
				acpi_tick_count += 0x100000000ULL;
		}
		else {
			acpi_tick_count &= 0xFFFFFFFFFF000000ULL;
			if ( !(timer_val & 0x00800000) )
				acpi_tick_count += 0x001000000ULL;
		}
		acpi_tick_count |= timer_val;
	}

	// ステータスをクリアする
	acpi_write_pm1_sts_reg( pm1_sts );

	return true;
}

static u32 acpi_read_pm1_sts_reg( void )
{
	if ( acpi_fadt == NULL )
		return 0;

	const u32 pm1_evt_blk = acpi_fadt->pm1a_evt_blk;
	const u32 pm1_evt_len = acpi_fadt->pm1_evt_len;

	if ( pm1_evt_blk == 0 )
		return 0;
	const u16 port = (u16)pm1_evt_blk;

	u32 data;
	switch ( pm1_evt_len >> 1 )
	{
	case 1:		data = inb( port );		break;
	case 2:		data = inw( port );		break;
	case 4:		data = ind( port );		break;
	default:	data = 0;
	}

	return data;
}

static void acpi_write_pm1_sts_reg( u32 data )
{
	if ( acpi_fadt == NULL )
		return;

	const u32 pm1_evt_blk = acpi_fadt->pm1a_evt_blk;
	const u32 pm1_evt_len = acpi_fadt->pm1_evt_len;

	if ( pm1_evt_blk == 0 )
		return;
	const u16 port = (u16)pm1_evt_blk;

	switch ( pm1_evt_len >> 1 )
	{
	case 1:		outb( port, (u8 )data );	break;
	case 2:		outw( port, (u16)data );	break;
	case 4:		outd( port,      data );	break;
	}
}

static u32 acpi_read_pm1_en_reg( void )
{
	if ( acpi_fadt == NULL )
		return 0;

	const u32 pm1_evt_blk = acpi_fadt->pm1a_evt_blk;
	const u32 pm1_evt_len = acpi_fadt->pm1_evt_len;

	if ( pm1_evt_blk == 0 )
		return 0;
	const u16 port = (u16)pm1_evt_blk + (pm1_evt_len >> 1);

	u32 data;
	switch ( pm1_evt_len >> 1 )
	{
	case 1:		data = inb( port );		break;
	case 2:		data = inw( port );		break;
	case 4:		data = ind( port );		break;
	default:	data = 0;
	}

	return data;
}

static void acpi_write_pm1_en_reg( u32 data )
{
	if ( acpi_fadt == NULL )
		return;

	const u32 pm1_evt_blk = acpi_fadt->pm1a_evt_blk;
	const u32 pm1_evt_len = acpi_fadt->pm1_evt_len;

	if ( pm1_evt_blk == 0 )
		return;
	const u16 port = (u16)pm1_evt_blk + (pm1_evt_len >> 1);

	switch ( pm1_evt_len >> 1 )
	{
	case 1:		outb( port, (u8 )data );	break;
	case 2:		outw( port, (u16)data );	break;
	case 4:		outd( port,      data );	break;
	}
}

u64 acpi_get_tick_freq( void )
{
	return acpi_tick_freq;
}

u64 acpi_get_tick_count( void )
{
	if ( acpi_fadt == NULL )
		return 0;

	if ( acpi_fadt->f.tmr_val_ext )
		acpi_tick_count &= 0xFFFFFFFF00000000ULL;
	else
		acpi_tick_count &= 0xFFFFFFFFFF000000ULL;
	acpi_tick_count |= ind( acpi_fadt->pm_tmr_blk );

	return acpi_tick_count;
}
