package org.unitedfront2.domain.communication;

import java.util.List;

import org.junit.Assert;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.unitedfront2.domain.SimpleUser;
import org.unitedfront2.test.TransactionalTestCaseWithInitialData;

public class MailTest extends TransactionalTestCaseWithInitialData {

    @Autowired private MailTable mailTable;

    @Test
    public void testHasUnread() {
        Mail mail0 = prototypeReadMail(simpleUser1);
        Mail mail1 = prototypeReadMail(simpleUser2);
        Mail mail2 = prototypeReadMail(simpleUser1);
        Mail mail3 = prototypeUnreadMail(simpleUser2);
        mail0.setNext(mail1);
        mail1.setNext(mail2);
        mail2.setNext(mail3);
        Assert.assertTrue(mail0.hasUnread(simpleUser2.getId()));
        Assert.assertFalse(mail0.hasUnread(simpleUser1.getId()));
    }

    @Test
    public void testGetCount() {
        Mail mail0 = prototypeReadMail(simpleUser1);
        Mail mail1 = prototypeReadMail(simpleUser2);
        Mail mail2 = prototypeReadMail(simpleUser1);
        Mail mail3 = prototypeUnreadMail(simpleUser2);
        mail0.setNext(mail1);
        mail1.setNext(mail2);
        mail2.setNext(mail3);
        Assert.assertEquals(4, mail0.getCount());
    }

    @Test
    public void testAsList() {
        Mail mail0 = prototypeReadMail(simpleUser1);
        Mail mail1 = prototypeReadMail(simpleUser2);
        Mail mail2 = prototypeReadMail(simpleUser1);
        Mail mail3 = prototypeUnreadMail(simpleUser2);
        mail0.setNext(mail1);
        mail1.setNext(mail2);
        mail2.setNext(mail3);
        List<Mail> mails = mail0.asList();
        Assert.assertEquals(mail0, mails.get(0));
        Assert.assertEquals(mail1, mails.get(1));
        Assert.assertEquals(mail2, mails.get(2));
        Assert.assertEquals(mail3, mails.get(3));
    }

    @Test
    public void testAsListDesc() {
        Mail mail0 = prototypeReadMail(simpleUser1);
        Mail mail1 = prototypeReadMail(simpleUser2);
        Mail mail2 = prototypeReadMail(simpleUser1);
        Mail mail3 = prototypeUnreadMail(simpleUser2);
        mail0.setNext(mail1);
        mail1.setNext(mail2);
        mail2.setNext(mail3);
        List<Mail> mails = mail0.asListDesc();
        Assert.assertEquals(mail3, mails.get(0));
        Assert.assertEquals(mail2, mails.get(1));
        Assert.assertEquals(mail1, mails.get(2));
        Assert.assertEquals(mail0, mails.get(3));
    }

    @Test
    public void testGetTail() {
        Mail mail0 = prototypeReadMail(simpleUser1);
        Mail mail1 = prototypeReadMail(simpleUser2);
        Mail mail2 = prototypeReadMail(simpleUser1);
        Mail mail3 = prototypeUnreadMail(simpleUser2);
        mail0.setNext(mail1);
        mail1.setNext(mail2);
        mail2.setNext(mail3);
        Mail tail = mail0.getTail();
        Assert.assertEquals(mail3, tail);
    }

    @Test
    public void testReadǂɂȂ() {
        Mail mail = domainFactory.prototype(new Mail(simpleUser1.getId(), simpleUser2.getId(), "Subject", "This is a body."), Mail.class);
        mail.send();
        Assert.assertFalse(mail.isRead());

        mail.read(simpleUser1.getId());
        Mail found = mailTable.find(mail.getId());
        Assert.assertTrue(found.isRead());
    }

    @Test
    public void testReadǂɂȂȂ() {
        Mail mail = domainFactory.prototype(new Mail(simpleUser1.getId(), simpleUser2.getId(), "Subject", "This is a body."), Mail.class);
        mail.send();
        Assert.assertFalse(mail.isRead());

        mail.read(simpleUser2.getId());
        Mail found = mailTable.find(mail.getId());
        Assert.assertFalse(found.isRead());
    }

    @Test
    public void testReadAll() {
        Mail mail0 = domainFactory.prototype(new Mail(simpleUser1.getId(), simpleUser2.getId(), "mail0", "This is a mail0."), Mail.class);
        Mail mail1 = domainFactory.prototype(new Mail(simpleUser2.getId(), simpleUser1.getId(), "mail1", "This is a mail1."), Mail.class);
        Mail mail2 = domainFactory.prototype(new Mail(simpleUser1.getId(), simpleUser2.getId(), "mail2", "This is a mail2."), Mail.class);
        mail0.send();
        mail1.send(mail0.getId());
        mail2.send(mail1.getId());
        Assert.assertFalse(mail0.isRead());
        Assert.assertFalse(mail1.isRead());
        Assert.assertFalse(mail2.isRead());

        mail0 = mailTable.findThread(mail0.getId());
        mail0.readAll(simpleUser1.getId());
        Assert.assertTrue(mail0.isRead());
        Assert.assertFalse(mail0.getNext().isRead());
        Assert.assertTrue(mail0.getNext().getNext().isRead());
        mail0 = mailTable.findThread(mail0.getId());
        Assert.assertTrue(mail0.isRead());
        Assert.assertFalse(mail0.getNext().isRead());
        Assert.assertTrue(mail0.getNext().getNext().isRead());
    }

    @Test
    public void testUnreadǂɂȂ() {
        Mail mail = domainFactory.prototype(new Mail(simpleUser1.getId(), simpleUser2.getId(), "Subject", "This is a body."), Mail.class);
        mail.send();
        Assert.assertFalse(mail.isRead());

        mail.read(simpleUser1.getId());
        Mail found = mailTable.find(mail.getId());
        Assert.assertTrue(found.isRead());

        mail.unread(simpleUser1.getId());
        found = mailTable.find(mail.getId());
        Assert.assertFalse(found.isRead());
    }

    @Test
    public void testUnreadǂɂȂȂ() {
        Mail mail = domainFactory.prototype(new Mail(simpleUser1.getId(), simpleUser2.getId(), "Subject", "This is a body."), Mail.class);
        mail.send();
        Assert.assertFalse(mail.isRead());

        mail.read(simpleUser1.getId());
        Mail found = mailTable.find(mail.getId());
        Assert.assertTrue(found.isRead());

        mail.unread(simpleUser2.getId());
        found = mailTable.find(mail.getId());
        Assert.assertTrue(found.isRead());
    }

    @Test
    public void testUneadAll() {
        Mail mail0 = domainFactory.prototype(new Mail(simpleUser1.getId(), simpleUser2.getId(), "mail0", "This is a mail0."), Mail.class);
        Mail mail1 = domainFactory.prototype(new Mail(simpleUser2.getId(), simpleUser1.getId(), "mail1", "This is a mail1."), Mail.class);
        Mail mail2 = domainFactory.prototype(new Mail(simpleUser1.getId(), simpleUser2.getId(), "mail2", "This is a mail2."), Mail.class);
        mail0.send();
        mail1.send(mail0.getId());
        mail2.send(mail1.getId());
        Assert.assertFalse(mail0.isRead());
        Assert.assertFalse(mail1.isRead());
        Assert.assertFalse(mail2.isRead());

        mail0 = mailTable.findThread(mail0.getId());
        mail0.readAll(simpleUser1.getId());
        Assert.assertTrue(mail0.isRead());
        Assert.assertFalse(mail0.getNext().isRead());
        Assert.assertTrue(mail0.getNext().getNext().isRead());
        mail0 = mailTable.findThread(mail0.getId());
        Assert.assertTrue(mail0.isRead());
        Assert.assertFalse(mail0.getNext().isRead());
        Assert.assertTrue(mail0.getNext().getNext().isRead());

        mail0 = mailTable.findThread(mail0.getId());
        mail0.unreadAll(simpleUser2.getId());
        Assert.assertTrue(mail0.isRead());
        Assert.assertFalse(mail0.getNext().isRead());
        Assert.assertTrue(mail0.getNext().getNext().isRead());
        mail0 = mailTable.findThread(mail0.getId());
        Assert.assertTrue(mail0.isRead());
        Assert.assertFalse(mail0.getNext().isRead());
        Assert.assertTrue(mail0.getNext().getNext().isRead());
    }

    @Test
    public void testOther() {
        Mail mail = domainFactory.prototype(new Mail(simpleUser1.getId(), simpleUser2.getId(), "mail", "This is a test mail."), Mail.class);
        mail.setTo(simpleUser1);
        mail.setFrom(simpleUser2);
        Assert.assertTrue(simpleUser1.identify(mail.other(simpleUser2)));
        Assert.assertTrue(simpleUser2.identify(mail.other(simpleUser1)));
    }

    private Mail prototypeReadMail(SimpleUser to) {
        Mail mail = domainFactory.prototype(Mail.class);
        mail.setRead(true);
        mail.setToId(to.getId());
        return mail;
    }

    private Mail prototypeUnreadMail(SimpleUser to) {
        Mail mail = domainFactory.prototype(new Mail(), Mail.class);
        mail.setToId(to.getId());
        return mail;
    }
}
