package org.unitedfront2.dao;

import java.util.List;
import java.util.Locale;

import org.junit.Assert;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.unitedfront2.domain.Domain;
import org.unitedfront2.domain.accesscontrol.AccessControl;
import org.unitedfront2.domain.accesscontrol.AccessControlTable;
import org.unitedfront2.domain.accesscontrol.OwnerOnly;
import org.unitedfront2.domain.accesscontrol.Public;
import org.unitedfront2.domain.accesscontrol.UserOnly;
import org.unitedfront2.domain.communication.Community;
import org.unitedfront2.domain.communication.Message;
import org.unitedfront2.domain.communication.MessageTable;
import org.unitedfront2.domain.communication.MultiThreadCommunity;
import org.unitedfront2.domain.communication.SingleThreadCommunity;
import org.unitedfront2.domain.communication.Thread;
import org.unitedfront2.test.TransactionalTestCaseWithInitialData;

public class CommunityDaoTest extends TransactionalTestCaseWithInitialData {

    @Autowired
    private CommunityDao communityDao;

    @Autowired
    private AccessControlTable accessControlTable;

    @Autowired
    private MessageTable messageTable;

    @Test
    public void testRegisterSingleThreadCommunity() {
        Community c = createCommunityTestData(SingleThreadCommunity.class, 1);
        communityDao.register(c);
        Community actual = communityDao.find(c.getId());
        Assert.assertEquals(c, actual);
        assertConsistency(c);
    }

    @Test
    public void testRegisterThread() {
        Community c = createCommunityTestData(SingleThreadCommunity.class, 1);
        communityDao.register(c);
        Thread t = createThread();
        t.store();
        communityDao.registerThread(c.getId(), t.getId());
        List<Thread> threads = communityDao.findThreads(c.getId());
        Assert.assertSame(1, threads.size());
        Assert.assertEquals(t, threads.get(0));
    }

    @Test
    public void testFindSingleThreadCommunity() {
        Community c = createCommunityTestData(SingleThreadCommunity.class, 1);
        communityDao.register(c);
        Community actual = communityDao.find(c.getId());
        Assert.assertEquals(c, actual);
    }

    @Test
    public void testFindȂSingleThreadCommunity() {
        Assert.assertNull(communityDao.find(-1));
    }

    @Test
    public void testFindByCodeSingleThreadCommunity() {
        Community c = createCommunityTestData(SingleThreadCommunity.class, 1);
        communityDao.register(c);
        Community actual = communityDao.findByCode(c.getCode());
        Assert.assertEquals(c, actual);
    }

    @Test
    public void testFindOrderByLastUpdateDateDesc() throws InterruptedException {
        Community c1 = createCommunityTestData(SingleThreadCommunity.class, 1);
        communityDao.register(c1);
        Thread t1 = createThread();
        t1.store();
        communityDao.registerThread(c1.getId(), t1.getId());
        Message e1 = createEntry(c1.getReadAccessControl(), c1.getWriteAccessControl());
        t1.post(e1);

        java.lang.Thread.sleep(1000);
        Thread t2 = createThread();
        t2.store();
        communityDao.registerThread(c1.getId(), t2.getId());
        Message e2 = createEntry(c1.getReadAccessControl(), c1.getWriteAccessControl());
        t2.post(e2);

        java.lang.Thread.sleep(1000);
        Community c2 = createCommunityTestData(SingleThreadCommunity.class, 1);
        c2.setCode(c2.getCode() + "2");
        communityDao.register(c2);
        Thread t3 = createThread();
        t3.store();
        communityDao.registerThread(c2.getId(), t3.getId());
        Message e3 = createEntry(c2.getReadAccessControl(), c2.getWriteAccessControl());
        t3.post(e3);

        java.lang.Thread.sleep(1000);
        Message e4 = createEntry(c1.getReadAccessControl(), c1.getWriteAccessControl());
        t2.post(e4);

        Community c3 = createCommunityTestData(SingleThreadCommunity.class, 1);
        c3.setCode(c3.getCode() + "3");
        communityDao.register(c3);

        List<Community> list = communityDao.findOrderByLastUpdateDateDesc(0, 2);
        Assert.assertSame(2, list.size());
        Assert.assertEquals(c1, list.get(0));
        Assert.assertNotNull(list.get(0).getLastUpdateDate());
        Assert.assertEquals(c2, list.get(1));
        Assert.assertNotNull(list.get(1).getLastUpdateDate());
    }

    @Test
    public void testFindByCodeȂSingleThreadCommunity() {
        Assert.assertNull(communityDao.findByCode(""));
    }

    @Test
    public void testFindPublicCommunities() {
        Community c1 = createCommunityTestData(SingleThreadCommunity.class, 1);
        c1.getReadAccessControl().changeTo(Public.class);
        communityDao.register(c1);
        Community c2 = createCommunityTestData(MultiThreadCommunity.class, 2);
        c2.getReadAccessControl().changeTo(Public.class);
        communityDao.register(c2);
        Community c3 = createCommunityTestData(SingleThreadCommunity.class, 3);
        communityDao.register(c3);
        List<Community> communities = communityDao.findPublicCommunities();
        Assert.assertSame(2, communities.size());
        Assert.assertEquals(c2, communities.get(0));
        Assert.assertEquals(c1, communities.get(1));
    }

    @Test
    public void testFindPublicCommunitiesRandomly() {
        Community c1 = createCommunityTestData(SingleThreadCommunity.class, 1);
        c1.getReadAccessControl().changeTo(Public.class);
        communityDao.register(c1);
        Community c2 = createCommunityTestData(MultiThreadCommunity.class, 2);
        c2.getReadAccessControl().changeTo(Public.class);
        communityDao.register(c2);
        Community c3 = createCommunityTestData(SingleThreadCommunity.class, 3);
        c3.getReadAccessControl().changeTo(Public.class);
        communityDao.register(c3);
        Community c4 = createCommunityTestData(MultiThreadCommunity.class, 4);
        communityDao.register(c4);
        List<Community> communities = communityDao.findPublicCommunitiesRandomly(2);
        Assert.assertSame(2, communities.size());
        Assert.assertFalse(communities.contains(c4));
    }

    @Test
    public void testFindThread() {
        Community c = createCommunityTestData(SingleThreadCommunity.class, 1);
        communityDao.register(c);
        Thread t = createThread();
        t.store();
        communityDao.registerThread(c.getId(), t.getId());
        Thread found = communityDao.findThread(c.getId(), t.getId());
        Assert.assertEquals(t, found);
    }

    @Test
    public void testFindThreadR~jeBɑĂȂΌȂ() {
        Community c1 = createCommunityTestData(SingleThreadCommunity.class, 1);
        communityDao.register(c1);
        Thread t = createThread();
        t.store();
        communityDao.registerThread(c1.getId(), t.getId());
        Community c2 = createCommunityTestData(SingleThreadCommunity.class, 1);
        c2.setCode("community2");
        communityDao.register(c2);
        Thread found = communityDao.findThread(c2.getId(), t.getId());
        Assert.assertNull(found);
    }

    @Test
    public void testFindThreads() {
        Community c = createCommunityTestData(SingleThreadCommunity.class, 1);
        communityDao.register(c);
        Thread t1 = createThread();
        t1.store();
        communityDao.registerThread(c.getId(), t1.getId());
        Thread t2 = createThread();
        t2.store();
        communityDao.registerThread(c.getId(), t2.getId());
        List<Thread> threads = communityDao.findThreads(c.getId());
        Assert.assertSame(2, threads.size());
        Assert.assertTrue(threads.contains(t1));
        Assert.assertTrue(threads.contains(t2));
    }

    @Test
    public void testUpdateSingleThreadCommunity() {
        Community c = createCommunityTestData(SingleThreadCommunity.class, 1);
        communityDao.register(c);
        c.setCode("new_" + c.getCode());
        c.getOverview().setSubject("new_" + c.getOverview().getSubject(Locale.JAPANESE), Locale.JAPANESE);
        c.getOverview().setBody("new_" + c.getOverview().getSubject(Locale.JAPANESE), Locale.JAPANESE);
        c.getReadAccessControl().changeTo(Public.class);
        communityDao.update(c);
        Community actual = communityDao.find(c.getId());
        Assert.assertEquals(c, actual);
        assertConsistency(c);
    }

    @Test
    public void testDeleteSingleThreadCommunity() {
        Community c = createCommunityTestData(SingleThreadCommunity.class, 1);
        communityDao.register(c);
        communityDao.delete(c.getId());
        Assert.assertNull(communityDao.find(c.getId()));
        Assert.assertNull(messageTable.find(c.getOverview().getId()));
        Assert.assertNull(accessControlTable.find(c.getOverview().getReadAccessControl().getId()));
        Assert.assertNull(accessControlTable.find(c.getOverview().getWriteAccessControl().getId()));
        Assert.assertNull(accessControlTable.find(c.getReadAccessControl().getId()));
        Assert.assertNull(accessControlTable.find(c.getWriteAccessControl().getId()));
        Assert.assertNull(accessControlTable.find(c.getPostAccessControl().getId()));        
    }

    @Test
    public void testRegisterMyCommunity() {
        Community c = createCommunityTestData(SingleThreadCommunity.class, 1);
        communityDao.register(c);
        communityDao.registerMyCommunity(simpleUser1.getId(), c.getId());
        List<Community> actual = communityDao.findMyCommunies(simpleUser1.getId());
        Assert.assertSame(1, actual.size());
        Assert.assertEquals(c, actual.get(0));
    }

    @Test
    public void testFindMyCommunities() {
        Community c1 = createCommunityTestData(SingleThreadCommunity.class, 1);
        communityDao.register(c1);
        communityDao.registerMyCommunity(simpleUser1.getId(), c1.getId());
        Community c2 = createCommunityTestData(SingleThreadCommunity.class, 1);
        c2.setCode(c1.getCode() + "2");
        communityDao.register(c2);
        communityDao.registerMyCommunity(simpleUser1.getId(), c2.getId());
        Community c3 = createCommunityTestData(SingleThreadCommunity.class, 1);
        c3.setCode(c1.getCode() + "3");
        communityDao.register(c3);
        communityDao.registerMyCommunity(simpleUser2.getId(), c3.getId());
        List<Community> actual = communityDao.findMyCommunies(simpleUser1.getId());
        Assert.assertSame(2, actual.size());
        Assert.assertTrue(actual.contains(c1));
        Assert.assertTrue(actual.contains(c2));
    }

    @Test
    public void testFindCommunityUserIds() {
        Community c = createCommunityTestData(SingleThreadCommunity.class, 1);
        communityDao.register(c);
        communityDao.registerMyCommunity(simpleUser1.getId(), c.getId());
        communityDao.registerMyCommunity(simpleUser2.getId(), c.getId());
        List<Integer> userIds = communityDao.findCommunityUserIds(c.getId());
        Assert.assertTrue(userIds.contains(simpleUser1.getId()));
        Assert.assertTrue(userIds.contains(simpleUser2.getId()));
        Assert.assertFalse(userIds.contains(simpleUser3.getId()));
    }

    @Test
    public void testDeleteMyCommunity() {
        Community c = createCommunityTestData(SingleThreadCommunity.class, 1);
        communityDao.register(c);
        communityDao.registerMyCommunity(simpleUser1.getId(), c.getId());
        communityDao.deleteMyCommunity(simpleUser1.getId(), c.getId());
        List<Community> actual = communityDao.findMyCommunies(simpleUser1.getId());
        Assert.assertSame(0, actual.size());
    }

    private Community createCommunityTestData(Class<? extends Community> clazz, int no) {
        AccessControl rac = domainFactory.prototype(UserOnly.class);
        AccessControl wac = domainFactory.prototype(OwnerOnly.class);
        AccessControl pac = domainFactory.prototype(UserOnly.class);
        Community c = (Community) domainFactory.prototype((Class<? extends Domain>) clazz);
        c.setCode("testCommunity" + no);
        c.setOverview(createOverview(rac, wac));
        c.setOwnerId(simpleUser1.getId());
        c.setReadAccessControl(rac);
        c.setWriteAccessControl(wac);
        c.setPostAccessControl(pac);
        return c;
    }

    private Message createOverview(AccessControl readAccessControl, AccessControl writeAccessControl) {
        Message o = domainFactory.prototype(Message.class);
        o.setOwnerId(simpleUser1.getId());
        o.setAuthorId(simpleUser1.getId());
        o.setSubject("eXgR~jeB", Locale.JAPANESE);
        o.setBody("̓eXgp̃R~jeBłB", Locale.JAPANESE);
        o.setReadAccessControl(readAccessControl);
        o.setWriteAccessControl(writeAccessControl);
        return o;
    }

    private Thread createThread() {
        Thread t = domainFactory.prototype(Thread.class);
        AccessControl rac = domainFactory.prototype(UserOnly.class);
        AccessControl wac = domainFactory.prototype(OwnerOnly.class);
        AccessControl pac = domainFactory.prototype(UserOnly.class);
        t.setOverview(createThreadOverview(rac, wac));
        t.setOwnerId(simpleUser1.getId());
        t.setReadAccessControl(rac);
        t.setWriteAccessControl(wac);
        t.setPostAccessControl(pac);
        return t;
    }

    private Message createThreadOverview(AccessControl readAccessControl, AccessControl writeAccessControl) {
        Message o = domainFactory.prototype(Message.class);
        o.setOwnerId(simpleUser1.getId());
        o.setAuthorId(simpleUser1.getId());
        o.setSubject("eXgXbh", Locale.JAPANESE);
        o.setBody("̓eXgp̃XbhłB", Locale.JAPANESE);
        o.setReadAccessControl(readAccessControl);
        o.setWriteAccessControl(writeAccessControl);
        return o;
    }

    private Message createEntry(AccessControl readAccessControl, AccessControl writeAccessControl) {
        Message e = domainFactory.prototype(Message.class);
        e.setOwnerId(simpleUser1.getId());
        e.setAuthorId(simpleUser1.getId());
        e.setSubject("eXgL", Locale.JAPANESE);
        e.setBody("̓eXgp̋LłB", Locale.JAPANESE);
        e.setReadAccessControl(readAccessControl);
        e.setWriteAccessControl(writeAccessControl);
        return e;
    }

    private void assertConsistency(Community community) {
        Assert.assertSame(community.getOwnerId(), community.getOverview().getOwnerId());
        Assert.assertSame(community.getOwnerId(), community.getOverview().getAuthorId());
        Assert.assertEquals(community.getReadAccessControl(), community.getOverview().getReadAccessControl());
        Assert.assertEquals(community.getWriteAccessControl(), community.getOverview().getWriteAccessControl());
        Assert.assertSame(OwnerOnly.class, community.getWriteAccessControl().getType());
    }
}
