package org.unitedfront2.validation;

import java.util.Locale;
import java.util.Stack;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang.ArrayUtils;
import org.springframework.web.util.HtmlUtils;

/**
 * HTML ֘Ǎ؂s[eBeBNXłB
 *
 * @author kurokkie
 *
 */
public final class HtmlValidate {

    /**
     * p, ol, ul, li, a, span, div, br, em, i, strong, dl, dt, dd, hr, table,
     * tr, th, td
     */
    private static final String[] ALLOWED_TAGS
        = {"p", "ol", "ul", "li", "a", "span", "div", "br", "em", "i",
            "strong", "dl", "dt", "dd", "hr", "table", "tr", "th", "td", };

    /** font-weight, font-style, text-decoration, text-align, margin-left */
    private static final String[] ALLOWED_STYLE
        = {"font-weight", "font-style", "text-decoration", "text-align"};

    /**
     * HTML `̃eLXg؂܂B^O {@link #ALLOWED_TAGS} ̂ƂłBS
     * ŕ\LKv܂Bꕔ̃X^C̎gpĂ܂BX^C
     * {@link #ALLOWED_STYLE} ̂ƂłB
     *
     * @param html HTML `̃eLXg
     * @throws ValidationException ؗO
     */
    public static void htmlText(String html) throws ValidationException {
        String text = html.toLowerCase(Locale.ENGLISH);
        Pattern pattern = Pattern.compile("<[^>]+>");
        Matcher matcher = pattern.matcher(text);
        Stack<String> tagStack = new Stack<String>();
        int pointer = 0;
        String[] nextExpected = null;
        while (matcher.find()) {
            String preString = text.substring(pointer, matcher.start());
            validateEscaped(preString);
            pointer = matcher.end();

            String tag = matcher.group();
            if (tag.startsWith("</")) {
                if (nextExpected != null) {
                    throw new ValidationException(
                        "validation.HtmlValidate.htmlText.error",
                        new Object[] {tag}, "Expected Tag '"
                        + ArrayUtils.toString(nextExpected) + "' but '"
                        + tag.toUpperCase(Locale.ENGLISH) + "'.");
                }
                validateEndTag(tag, tagStack);
            } else if (tag.endsWith("/>")) {
                if (nextExpected != null) {
                    throw new ValidationException(
                        "validation.HtmlValidate.htmlText.error",
                        new Object[] {tag}, "Expected Tag '"
                        + ArrayUtils.toString(nextExpected) + "' but '"
                        + tag.toUpperCase(Locale.ENGLISH) + "'.");
                }
                validateNoValueTag(tag);
            } else {
                nextExpected = validateStartTag(tag, tagStack, nextExpected);
            }
        }
        validateEscaped(text.substring(pointer));
        if (!tagStack.empty()) {
            throw new ValidationException(
                    "validation.HtmlValidate.htmlText.error",
                    new Object[] {tagStack},
                    "Tag " + tagStack + " not closed.");
        }
    }

    private static void validateEscaped(String s) throws ValidationException {
        if (!s.matches("^[^<>\"]*$")) {
            throw new ValidationException(
                    "validation.HtmlValidate.htmlText.error",
                    new Object[] {HtmlUtils.htmlEscape(s)},
                    "Text '" + s + "' contains invalid character '<>\"'");
        }
    }

    private static String[] validateStartTag(String tag,
        Stack<String> tagStack, String[] nextExpected)
        throws ValidationException {

        String tagName = tag.substring(1, tag.length() - 1).trim();
        if (tagName.contains(" ")) {
            tagName = tagName.substring(0, tag.indexOf(' ') - 1);
        }
        if (nextExpected != null
            && !ArrayUtils.contains(nextExpected, tagName)) {

            throw new ValidationException(
                "validation.HtmlValidate.htmlText.error",
                new Object[] {tagName}, "Expected Tag '"
                + ArrayUtils.toString(nextExpected) + "' but '"
                + tagName.toUpperCase(Locale.ENGLISH) + "'.");
        }
        validateTagName(tagName);
        validateOrder(tagName, tagStack);
        tagStack.push(tagName);
        validateAttributes(tag, tagName);
        if ("table".equals(tagName)) {
            return new String[] {"tr"};
        } else if ("ul".equals(tagName) || "ol".equals(tagName)) {
            return new String[] {"li"};
        } else if ("tr".equals(tagName)) {
            return new String[] {"th", "td"};
        } else if ("dl".equals(tagName)) {
            return new String[] {"dt", "dd"};
        } else {
            return null;
        }
    }

    private static void validateOrder(String tagName, Stack<String> tagStack)
        throws ValidationException {
        validateOrderForLI(tagName, tagStack);
        validateOrderForTR(tagName, tagStack);
        validateOrderForTH(tagName, tagStack);
        validateOrderForDT(tagName, tagStack);
    }

    private static void validateOrderForLI(String tagName,
            Stack<String> tagStack) throws ValidationException {
        if ("li".equals(tagName) && !tagStack.contains("ul")
                && !tagStack.contains("ol")) {
            throw new ValidationException(
                "validation.HtmlValidate.htmlText.error",
                new Object[] {tagName}, "Tag 'UL' or 'OL' not found before '"
                    + tagName.toUpperCase(Locale.ENGLISH) + "'.");
        }
    }

    private static void validateOrderForTR(String tagName,
            Stack<String> tagStack) throws ValidationException {
        if ("tr".equals(tagName) && !tagStack.contains("table")) {
            throw new ValidationException(
                "validation.HtmlValidate.htmlText.error",
                new Object[] {tagName}, "Tag 'TABLE' not found before '"
                    + tagName.toUpperCase(Locale.ENGLISH) + "'.");
        }
    }

    private static void validateOrderForTH(String tagName,
            Stack<String> tagStack) throws ValidationException {
        if (("th".equals(tagName) || "td".equals(tagName))
                && !tagStack.contains("tr")) {
            throw new ValidationException(
                "validation.HtmlValidate.htmlText.error",
                new Object[] {tagName}, "Tag 'TR' not found before '"
                    + tagName.toUpperCase(Locale.ENGLISH) + "'.");
        }
    }

    private static void validateOrderForDT(String tagName,
            Stack<String> tagStack) throws ValidationException {
        if (("dt".equals(tagName) || "dd".equals(tagName))
                && !tagStack.contains("dl")) {
            throw new ValidationException(
                "validation.HtmlValidate.htmlText.error",
                new Object[] {tagName}, "Tag 'TR' not found before '"
                    + tagName.toUpperCase(Locale.ENGLISH) + "'.");
        }
    }

    private static void validateTagName(String tagName)
        throws ValidationException {

        if (!ArrayUtils.contains(ALLOWED_TAGS, tagName)) {
            throw new ValidationException(
                    "validation.HtmlValidate.htmlText.error",
                    new Object[] {tagName},
                    "Tag '" + tagName + "' not allowed.");
        }
    }

    private static void validateAttributes(String tag, String tagName)
        throws ValidationException {

        Pattern pattern = Pattern.compile("\\s([a-z]+)=\"([^\"]+)\"");
        Matcher matcher = pattern.matcher(tag);
        while (matcher.find()) {
            String attrName = matcher.group(1);
            String value = matcher.group(2);
            if ("style".equals(attrName)) {
                validateStyleValue(value);
            } else if ("href".equals(attrName) && "a".equals(tagName)) {
                continue;
            } else {
                throw new ValidationException(
                        "validation.HtmlValidate.htmlText.error",
                        new Object[] {attrName},
                        "Attribute '" + attrName + "' not allowed.");
            }
        }
    }

    private static void validateStyleValue(String styleValue)
        throws ValidationException {

        String[] styles = styleValue.split(";\\s?");
        for (String style : styles) {
            String styleName = style.substring(0, style.indexOf(':'));
            if (!ArrayUtils.contains(ALLOWED_STYLE, styleName)) {
                throw new ValidationException(
                        "validation.HtmlValidate.htmlText.error",
                        new Object[] {styleValue},
                        "Style '" + styleName + "' not allowed.");
            }
        }
    }

    private static void validateEndTag(String tag,
            Stack<String> tagStack) throws ValidationException {
        String tagName = tag.substring(2, tag.length() - 1).trim();

        validateTagName(tagName);

        String startTag = tagStack.pop();
        if (!tagName.equals(startTag)) {
            throw new ValidationException(
                    "validation.HtmlValidate.htmlText.error",
                    new Object[] {tagName},
                    "Invalid order between '" + startTag + "' and '" + tagName
                    + "'.");
        }
    }

    private static void validateNoValueTag(String tag)
        throws ValidationException {

        String tagName = tag.substring(1, tag.length() - 2).trim();
        if (tagName.contains(" ")) {
            tagName = tagName.substring(0, tag.indexOf(' ') - 1);
        }
        validateTagName(tagName);
        validateAttributes(tag, tagName);
    }

    private HtmlValidate() {
        super();
    }
}
