package org.unitedfront2.domain.communication;

import java.io.Serializable;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.unitedfront2.dao.DateDao;
import org.unitedfront2.dao.MessageDao;
import org.unitedfront2.domain.Deletable;
import org.unitedfront2.domain.Domain;
import org.unitedfront2.domain.Identifiable;
import org.unitedfront2.domain.Storable;
import org.unitedfront2.domain.UnsupportedLocaleException;
import org.unitedfront2.domain.accesscontrol.AbstractCopyrightedResource;
import org.unitedfront2.domain.accesscontrol.AccessControl;

/**
 * bZ[W\hCfłBі{͍ۉɑΉĂ܂B<p>
 *
 * bZ[ẂA[Ũb`ȓ͂邽߂ɔėpIɗpł܂BႦ΁AfAuOȂǂ
 * p[cƂč̗pł܂B̃R[h́AbZ[WVɐAiĂ܂B<p>
 *
 * <pre>
 * DomainFactory factory = ...
 * Message message = factory.prototype(Message.class);
 * ...
 * message.store();
 * </pre>
 *
 * bZ[Wǂ̂悤ɗp邩͗p鑤Ō߂܂B<p>
 *
 * @author kurokkie
 *
 */
public class Message extends AbstractCopyrightedResource
    implements Identifiable<Message>, Storable, Serializable, Deletable,
    Domain {

    /** _ŐƂ̃R[h̒ (32) */
    public static final int GENERATED_CODE_LENGTH = 32;

    /** VAԍ */
    private static final long serialVersionUID = 896369281666966593L;

    /** _R[hdۂɁAĐčă`W (10) */
    private static final int MAX_CHALLENGE_COUNT = 10;

    /** ID */
    private Integer id;

    /**
     * R[h
     *
     * @invariant ӂȒl
     */
    private String code;

    /**
     * ΉGg̈ꗗBAvP[VΉĂSĂ̌ɂĂ {@link Map} I
     * uWFNg܂B
     *
     * @invariant ΉSČɂẴ}bvێ܂BΉĂ錾L[Ƃ
     * <code>get</code> \bhĂяoƂÃ}bvԂ܂B<code>null</code> Ԃ
     * Ƃ͂܂B
     */
    private Map<Locale, MessageEntry> entryMap;

    /** o^ */
    private Date registrationDate;

    /** ŏIXV */
    private Date lastUpdateDate;

    /** ̃bZ[Wւ URI */
    private transient String uri;

    /** f[^ANZXIuWFNg */
    private transient DateDao dateDao;

    /** bZ[Wf[^ANZXIuWFNg */
    private transient MessageDao messageDao;

    /** VXeT|[g郍P[̃Xg */
    private transient List<Locale> availableLocales;

    public Message() {
        super();
    }

    public Message(Integer ownerId, Integer authorId,
            AccessControl readAccessControl, AccessControl writeAccessControl) {
        super(ownerId, authorId, readAccessControl, writeAccessControl);
    }

    public Message(String code, Integer ownerId, Integer authorId,
            AccessControl readAccessControl, AccessControl writeAccessControl) {
        this(code, null, ownerId, authorId, readAccessControl,
                writeAccessControl);
    }

    public Message(String code, Map<Locale, MessageEntry> entryMap,
            Integer ownerId, Integer authorId, AccessControl readAccessControl,
            AccessControl writeAccessControl) {
        super(ownerId, authorId, readAccessControl, writeAccessControl);
        this.code = code;
        this.entryMap = entryMap;
    }

    public Message(Integer id, String code, Map<Locale, MessageEntry> entryMap,
            Integer ownerId, Integer authorId, AccessControl readAccessControl,
            AccessControl writeAccessControl, Date registrationDate,
            Date lastUpdateDate) {
        this(code, entryMap, ownerId, authorId, readAccessControl,
                writeAccessControl);
        this.id = id;
        this.registrationDate = (Date) registrationDate.clone();
        this.lastUpdateDate = (Date) lastUpdateDate.clone();
    }

    @Override
    protected boolean buildEqualsBuilder(EqualsBuilder eb, Object other) {
        if (!(other instanceof Message)) {
            return false;
        }
        Message castOther = (Message) other;
        eb.append(id, castOther.id)
            .append(code, castOther.code)
            .append(entryMap, castOther.entryMap)
            .append(registrationDate, castOther.registrationDate)
            .append(lastUpdateDate, castOther.lastUpdateDate);
        return super.buildEqualsBuilder(eb, other);
    }

    @Override
    protected void buildHashCodeBuilder(HashCodeBuilder hcb) {
        hcb.append(id)
            .append(code)
            .append(entryMap)
            .append(registrationDate)
            .append(lastUpdateDate);
        super.buildHashCodeBuilder(hcb);
    }

    @Override
    protected void buildToStringBuilder(ToStringBuilder tsb) {
        tsb.append("id", id)
            .append("code", code)
            .append("entryMap", entryMap)
            .append("registrationDate", registrationDate)
            .append("lastUpdateDate", lastUpdateDate);
        super.buildToStringBuilder(tsb);
    }

    @Override
    public boolean identify(Message message) {
        if (message == null || id == null) {
            return false;
        }
        return id.equals(message.getId());
    }

    /**
     * {@link Storable#store()}<p>
     *
     * VKo^ہAR[hݒ肳ĂȂ΁AŔs܂BVKo^ɂ́Ao^Ɍݓ
     * ݒ肳܂Bo^͈ȌύXł܂BƂXVɓo^ݒ肳ĂƂĂ
     * ͖܂BXV͌ݓݒ肳܂B
     *
     * @ensure if ${this.code} is null, ${this.code} is auto generated.
     * @ensure ${this.lastUpdateDate} is current date.
     * @throws MessageCodeUsedByOtherException bZ[WR[hdĂ
     * @see #GENERATED_CODE_LENGTH
     * @see #MAX_CHALLENGE_COUNT
     */
    @Override
    public void store() throws MessageCodeUsedByOtherException {
        if (this.id == null) {
            // VKo^
            if (this.code == null) {
                this.code = generateCode();
            } else {
                if (messageDao.findByCode(code) != null) {
                    String errorMessage = "The message code '" + code
                        + "' already used by other.";
                    logger.warn(errorMessage);
                    throw new MessageCodeUsedByOtherException(errorMessage);
                }
            }
            Date currentDate = dateDao.getCurrentDate();
            this.registrationDate = currentDate;
            this.lastUpdateDate = currentDate;
            messageDao.register(this);
        } else {
            // XV
            Message found = messageDao.findByCode(code);
            if (found != null && !found.identify(this)) {
                String errorMessage = "The message code '" + code
                    + "' already used by other.";
                logger.warn(errorMessage);
                throw new MessageCodeUsedByOtherException(errorMessage);
            }
            Message current;
            if (identify(found)) {
                current = found;
            } else {
                current = messageDao.find(this.id);
            }
            this.registrationDate = current.registrationDate;
            this.lastUpdateDate = dateDao.getCurrentDate();
            messageDao.update(this);
        }
    }

    @Override
    public void delete() {
        messageDao.delete(id);
    }

    private String generateCode() {
        for (int i = 0; i < MAX_CHALLENGE_COUNT; i++) {
            String code = RandomStringUtils.randomAlphanumeric(
                    GENERATED_CODE_LENGTH).toLowerCase(Locale.ENGLISH);
            if (messageDao.findByCode(code) == null) {
                return code;
            }
        }
        String message = "Failed to generate code.";
        logger.error(message);
        throw new IllegalStateException(message);
    }

    /**
     * łȂ擾܂B
     *
     * @return 
     */
    public String getSubject() {
        return getRequiredEntry().getSubject();
    }

    /**
     * w肵P[ɑΉ擾܂Bw肵P[ɑΉ錏ȂA܂
     * ̏ꍇ {@link #getRequiredEntry()} ŕԂGǧԂ܂B
     *
     * @param locale P[
     * @return 
     */
    public String getSubject(Locale locale) {
        MessageEntry entry = entryMap.get(locale);
        if (entry == null || StringUtils.isBlank(entry.getSubject())) {
            entry = getRequiredEntry();
        }
        return entry.getSubject();
    }

    /**
     * w肵P[ɑΉݒ肵܂B
     *
     * @param subject 
     * @param locale P[
     */
    public void setSubject(String subject, Locale locale) {
        if (entryMap == null) {
            fillEntryMap();
        }
        entryMap.get(locale).setSubject(subject);
    }

    /**
     * łȂ{擾܂B
     *
     * @return {
     */
    public String getBody() {
        return getRequiredEntry().getBody();
    }

    /**
     * w肵P[ɑΉ{擾܂Bw肵P[ɑΉ{Ȃꍇ
     * {@link #getRequiredEntry()} ŕԂGg̖{Ԃ܂B
     *
     * @param locale P[
     * @return {
     */
    public String getBody(Locale locale) {
        MessageEntry entry = entryMap.get(locale);
        if (entry == null) {
            entry = getRequiredEntry();
        }
        return entry.getBody();
    }

    /**
     * w肵P[ɑΉ{ݒ肵܂B
     *
     * @param body {
     * @param locale P[
     */
    public void setBody(String body, Locale locale) {
        if (entryMap == null) {
            fillEntryMap();
        }
        entryMap.get(locale).setBody(body);
    }

    /**
     * Ggł邩ǂ肵܂B
     *
     * @return GgȂ <code>true</code> AłȂȂ <code>false</code>
     */
    public boolean isEmptyEntry() {
        for (MessageEntry entry : entryMap.values()) {
            if (StringUtils.isNotEmpty(entry.getSubject())
                    || StringUtils.isNotEmpty(entry.getBody())) {
                return false;
            }
        }
        return true;
    }

    public Integer getId() {
        return id;
    }

    public void setId(Integer id) {
        this.id = id;
    }

    public String getCode() {
        return code;
    }

    public void setCode(String code) {
        this.code = code;
    }

    /**
     * Gg}bv擾܂BύXs\ȃ}bvłB
     *
     * @return Gg}bv
     */
    public Map<Locale, MessageEntry> getEntryMap() {
        fillEntryMap();
        return new HashMap<Locale, MessageEntry>(entryMap);
    }

    /**
     * ^CgłȂCӂ̃Ggv܂B
     *
     * @return Cӂ̃GgAȂ΋̃Gg
     */
    public MessageEntry getRequiredEntry() {
        if (getEntryMap() == null || getEntryMap().size() == 0) {
            return new MessageEntry();
        } else {
            for (MessageEntry entry : getEntryMap().values()) {
                if (StringUtils.isNotBlank(entry.getSubject())) {
                    return entry;
                }
            }
            return getEntryMap().values().iterator().next();
        }
    }

    /**
     * Gg}bvݒ肵܂BȂP[΁ÃGgǉ܂B
     *
     * @param entryMap Gg}bv
     * @ensure VXeSẴP[Gg}bvɐݒ肳
     * @throws IllegalArgumentException VXeĂȂP[܂܂Ă
     */
    public void setEntryMap(Map<Locale, MessageEntry> entryMap)
        throws IllegalArgumentException {
        if (entryMap != null && availableLocales != null) {
            for (Locale l : entryMap.keySet()) {
                if (!availableLocales.contains(l)) {
                    String message = "The locale '" + l + "' is not supported.";
                    logger.error(message);
                    throw new IllegalArgumentException(
                            new UnsupportedLocaleException(message));
                }
            }
        }
        this.entryMap = entryMap;
        fillEntryMap();
    }

    private void fillEntryMap() {
        if (entryMap == null) {
            entryMap = new HashMap<Locale, MessageEntry>();
        }
        if (availableLocales != null) {
            for (Locale locale : availableLocales) {
                if (!entryMap.containsKey(locale)) {
                    entryMap.put(locale, new MessageEntry());
                }
            }
        }
    }

    public Date getRegistrationDate() {
        if (registrationDate != null) {
            return (Date) registrationDate.clone();
        } else {
            return null;
        }
    }

    public void setRegistrationDate(Date registrationDate) {
        if (registrationDate != null) {
            this.registrationDate = (Date) registrationDate.clone();
        } else {
            this.registrationDate = null;
        }
    }

    public Date getLastUpdateDate() {
        if (lastUpdateDate != null) {
            return (Date) lastUpdateDate.clone();
        } else {
            return null;
        }
    }

    public void setLastUpdateDate(Date lastUpdateDate) {
        if (lastUpdateDate != null) {
            this.lastUpdateDate = (Date) lastUpdateDate.clone();
        } else {
            this.lastUpdateDate = null;
        }
    }

    /**
     * T|[g錾ݒ肵܂BT|[gȂꂪ܂܂ĂꍇÃbZ[WGg͍폜
     * BT|[g錾ꂪ ${this.entryMap} Ɋ܂܂ĂȂꍇÃbZ[WGg͒ǉ
     * ܂B
     *
     * @param availableLocales T|[g錾ꃊXg
     */
    public void setAvailableLocales(List<Locale> availableLocales) {
        this.availableLocales = availableLocales;
        fillEntryMap();
        deleteUnsupportedLocales();
    }

    private void deleteUnsupportedLocales() {
        if (entryMap != null) {
            Set<Locale> del = new HashSet<Locale>();
            for (Locale l : entryMap.keySet()) {
                if (!availableLocales.contains(l)) {
                    del.add(l);
                }
            }
            for (Locale l : del) {
                entryMap.remove(l);
            }
        }
    }

    public String getUri() {
        return uri;
    }

    public void setUri(String uri) {
        this.uri = uri;
    }

    public void setDateDao(DateDao dateDao) {
        this.dateDao = dateDao;
    }

    public void setMessageDao(MessageDao messageDao) {
        this.messageDao = messageDao;
    }
}
