package org.unitedfront2.dao.jdbc;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Date;
import java.util.List;

import javax.sql.DataSource;

import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.jdbc.core.simple.SimpleJdbcInsert;
import org.springframework.stereotype.Repository;
import org.unitedfront2.dao.SimpleUserDao;
import org.unitedfront2.domain.DomainFactory;
import org.unitedfront2.domain.SimpleUser;

/**
 * {@link SimpleUserDao}  JDBC NXłB
 *
 * @author kurokkie
 *
 */
@Repository(value = "simpleUserDao")
public class SimpleUserDaoImpl extends SimpleDaoSupport<SimpleUser>
    implements SimpleUserDao {

    /** [U}bp[NX */
    private static class SimpleUserRowMapper
        implements ParameterizedRowMapper<SimpleUser> {

        /** hCt@Ng */
        private DomainFactory domainFactory;

        public SimpleUserRowMapper(DomainFactory domainFactory) {
            super();
            this.domainFactory = domainFactory;
        }

        @Override
        public SimpleUser mapRow(ResultSet rs, int rowNum) throws SQLException {
            SimpleUser user = domainFactory.prototype(SimpleUser.class);
            user.setId(rs.getInt("Id"));
            user.setCode(rs.getString("Code"));
            user.setName(rs.getString("Name"));
            user.setLastAccessDate(rs.getTimestamp("LastAccessDate"));
            return user;
        }
    }

    //**************************************************************************
    // SimpleDaoSupport
    //**************************************************************************
    @Override
    public ParameterizedRowMapper<SimpleUser> createRowMapper() {
        return new SimpleUserRowMapper(getDomainFactory());
    }

    @Override
    protected SimpleJdbcInsert createSimpleJdbcInsert(DataSource dataSource) {
        return new SimpleJdbcInsert(dataSource).withTableName(getTableName());
    }

    @Override
    protected Class<SimpleUser> getDomainClass() {
        return SimpleUser.class;
    }

    @Override
    protected String getTableName() {
        return "User";
    }

    //**************************************************************************
    // SimpleUserDao
    //**************************************************************************
    @Override
    public void register(SimpleUser user) {
        super.registerNoReturnKey(user);
    }

    @Override
    public Date findLastAccessDate(int id) {
        Date lastAccessDate = (Date) getJdbcTemplate().queryForObject(
            "SELECT LastAccessDate FROM User WHERE Id = ?",
            new Object[] {id}, Date.class);
        return lastAccessDate;
    }

    @Override
    public SimpleUser findByCode(String code) {
        SimpleUser user;
        try {
            user = (SimpleUser) getJdbcTemplate().queryForObject(
                "SELECT * FROM User WHERE Code = ?", new Object[] {code},
                new SimpleUserRowMapper(getDomainFactory()));
        } catch (EmptyResultDataAccessException e) {
            logger.debug(e.getMessage());
            return null;
        }
        user.setSimpleUserDao(this);
        return user;
    }

    @Override
    public SimpleUser findByName(String name) {
        SimpleUser user;
        try {
            user = (SimpleUser) getJdbcTemplate().queryForObject(
                "SELECT * FROM User WHERE Name = ?", new Object[] {name},
                new SimpleUserRowMapper(getDomainFactory()));
        } catch (EmptyResultDataAccessException e) {
            logger.debug(e.getMessage());
            return null;
        }
        user.setSimpleUserDao(this);
        return user;
    }

    @Override
    public List<SimpleUser> findByLastAccessDateAfter(Date date) {
        return getSimpleJdbcTemplate().query(
                "SELECT * FROM User WHERE LastAccessDate >= ?",
                new SimpleUserRowMapper(getDomainFactory()), date);
    }

    @Override
    public int countByLastAccessDateAfter(Date date) {
        return getSimpleJdbcTemplate().queryForInt(
                "SELECT COUNT(*) FROM User WHERE LastAccessDate >= ?", date);
    }

    @Override
    public void updateLastAccessDate(int id, Date date) {
        getJdbcTemplate().update(
            "UPDATE User SET LastAccessDate = ? WHERE Id = ?",
            new Object[] {date, id});
    }
}
