package org.unitedfront2.dao.jdbc;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.stereotype.Repository;
import org.unitedfront2.dao.BlogDao;
import org.unitedfront2.domain.DomainFactory;
import org.unitedfront2.domain.accesscontrol.AccessControlTable;
import org.unitedfront2.domain.accesscontrol.Public;
import org.unitedfront2.domain.communication.Blog;
import org.unitedfront2.domain.communication.BlogEntry;
import org.unitedfront2.domain.communication.Comment;
import org.unitedfront2.domain.communication.MessageCodeUsedByOtherException;
import org.unitedfront2.domain.communication.MessageTable;

/**
 * {@link BlogDao} ̎NXłB
 *
 * @author kurokkie
 *
 */
@Repository(value = "blogDao")
public class BlogDaoImpl extends SimpleDaoSupport<Blog> implements BlogDao {

    /** uO}bp[NX */
    private static class BlogRowMapper implements ParameterizedRowMapper<Blog> {

        /** uOt@Ng */
        private DomainFactory domainFactory;

        /** bZ[We[u */
        private MessageTable messageTable;

        /** ANZXe[u */
        private AccessControlTable accessControlTable;

        public BlogRowMapper(DomainFactory domainFactory,
                MessageTable messageTable,
                AccessControlTable accessControlTable) {
            super();
            this.domainFactory = domainFactory;
            this.messageTable = messageTable;
            this.accessControlTable = accessControlTable;
        }

        @Override
        public Blog mapRow(ResultSet rs, int rowNum) throws SQLException {
            Blog blog = domainFactory.prototype(Blog.class);
            blog.setId(rs.getInt("Id"));
            blog.setCode(rs.getString("Code"));
            blog.setOwnerId(rs.getInt("OwnerId"));
            blog.setOverview(messageTable.find(rs.getInt("OverviewId")));
            blog.setReadAccessControl(accessControlTable.find(rs.getInt(
                    "ReadAccessControlId")));
            blog.setWriteAccessControl(accessControlTable.find(rs.getInt(
                    "WriteAccessControlId")));
            blog.setCommentAccessControl(accessControlTable.find(rs.getInt(
                    "CommentAccessControlId")));
            return blog;
        }
    }

    /** uOL}bp[ */
    private static class BlogEntryRowMapper
        implements ParameterizedRowMapper<BlogEntry> {

        /** uOLhCt@Ng */
        private DomainFactory domainFactory;

        /** bZ[We[u */
        private MessageTable messageTable;

        public BlogEntryRowMapper(DomainFactory domainFactory,
                MessageTable messageTable) {
            super();
            this.domainFactory = domainFactory;
            this.messageTable = messageTable;
        }

        @Override
        public BlogEntry mapRow(ResultSet rs, int rowNum) throws SQLException {
            BlogEntry blogEntry = domainFactory.prototype(BlogEntry.class);
            blogEntry.setEntry(messageTable.find(rs.getInt("EntryId")));
            return blogEntry;
        }
    }

    /** bZ[We[u */
    @Autowired private MessageTable messageTable;

    /** ANZXe[u */
    @Autowired private AccessControlTable accessControlTable;

    /** Rgf[^ANZXIuWFNg */
    @Resource(name = "commentDao")
    private RowMapperFactory<Comment> commentRowMapperFactory;

    //**************************************************************************
    // SimpleDaoSupport
    //**************************************************************************
    @Override
    public ParameterizedRowMapper<Blog> createRowMapper() {
        return new BlogRowMapper(getDomainFactory(), messageTable,
                accessControlTable);
    }

    @Override
    protected Class<Blog> getDomainClass() {
        return Blog.class;
    }

    /**
     * uOL̃}bp[NX𐶐܂B
     *
     * @return uOL̃}bp[NX
     */
    protected ParameterizedRowMapper<BlogEntry> createBlogEntryRowMapper() {
        return new BlogEntryRowMapper(getDomainFactory(), messageTable);
    }

    @Override
    protected Map<String, Object> toColumnValueMap(Blog blog) {
        Map<String, Object> map = super.toColumnValueMap(blog);
        map.put("OverviewId", blog.getOverview().getId());
        map.put("ReadAccessControlId", blog.getReadAccessControl().getId());
        map.put("WriteAccessControlId", blog.getWriteAccessControl().getId());
        map.put("CommentAccessControlId",
                blog.getCommentAccessControl().getId());
        return map;
    }

    //**************************************************************************
    // BlogDao
    //**************************************************************************
    @Override
    public void register(Blog blog) {
        try {
            blog.getOverview().store();
        } catch (MessageCodeUsedByOtherException e) {
            // bZ[WR[h͎s邽߁AN肦Ȃ
            logger.error(e);
            throw new IllegalStateException(e);
        }
        blog.getReadAccessControl().store();
        blog.getWriteAccessControl().store();
        blog.getCommentAccessControl().store();
        super.register(blog);
    }

    @Override
    public void registerBlogEntry(int blogId, int entryId) {
        getSimpleJdbcTemplate().update(
                "INSERT INTO BlogEntry(BlogId, EntryId) VALUES(?, ?)", blogId,
                entryId);
    }

    @Override
    public void registerComment(int entryId, int commentId) {
        getSimpleJdbcTemplate().update(
                "INSERT INTO BlogComment(EntryId, CommentId) VALUES(?, ?)",
                entryId, commentId);
    }

    @Override
    public Blog findByCode(String code) {
        try {
            return getSimpleJdbcTemplate().queryForObject(
                    "SELECT * FROM Blog WHERE Code = ?", createRowMapper(),
                    code);
        } catch (EmptyResultDataAccessException e) {
            logger.debug(e.getMessage());
            return null;
        }
    }

    @Override
    public List<Blog> findByOwnerId(int ownerId) {
        return getSimpleJdbcTemplate().query(
                "SELECT * FROM Blog WHERE OwnerId = ?", createRowMapper(),
                ownerId);
    }

    @Override
    public List<Blog> findPublicBlogs() {
        return getSimpleJdbcTemplate().query("SELECT b.* "
                + "FROM Blog b, AccessControl a "
                + "WHERE b.ReadAccessControlId = a.Id AND a.Class = '"
                + Public.class.getName() + "' " + "ORDER BY b.Id DESC",
                createRowMapper());
    }

    @Override
    public List<Blog> findPublicBlogsRandomly(int max) {
        return getSimpleJdbcTemplate().query("SELECT b.* "
                + "FROM Blog b, AccessControl a "
                + "WHERE b.ReadAccessControlId = a.Id AND a.Class = '"
                + Public.class.getName() + "' ORDER BY RAND() LIMIT 0, ?",
                createRowMapper(), max);
    }

    @Override
    public List<BlogEntry> findBlogEntries(int blogId, int no, int num) {
        return getSimpleJdbcTemplate().query(
                "SELECT * FROM BlogEntry WHERE BlogId = ? "
                + "ORDER BY EntryId DESC LIMIT ?, ?",
                createBlogEntryRowMapper(), blogId, no, num);
    }

    @Override
    public List<Comment> findComments(int entryId) {
        return getSimpleJdbcTemplate().query(
                "SELECT * FROM BlogComment bc, Comment c WHERE bc.EntryId = ? "
                + "AND bc.CommentId = c.Id",
                commentRowMapperFactory.createRowMapper(), entryId);
    }

    @Override
    public int countBlogEntry(int blogId) {
        return getSimpleJdbcTemplate().queryForInt(
                "SELECT COUNT(*) FROM BlogEntry WHERE BlogId = ?", blogId);
    }

    @Override
    public int countComment(int blogEntryId) {
        return getSimpleJdbcTemplate().queryForInt(
                "SELECT COUNT(*) FROM BlogComment WHERE EntryId = ?",
                blogEntryId);
    }

    @Override
    public void update(Blog blog) {
        try {
            blog.getOverview().store();
        } catch (MessageCodeUsedByOtherException e) {
            // bZ[WR[h͕ύXȂ߁AN肦Ȃ
            logger.error(e);
            throw new IllegalStateException(e);
        }
        blog.getReadAccessControl().store();
        blog.getWriteAccessControl().store();
        blog.getCommentAccessControl().store();
        super.update(blog);
    }

    @Override
    public void delete(int id) {
        Blog b = find(id);
        super.delete(id);
        try {
            b.getOverview().delete();
        } catch (DataIntegrityViolationException e) {
            logger.debug(e.getMessage());
        }
        try {
            b.getReadAccessControl().delete();
        } catch (DataIntegrityViolationException e) {
            logger.debug(e.getMessage());
        }
        try {
            b.getWriteAccessControl().delete();
        } catch (DataIntegrityViolationException e) {
            logger.debug(e.getMessage());
        }
        try {
            b.getCommentAccessControl().delete();
        } catch (DataIntegrityViolationException e) {
            logger.debug(e.getMessage());
        }
    }

    @Override
    public void deleteBlogEntry(int entryId) {
        deleteComments(entryId);
        messageTable.get(entryId).delete();
    }

    private void deleteComments(int entryId) {
        getSimpleJdbcTemplate().update(
                "DELETE FROM Comment WHERE Id IN (SELECT CommentId FROM "
                + "BlogComment WHERE EntryId = ?)", entryId);
    }
}
