package org.unitedfront2.domain.communication;

import java.util.Date;
import java.util.List;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.unitedfront2.dao.CommunityDao;
import org.unitedfront2.domain.DomainFactory;
import org.unitedfront2.domain.User;
import org.unitedfront2.domain.accesscontrol.AbstractResource;
import org.unitedfront2.domain.accesscontrol.AccessControl;
import org.unitedfront2.domain.accesscontrol.AccessDeniedException;
import org.unitedfront2.domain.accesscontrol.OwnerOnly;

/**
 * ۓI {@link Community} łB
 *
 * @author kurokkie
 *
 */
public abstract class AbstractCommunity extends AbstractResource
    implements Community {

    /** hCt@Ng */
    private transient DomainFactory domainFactory;

    /**
     * ID
     *
     * @invariant ӂȒl
     */
    private Integer id;

    /**
     * R[h
     *
     * @invariant ӂȒl
     */
    private String code;

    /** Tv */
    private Message overview;

    /** eɑ΂ANZX */
    private AccessControl postAccessControl;

    /** ŏIXV */
    private transient Date lastUpdateDate;

    /** Q̃R~jeBȂ <code>true</code> AłȂ <code>false</code> */
    private transient Boolean myCommunity;

    /** R~jeBf[^ANZXIuWFNg */
    private transient CommunityDao communityDao;

    public AbstractCommunity() {
        super();
    }

    public AbstractCommunity(String code, Message overview, Integer ownerId,
            AccessControl readAccessControl, AccessControl writeAccessControl,
            AccessControl postAccessControl) {
        super(ownerId, readAccessControl, writeAccessControl);
        this.code = code;
        this.overview = overview;
        this.postAccessControl = postAccessControl;
    }

    public AbstractCommunity(Integer id, String code, Message overview,
            Integer ownerId, AccessControl readAccessControl,
            AccessControl writeAccessControl, AccessControl postAccessControl) {
        this(code, overview, ownerId, readAccessControl, writeAccessControl,
                postAccessControl);
        this.id = id;
    }

    @Override
    protected final boolean buildEqualsBuilder(EqualsBuilder eb, Object other) {
        if (!(other instanceof Community)) {
            return false;
        }
        Community castOther = (Community) other;
        eb.append(id, castOther.getId())
            .append(code, castOther.getCode())
            .append(overview, castOther.getOverview());
        if (!super.buildEqualsBuilder(eb, other)) {
            return false;
        }
        eb.append(postAccessControl, castOther.getPostAccessControl());
        return super.buildEqualsBuilder(eb, other);
    }

    @Override
    protected final void buildHashCodeBuilder(HashCodeBuilder hcb) {
        hcb.append(id).append(code).append(getClass()).append(overview);
        super.buildHashCodeBuilder(hcb);
        hcb.append(postAccessControl);
    }

    @Override
    protected final void buildToStringBuilder(ToStringBuilder tsb) {
        tsb.append("id", id).append("code", code).append("overview", overview);
        super.buildToStringBuilder(tsb);
        tsb.append("postAccessControl", postAccessControl);
    }

    @Override
    public boolean identify(Community community) {
        if (id == null) {
            return false;
        }
        return id.equals(community.getId());
    }

    @Override
    public void store() throws CommunityCodeUsedByOtherException {
        Community c = communityDao.findByCode(code);
        if (c != null && !c.identify(this)) {
            String message = "The code '" + code
                + "' has already been used by other.";
            logger.warn(message);
            throw new CommunityCodeUsedByOtherException(message);
        }
        setWriteAccessControl(domainFactory.prototype(OwnerOnly.class));
        getOverview().setOwnerId(getOwnerId());
        getOverview().setAuthorId(getOwnerId());
        if (id == null) {
            communityDao.register(this);
            communityDao.registerMyCommunity(getOwnerId(), this.getId());
        } else {
            communityDao.update(this);

            // Qƌ[U̓R~jeBމ
            for (Integer userId : communityDao.findCommunityUserIds(this.id)) {
                if (!canRead(userId)) {
                    try {
                        cancel(userId);
                    } catch (CannotCancelCommunityException e) {
                        // N肦Ȃ
                        logger.error(e.getMessage(), e);
                        throw new IllegalStateException(e);
                    }
                }
            }
        }
    }

    @Override
    public void delete() throws CannotDeleteCommunityException {
        if (!isDeletable()) {
            String message = "The community '" + this + "' cannot delete.";
            logger.warn(message);
            throw new CannotDeleteCommunityException(message);
        }
        communityDao.delete(id);
    }

    @Override
    public boolean isMyCommunity() {
        return myCommunity;
    }

    @Override
    public void retrieveMyCommunity(User user) {
        if (user == null) {
            myCommunity = false;
        } else {
            List<Integer> users = communityDao.findCommunityUserIds(this.id);
            this.myCommunity = users.contains(user.getId());
        }
    }

    @Override
    public void postAccess() throws AccessDeniedException {
        postAccessControl.access(this);
    }

    @Override
    public void postAccess(int userId) throws AccessDeniedException {
        postAccessControl.access(this, userId);
    }

    @Override
    public void postAccess(User user) throws AccessDeniedException {
        if (user == null) {
            postAccess();
        } else {
            postAccess(user.getId());
        }
    }

    @Override
    public boolean canPost() {
        try {
            postAccess();
            return true;
        } catch (AccessDeniedException e) {
            return false;
        }
    }

    @Override
    public boolean canPost(int userId) {
        try {
            postAccess(userId);
            return true;
        } catch (AccessDeniedException e) {
            return false;
        }
    }

    @Override
    public boolean canPost(User user) {
        if (user == null) {
            return canPost();
        } else {
            return canPost(user.getId());
        }
    }

    @Override
    public void entry(int userId) throws CannotEntryCommunityException {
        if (communityDao.findMyCommunies(userId).contains(this)) {
            String message = "The user " + userId
                + " has already been a member of this community '" + this.code
                + "'.";
            logger.warn(message);
            throw new CannotEntryCommunityException(message);
        }
        communityDao.registerMyCommunity(userId, this.id);
    }

    @Override
    public void entry(User user) throws CannotEntryCommunityException {
        if (user == null) {
            String message
                = "The anonymous user cannot entry into the community.";
            logger.warn(message);
            throw new CannotEntryCommunityException(message);
        }
        entry(user.getId());
    }

    @Override
    public void cancel(int userId) throws CannotCancelCommunityException {
        if (getOwnerId().intValue() == userId) {
            String message = "The user " + userId
                + " is the owner of this community '" + this.code + "'.";
            logger.warn(message);
            throw new CannotCancelCommunityException(message);
        }
        if (!communityDao.findCommunityUserIds(this.id).contains(userId)) {
            String message = "The user " + userId
                + " is not member of this community '" + this.code + "'";
            logger.warn(message);
            throw new CannotCancelCommunityException(message);
        }
        communityDao.deleteMyCommunity(userId, this.id);
    }

    @Override
    public void cancel(User user) throws CannotCancelCommunityException {
        if (user == null) {
            String message = "The anonymous user have no community.";
            logger.warn(message);
            throw new CannotCancelCommunityException(message);
        }
        cancel(user.getId());
    }

    @Override
    public Date getLastUpdateDate() {
        if (this.lastUpdateDate == null) {
            return null;
        } else {
            return (Date) this.lastUpdateDate.clone();
        }
    }

    @Override
    public void setLastUpdateDate(Date lastUpdateDate) {
        if (lastUpdateDate == null) {
            this.lastUpdateDate = null;
        } else {
            this.lastUpdateDate = (Date) lastUpdateDate.clone();
        }
    }

    @Override
    public final <C extends Community> C changeTo(Class<C> clazz)
        throws CannotChangeCommunityException {
        if (!canChangeTo(clazz)) {
            String message = "Cannot change this community '" + this.code
                + "' to clazz '" + clazz.getClass().getName() + "'.";
            logger.warn(message);
            throw new CannotChangeCommunityException(message);
        }
        return doChangeTo(clazz);
    }

    /**
     * ۂ̃R~jeBύXWbN܂B
     *
     * @param <C> R~jeB
     * @param clazz ύX̃R~jeBNX
     * @return ύX̃R~jeB
     */
    protected abstract <C extends Community> C doChangeTo(Class<C> clazz);

    /**
     * Xbh擾܂Bw肵Xbh̃R~jeBɑ݂Ȃꍇ <code>null</code> 
     * Ԃ܂B
     *
     * @param threadId Xbh ID
     * @return Xbh
     */
    public Thread findThread(int threadId) {
        return communityDao.findThread(this.id, threadId);
    }

    @Override
    public Integer getId() {
        return id;
    }

    @Override
    public void setId(Integer id) {
        this.id = id;
    }

    @Override
    public String getCode() {
        return code;
    }

    @Override
    public void setCode(String code) {
        this.code = code;
    }

    @Override
    public Message getOverview() {
        return overview;
    }

    @Override
    public void setOverview(Message overview) {
        this.overview = overview;
    }

    @Override
    public AccessControl getPostAccessControl() {
        return postAccessControl;
    }

    @Override
    public void setPostAccessControl(AccessControl postAccessControl) {
        this.postAccessControl = postAccessControl;
    }

    protected CommunityDao getCommunityDao() {
        return communityDao;
    }

    public void setCommunityDao(CommunityDao communityDao) {
        this.communityDao = communityDao;
    }

    protected DomainFactory getDomainFactory() {
        return domainFactory;
    }

    public void setDomainFactory(DomainFactory domainFactory) {
        this.domainFactory = domainFactory;
    }
}
