/*
 * @file  l7vsadm_main.c
 * @brief the main module of l7vsadm 
 * @brief it interrupts command line option 
 * @brief and communucate with l7vsd 
 *
 * L7VSADM: Virtual Server Administration Program for L7vsd
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <sys/un.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/file.h>
#include <errno.h>

#include "l7vs.h"
#include "l7vs_config.h"
#include "l7vs_replication.h"
#include "l7vs_module.h"
#include "l7vs_service.h"
#include "l7vs_dest.h"
#include "l7vs_conn.h"

#include "l7vsadm.h"

//! Prototype of functions.
static int parse_command(int argc, char *argv[]);

static int parse_option_list(int argc, char *argv[]);
static int parse_option_vs(int argc, char *argv[]);
static int parse_option_rs(int argc, char *argv[]);
static int parse_option_replication(int argc, char *argv[]);
static int parse_option_log(int argc, char *argv[]);
static int parse_option_snmp(int argc, char *argv[]);
static int parse_option_parameter(int argc, char *argv[]);
static int parse_option_help(int argc, char *argv[]);

static int parse_endpoint(struct sockaddr_in *addr, const char *endpoint_str, int allow_inaddr_any);
static int parse_digit(const char *digit_str);
static long long parse_qos(const char *qos_str);
static int parse_option_protomod(int argc, char *argv[], int m_idx_begin, int m_idx_end);

static int operate_list(void);
static int operate_add_vs(void);
static int operate_edit_vs(void);
static int operate_del_vs(void);
static int operate_flush_vs(void);
static int operate_add_rs(void);
static int operate_edit_rs(void);
static int operate_del_rs(void);
static int operate_replication(void);
static int operate_log(void);
static int operate_snmp(void);
static int operate_parameter(void);
static int operate_help(void);

static int list_replication_info(void);
static int list_log_info(void);
static int list_snmp_info(void);

static int get_vs(void *vs_buffer, int vs_bufsize);
static int get_rs(struct l7vs_dest_arg *darg_buffer, int rs_num, struct l7vs_service_arg *sarg);
static int list_vs(struct l7vs_service_arg *sarg);
static int list_rs(struct l7vs_dest_arg *darg);

static char *get_endpoint_name(struct sockaddr_storage *addr);

static int create_socket(void);
static void destroy_socket(void);
static int communicate_to_l7vsd(struct l7vsadm_iov_data *iov_data);
static int send_request(struct iovec *req_iov, int req_cnt);
static int recv_response(struct iovec *rsp_iov, int rsp_cnt);

static void usage(FILE *fp);

static void set_parameter(void);
static char *config_strerror(int rsp_code);
static void sig_exit_handler(int sig);
static int set_signal(void);

//! L7vsadm option data that has been parsed.
static struct l7vsadm_option_data l7vsadm_option_data;
static char tmp_endpoint_name[NI_MAXHOST + NI_MAXSERV + 2];

//! Table for command argument check.
static struct l7vsadm_command_list l7vsadm_command_table[] = {
	{CMD_LIST,         "-l", "--list",           parse_option_list,        operate_list},
	{CMD_LIST_VERBOSE, "-V", "--verbose",        parse_option_list,        operate_list},
	{CMD_LIST_KEY,     "-K", "--key",            parse_option_list,        operate_list},
	{CMD_ADD_VS,       "-A", "--add-service",    parse_option_vs,          operate_add_vs},
	{CMD_DEL_VS,       "-D", "--delete-service", parse_option_vs,          operate_del_vs},
	{CMD_EDIT_VS,      "-E", "--edit-service",   parse_option_vs,          operate_edit_vs},
	{CMD_FLUSH_VS,     "-C", "--flush",          parse_option_vs,          operate_flush_vs},
	{CMD_ADD_RS,       "-a", "--add-server",     parse_option_rs,          operate_add_rs},
	{CMD_DEL_RS,       "-d", "--delete-server",  parse_option_rs,          operate_del_rs},
	{CMD_EDIT_RS,      "-e", "--edit-server",    parse_option_rs,          operate_edit_rs},
	{CMD_REPLICATION,  "-R", "--replication",    parse_option_replication, operate_replication},
	{CMD_LOG,          "-L", "--log",            parse_option_log,         operate_log},
	{CMD_SNMP,         "-S", "--snmp",           parse_option_snmp,        operate_snmp},
	{CMD_PARAMETER,    "-P", "--parameter",      parse_option_parameter,   operate_parameter},
	{CMD_HELP,         "-h", "--help",           parse_option_help,        operate_help},
	{CMD_NONE,         NULL, NULL,               NULL,                     NULL}
};

//! Table for List command argument check.(optional argument)
static struct l7vsadm_option_list l7vsadm_option_table_list[] = {
	{OPT_NUMERIC,		"-n",	"--numeric",		0},
	{OPT_NONE,		NULL,	NULL,			0}
};

//! Table for VirtualService command argument check.(optional argument)
static struct l7vsadm_option_list l7vsadm_option_table_vs[] = {
	{OPT_SCHEDULER,		"-s",	"--scheduler",		0},
	{OPT_UPPER,		"-u",	"--upper",		0},
	{OPT_BYPASS,		"-b",	"--bypass",		0},
	{OPT_FLAG,		"-f",	"--flag",		0},
	{OPT_QOS_UP,	"-Q",	"--qos-up",	0},
	{OPT_QOS_DOWN,	"-q",	"--qos-down",	0},
	{OPT_NONE,		NULL,	NULL,			0}
};

//! Table for RealServer command argument check.(optional argument)
static struct l7vsadm_option_list l7vsadm_option_table_rs[] = {
	{OPT_WEIGHT,		"-w",	"--weight",		0},
	{OPT_NONE,		NULL,	NULL,			0}
};

//! Table for Replication command argument check.
static struct l7vsadm_option_list l7vsadm_option_table_replication[] = {
	{OPT_SWITCH,		"-s",	"--switch",		0},
	{OPT_FORCE,		"-f",	"--force",		0},
	{OPT_DUMP,		"-d",	"--dump",		0},
	{OPT_NONE,		NULL,	NULL,			0}
};

//! Table for Logger command argument check.
static struct l7vsadm_option_list l7vsadm_option_table_log[] = {
	{OPT_CATEGORY,		"-c",	"--category",		0},
	{OPT_LEVEL,		"-l",	"--level",		0},
	{OPT_NONE,		NULL,	NULL,			0}
};

//! Table for SNMPAgent command argument check.
static struct l7vsadm_option_list l7vsadm_option_table_snmp[] = {
	{OPT_CATEGORY,		"-c",	"--category",		0},
	{OPT_LEVEL,		"-l",	"--level",		0},
	{OPT_NONE,		NULL,	NULL,			0}
};

//! Table for Parameter command argument check.
static struct l7vsadm_option_list l7vsadm_option_table_parameter[] = {
	{OPT_RELOAD,		"-r",	"--reload",		0},
	{OPT_NONE,		NULL,	NULL,			0}
};

//! Table for L7vsd Log-Category string check.
static struct l7vsadm_category_name_list l7vsd_category_name_table[] = {
	{LOG_CAT_L7VSD_NETWORK,			"l7vsd_network",		"nw"},
	{LOG_CAT_L7VSD_NETWORK_BANDWIDTH,	"l7vsd_network.bandwidth",	"nw.bw"},
	{LOG_CAT_L7VSD_NETWORK_NUM_CONNECTION,	"l7vsd_network.num_connection",	"nw.conn"},
	{LOG_CAT_L7VSD_NETWORK_QOS,		"l7vsd_network.qos",		"nw.qos"},
	{LOG_CAT_L7VSD_VIRTUAL_SERVICE,		"l7vsd_virtual_service",	"vs"},
	{LOG_CAT_L7VSD_REAL_SERVER,		"l7vsd_real_server",		"rs"},
	{LOG_CAT_L7VSD_SORRY_SERVER,		"l7vsd_sorry_server",		"sorry"},
	{LOG_CAT_L7VSD_REAL_SERVER_BALANCING,	"l7vsd_real_server.balancing",	"rs.bal"},
	{LOG_CAT_L7VSD_REPLICATION,		"l7vsd_replication",		"rep"},
	{LOG_CAT_L7VSD_START_STOP,		"l7vsd_start_stop",		"stastp"},
	{LOG_CAT_L7VSD_SYSTEM,			"l7vsd_system",			"sys"},
	{LOG_CAT_L7VSD_SYSTEM_MEMORY,		"l7vsd_system.memory",		"sys.mem"},
	{LOG_CAT_L7VSD_SYSTEM_SOCKET,		"l7vsd_system.socket",		"sys.sock"},
	{LOG_CAT_L7VSD_SYSTEM_SIGNAL,		"l7vsd_system.signal",		"sys.sig"},
	{LOG_CAT_L7VSD_ENVIRONMENT,		"l7vsd_environment",		"env"},
	{LOG_CAT_L7VSD_ENVIRONMENT_PARAMETER,	"l7vsd_environment.parameter",	"env.para"},
	{LOG_CAT_L7VSD_LOGGER,			"l7vsd_logger",			"logger"},
	{LOG_CAT_L7VSD_PARAMETER,		"l7vsd_parameter",		"para"},
	{LOG_CAT_L7VSD_EVENT,			"l7vsd_event",			"event"},
	{LOG_CAT_L7VSD_SCHEDULE,		"l7vsd_schedule",		"sched"},
	{LOG_CAT_L7VSD_PROGRAM,			"l7vsd_program",		"prog"},
	{LOG_CAT_L7VSD_PROTOCOL,		"l7vsd_protocol",		"prot"},
	{LOG_CAT_L7VSD_MODULE,			"l7vsd_module",			"mod"},
	{(enum LOG_CATEGORY_TAG)0,		NULL,				NULL}
};

//! Table for SNMPAgent Log-Category string check.
static struct l7vsadm_category_name_list snmp_category_name_table[] = {
	{LOG_CAT_SNMPAGENT_START_STOP,		"snmpagent_start_stop",		"snmp_stastp"},
	{LOG_CAT_SNMPAGENT_MANAGER_RECEIVE,	"snmpagent_manager_receive",	"snmp_mngrcv"},
	{LOG_CAT_SNMPAGENT_MANAGER_SEND,	"snmpagent_manager_send",	"snmp_mngsnd"},
	{LOG_CAT_SNMPAGENT_L7VSD_RECEIVE,	"snmpagent_l7vsd_receive",	"snmp_vsdrcv"},
	{LOG_CAT_SNMPAGENT_L7VSD_SEND,		"snmpagent_l7vsd_send",		"snmp_vsdsnd"},
	{LOG_CAT_SNMPAGENT_LOGGER,		"snmpagent_logger",		"snmp_logger"},
	{LOG_CAT_SNMPAGENT_PARAMETER,		"snmpagent_parameter",		"snmp_para"},
	{(enum LOG_CATEGORY_TAG)0,		NULL,				NULL}
};

//! Table for Log-Level string check.
static struct l7vsadm_level_name_list level_name_table[] = {
	{LOG_LV_DEBUG,	"debug"},
	{LOG_LV_INFO,	"info"},
	{LOG_LV_WARN,	"warn"},
	{LOG_LV_ERROR,	"error"},
	{LOG_LV_FATAL,	"fatal"},
	{LOG_LV_NONE,	NULL}
};

//! Interval of l7vsadm command conflict check.
static int command_wait_interval = L7VSADM_DEFAULT_WAIT_INTERVAL;
//! Number of times of l7vsadm command conflict check.
static int command_wait_count = L7VSADM_DEFAULT_WAIT_COUNT;

//! Interval of connected check to l7vsd.
static int connect_wait_interval = L7VSADM_DEFAULT_WAIT_INTERVAL;
//! Number of times of connected check to l7vsd.
static int connect_wait_count = L7VSADM_DEFAULT_WAIT_COUNT;

//! Log-Level of Parse-Category.
static int log_level_parse;
//! Log-Level of Operate-Category.
static int log_level_operate;
//! Log-Level of Communicate-Category.
static int log_level_communicate;
//! Log-Level of Config_Result-Category.
static int log_level_config_result;
//! Log-Level of Common-Category.
static int log_level_common;

//! Received signal count.
static int signal_flg = 0;
//! Received signal number.
static int received_sig = 0;

//! Local socket name.
static char local_sockname[sizeof(L7VS_CONFIG_SOCK_PATH) + sizeof(L7VSADM_NAME) + 10];
//! Local socket file discripter.
static int local_sock_fd = -1;

/*!
 * Parse command option.
 * Check 1st argment
 * @param[in]	argc	number of argument
 * @param[in]	*argv[]	array of argument string
 * @retval	0	parse result OK
 * @retval	-1	parse result NG
 */
static int
parse_command(int argc, char *argv[])
{
	struct l7vsadm_command_list *list;

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		char arg_str[DEBUG_STR_LEN] = {0};
		argc_argv_c_str(arg_str, argc, argv);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,1,
			"in_function : static int parse_command(int argc, char *argv[]) : "
			"argc = %d, %s",
			argc,
			arg_str);
	}
	/*------ DEBUG LOG END ------*/

	// Argument check.
	if (argc < 2) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,1, "Argument argc is illegal.");
		fprintf(stderr, "PARSE ERROR : Argument argc is illegal.\n");

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,2, 
			"out_function : static int parse_command(int argc, char *argv[]) : "
			"return = -1");
	}
	/*------ DEBUG LOG END ------*/

		return -1;
	}

	// Search command option on l7vsadm_command_table.
	l7vsadm_option_data.command_code = CMD_NONE;
	for (list = l7vsadm_command_table; list->command_name != NULL; list++) {
		if (strcmp(argv[1], list->command_name) == 0 || 
		    strcmp(argv[1], list->command_name_long) == 0) {
			l7vsadm_option_data.command_code = list->command_code;
			l7vsadm_option_data.parse_func = list->parse_func;
			l7vsadm_option_data.operate_func = list->operate_func;
			break;
		}
	}
	if (l7vsadm_option_data.command_code == CMD_NONE) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,2,
			"No such command option %s.", argv[1]);
		fprintf(stderr, "PARSE ERROR : "
			"No such command option %s.\n", argv[1]);

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,3, 
			"out_function : static int parse_command(int argc, char *argv[]) : "
			"return = -1");
	}
	/*------ DEBUG LOG END ------*/

		return -1;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,4, 
			"out_function : static int parse_command(int argc, char *argv[]) : "
			"return = 0 : result l7vsadm_option_data.command_code = %d", 
			l7vsadm_option_data.command_code);
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * Parse list command option.
 * Check 2nd argument and later of list command
 * @param[in]	argc	number of argument
 * @param[in]	*argv[]	array of argument string
 * @return	int	parse result OK=0, NG=-1
 */
static int
parse_option_list(int argc, char *argv[])
{
	struct l7vsadm_option_list *list;
	int idx;
	int match_option_f;

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		char arg_str[DEBUG_STR_LEN] = {0};
		argc_argv_c_str(arg_str, argc, argv);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,5,
			"in_function : static int parse_option_list(int argc, char *argv[]) : "
			"argc = %d, %s",
			argc,
			arg_str);
	}
	/*------ DEBUG LOG END ------*/

	// Argument check.
	if (argc > 3) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,3,
			"Argument argc is illegal for %s command.", argv[1]);
		fprintf(stderr, "PARSE ERROR : "
			"Argument argc is illegal for %s command.\n", argv[1]);

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,6, 
			"out_function : static int "
			"parse_option_list(int argc, char *argv[]) : "
			"return = -1");
	}
	/*------ DEBUG LOG END ------*/

		return -1;
	}
	idx = 2;
	while (idx < argc) {
		match_option_f = 0;
		// Search optional option on l7vsadm_option_table_list.
		for (list = l7vsadm_option_table_list; list->option_name != NULL; list++) {
			if (strcmp(argv[idx], list->option_name) == 0 || 
			    strcmp(argv[idx], list->option_name_long) == 0) {
				match_option_f = 1;
				// Set each option value.
				switch (list->option_code) {
				case OPT_NUMERIC:
					l7vsadm_option_data.isnumeric = 1;
					break;
				default:
					break;
				}
			}
		}
		if (!match_option_f) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,4,
				"Invalid option for %s command.", argv[1]);
			fprintf(stderr, "PARSE ERROR : "
				"Invalid option for %s command.\n", argv[1]);

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,7, 
				"out_function : static int "
				"parse_option_list(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

			return -1;
		}
		idx++;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,8, 
			"out_function : static int parse_option_list(int argc, char *argv[]) : "
			"return = 0 : result l7vsadm_option_data.isnumeric = %d", 
			l7vsadm_option_data.isnumeric);
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * Parse virtual service command option.
 * Check 2nd argument and later of vs command
 * @param[in]	argc	number of argument
 * @param[in]	*argv[]	array of argument string
 * @return	int	parse result OK=0, NG=-1
 */
static int
parse_option_vs(int argc, char *argv[])
{
	struct l7vsadm_option_list *list;
	int idx;
	int ret;
	long long ret_ll;
	int m_idx_begin,m_idx_end;
	int break_f;
	int match_option_f;

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		char arg_str[DEBUG_STR_LEN] = {0};
		argc_argv_c_str(arg_str, argc, argv);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,9,
			"in_function : static int parse_option_vs(int argc, char *argv[]) : "
			"argc = %d, %s",
			argc,
			arg_str);
	}
	/*------ DEBUG LOG END ------*/

	// Argument check for flush_vs command.
	if (l7vsadm_option_data.command_code == CMD_FLUSH_VS) {
		if (argc != 2) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,5,
				"Argument argc is illegal for -C command.");
			fprintf(stderr, "PARSE ERROR : "
				"Argument argc is illegal for -C command.\n");

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,10, 
					"out_function : static int "
					"parse_option_vs(int argc, char *argv[]) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,11, 
				"out_function : static int "
				"parse_option_vs(int argc, char *argv[]) : "
				"return = 0 : result -C");
		}
		/*------ DEBUG LOG END ------*/

		return 0;
	}

	// Argument check for add_vs/edit_vs/delete_vs command.
	if (argc < 6) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,6,
			"Argument argc is illegal for (%s) command.", argv[1]);
		fprintf(stderr, "PARSE ERROR : "
			"Argument argc is illegal for (%s) command.\n", argv[1]);

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,12, 
				"out_function : static int "
				"parse_option_vs(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	idx = 2;
	if (strcmp(argv[idx], "-t") == 0 || strcmp(argv[idx], "--tcp-service") == 0) {
		idx++;
		ret = parse_endpoint(&l7vsadm_option_data.vs_addr,
				     argv[idx], TRUE);
		if (ret < 0) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,7,
				"Parse VirtualService endpoint error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,13, 
				"out_function : static int "
				"parse_option_vs(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

			return -1;
		}
		idx++;
	} else {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,8,
			"Invalid option %s. Need -t option for %s command.",
			argv[idx],
			argv[1]);
		fprintf(stderr, "PARSE ERROR : "
			"Invalid option %s. Need -t option for %s command.\n",
			argv[idx],
			argv[1]);

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,14, 
				"out_function : static int "
				"parse_option_vs(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	if (strcmp(argv[idx], "-m") == 0 || strcmp(argv[idx], "--proto-module") == 0) {
		idx++;
		if (strlen(argv[idx]) >= L7VS_MODNAME_LEN - 1) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,9,
				"ProtocolModule name too long. (max %d)", L7VS_MODNAME_LEN);
			fprintf(stderr, "PARSE ERROR : "
				"ProtocolModule name too long. (max %d)\n", L7VS_MODNAME_LEN);

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,15, 
					"out_function : static int "
					"parse_option_vs(int argc, char *argv[]) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}
		strncpy(l7vsadm_option_data.protomod_name, argv[idx], strlen(argv[idx]));
		idx++;
	} else {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,10,
			"Invalid option %s. Need -m option for %s command.",
			argv[idx],
			argv[1]);
		fprintf(stderr, "PARSE ERROR : "
			"Invalid option %s. Need -m option for %s command.\n",
			argv[idx],
			argv[1]);

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,16, 
				"out_function : static int "
				"parse_option_vs(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	// Search begin and end of ProtocolModule option.
	m_idx_begin = idx;
	break_f = 0;
	while (idx < argc) {
		// End of ProtocolModule option is next valid option or end of argument.
		for (list = l7vsadm_option_table_vs; list->option_name != NULL; list++) {
			if (strcmp(argv[idx], list->option_name) == 0 || 
			    strcmp(argv[idx], list->option_name_long) == 0) {
				break_f = 1;
				break;
			}
		}
		if (break_f) {
			break;
		}
		idx++;
	}
	m_idx_end = idx;
	ret = parse_option_protomod(argc, argv, m_idx_begin, m_idx_end);
	if (ret < 0) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,11,
			"Parse ProtocolModule option error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,17, 
				"out_function : static int "
				"parse_option_vs(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	// Argument check for delete_vs command.
	if (l7vsadm_option_data.command_code == CMD_DEL_VS) {
		if (idx < argc) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,12,
				"Argument argc is illegal for -D command.");
			fprintf(stderr, "PARSE ERROR : "
				"Argument argc is illegal for -D command.\n");

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,18, 
					"out_function : static int "
					"parse_option_vs(int argc, char *argv[]) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,19, 
				"out_function : static int "
				"parse_option_vs(int argc, char *argv[]) : "
				"return = 0 : result -D");
		}
		/*------ DEBUG LOG END ------*/

		return 0;
	}

	// Set default ScheduleModule "rr".
	strncpy(l7vsadm_option_data.schedmod_name,
		L7VSADM_DEFAULT_SCHEDULER,
		strlen(L7VSADM_DEFAULT_SCHEDULER));

	while (idx < argc) {
		match_option_f = 0;
		// Search optional option on l7vsadm_option_table_vs.
		for (list = l7vsadm_option_table_vs; list->option_name != NULL; list++) {
			if (strcmp(argv[idx], list->option_name) == 0 || 
			    strcmp(argv[idx], list->option_name_long) == 0) {
				match_option_f = 1;
				if (list->check_flag) {
					LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,13,
						"Option %s is conflict.",
						argv[idx]);
					fprintf(stderr, "PARSE ERROR : "
						"Option %s is conflict.\n",
						argv[idx]);

					/*-------- DEBUG LOG --------*/
					if (log_level_parse == LOG_LV_DEBUG) {
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,20, 
							"out_function : static int "
							"parse_option_vs(int argc, char *argv[]) : "
							"return = -1");
					}
					/*------ DEBUG LOG END ------*/

					return -1;
				}
				list->check_flag = 1;
				// Check and Set each option value.
				switch (list->option_code) {
				case OPT_SCHEDULER:
					idx++;
					if (idx >= argc) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,14,
							"Need option value for -s option.");
						fprintf(stderr, "PARSE ERROR : "
							"Need option value for -s option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,21, 
								"out_function : static int "
								"parse_option_vs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					if (strlen(argv[idx]) >= L7VS_MODNAME_LEN - 1) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,15,
							"ScheduleModule name too long. (max %d)",
							L7VS_MODNAME_LEN);
						fprintf(stderr, "PARSE ERROR : "
							"ScheduleModule name too long. (max %d)\n",
							L7VS_MODNAME_LEN);

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,22, 
								"out_function : static int "
								"parse_option_vs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					strncpy(l7vsadm_option_data.schedmod_name,
						argv[idx],
						strlen(argv[idx]));
					break;
				case OPT_UPPER:
					idx++;
					if (idx >= argc) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,16,
							"Need option value for -u option.");
						fprintf(stderr, "PARSE ERROR : "
							"Need option value for -u option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,23, 
								"out_function : static int "
								"parse_option_vs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					ret = parse_digit(argv[idx]);
					if (ret < 0) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,17,
							"-u option value is only digit.");
						return -1;
					}
					sscanf(argv[idx], "%d", &l7vsadm_option_data.max_connection);
					if (l7vsadm_option_data.max_connection < 0 ||
					    l7vsadm_option_data.max_connection > 100000) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,18,
							"Invalid option value for -u option.");
						fprintf(stderr, "PARSE ERROR : "
							"Invalid option value for -u option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,24, 
								"out_function : static int "
								"parse_option_vs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					break;
				case OPT_BYPASS:
					idx++;
					if (idx >= argc) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,19,
							"Need option value for -b option.");
						fprintf(stderr, "PARSE ERROR : "
							"Need option value for -b option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,25, 
								"out_function : static int "
								"parse_option_vs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					// When specify the initialization of the sorry endpoint.
					if (strncmp(L7VSADM_INIT_ENDPOINT_STR, argv[idx],
					    strlen(L7VSADM_INIT_ENDPOINT_STR)) == 0) {
						// Sorry endpoint is 0 clear data.
						break;
					}
					ret = parse_endpoint(
						(struct sockaddr_in *)(&l7vsadm_option_data.sorry_addr),
						argv[idx], FALSE);
					if (ret < 0) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,20,
							"Parse SorryServer endpoint error.");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,26, 
								"out_function : static int "
								"parse_option_vs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					break;
				case OPT_FLAG:
					idx++;
					if (idx >= argc) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,21,
							"Need option value for -f option.");
						fprintf(stderr, "PARSE ERROR : "
							"Need option value for -f option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,27, 
								"out_function : static int "
								"parse_option_vs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					ret = parse_digit(argv[idx]);
					if (ret < 0) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,22,
							"-f option value is only digit.");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,28, 
								"out_function : static int "
								"parse_option_vs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					sscanf(argv[idx], "%d", &l7vsadm_option_data.sorry_flag);
					if (l7vsadm_option_data.sorry_flag < 0 ||
					    l7vsadm_option_data.sorry_flag > 1) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,23,
							"Invalid option value for -f option.");
						fprintf(stderr, "PARSE ERROR : "
							"Invalid option value for -f option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,29, 
								"out_function : static int "
								"parse_option_vs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					break;
				case OPT_QOS_UP:
					idx++;
					if (idx >= argc) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,24,
							"Need option value for -Q option.");
						fprintf(stderr, "PARSE ERROR : "
							"Need option value for -Q option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,30, 
								"out_function : static int "
								"parse_option_vs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					ret_ll = parse_qos(argv[idx]);
					if (ret_ll < 0) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,25,
							"Parse QoS-up value error.");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,31, 
								"out_function : static int "
								"parse_option_vs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					l7vsadm_option_data.qos_threshold_up = (unsigned long long)ret_ll;
					break;
				case OPT_QOS_DOWN:
					idx++;
					if (idx >= argc) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,26,
							"Need option value for -q option.");
						fprintf(stderr, "PARSE ERROR : "
							"Need option value for -q option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,32, 
								"out_function : static int "
								"parse_option_vs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					ret_ll = parse_qos(argv[idx]);
					if (ret_ll < 0) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,27,
							"Parse QoS-down value error.");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,33, 
								"out_function : static int "
								"parse_option_vs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					l7vsadm_option_data.qos_threshold_down = (unsigned long long)ret_ll;
					break;
				default:
					break;
				}
				break;
			}
		}
		if (!match_option_f) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,28,
				"Invalid option for %s command.", argv[1]);
			fprintf(stderr, "PARSE ERROR : "
				"Invalid option for %s command.\n", argv[1]);

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,34, 
					"out_function : static int "
					"parse_option_vs(int argc, char *argv[]) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}
		idx++;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,35, 
			"out_function : static int "
			"parse_option_vs(int argc, char *argv[]) : "
			"return = 0 : result -A or -E");
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * Parse real server command option.
 * Check 2nd argument and later of rs command
 * @param[in]	argc	number of argument
 * @param[in]	*argv[]	array of argument string
 * @return	int	parse result OK=0, NG=-1
 */
static int
parse_option_rs(int argc, char *argv[])
{
	struct l7vsadm_option_list *list;
	int idx;
	int ret;
	int m_idx_begin,m_idx_end;
	int break_f;
	int match_option_f;

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		char arg_str[DEBUG_STR_LEN] = {0};
		argc_argv_c_str(arg_str, argc, argv);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,36,
			"in_function : static int parse_option_rs(int argc, char *argv[]) : "
			"argc = %d, %s",
			argc,
			arg_str);
	}
	/*------ DEBUG LOG END ------*/

	// Argument check for add_rs/edit_rs/delete_rs command.
	if (argc < 8) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,29,
			"Argument argc is illegal for %s command.", argv[1]);
		fprintf(stderr, "PARSE ERROR : "
			"Argument argc is illegal for %s command.\n", argv[1]);

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,37, 
				"out_function : static int "
				"parse_option_rs(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	idx = 2;
	if (strcmp(argv[idx], "-t") == 0 || strcmp(argv[idx], "--tcp-service") == 0) {
		idx++;
		ret = parse_endpoint(&l7vsadm_option_data.vs_addr,
				     argv[idx], TRUE);
		if (ret < 0) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,30,
				"Parse VirtualService endpoint error.");

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,38, 
					"out_function : static int "
					"parse_option_rs(int argc, char *argv[]) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}
		idx++;
	} else {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,31,
			"Invalid option %s. Need -t option for %s command.",
			argv[idx],
			argv[1]);
		fprintf(stderr, "PARSE ERROR : "
			"Invalid option %s. Need -t option for %s command.\n",
			argv[idx],
			argv[1]);

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,39, 
				"out_function : static int "
				"parse_option_rs(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	if (strcmp(argv[idx], "-m") == 0 || strcmp(argv[idx], "--proto-module") == 0) {
		idx++;
		if (strlen(argv[idx]) >= L7VS_MODNAME_LEN - 1) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,32,
				"ProtocolModule name too long. (max %d)", L7VS_MODNAME_LEN);
			fprintf(stderr, "PARSE ERROR : "
				"ProtocolModule name too long. (max %d)\n", L7VS_MODNAME_LEN);

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,40, 
					"out_function : static int "
					"parse_option_rs(int argc, char *argv[]) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}
		strncpy(l7vsadm_option_data.protomod_name, argv[idx], strlen(argv[idx]));
		idx++;
	} else {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,33,
			"Invalid option %s. Need -m option for %s command.",
			argv[idx],
			argv[1]);
		fprintf(stderr, "PARSE ERROR : "
			"Invalid option %s. Need -m option for %s command.\n",
			argv[idx],
			argv[1]);

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,41, 
				"out_function : static int "
				"parse_option_rs(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	// Search begin and end of ProtocolModule option.
	m_idx_begin = idx;
	break_f = 0;
	while (idx < argc) {
		// End of ProtocolModule option is -r option.
		if (strcmp(argv[idx], "-r") == 0 || strcmp(argv[idx], "--real-server") == 0) {
			break_f = 1;
			break;
		}
		idx++;
	}
	if (!break_f) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,34,
			"Invalid option %s. Need -r option for %s command.",
			argv[idx],
			argv[1]);
		fprintf(stderr, "PARSE ERROR : "
			"Invalid option %s. Need -r option for %s command.\n",
			argv[idx],
			argv[1]);

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,42, 
				"out_function : static int "
				"parse_option_rs(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	m_idx_end = idx;
	ret = parse_option_protomod(argc, argv, m_idx_begin, m_idx_end);
	if (ret < 0) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,35,
			"Parse ProtocolModule option error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,43, 
				"out_function : static int "
				"parse_option_rs(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	idx++;
	ret = parse_endpoint(&l7vsadm_option_data.rs_addr, argv[idx], FALSE);
	if (ret < 0) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,36,
			"Parse RealServer endpoint error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,44, 
				"out_function : static int "
				"parse_option_rs(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	idx++;

	// Argument check for delete_rs command.
	if (l7vsadm_option_data.command_code == CMD_DEL_RS) {
		if (idx < argc) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,37,
				"Argument argc is illegal for -d command.");
			fprintf(stderr, "PARSE ERROR : "
				"Argument argc is illegal for -d command.\n");

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,45, 
					"out_function : static int "
					"parse_option_rs(int argc, char *argv[]) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,46, 
				"out_function : static int "
				"parse_option_rs(int argc, char *argv[]) : "
				"return = 0 : result -d");
		}
		/*------ DEBUG LOG END ------*/

		return 0;
	}

	// Set default RealServer weight = 1.
	l7vsadm_option_data.weight = 1;

	while (idx < argc) {
		match_option_f = 0;
		// Search optional option on l7vsadm_option_table_rs.
		for (list = l7vsadm_option_table_rs; list->option_name != NULL; list++) {
			if (strcmp(argv[idx], list->option_name) == 0 || 
			    strcmp(argv[idx], list->option_name_long) == 0) {
				match_option_f = 1;
				if (list->check_flag) {
					LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,38,
						"Option %s is conflict.",
						argv[idx]);
					fprintf(stderr, "PARSE ERROR : "
						"Option %s is conflict.\n",
						argv[idx]);

					/*-------- DEBUG LOG --------*/
					if (log_level_parse == LOG_LV_DEBUG) {
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,47, 
							"out_function : static int "
							"parse_option_rs(int argc, char *argv[]) : "
							"return = -1");
					}
					/*------ DEBUG LOG END ------*/

					return -1;
				}
				list->check_flag = 1;
				// Check and Set each option value.
				switch (list->option_code) {
				case OPT_WEIGHT:
					idx++;
					if (idx >= argc) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,39,
							"Need option value for -w option.");
						fprintf(stderr, "PARSE ERROR : "
							"Need option value for -w option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,48, 
								"out_function : static int "
								"parse_option_rs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					ret = parse_digit(argv[idx]);
					if (ret < 0) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,40,
							"-w option value is only digit.");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,49, 
								"out_function : static int "
								"parse_option_rs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					sscanf(argv[idx], "%d", &l7vsadm_option_data.weight);
					if (l7vsadm_option_data.weight < 0 ||
					    l7vsadm_option_data.weight > 100) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,41,
							"Invalid option value for -w option.");
						fprintf(stderr, "PARSE ERROR : "
							"Invalid option value for -w option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,50, 
								"out_function : static int "
								"parse_option_rs(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					break;
				default:
					break;
				}
				break;
			}
		}
		if (!match_option_f) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,42,
				"Invalid option for %s command.", argv[1]);
			fprintf(stderr, "PARSE ERROR : "
				"Invalid option for %s command.\n", argv[1]);

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,51, 
					"out_function : static int "
					"parse_option_rs(int argc, char *argv[]) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}
		idx++;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,52, 
			"out_function : static int "
			"parse_option_rs(int argc, char *argv[]) : "
			"return = 0 : result -a or -e");
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * Parse replication command option.
 * Check 2nd argument and later of replication command
 * @param[in]	argc	number of argument
 * @param[in]	*argv[]	array of argument string
 * @return	int	parse result OK=0, NG=-1
 */
static int
parse_option_replication(int argc, char *argv[])
{
	struct l7vsadm_option_list *list;
	int idx;
	int match_option_f;

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		char arg_str[DEBUG_STR_LEN] = {0};
		argc_argv_c_str(arg_str, argc, argv);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,53,
			"in_function : static int parse_option_replication(int argc, char *argv[]) : "
			"argc = %d, %s",
			argc,
			arg_str);
	}
	/*------ DEBUG LOG END ------*/

	// Argument check for replication command.
	if (argc < 3 || argc > 4) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,43,
			"Argument argc is illegal for -R command.");
		fprintf(stderr, "PARSE ERROR : "
			"Argument argc is illegal for -R command.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,54, 
				"out_function : static int "
				"parse_option_replication(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	idx = 2;
	match_option_f = 0;
	// Search option on l7vsadm_option_table_replication.
	for (list = l7vsadm_option_table_replication; list->option_name != NULL; list++) {
		if (strcmp(argv[idx], list->option_name) == 0 || 
		    strcmp(argv[idx], list->option_name_long) == 0) {
			match_option_f = 1;
			if (list->check_flag) {
				LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,44,
					"Option %s is conflict.",
					argv[idx]);
				fprintf(stderr, "PARSE ERROR : "
					"Option %s is conflict.\n",
					argv[idx]);

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,55, 
					"out_function : static int "
					"parse_option_replication(int argc, char *argv[]) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

				return -1;
			}
			list->check_flag = 1;
			// Check and Set each option value.
			switch (list->option_code) {
			case OPT_SWITCH:
				idx++;
				if (idx >= argc) {
					LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,45,
						"Need option value for -s option.");
					fprintf(stderr, "PARSE ERROR : "
						"Need option value for -s option.\n");

					/*-------- DEBUG LOG --------*/
					if (log_level_parse == LOG_LV_DEBUG) {
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,56, 
							"out_function : static int "
							"parse_option_replication(int argc, char *argv[]) : "
							"return = -1");
					}
					/*------ DEBUG LOG END ------*/

					return -1;
				}
				l7vsadm_option_data.control_code = list->option_code;
				if (strcmp(argv[idx], "start") == 0) {
					l7vsadm_option_data.replication_start_flag = 1;
				} else if (strcmp(argv[idx], "stop") == 0) {
					l7vsadm_option_data.replication_start_flag = 0;
				} else {
					LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,46,
						"Invalid option value for -s option.");
					fprintf(stderr, "PARSE ERROR : "
						"Invalid option value for -s option.\n");

					/*-------- DEBUG LOG --------*/
					if (log_level_parse == LOG_LV_DEBUG) {
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,57, 
							"out_function : static int "
							"parse_option_replication(int argc, char *argv[]) : "
							"return = -1");
					}
					/*------ DEBUG LOG END ------*/

					return -1;
				}
				break;
			case OPT_FORCE:
				if (idx + 1 < argc) {
					LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,47,
						"-f option is no argument option.");
					fprintf(stderr, "PARSE ERROR : "
						"-f option is no argument option.\n");

					/*-------- DEBUG LOG --------*/
					if (log_level_parse == LOG_LV_DEBUG) {
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,58, 
							"out_function : static int "
							"parse_option_replication(int argc, char *argv[]) : "
							"return = -1");
					}
					/*------ DEBUG LOG END ------*/

					return -1;
				}
				l7vsadm_option_data.control_code = list->option_code;
				break;
			case OPT_DUMP:
				if (idx + 1 < argc) {
					LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,48,
						"-d option is no argument option.");
					fprintf(stderr, "PARSE ERROR : "
						"-d option is no argument option.\n");

					/*-------- DEBUG LOG --------*/
					if (log_level_parse == LOG_LV_DEBUG) {
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,59, 
							"out_function : static int "
							"parse_option_replication(int argc, char *argv[]) : "
							"return = -1");
					}
					/*------ DEBUG LOG END ------*/

					return -1;
				}
				l7vsadm_option_data.control_code = list->option_code;
				break;
			default:
				break;
			}
			break;
		}
	}
	if (!match_option_f) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,49,
			"Invalid option for -R command.");
		fprintf(stderr, "PARSE ERROR : "
			"Invalid option for -R command.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,60, 
				"out_function : static int "
				"parse_option_replication(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,61, 
			"out_function : static int "
			"parse_option_replication(int argc, char *argv[]) : "
			"return = 0 : result "
			"l7vsadm_option_data.control_code = %d "
			"l7vsadm_option_data.replication_start_flag = %d",
			l7vsadm_option_data.control_code,
			l7vsadm_option_data.replication_start_flag);
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * Parse logger command option.
 * Check 2nd argument and later of logger command
 * @param[in]	argc	number of argument
 * @param[in]	*argv[]	array of argument string
 * @return	int	parse result OK=0, NG=-1
 */
static int
parse_option_log(int argc, char *argv[])
{
	struct l7vsadm_option_list *op_list;
	struct l7vsadm_category_name_list *cat_list;
	struct l7vsadm_level_name_list *lv_list;
	int idx;
	int match_option_f;
	int match_value_f;

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		char arg_str[DEBUG_STR_LEN] = {0};
		argc_argv_c_str(arg_str, argc, argv);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,62,
			"in_function : static int parse_option_log(int argc, char *argv[]) : "
			"argc = %d, %s",
			argc,
			arg_str);
	}
	/*------ DEBUG LOG END ------*/

	// Argument check for logger command.
	if (argc != 6) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,50,
			"Argument argc is illegal for -L command.");
		fprintf(stderr, "PARSE ERROR : "
			"Argument argc is illegal for -L command.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,63, 
				"out_function : static int "
				"parse_option_log(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	idx = 2;
	while (idx < argc) {
		match_option_f = 0;
		// Search option on l7vsadm_option_table_log.
		for (op_list = l7vsadm_option_table_log; op_list->option_name != NULL; op_list++) {
			if (strcmp(argv[idx], op_list->option_name) == 0 || 
			    strcmp(argv[idx], op_list->option_name_long) == 0) {
				match_option_f = 1;
				if (op_list->check_flag) {
					LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,51,
						"Option %s is conflict.",
						argv[idx]);
					fprintf(stderr, "PARSE ERROR : "
						"Option %s is conflict.\n",
						argv[idx]);

					/*-------- DEBUG LOG --------*/
					if (log_level_parse == LOG_LV_DEBUG) {
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,64, 
							"out_function : static int "
							"parse_option_log(int argc, char *argv[]) : "
							"return = -1");
					}
					/*------ DEBUG LOG END ------*/

					return -1;
				}
				op_list->check_flag = 1;
				// Check and Set each option value.
				switch (op_list->option_code) {
				case OPT_CATEGORY:
					idx++;
					if (idx >= argc) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,52,
							"Need option value for -c option.");
						fprintf(stderr, "PARSE ERROR : "
							"Need option value for -c option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,65, 
								"out_function : static int "
								"parse_option_log(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					// Check option value "all".
					if (strcmp(argv[idx], "all") == 0) {
						l7vsadm_option_data.category_all_flag = 1;
						break;
					}
					match_value_f = 0;
					// Search category name on l7vsd_category_name_table
					// and Convert category name to category code.
					for (cat_list = l7vsd_category_name_table;
					     cat_list->category_name != NULL;
					     cat_list++) {
						if (strcmp(argv[idx],
							   cat_list->category_name) == 0 ||
						    strcmp(argv[idx],
							   cat_list->category_name_short) == 0) {
							l7vsadm_option_data.log_category = 
								cat_list->category_code;
							match_value_f = 1;
							break;
						}
					}
					if (!match_value_f) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,53,
							"No such Log-Category %s.", argv[idx]);
						fprintf(stderr, "PARSE ERROR : "
							"No such Log-Category %s.\n", argv[idx]);

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,66, 
								"out_function : static int "
								"parse_option_log(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					break;
				case OPT_LEVEL:
					idx++;
					if (idx >= argc) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,54,
							"Need option value for -l option.");
						fprintf(stderr, "PARSE ERROR : "
							"Need option value for -l option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,67, 
								"out_function : static int "
								"parse_option_log(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					match_value_f = 0;
					// Search level name on level_name_table
					// and Convert level name to level code.
					for (lv_list = level_name_table;
					     lv_list->level_name != NULL;
					     lv_list++) {
						if (strcmp(argv[idx], lv_list->level_name) == 0) {
							l7vsadm_option_data.log_level = 
								lv_list->level_code;
							match_value_f = 1;
							break;
						}
					}
					if (!match_value_f) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,55,
							"No such Log-Level %s.", argv[idx]);
						fprintf(stderr, "PARSE ERROR : "
							"No such Log-Level %s.\n", argv[idx]);

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,68, 
								"out_function : static int "
								"parse_option_log(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					break;
				default:
					break;
				}
				break;
			}
		}
		if (!match_option_f) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,56,
				"Invalid option for -L command.");
			fprintf(stderr, "PARSE ERROR : "
				"Invalid option for -L command.\n");

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,69, 
					"out_function : static int "
					"parse_option_log(int argc, char *argv[]) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}
		idx++;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,70, 
			"out_function : static int "
			"parse_option_log(int argc, char *argv[]) : "
			"return = 0 : result "
			"l7vsadm_option_data.category_all_flag = %d "
			"l7vsadm_option_data.log_category = %d "
			"l7vsadm_option_data.log_level = %d",
			l7vsadm_option_data.category_all_flag,
			l7vsadm_option_data.log_category,
			l7vsadm_option_data.log_level);
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * Parse SNMPAgent command option.
 * Check 2nd argument and later of SNMPAgent command
 * @param[in]	argc	number of argument
 * @param[in]	*argv[]	array of argument string
 * @return	int	parse result OK=0, NG=-1
 */
static int
parse_option_snmp(int argc, char *argv[])
{
	struct l7vsadm_option_list *op_list;
	struct l7vsadm_category_name_list *cat_list;
	struct l7vsadm_level_name_list *lv_list;
	int idx;
	int match_option_f;
	int match_value_f;

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		char arg_str[DEBUG_STR_LEN] = {0};
		argc_argv_c_str(arg_str, argc, argv);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,71,
			"in_function : static int parse_option_snmp(int argc, char *argv[]) : "
			"argc = %d, %s",
			argc,
			arg_str);
	}
	/*------ DEBUG LOG END ------*/

	// Argument check for SNMPAgent command.
	if (argc != 6) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,57,
			"Argument argc is illegal for -S command.");
		fprintf(stderr, "PARSE ERROR : "
			"Argument argc is illegal for -S command.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,72, 
				"out_function : static int "
				"parse_option_snmp(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	idx = 2;
	while (idx < argc) {
		match_option_f = 0;
		// Search option on l7vsadm_option_table_snmp.
		for (op_list = l7vsadm_option_table_snmp;
		     op_list->option_name != NULL;
		     op_list++) {
			if (strcmp(argv[idx], op_list->option_name) == 0 || 
			    strcmp(argv[idx], op_list->option_name_long) == 0) {
				match_option_f = 1;
				if (op_list->check_flag) {
					LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,58,
						"Option %s is conflict.",
						argv[idx]);
					fprintf(stderr, "PARSE ERROR : "
						"Option %s is conflict.\n",
						argv[idx]);

					/*-------- DEBUG LOG --------*/
					if (log_level_parse == LOG_LV_DEBUG) {
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,73, 
							"out_function : static int "
							"parse_option_snmp(int argc, char *argv[]) : "
							"return = -1");
					}
					/*------ DEBUG LOG END ------*/

					return -1;
				}
				op_list->check_flag = 1;
				// Check and Set each option value.
				switch (op_list->option_code) {
				case OPT_CATEGORY:
					idx++;
					if (idx >= argc) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,59,
							"Need option value for -c option.");
						fprintf(stderr, "PARSE ERROR : "
							"Need option value for -c option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,74, 
								"out_function : static int "
								"parse_option_snmp(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					// Check option value "all".
					if (strcmp(argv[idx], "all") == 0) {
						l7vsadm_option_data.category_all_flag = 1;
						break;
					}
					match_value_f = 0;
					// Search category name on snmp_category_name_table
					// and Convert category name to category code.
					for (cat_list = snmp_category_name_table;
					     cat_list->category_name != NULL;
					     cat_list++) {
						if (strcmp(argv[idx],
							   cat_list->category_name) == 0 ||
						    strcmp(argv[idx],
							   cat_list->category_name_short) == 0) {
							l7vsadm_option_data.log_category = 
								cat_list->category_code;
							match_value_f = 1;
							break;
						}
					}
					if (!match_value_f) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,60,
							"No such Log-Category %s.", argv[idx]);
						fprintf(stderr, "PARSE ERROR : "
							"No such Log-Category %s.\n", argv[idx]);

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,75, 
								"out_function : static int "
								"parse_option_snmp(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					break;
				case OPT_LEVEL:
					idx++;
					if (idx >= argc) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,61,
							"Need option value for -l option.");
						fprintf(stderr, "PARSE ERROR : "
							"Need option value for -l option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,76, 
								"out_function : static int "
								"parse_option_snmp(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					match_value_f = 0;
					// Search level name on level_name_table
					// and Convert level name to level code.
					for (lv_list = level_name_table;
					     lv_list->level_name != NULL;
					     lv_list++) {
						if (strcmp(argv[idx],
							   lv_list->level_name) == 0) {
							l7vsadm_option_data.log_level = 
								lv_list->level_code;
							match_value_f = 1;
							break;
						}
					}
					if (!match_value_f) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,62,
							"No such Log-Level %s.", argv[idx]);
						fprintf(stderr, "PARSE ERROR : "
							"No such Log-Level %s.\n", argv[idx]);

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,77, 
								"out_function : static int "
								"parse_option_snmp(int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					break;
				default:
					break;
				}
				break;
			}
		}
		if (!match_option_f) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,63,
				"Invalid option for -S command.");
			fprintf(stderr, "PARSE ERROR : "
				"Invalid option for -S command.\n");

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,78, 
					"out_function : static int "
					"parse_option_snmp(int argc, char *argv[]) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}
		idx++;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,79, 
			"out_function : static int "
			"parse_option_snmp(int argc, char *argv[]) : "
			"return = 0 : result "
			"l7vsadm_option_data.category_all_flag = %d "
			"l7vsadm_option_data.log_category = %d "
			"l7vsadm_option_data.log_level = %d",
			l7vsadm_option_data.category_all_flag,
			l7vsadm_option_data.log_category,
			l7vsadm_option_data.log_level);
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * Parse parameter command option.
 * Check 2nd argument and later of parameter command
 * @param[in]	argc	number of argument
 * @param[in]	*argv[]	array of argument string
 * @return	int	parse result OK=0, NG=-1
 */
static int
parse_option_parameter(int argc, char *argv[])
{
	struct l7vsadm_option_list *op_list;
	static struct l7vsadm_component_name_list component_name_table[] = {
		{PARAM_COMP_ALL,			"all"},
		{PARAM_COMP_L7VSD,			"l7vsd"},
		{PARAM_COMP_IOMUX,			"iomux"},
		{PARAM_COMP_LSOCK,			"lsock"},
		{PARAM_COMP_CONN,			"conn"},
		{PARAM_COMP_DEST,			"dest"},
		{PARAM_COMP_SERVICE,			"service"},
		{PARAM_COMP_MODULE,			"module"},
		{PARAM_COMP_REPLICATION,		"replication"},
		{PARAM_COMP_LOGGER,			"logger"},
		{PARAM_COMP_L7VSADM,			"l7vsadm"},
		{PARAM_COMP_SNMPAGENT,			"snmpagent"},
		{(enum PARAMETER_COMPONENT_TAG)0,	NULL}
	};
	struct l7vsadm_component_name_list *cmp_list;
	int idx;
	int match_option_f;
	int match_value_f;

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		char arg_str[DEBUG_STR_LEN] = {0};
		argc_argv_c_str(arg_str, argc, argv);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,80,
			"in_function : static int parse_option_parameter(int argc, char *argv[]) : "
			"argc = %d, %s",
			argc,
			arg_str);
	}
	/*------ DEBUG LOG END ------*/

	// Argument check for parameter command.
	if (argc != 4) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,64,
			"Argument argc is illegal for -P command.");
		fprintf(stderr, "PARSE ERROR : "
			"Argument argc is illegal for -P command.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,81, 
				"out_function : static int "
				"parse_option_parameter(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	idx = 2;
	while (idx < argc) {
		match_option_f = 0;
		// Search option on l7vsadm_option_table_snmp.
		for (op_list = l7vsadm_option_table_parameter;
		     op_list->option_name != NULL;
		     op_list++) {
			if (strcmp(argv[idx], op_list->option_name) == 0 || 
			    strcmp(argv[idx], op_list->option_name_long) == 0) {
				match_option_f = 1;
				if (op_list->check_flag) {
					LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,65,
						"Option %s is conflict.",
						argv[idx]);
					fprintf(stderr, "PARSE ERROR : "
						"Option %s is conflict.\n",
						argv[idx]);

					/*-------- DEBUG LOG --------*/
					if (log_level_parse == LOG_LV_DEBUG) {
						LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,82, 
							"out_function : static int "
							"parse_option_parameter(int argc, char *argv[]) : "
							"return = -1");
					}
					/*------ DEBUG LOG END ------*/

					return -1;
				}
				op_list->check_flag = 1;
				// Check and Set each option value.
				switch (op_list->option_code) {
				case OPT_RELOAD:
					idx++;
					if (idx >= argc) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,66,
							"Need option value for -r option.");
						fprintf(stderr, "PARSE ERROR : "
							"Need option value for -r option.\n");

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,83, 
								"out_function : static int "
								"parse_option_parameter("
								"int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}
					match_value_f = 0;
					// Search component name on component_name_table
					// and Convert component name to component code.
					for (cmp_list = component_name_table;
					     cmp_list->component_name != NULL;
					     cmp_list++) {
						if (strcmp(argv[idx],
						    	   cmp_list->component_name) == 0) {
							l7vsadm_option_data.reload_param = 
								cmp_list->component_code;
							match_value_f = 1;
							break;
						}
					}
					if (!match_value_f) {
						LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,67,
							"No such Component %s.", argv[idx]);
						fprintf(stderr, "PARSE ERROR : "
							"No such Component %s.\n", argv[idx]);

						/*-------- DEBUG LOG --------*/
						if (log_level_parse == LOG_LV_DEBUG) {
							LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,84, 
								"out_function : static int "
								"parse_option_parameter("
								"int argc, char *argv[]) : "
								"return = -1");
						}
						/*------ DEBUG LOG END ------*/

						return -1;
					}

					break;
				default:
					break;
				}
				break;
			}
		}
		if (!match_option_f) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,68,
				"Invalid option for -P command.");
			fprintf(stderr, "PARSE ERROR : "
				"Invalid option for -P command.\n");

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,85, 
					"out_function : static int "
					"parse_option_parameter(int argc, char *argv[]) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}
		idx++;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,86, 
			"out_function : static int "
			"parse_option_parameter(int argc, char *argv[]) : "
			"return = 0 : result "
			"l7vsadm_option_data.reload_param = %d",
			l7vsadm_option_data.reload_param);
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * Parse help command option.
 * Check 2nd argument and later of help command
 * @param[in]	argc	number of argument
 * @param[in]	*argv[]	array of argument string
 * @return	int	parse result OK=0, NG=-1
 */
static int
parse_option_help(int argc, char *argv[])
{
	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		char arg_str[DEBUG_STR_LEN] = {0};
		argc_argv_c_str(arg_str, argc, argv);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,87,
			"in_function : static int parse_option_help(int argc, char *argv[]) : "
			"argc = %d, %s",
			argc,
			arg_str);
	}
	/*------ DEBUG LOG END ------*/

	// Argument check for help command.
	if (argc != 2) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,69,
			"Argument argc is illegal for -h command.");
		fprintf(stderr, "PARSE ERROR : "
			"Argument argc is illegal for -h command.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,88, 
				"out_function : static int "
				"parse_option_help(int argc, char *argv[]) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,89, 
			"out_function : static int "
			"parse_option_help(int argc, char *argv[]) : "
			"return = 0");
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * Parse endpoint option.
 * Check endpoint(IP-Address:port) data
 * @param[out]	*addr	socket address struct
 * @param[in]	*endpoint_str	endpoint data string
 * @return	int	parse result OK=0, NG=-1
 */
static int
parse_endpoint(struct sockaddr_in *addr, const char *endpoint_str, int allow_inaddr_any)
{
	struct addrinfo *res, hints;
	struct sockaddr_in *sin;
	char s[NI_MAXHOST + NI_MAXSERV], *t;
	int ret;

	// Argment check.
	if (endpoint_str == NULL) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,70,
			"Argument endpoint_str is NULL.");
		fprintf(stderr, "PARSE ERROR : "
			"Argument endpoint_str is NULL.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,90,
				"out_function : static int parse_endpoint"
				"(struct sockaddr_in *addr, const char *endpoint_str) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,91,
			"in_function : static int parse_endpoint"
			"(struct sockaddr_in *addr, const char *endpoint_str) : "
			"addr = \"\" endpoint_str = %s",
			endpoint_str);
	}
	/*------ DEBUG LOG END ------*/

	// Separate host and port string.
	strncpy(s, endpoint_str, strlen(endpoint_str));
	s[strlen(endpoint_str)] = '\0';
	t = strrchr(s, ':');
	if (t == NULL) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,71,
			"Endpoint string is illegal. (':' not found)");
		fprintf(stderr, "PARSE ERROR : "
			"Endpoint string is illegal. (':' not found)\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,92,
				"out_function : static int parse_endpoint"
				"(struct sockaddr_in *addr, const char *endpoint_str) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	*t++ = '\0';

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_INET; /* inet only. no support for inet6 (yet) */
	hints.ai_socktype = SOCK_STREAM;
	ret = getaddrinfo(s, t, &hints, &res);
	if (ret != 0) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,72,
			"getaddrinfo() error. %s:%s: %s", s, t, gai_strerror(ret));
		fprintf(stderr, "PARSE ERROR : "
			"getaddrinfo() error. %s:%s: %s\n", s, t, gai_strerror(ret));

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,93,
				"out_function : static int parse_endpoint"
				"(struct sockaddr_in *addr, const char *endpoint_str) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	/*
	 * We always use the first entry, because we can't distinguish
	 * which entry you want to specify if we have multiple entries.
	 */
	sin = (struct sockaddr_in *)res->ai_addr;
	if (!allow_inaddr_any && sin->sin_addr.s_addr == htonl(INADDR_ANY)) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,73,
			"You can't specify INADDR_ANY for endpoint.");
		fprintf(stderr, "PARSE ERROR : "
			"You can't specify INADDR_ANY for endpoint.\n");
		freeaddrinfo(res);

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,94,
				"out_function : static int parse_endpoint"
				"(struct sockaddr_in *addr, const char *endpoint_str) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
        
	if (sin->sin_port == htons(0)) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,74,
			"You can't specify port number 0.");
		fprintf(stderr, "PARSE ERROR : "
			"You can't specify port number 0.\n");
		freeaddrinfo(res);

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,95,
				"out_function : static int parse_endpoint"
				"(struct sockaddr_in *addr, const char *endpoint_str) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	*addr = *sin;
	freeaddrinfo(res);

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		char ep_str[DEBUG_STR_LEN] = {0};
		endpoint_c_str(ep_str, (void *)addr);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,96,
			"out_function : static int parse_endpoint"
			"(struct sockaddr_in *addr, const char *endpoint_str) : "
			"return = 0 : result "
			"addr = (%s)",
			ep_str);
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * Parse digit option.
 * Check digit data
 * @param[in]	*digit_str	digit data string
 * @return	int	parse result OK=0, NG=-1
 */
static int
parse_digit(const char *digit_str)
{
	int i;

	// Argment check.
	if (digit_str == NULL) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,75,
			"Argument digit_str is NULL.");
		fprintf(stderr, "PARSE ERROR : "
			"Argument digit_str is NULL.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,97,
				"out_function : static int parse_digit"
				"(const char *digit_str) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,98,
			"in_function : static int parse_digit(const char *digit_str) : "
			"digit_str = %s",
			digit_str);
	}
	/*------ DEBUG LOG END ------*/

	for (i = 0; i < (int)strlen(digit_str); i++) {
		// Digit check.
		if (isdigit(digit_str[i]) == 0) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,76,
				"Non digit character is included.");
			fprintf(stderr, "PARSE ERROR : "
				"Non digit character is included.\n");

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,99,
					"out_function : static int parse_digit"
					"(const char *digit_str) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,100,
			"out_function : static int parse_digit"
			"(const char *digit_str) : "
			"return = 0");
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * Parse qos option.
 * Check qos data (xxx,xxxK,xxxM,xxxG)
 * @param[in]	*qos_str	qos data string
 * @return	long long	qos value, parse result NG=-1
 */
static long long
parse_qos(const char *qos_str)
{
	int i;
	int unit_f = 0;
	char numstr[4];
	long long qos_value = 0;
	long long multiply_value = 0;

	// Argment check.
	if (qos_str == NULL) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,77,
			"Argument qos_str is NULL.");
		fprintf(stderr, "PARSE ERROR : "
			"Argument qos_str is NULL.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,101,
				"out_function : static long long parse_qos"
				"(const char *qos_str) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	if ((int)strlen(qos_str) > 4) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,78,
			"Argument qos_str is too long.");
		fprintf(stderr, "PARSE ERROR : "
			"Argument qos_str is too long.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,102,
				"out_function : static long long parse_qos"
				"(const char *qos_str) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,103,
			"in_function : static long long parse_qos"
			"(const char *qos_str) : "
			"qos_str = %s",
			qos_str);
	}
	/*------ DEBUG LOG END ------*/


	for (i = 0; i < (int)strlen(qos_str); i++) {
		// Check QoS data format. (xxx or xxx + one alphabet)
		if (isdigit(qos_str[i]) == 0) {
			if ((int)strlen(qos_str) > 1 && 
			    i == (int)strlen(qos_str) -1 && 
			    isalpha(qos_str[i]) != 0) {
				// xxx + one alphabet (len > 1)
				unit_f = 1;
			} else {
				LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,79,
					"Wrong QoS value format.");
				fprintf(stderr, "PARSE ERROR : "
					"Wrong QoS value format.\n");

				/*-------- DEBUG LOG --------*/
				if (log_level_parse == LOG_LV_DEBUG) {
					LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,104,
						"out_function : static long long parse_qos"
						"(const char *qos_str) : "
						"return = -1");
				}
				/*------ DEBUG LOG END ------*/

				return -1;
			}
		}
	}
	if (unit_f) {
		// Format xxxK,xxxM,xxxG,xxxk,xxxm,xxxg
		strncpy(numstr, qos_str, strlen(qos_str) - 1);
		numstr[strlen(qos_str) - 1] = '\0';
		sscanf(numstr, "%lld", &qos_value);
		if (qos_value < 0LL || qos_value > 999LL) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,80,
				"Invalid QoS value.");
			fprintf(stderr, "PARSE ERROR : "
				"Invalid QoS value.\n");

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,105,
					"out_function : static long long parse_qos"
					"(const char *qos_str) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}
		if (qos_str[strlen(qos_str) - 1] == 'K' || 
		    qos_str[strlen(qos_str) - 1] == 'k') {
			multiply_value = 1000LL;
		} else if (qos_str[strlen(qos_str) - 1] == 'M' || 
			   qos_str[strlen(qos_str) - 1] == 'm') {
			multiply_value = 1000LL * 1000LL;
		} else if (qos_str[strlen(qos_str) - 1] == 'G' || 
			   qos_str[strlen(qos_str) - 1] == 'g') {
			multiply_value = 1000LL * 1000LL * 1000LL;
		} else {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,81,
				"Invalid QoS value unit character.");
			fprintf(stderr, "PARSE ERROR : "
				"Invalid QoS value character.\n");

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,106,
					"out_function : static long long parse_qos"
					"(const char *qos_str) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}
		// Convert from bps to byte/s.
		//qos_value = qos_value * multiply_value / 8LL;
		qos_value = qos_value * multiply_value;
	} else {
		// Format xxx only.
		sscanf(qos_str, "%lld", &qos_value);
		if (qos_value < 0LL || qos_value > 999LL) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,82,
				"Invalid QoS value.");
			fprintf(stderr, "PARSE ERROR : "
				"Invalid QoS value.\n");

			/*-------- DEBUG LOG --------*/
			if (log_level_parse == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,107,
					"out_function : static long long parse_qos"
					"(const char *qos_str) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}
		// Convert from bps to byte/s.
//		qos_value = qos_value / 8LL;
	}
	
	// compare l7vs_conn_read_bufsize <= qos_value by bps
	if (qos_value != 0) {
		int l7vs_conn_read_bufsize_bps = 20480 * 8;			//! conn read buffer size
	    int ret = parameter_is_int_exist( PARAM_COMP_CONN, "read_bufsize" );
	    if (ret) {
	        int bufsize = parameter_get_int_value( PARAM_COMP_CONN, "read_bufsize" );
	        if (bufsize >= L7VS_CONN_READ_BUFSIZE_MIN && bufsize <= L7VS_CONN_READ_BUFSIZE_MAX)
	            l7vs_conn_read_bufsize_bps = bufsize * 8;
	        else if (bufsize > L7VS_CONN_READ_BUFSIZE_MAX) {
	        	l7vs_conn_read_bufsize_bps = L7VS_CONN_READ_BUFSIZE_MAX * 8;
	        }
	        else {
	        	l7vs_conn_read_bufsize_bps = L7VS_CONN_READ_BUFSIZE_MIN * 8;
	        }
	    }
	    if (l7vs_conn_read_bufsize_bps > qos_value) {
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,86,
				"Invalid QoS value. qos >= read_bufsize is must");
			fprintf(stderr, "PARSE ERROR : "
				"Invalid QoS value. qos >= read_bufsize is must\n");

			return -1;
	    }
	}

	// Convert from bps to byte/s.
	qos_value = qos_value / 8LL;

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,108,
			"out_function : static long long parse_qos"
			"(const char *qos_str) : "
			"return = %lld",
			qos_value);
	}
	/*------ DEBUG LOG END ------*/

	return qos_value;
}

/*!
 * Parse protocol module option.
 * Check protocol module option data
 * @param[in]	argc	number of argument
 * @param[in]	*argv[]	array of argument string
 * @param[in]	m_idx_begin	start of module option
 * @param[in]	m_idx_end	end of module option
 * @return	int	parse result OK=0, NG=-1
 */
static int
parse_option_protomod(int argc, char *argv[], int m_idx_begin, int m_idx_end)
{
	struct l7vs_service_arg_multi sarg;
	void *protomod_service_arg;
	int ret;
	int i;

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		char arg_str[DEBUG_STR_LEN] = {0};
		argc_argv_c_str(arg_str, argc, argv);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,109,
			"in_function : static int parse_option_protomod"
			"(int argc, char *argv[], int m_idx_begin, int m_idx_end) : "
			"argc = %d, %s, m_idx_begin = %d, m_idx_end = %d",
			argc,
			arg_str,
			m_idx_begin,
			m_idx_end);
	}
	/*------ DEBUG LOG END ------*/

	// Check signal.
	if (signal_flg > 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,1,
			"Signal (%d) Received.", received_sig);
		fprintf(stderr, "COMMON ERROR : "
			"Signal (%d) Received.\n", received_sig);

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,110,
				"out_function : static int parse_option_protomod"
				"(int argc, char *argv[], int m_idx_begin, int m_idx_end) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	// Clear sarg.
	memset(&sarg, 0, sizeof(l7vs_service_arg_multi));

	// Create GArray.
	l7vsadm_option_data.protomod_options = g_array_new(FALSE, TRUE, sizeof(char*));
	if (l7vsadm_option_data.protomod_options == NULL) {
		LOGGER_PUT_LOG_FATAL(LOG_CAT_L7VSADM_COMMON,1,
			" GArray create error. (Could not allocate memory)");
		fprintf(stderr, "COMMON ERROR : "
			" GArray create error. (Could not allocate memory)\n");
		ret = -1;
		goto END;
	}

	/* Add argv[0] first because getopt_long() skip argv[0]... */
	if (g_array_append_val(l7vsadm_option_data.protomod_options, argv[0]) == NULL) {
		LOGGER_PUT_LOG_FATAL(LOG_CAT_L7VSADM_COMMON,2,
			"Add to GArray error. (Could not allocate memory)");
		fprintf(stderr, "COMMON ERROR : "
			"Add to GArray error. (Could not allocate memory)\n");
		ret = -1;
		goto END;
	}
	// Add Protocol Module option to GArray. (from m_idx_begin to m_idx_end)
	for (i = m_idx_begin; i < m_idx_end; i++) {
		if (g_array_append_val(l7vsadm_option_data.protomod_options, argv[i]) == NULL) {
			LOGGER_PUT_LOG_FATAL(LOG_CAT_L7VSADM_COMMON,3,
				"Add to GArray error. (Could not allocate memory)");
			fprintf(stderr, "COMMON ERROR : "
				"Add to GArray error. (Could not allocate memory)\n");
			ret = -1;
			goto END;
		}
	}

	// Protocol Module Initialize.
	ret = l7vs_module_init(NULL);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_PARSE,1,
			"Protocol Module Initialize error.");
		fprintf(stderr, "PARSE ERROR : "
			"Protocol Module Initialize error.\n");
		goto END;
	}

	// Protocol Module Load.
	l7vsadm_option_data.protomod =
		(struct l7vs_protomod *)l7vs_protomod_get(l7vsadm_option_data.protomod_name);
	if (l7vsadm_option_data.protomod == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_PARSE,2,
			"Protocol Module Load error.");
		fprintf(stderr, "PARSE ERROR : "
			"Protocol Module Load error.\n");
		ret = -1;
		goto END;
	}

	//  Check Protocol Module function pointer
	if (l7vsadm_option_data.protomod->create_sa  == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_PARSE,3,
			"create_sa() function pointer is NULL.");
		fprintf(stderr, "PARSE ERROR : "
			"create_sa() function pointer is NULL.\n");
		ret = -1;
		goto END;
	}
	if (l7vsadm_option_data.protomod->destroy_sa == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_PARSE,4,
			"destroy_sa() function pointer is NULL.");
		fprintf(stderr, "PARSE ERROR : "
			"destroy_sa() function pointer is NULL.\n");
		ret = -1;
		goto END;
	}
	if (l7vsadm_option_data.protomod->parse == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_PARSE,5,
			"parse() function pointer is NULL.");
		fprintf(stderr, "PARSE ERROR : "
			"parse() function pointer is NULL.\n");
		ret = -1;
		goto END;
	}

	// Create Protocol Module service_arg
	protomod_service_arg = l7vsadm_option_data.protomod->create_sa(&sarg.srv_arg);
	if (protomod_service_arg == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_PARSE,6,
			"Create Protocol Module service_arg error.");
		fprintf(stderr, "PARSE ERROR : "
			"Create Protocol Module service_arg error.\n");
		ret = -1;
		goto END;
	}
	// Copy protomod_service_arg to sarg.protomod_arg
	memcpy(&sarg.protomod_arg, protomod_service_arg, sizeof(sarg.protomod_arg));

	// Destroy protomod_service_arg. (because copy end)
	l7vsadm_option_data.protomod->destroy_sa(&protomod_service_arg);

	// Parse Protocol Module option. (parse result saved sarg.protomod_arg)
	ret = l7vsadm_option_data.protomod->parse(sarg.protomod_arg,
				l7vsadm_option_data.protomod_options->len,
				(char **)l7vsadm_option_data.protomod_options->data);
	if (ret < 0) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,83,
			"Parse %s Protocol Module option error.",
			l7vsadm_option_data.protomod_name);
		fprintf(stderr, "PARSE ERROR : "
			"Parse %s Protocol Module option error.\n",
			l7vsadm_option_data.protomod_name);
		goto END;
	}

	// Copy sarg.protomod_arg to l7vsadm_option_data. (for operate)
	memcpy(&l7vsadm_option_data.protomod_sarg,
	       &sarg,
	       sizeof(struct l7vs_service_arg_multi));

	ret = 0;

END:
	if (l7vsadm_option_data.protomod_options != NULL) {
		g_array_free(l7vsadm_option_data.protomod_options, TRUE);
	}

	if (l7vsadm_option_data.protomod != NULL) {
		l7vs_protomod_put(l7vsadm_option_data.protomod);
	}

	l7vs_module_fini();

	/*-------- DEBUG LOG --------*/
	if (log_level_parse == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,111,
			"out_function : static int parse_option_protomod"
			"(int argc, char *argv[], int m_idx_begin, int m_idx_end) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Operate List command.
 * @return	int	result OK=0, NG=<0
 */
static int
operate_list(void)
{
	struct l7vs_service_arg *sarg;
	void *vs_buffer = NULL;
	int vs_bufsize = L7VSADM_DEFAULT_LIST_VS_BUFSIZE;
	int vs_num;
	int ret_num;
	int vs_offset;
	int i;
	int ret;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,1,
			"in_function : static int operate_list(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	ret = create_socket();
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,1,
			"Create socket error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,2,
				"out_function : static int operate_list"
				"(void) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	// Print Version.
	printf("Layer-7 Virtual Server version %s\n", L7VS_VERSION_STRING);

	if (l7vsadm_option_data.command_code == CMD_LIST_VERBOSE) {
		// Logger info.
		ret = list_log_info();
		if (ret < 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,2,
				"List logger info error.");
			goto END;
		}

		// Replication info.
		ret = list_replication_info();
		if (ret < 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,3,
				"List replication info error.");
			goto END;
		}

		// SNMPAgent info.
		ret = list_snmp_info();
		if (ret < 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,4,
				"List SNMPAgent info error.");
			goto END;
		}
	}

	// Display VS header.
	if (l7vsadm_option_data.command_code == CMD_LIST_VERBOSE) {
		printf("Prot LocalAddress:Port ProtoMod Scheduler Reschedule "
		       "Protomod_opt_string\n");
		printf("     SorryAddress:Port Sorry_cc Sorry_flag\n");
		printf("     QoS-up   Throughput-up\n");
		printf("     QoS-down Throughput-down\n");
	} else if (l7vsadm_option_data.command_code == CMD_LIST_KEY) {
		printf("Prot LocalAddress:Port ProtoMod Scheduler Reschedule "
		       "Protomod_key_string\n");
	} else {
		printf("Prot LocalAddress:Port ProtoMod Scheduler\n");
	}
	printf("  -> RemoteAddress:Port           Forward Weight ActiveConn InactConn\n");

RETRY:
	// Alloc VirtualService list memory.
	vs_buffer = malloc(vs_bufsize);

	/*-------- DEBUG LOG --------*/
	if (log_level_common == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,1,
			"malloc vs_buffer : "
			"allocate address = %p "
			"allocate size = %d",
			vs_buffer,
			vs_bufsize);
	}
	/*------ DEBUG LOG END ------*/

	if (vs_buffer == NULL) {
		LOGGER_PUT_LOG_FATAL(LOG_CAT_L7VSADM_COMMON,4,
			"Could not allocate vs_buffer memory.");
		fprintf(stderr, "COMMON ERROR : "
			"Could not allocate vs_buffer memory.\n");
		ret = -1;
		goto END;
	}
	// Get all VirtualService list.
	ret_num = get_vs(vs_buffer, vs_bufsize);
	// If return L7VSADM_RECV_TOO_LARGE (-2) when extend vs_buffer size and retry.
	if (ret_num < 0) {
		free(vs_buffer);

		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,2,
				"free vs_buffer : "
				"free address = %p",
				vs_buffer);
		}
		/*------ DEBUG LOG END ------*/

		vs_buffer = NULL;
		if (ret_num == L7VSADM_RECV_TOO_LARGE) {
			vs_bufsize += L7VSADM_DEFAULT_LIST_VS_BUFSIZE;
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_OPERATE,1,
				"Receive VirtualService data too large. retrying.");
			goto RETRY;
		}
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,5,
			"Get VirtualService data error.");
		fprintf(stderr, "OPERATE ERROR : "
			"Get VirtualService data error.\n");
		ret = -1;
		goto END;
	}
	vs_num = ret_num;
	// List all VirtualService.
	sarg = (struct l7vs_service_arg *)vs_buffer;
	vs_offset = 0;
	for (i = 0; i < vs_num; i++) {
		ret = list_vs(sarg);
		if (ret < 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,6,
				"List VirtualService data error.");
			fprintf(stderr, "OPERATE ERROR : "
				"List VirtualService data error.\n");
			goto END;
		}
		vs_offset += sarg->len;
		if (vs_offset >= vs_bufsize) {
			// Message size did not match.
			// Version mismatch between l7vsd and l7vsadm?
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,7,
				"VirtualService message size did not match.");
			fprintf(stderr, "OPERATE ERROR : "
				"VirtualService message size did not match.\n");
			ret = -1;
			goto END;
		}
		sarg = (struct l7vs_service_arg *)((char *)vs_buffer + vs_offset);
	}
	ret = 0;

END:
	if (vs_buffer != NULL) {
		free(vs_buffer);

		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,3,
				"free vs_buffer : "
				"free address = %p",
				vs_buffer);
		}
		/*------ DEBUG LOG END ------*/

	}

        destroy_socket();

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,3,
			"out_function : static int operate_list"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Operate Add-VirtualService command.
 * @return	int	result OK=0, NG=<0
 */
static int
operate_add_vs(void)
{
	struct l7vs_config_req_operate_vs req;
	struct l7vs_config_rsp_operate_vs rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,4,
			"in_function : static int operate_add_vs(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	ret = create_socket();
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,8,
			"Create socket error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,5,
				"out_function : static int operate_add_vs"
				"(void) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	memset(&req, 0, sizeof(struct l7vs_config_req_operate_vs));
	memset(&rsp, 0, sizeof(struct l7vs_config_rsp_operate_vs));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	req.cmd = L7VS_CONFIG_ADD_VS;

	l7vsadm_option_data.protomod_sarg.srv_arg.addr = l7vsadm_option_data.vs_addr;
	l7vsadm_option_data.protomod_sarg.srv_arg.proto = IPPROTO_TCP;
	l7vsadm_option_data.protomod_sarg.srv_arg.backlog = L7VSADM_DEFAULT_BACKLOG;

	strncpy(l7vsadm_option_data.protomod_sarg.srv_arg.schedmod, 
		l7vsadm_option_data.schedmod_name, 
		strlen(l7vsadm_option_data.schedmod_name));

	// set sorry data
	l7vsadm_option_data.protomod_sarg.srv_arg.sorry_cc = l7vsadm_option_data.max_connection;
	l7vsadm_option_data.protomod_sarg.srv_arg.sorry_addr = l7vsadm_option_data.sorry_addr;
	l7vsadm_option_data.protomod_sarg.srv_arg.sorry_flag = l7vsadm_option_data.sorry_flag;
        
	// QoS Threshold data
	l7vsadm_option_data.protomod_sarg.srv_arg.qos_threshold_up = l7vsadm_option_data.qos_threshold_up;
	l7vsadm_option_data.protomod_sarg.srv_arg.qos_threshold_down = l7vsadm_option_data.qos_threshold_down;

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_iov[1].iov_base = &l7vsadm_option_data.protomod_sarg;
	iov_data.req_iov[1].iov_len = sizeof(struct l7vs_service_arg_multi);
	iov_data.req_cnt = 2;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_cnt = 1;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,9,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,1,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}
	ret = 0;

END:
	destroy_socket();

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,6,
			"out_function : static int operate_add_vs"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Operate Edit-VirtualService command.
 * @return	int	result OK=0, NG=<0
 */
static int
operate_edit_vs(void)
{
	struct l7vs_config_req_operate_vs req;
	struct l7vs_config_rsp_operate_vs rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;
	struct l7vsadm_option_list *list;
	int specified_f = 0;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,7,
			"in_function : static int operate_edit_vs(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	ret = create_socket();
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,10,
			"Create socket error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,8,
				"out_function : static int operate_edit_vs"
				"(void) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	memset(&req, 0, sizeof(struct l7vs_config_req_operate_vs));
	memset(&rsp, 0, sizeof(struct l7vs_config_rsp_operate_vs));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	req.cmd = L7VS_CONFIG_EDIT_VS;

	l7vsadm_option_data.protomod_sarg.srv_arg.addr = l7vsadm_option_data.vs_addr;
	l7vsadm_option_data.protomod_sarg.srv_arg.proto = IPPROTO_TCP;
	l7vsadm_option_data.protomod_sarg.srv_arg.backlog = L7VSADM_DEFAULT_BACKLOG;

	// Check Specified option for edit. Set -1 or NULL or etc. to omitted option.
	for (list = l7vsadm_option_table_vs; list->option_name != NULL; list++) {
		switch (list->option_code) {
		case OPT_SCHEDULER:
			if (list->check_flag) {
				// Option specified.
				strncpy(l7vsadm_option_data.protomod_sarg.srv_arg.schedmod, 
					l7vsadm_option_data.schedmod_name, 
					strlen(l7vsadm_option_data.schedmod_name));
			} else {
				// Option omitted. Set all "0x00".
				memset(&l7vsadm_option_data.protomod_sarg.srv_arg.schedmod, 0, 
				       sizeof(l7vsadm_option_data.protomod_sarg.srv_arg.schedmod));
			}
			break;
		case OPT_UPPER:
			if (list->check_flag) {
				// Option specified.
				l7vsadm_option_data.protomod_sarg.srv_arg.sorry_cc = 
				l7vsadm_option_data.max_connection;
			} else {
				// Option omitted.
				l7vsadm_option_data.protomod_sarg.srv_arg.sorry_cc = -1;
			}
			break;
		case OPT_BYPASS:
			if (list->check_flag) {
				// Option specified.
				l7vsadm_option_data.protomod_sarg.srv_arg.sorry_addr = 
				l7vsadm_option_data.sorry_addr;
			} else {
				// Option omitted. Set all "0xff".
				memset(&l7vsadm_option_data.protomod_sarg.srv_arg.sorry_addr, 0xff, 
				       sizeof(struct sockaddr_storage));
			}
			break;
		case OPT_FLAG:
			if (list->check_flag) {
				// Option specified.
				l7vsadm_option_data.protomod_sarg.srv_arg.sorry_flag = 
				l7vsadm_option_data.sorry_flag;
			} else {
				// Option omitted. Set -1.
				l7vsadm_option_data.protomod_sarg.srv_arg.sorry_flag = -1;
			}
			break;
		case OPT_QOS_UP:
			if (list->check_flag) {
				// Option specified.
				l7vsadm_option_data.protomod_sarg.srv_arg.qos_threshold_up = 
				l7vsadm_option_data.qos_threshold_up;
			} else {
				// Option omitted. Set ULLONG_MAX.
				l7vsadm_option_data.protomod_sarg.srv_arg.qos_threshold_up = ULLONG_MAX;
			}
			break;
		case OPT_QOS_DOWN:
			if (list->check_flag) {
				// Option specified.
				l7vsadm_option_data.protomod_sarg.srv_arg.qos_threshold_down = 
				l7vsadm_option_data.qos_threshold_down;
			} else {
				// Option omitted. Set ULLONG_MAX.
				l7vsadm_option_data.protomod_sarg.srv_arg.qos_threshold_down = ULLONG_MAX;
			}
			break;
		default:
			break;
		}
		specified_f += list->check_flag;
	}

	// All option omitted.
	if (specified_f == 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,11,
			"All option omitted for edit vs command.");
		fprintf(stderr, "OPERATE ERROR : "
			"All option omitted for edit vs command.\n");
		ret = -1;
		goto END;
	}

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_iov[1].iov_base = &l7vsadm_option_data.protomod_sarg;
	iov_data.req_iov[1].iov_len = sizeof(struct l7vs_service_arg_multi);
	iov_data.req_cnt = 2;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_cnt = 1;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,12,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,2,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}
	ret = 0;

END:
	destroy_socket();

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,9,
			"out_function : static int operate_edit_vs"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Operate Delete-VirtualService command.
 * @return	int	result OK=0, NG=<0
 */
static int
operate_del_vs(void)
{
	struct l7vs_config_req_operate_vs req;
	struct l7vs_config_rsp_operate_vs rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,10,
			"in_function : static int operate_del_vs(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	ret = create_socket();
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,13,
			"Create socket error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,11,
				"out_function : static int operate_del_vs"
				"(void) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	memset(&req, 0, sizeof(struct l7vs_config_req_operate_vs));
	memset(&rsp, 0, sizeof(struct l7vs_config_rsp_operate_vs));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	req.cmd = L7VS_CONFIG_DEL_VS;

	l7vsadm_option_data.protomod_sarg.srv_arg.addr = l7vsadm_option_data.vs_addr;
	l7vsadm_option_data.protomod_sarg.srv_arg.proto = IPPROTO_TCP;
	l7vsadm_option_data.protomod_sarg.srv_arg.backlog = L7VSADM_DEFAULT_BACKLOG;

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_iov[1].iov_base = &l7vsadm_option_data.protomod_sarg;
	iov_data.req_iov[1].iov_len = sizeof(struct l7vs_service_arg_multi);
	iov_data.req_cnt = 2;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_cnt = 1;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,14,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,3,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}
	ret = 0;

END:
	destroy_socket();

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,12,
			"out_function : static int operate_del_vs"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Operate Flush-VirtualService command.
 * @return	int	result OK=0, NG=<0
 */
static int
operate_flush_vs(void)
{
	struct l7vs_config_req_flush_vs req;
	struct l7vs_config_rsp_flush_vs rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,13,
			"in_function : static int operate_flush_vs(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	ret = create_socket();
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,15,
			"Create socket error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,14,
				"out_function : static int operate_flush_vs"
				"(void) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	memset(&req, 0, sizeof(struct l7vs_config_req_flush_vs));
	memset(&rsp, 0, sizeof(struct l7vs_config_rsp_flush_vs));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	req.cmd = L7VS_CONFIG_FLUSH_VS;

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_cnt = 1;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_cnt = 1;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,16,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,4,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}
	ret = 0;

END:
	destroy_socket();

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,15,
			"out_function : static int operate_flush_vs"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Operate Add-RealServer command.
 * @return	int	result OK=0, NG=<0
 */
static int
operate_add_rs(void)
{
	struct l7vs_config_req_operate_rs req;
	struct l7vs_config_rsp_operate_rs rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,16,
			"in_function : static int operate_add_rs(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	ret = create_socket();
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,17,
			"Create socket error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,17,
				"out_function : static int operate_add_rs"
				"(void) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	memset(&req, 0, sizeof(struct l7vs_config_req_operate_rs));
	memset(&rsp, 0, sizeof(struct l7vs_config_rsp_operate_rs));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	req.cmd = L7VS_CONFIG_ADD_RS;

	l7vsadm_option_data.protomod_sarg.srv_arg.addr = l7vsadm_option_data.vs_addr;
	l7vsadm_option_data.protomod_sarg.srv_arg.proto = IPPROTO_TCP;

	req.darg.addr = l7vsadm_option_data.rs_addr;
	req.darg.weight = l7vsadm_option_data.weight;

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_iov[1].iov_base = &l7vsadm_option_data.protomod_sarg;
	iov_data.req_iov[1].iov_len = sizeof(struct l7vs_service_arg_multi);
	iov_data.req_cnt = 2;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_cnt = 1;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,18,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,5,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}
	ret = 0;

END:
	destroy_socket();

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,18,
			"out_function : static int operate_add_rs"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Operate Edit-RealServer command.
 * @return	int	result OK=0, NG=<0
 */
static int
operate_edit_rs(void)
{
	struct l7vs_config_req_operate_rs req;
	struct l7vs_config_rsp_operate_rs rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;
	struct l7vsadm_option_list *list;
	int specified_f = 0;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,19,
			"in_function : static int operate_edit_rs(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	ret = create_socket();
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,19,
			"Create socket error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,20,
				"out_function : static int operate_edit_rs"
				"(void) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	memset(&req, 0, sizeof(struct l7vs_config_req_operate_rs));
	memset(&rsp, 0, sizeof(struct l7vs_config_rsp_operate_rs));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	req.cmd = L7VS_CONFIG_EDIT_RS;

	l7vsadm_option_data.protomod_sarg.srv_arg.addr = l7vsadm_option_data.vs_addr;
	l7vsadm_option_data.protomod_sarg.srv_arg.proto = IPPROTO_TCP;

	req.darg.addr = l7vsadm_option_data.rs_addr;

	// Check Specified option for edit. Set -1 or NULL or etc. to omitted option.
	for (list = l7vsadm_option_table_rs; list->option_name != NULL; list++) {
		switch (list->option_code) {
		case OPT_WEIGHT:
			if (list->check_flag) {
				// Option specified.
				req.darg.weight = l7vsadm_option_data.weight;
			} else {
				// Option omitted. Set -1.
				req.darg.weight = -1;
			}
			break;
		default:
			break;
		}
		specified_f += list->check_flag;
	}

	// All option omitted.
	if (specified_f == 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,20,
			"All option omitted for edit rs command.");
		fprintf(stderr, "OPERATE ERROR : "
			"All option omitted for edit rs command.\n");
		ret = -1;
		goto END;
	}

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_iov[1].iov_base = &l7vsadm_option_data.protomod_sarg;
	iov_data.req_iov[1].iov_len = sizeof(struct l7vs_service_arg_multi);
	iov_data.req_cnt = 2;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_cnt = 1;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,21,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,6,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}
	ret = 0;

END:
	destroy_socket();

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,21,
			"out_function : static int operate_edit_rs"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Operate Delete-RealServer command.
 * @return	int	result OK=0, NG=<0
 */
static int
operate_del_rs(void)
{
	struct l7vs_config_req_operate_rs req;
	struct l7vs_config_rsp_operate_rs rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,22,
			"in_function : static int operate_del_rs(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	ret = create_socket();
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,22,
			"Create socket error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,23,
				"out_function : static int operate_del_rs"
				"(void) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	memset(&req, 0, sizeof(struct l7vs_config_req_operate_rs));
	memset(&rsp, 0, sizeof(struct l7vs_config_rsp_operate_rs));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	req.cmd = L7VS_CONFIG_DEL_RS;

	l7vsadm_option_data.protomod_sarg.srv_arg.addr = l7vsadm_option_data.vs_addr;
	l7vsadm_option_data.protomod_sarg.srv_arg.proto = IPPROTO_TCP;

	req.darg.addr = l7vsadm_option_data.rs_addr;

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_iov[1].iov_base = &l7vsadm_option_data.protomod_sarg;
	iov_data.req_iov[1].iov_len = sizeof(struct l7vs_service_arg_multi);
	iov_data.req_cnt = 2;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_cnt = 1;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,23,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,7,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}
	ret = 0;

END:
	destroy_socket();

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,24,
			"out_function : static int operate_del_rs"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Operate Replication command.
 * @return	int	result OK=0, NG=<0
 */
static int
operate_replication(void)
{
	struct l7vs_config_msg_replication req;
	struct l7vs_config_msg_replication rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,25,
			"in_function : static int operate_replication(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	ret = create_socket();
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,24,
			"Create socket error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,26,
				"out_function : static int operate_replication"
				"(void) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	memset(&req, 0, sizeof(struct l7vs_config_msg_replication));
	memset(&rsp, 0, sizeof(struct l7vs_config_msg_replication));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	req.cmd = L7VS_CONFIG_REPLICATION;

	switch (l7vsadm_option_data.control_code) {
	case OPT_SWITCH:
		if (l7vsadm_option_data.replication_start_flag == 1) {
			req.kind = L7VS_CONFIG_REPLICATION_START;
		} else {
			req.kind = L7VS_CONFIG_REPLICATION_STOP;
		}
		break;
	case OPT_FORCE:
		req.kind = L7VS_CONFIG_REPLICATION_COMPULSORILY;
		break;
	case OPT_DUMP:
		req.kind = L7VS_CONFIG_REPLICATION_DUMP;
		break;
	default:
		break;
	}

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_cnt = 1;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_cnt = 1;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,25,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,8,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}
	ret = 0;

END:
	destroy_socket();

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,27,
			"out_function : static int operate_replication"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Operate Logger command.
 * @return	int	result OK=0, NG=<0
 */
static int
operate_log(void)
{
	struct l7vs_config_msg_log req;
	struct l7vs_config_msg_log rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,28,
			"in_function : static int operate_log(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	ret = create_socket();
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,26,
			"Create socket error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,29,
				"out_function : static int operate_log"
				"(void) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	memset(&req, 0, sizeof(struct l7vs_config_msg_log));
	memset(&rsp, 0, sizeof(struct l7vs_config_msg_log));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	if (l7vsadm_option_data.category_all_flag == 1) {
		req.cmd = L7VS_CONFIG_SET_LOG_LEVEL_ALL;
	} else {
		req.cmd = L7VS_CONFIG_SET_LOG_LEVEL;
	}
	req.kind = l7vsadm_option_data.log_category;
	req.vsd_log_level = l7vsadm_option_data.log_level;

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_cnt = 1;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_cnt = 1;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,27,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,9,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}
	ret = 0;

END:
	destroy_socket();

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,30,
			"out_function : static int operate_log"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Operate SNMPAgent command.
 * @return	int	result OK=0, NG=<0
 */
static int
operate_snmp(void)
{
	struct l7vs_config_msg_snmp req;
	struct l7vs_config_msg_snmp rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,31,
			"in_function : static int operate_snmp(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	ret = create_socket();
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,28,
			"Create socket error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,32,
				"out_function : static int operate_snmp"
				"(void) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	memset(&req, 0, sizeof(struct l7vs_config_msg_snmp));
	memset(&rsp, 0, sizeof(struct l7vs_config_msg_snmp));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	if (l7vsadm_option_data.category_all_flag == 1) {
		req.cmd = L7VS_CONFIG_SET_SNMP_LOG_LEVEL_ALL;
	} else {
		req.cmd = L7VS_CONFIG_SET_SNMP_LOG_LEVEL;
	}
	req.kind = l7vsadm_option_data.log_category;
	req.snmp_log_level = l7vsadm_option_data.log_level;

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_cnt = 1;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_cnt = 1;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,29,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,10,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}
	ret = 0;

END:
	destroy_socket();

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,33,
			"out_function : static int operate_snmp"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Operate Parameter command.
 * @return	int	result OK=0, NG=<0
 */
static int
operate_parameter(void)
{
	struct l7vs_config_msg_parameter req;
	struct l7vs_config_msg_parameter rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,34,
			"in_function : static int operate_parameter(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	ret = create_socket();
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,30,
			"Create socket error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,35,
				"out_function : static int operate_parameter"
				"(void) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	memset(&req, 0, sizeof(struct l7vs_config_msg_parameter));
	memset(&rsp, 0, sizeof(struct l7vs_config_msg_parameter));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	req.cmd = L7VS_CONFIG_PARAMETER;
	req.category = l7vsadm_option_data.reload_param;

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_cnt = 1;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_cnt = 1;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,31,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,11,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}
	ret = 0;

END:
	destroy_socket();

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,36,
			"out_function : static int operate_parameter"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Operate help command.
 * @return	int	result OK=0
 */
static int
operate_help(void)
{
	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,37,
			"in_function : static int operate_help(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	usage(stdout);

	fprintf(stdout,
	"Commands:\n"
	"  --add-service     -A        add virtual service with options\n"
	"  --edit-service    -E        edit virtual service with options\n"
	"  --delete-service  -D        delete virtual service with options\n"
	"  --flush           -C        flush virtual service\n"
	"  --add-server      -a        add real server with options\n"
	"  --edit-server     -e        edit real server with options\n"
	"  --delete-server   -d        delete real server with options\n"
	"  --replication     -R        control replication-function\n"
	"  --log             -L        control logger-function\n"
	"  --snmp            -S        control SNMP Agent-function\n"
	"  --parameter       -P        control parameter-function\n"
	"  --list            -l        list the table\n"
	"  --verbose         -V        list the table in verbose format\n"
	"  --key             -K        list the table in key setting format\n"
	"  --help            -h        show usage\n"
	"\n");

	fprintf(stdout,
	"Options:\n"
	"  --tcp-service   -t service-address     service-address is host:port\n"
	"  --proto-module  -m proto-module        protocol module name and module argment\n"
	"                     [module-args]\n"
	"  --scheduler     -s scheduler           one of rr,lc,wrr\n"
	"  --upper         -u connection-count    maximum number of connections\n"
	"  --bypass        -b sorry-server        sorry server address is host:port\n"
	"  --flag          -f sorry-flag          sorry status set to virtual service\n"
	"  --qos-up        -Q QoSval-up           QoS Threshold(bps) set to real server direction\n"
	"  --qos-down      -q QoSval-down         QoS Threshold(bps) set to client direction\n"
	"  --real-server   -r server-address      server-address is host:port\n"
	"  --weight        -w weight              scheduling weight set to real server\n"
	"  --switch        -s replication-switch  start or stop replication\n"
	"  --force         -f                     force replication start\n"
	"  --dump          -d                     dump replication memory\n"
	"  --category      -c log-category        set log category for l7vsd or SNMP Agent\n"
	"  --level         -l log-level           set log level for l7vsd or SNMP Agent\n"
	"  --reload        -r reload-parameter    reload specified config parameter\n"
	"  --numeric       -n                     list the table in numeric\n"
	"\n");

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,38,
			"out_function : static int operate_help"
			"(void) : "
			"return = 0");
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * List Replication information.
 * @return	int	list result OK=0, NG=-1
 */
static int
list_replication_info(void)
{
	struct l7vs_config_msg_replication req;
	struct l7vs_config_msg_replication rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;
	char *mode = NULL;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,39,
			"in_function : static int list_replication_info(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	memset(&req, 0, sizeof(struct l7vs_config_msg_replication));
	memset(&rsp, 0, sizeof(struct l7vs_config_msg_replication));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	req.cmd = L7VS_CONFIG_GET_REPLICATION_INFO;

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_cnt = 1;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_cnt = 1;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,32,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,12,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}

	// Convert Replication mode to string.
	switch (rsp.replication_mode) {
	case REPLICATION_OUT:
		mode = "OUT";
		break;
	case REPLICATION_SINGLE:
		mode = "SINGLE";
		break;
	case REPLICATION_MASTER:
		mode = "MASTER";
		break;
	case REPLICATION_SLAVE:
		mode = "SLAVE";
		break;
	case REPLICATION_MASTER_STOP:
		mode = "MASTER_STOP";
		break;
	case REPLICATION_SLAVE_STOP:
		mode = "SLAVE_STOP";
		break;
	default:
		break;
	}

	// Print Replication info.
	printf("Replication Mode:\n");
	if (mode != NULL) {
		printf("%s\n", mode);
	} else {
		// Mode is NULL.
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_OPERATE,2,
			"Invalid Replication Mode.");
	}
	printf("\n");

	ret = 0;

END:
	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,40,
			"out_function : static int list_replication_info"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * List Logger information.
 * @return	int	list result OK=0, NG=-1
 */
static int
list_log_info(void)
{
	struct l7vs_config_msg_log req;
	struct l7vs_config_msg_log rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;
	struct l7vsadm_category_name_list *cat_list;
	struct l7vsadm_level_name_list *lv_list;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,41,
			"in_function : static int list_log_info(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	// Print Logger info header.
	printf("L7vsd Log Level:\n");
	printf("Category                       Level\n");

	// Get and List all category log level.
	for (cat_list = l7vsd_category_name_table;
	     cat_list->category_name != NULL;
	     cat_list++) {
		memset(&req, 0, sizeof(struct l7vs_config_msg_log));
		memset(&rsp, 0, sizeof(struct l7vs_config_msg_log));
		memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

		req.cmd = L7VS_CONFIG_GET_LOG_INFO;
		req.kind = cat_list->category_code;

		// Set data for request
		iov_data.req_iov[0].iov_base = &req;
		iov_data.req_iov[0].iov_len = sizeof(req);
		iov_data.req_cnt = 1;

		// Set data for response
		iov_data.rsp_iov[0].iov_base = &rsp;
		iov_data.rsp_iov[0].iov_len = sizeof(rsp);
		iov_data.rsp_cnt = 1;

		ret = communicate_to_l7vsd(&iov_data);
		if (ret < 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,33,
				"Communicate to l7vsd error.");
			goto END;
		}

		if (rsp.code != 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,13,
				"%s.", config_strerror(rsp.code));
			fprintf(stderr, "CONFIG ERROR : "
				"%s.\n", config_strerror(rsp.code));
			ret = -1;
			goto END;
		}

		// Convert log level to string.
		for (lv_list = level_name_table;
		     lv_list->level_name != NULL;
		     lv_list++) {
			if (rsp.vsd_log_level == lv_list->level_code) {
				printf("%-30s %s\n",
					cat_list->category_name,
					lv_list->level_name);
				break;
			}
		}
	}
	printf("\n");

	ret = 0;

END:
	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,42,
			"out_function : static int list_log_info"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * List SNMPAgent information.
 * @return	int	list result OK=0, NG=-1
 */
static int
list_snmp_info(void)
{
	struct l7vs_config_msg_snmp req;
	struct l7vs_config_msg_snmp rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;
	struct l7vsadm_category_name_list *cat_list;
	struct l7vsadm_level_name_list *lv_list;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,43,
			"in_function : static int list_snmp_info(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	memset(&req, 0, sizeof(struct l7vs_config_msg_snmp));
	memset(&rsp, 0, sizeof(struct l7vs_config_msg_snmp));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	req.cmd = L7VS_CONFIG_GET_SNMP_CONNECT_STATUS;

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_cnt = 1;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_cnt = 1;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,34,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,14,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}

	// Print SNMP connection status header.
	printf("SNMPAgent Connection Status:\n");
	// Print SNMP connection status.
	if (rsp.connection_status == 1) {
		printf("connecting\n");
	} else {
		printf("non-connecting\n");
	}
	printf("\n");

	// Print SNMPAgent Log header.
	printf("SNMPAgent Log Level:\n");
	printf("Category                       Level\n");

	// Get and List all category SNMPAgent log level.
	for (cat_list = snmp_category_name_table;
	     cat_list->category_name != NULL;
	     cat_list++) {
		memset(&req, 0, sizeof(struct l7vs_config_msg_snmp));
		memset(&rsp, 0, sizeof(struct l7vs_config_msg_snmp));
		memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

		req.cmd = L7VS_CONFIG_GET_SNMP_LOG_INFO;
		req.kind = cat_list->category_code;

		// Set data for request
		iov_data.req_iov[0].iov_base = &req;
		iov_data.req_iov[0].iov_len = sizeof(req);
		iov_data.req_cnt = 1;

		// Set data for response
		iov_data.rsp_iov[0].iov_base = &rsp;
		iov_data.rsp_iov[0].iov_len = sizeof(rsp);
		iov_data.rsp_cnt = 1;

		ret = communicate_to_l7vsd(&iov_data);
		if (ret < 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,35,
				"Communicate to l7vsd error.");
			goto END;
		}

		if (rsp.code != 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,15,
				"%s.", config_strerror(rsp.code));
			fprintf(stderr, "CONFIG ERROR : "
				"%s.\n", config_strerror(rsp.code));
			ret = -1;
			goto END;
		}

		// Convert log level to string.
		for (lv_list = level_name_table;
		     lv_list->level_name != NULL;
		     lv_list++) {
			if (rsp.snmp_log_level == lv_list->level_code) {
				printf("%-30s %s\n",
					cat_list->category_name,
					lv_list->level_name);
				break;
			}
		}
	}
	printf("\n");

	ret = 0;

END:
	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,44,
			"out_function : static int list_snmp_info"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Get list of VirtualService information.
 * @param[in,out]	*vs_buffer	VirtualService list buffer pointer
 * @param[in]		vs_bufsize	size of vs_buffer
 * @return		int		number of VirtualService, NG<0
 */
static int
get_vs(void *vs_buffer, int vs_bufsize)
{
	struct l7vs_config_req_list_vs req;
	struct l7vs_config_rsp_list_vs rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;

	// Argment check.
	if (vs_buffer == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,36,
			"Argument vs_buffer is NULL.");
		fprintf(stderr, "OPERATE ERROR : "
			"Argument vs_buffer is NULL.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,45,
				"out_function : static int get_vs"
				"(void *vs_buffer, int vs_bufsize) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,46,
			"in_function : static int get_vs"
			"(void *vs_buffer, int vs_bufsize) : "
			"vs_buffer = %p, vs_bufsize = %d",
			vs_buffer,
			vs_bufsize);
	}
	/*------ DEBUG LOG END ------*/

	memset(&req, 0, sizeof(struct l7vs_config_req_list_vs));
	memset(&rsp, 0, sizeof(struct l7vs_config_rsp_list_vs));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	req.cmd = L7VS_CONFIG_LIST_VS;

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_cnt = 1;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_iov[1].iov_base = vs_buffer;
	iov_data.rsp_iov[1].iov_len = vs_bufsize;
	iov_data.rsp_cnt = 2;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,37,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,16,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}
	ret = rsp.num;

END:
	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,47,
			"out_function : static int get_vs"
			"(void *vs_buffer, int vs_bufsize) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Get list of RealServer information in the VirtualService.
 * @param[in,out]	*darg_buffer	l7vs_dest_arg list pointer
 * @param[in]		rs_num		number of l7vs_dest_arg
 * @param[in]		*sarg		l7vs_service_arg pointer
 * @return		int		number of RealServer, NG<0
 */
static int
get_rs(struct l7vs_dest_arg *darg_buffer, int rs_num, struct l7vs_service_arg *sarg)
{
	struct l7vs_config_req_list_rs req;
	struct l7vs_config_rsp_list_rs rsp;
	struct l7vsadm_iov_data iov_data;
	int ret;

	// Argment check.
	if (darg_buffer == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,38,
			"Argument darg_buffer is NULL.");
		fprintf(stderr, "OPERATE ERROR : "
			"Argument darg_buffer is NULL.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,48,
				"out_function : static int get_rs"
				"(struct l7vs_dest_arg *darg_buffer, "
				"int rs_num, struct l7vs_service_arg *sarg) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	if (sarg == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,39,
			"Argument sarg is NULL.");
		fprintf(stderr, "OPERATE ERROR : "
			"Argument sarg is NULL.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,49,
				"out_function : static int get_rs"
				"(struct l7vs_dest_arg *darg_buffer, "
				"int rs_num, struct l7vs_service_arg *sarg) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		char sarg_str[DEBUG_STR_LEN] = {0};
		l7vs_service_arg_c_str(sarg_str, sarg);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,50,
			"in_function : static int get_rs"
			"(struct l7vs_dest_arg *darg_buffer, "
			"int rs_num, "
			"struct l7vs_service_arg *sarg) : "
			"darg_buffer = %p, "
			"rs_num = %d, "
			"sarg = (%s)",
			darg_buffer,
			rs_num,
			sarg_str);
	}
	/*------ DEBUG LOG END ------*/

	memset(&req, 0, sizeof(struct l7vs_config_req_list_rs));
	memset(&rsp, 0, sizeof(struct l7vs_config_rsp_list_rs));
	memset(&iov_data, 0, sizeof(struct l7vsadm_iov_data));

	req.cmd = L7VS_CONFIG_LIST_RS;

	// Set data for request
	iov_data.req_iov[0].iov_base = &req;
	iov_data.req_iov[0].iov_len = sizeof(req);
	iov_data.req_iov[1].iov_base = sarg;
	iov_data.req_iov[1].iov_len = sizeof(struct l7vs_service_arg_multi);
	iov_data.req_cnt = 2;

	// Set data for response
	iov_data.rsp_iov[0].iov_base = &rsp;
	iov_data.rsp_iov[0].iov_len = sizeof(rsp);
	iov_data.rsp_iov[1].iov_base = darg_buffer;
	iov_data.rsp_iov[1].iov_len = rs_num * sizeof(struct l7vs_dest_arg);
	iov_data.rsp_cnt = 2;

	ret = communicate_to_l7vsd(&iov_data);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,40,
			"Communicate to l7vsd error.");
		goto END;
	}

	if (rsp.code != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_CONFIG_RESULT,17,
			"%s.", config_strerror(rsp.code));
		fprintf(stderr, "CONFIG ERROR : "
			"%s.\n", config_strerror(rsp.code));
		ret = -1;
		goto END;
	}
	ret = rsp.num;

END:
	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,51,
			"out_function : static int get_rs"
			"(struct l7vs_dest_arg *darg_buffer, "
			"int rs_num, struct l7vs_service_arg *sarg) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * List one of VirtualService information.
 * @param[in]	*sarg	l7vs_service_arg pointer
 * @return	int	list result OK=0, NG=-1
 */
static int
list_vs(struct l7vs_service_arg *sarg)
{
	struct sockaddr_in vs_addr_0;
	struct sockaddr_in sorry_addr_in_0;
	struct l7vs_dest_arg *darg_buffer;
	char *endpoint_name;
	int rs_num = L7VSADM_DEFAULT_LIST_RS_RSNUM;
	int ret_num;
	int ret;
	int i;

	// Argment check.
	if (sarg == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,41,
			"Argument sarg is NULL.");
		fprintf(stderr, "OPERATE ERROR : "
			"Argument sarg is NULL.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,52,
				"out_function : static int list_vs"
				"(struct l7vs_service_arg *sarg) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		char sarg_str[DEBUG_STR_LEN] = {0};
		l7vs_service_arg_c_str(sarg_str, sarg);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,53,
			"in_function : static int list_vs"
			"(struct l7vs_service_arg *sarg) : "
			"sarg = (%s)",
			sarg_str);
	}
	/*------ DEBUG LOG END ------*/

	// Check VirtualService address.(Compare vs_addr_0)
	memset(&vs_addr_0, 0, sizeof(struct sockaddr_in));
	if (memcmp(&sarg->addr, &vs_addr_0, sizeof(struct sockaddr_in)) == 0) {
		// sarg->addr not set.
		endpoint_name = NULL;
	} else {
		endpoint_name = 
			get_endpoint_name((struct sockaddr_storage *)(&sarg->addr));
	}
	if (endpoint_name == NULL) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_OPERATE,3,
			"Invalid VirtualService address for list_vs.");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,54,
				"out_function : static int list_vs"
				"(struct l7vs_service_arg *sarg) : "
				"return = 0");
		}
		/*------ DEBUG LOG END ------*/

		return 0;
	}

	// If it ie verbose, display more details else simple details.
	if (l7vsadm_option_data.command_code == CMD_LIST_VERBOSE) {
		printf("TCP %s %s %s %d %s\n",
			endpoint_name,
			sarg->protomod,
			sarg->schedmod,
			sarg->reschedule,
			sarg->protomod_opt_string);
	} else if (l7vsadm_option_data.command_code == CMD_LIST_KEY) {
		printf("TCP %s %s %s %d %s\n",
			endpoint_name,
			sarg->protomod,
			sarg->schedmod,
			sarg->reschedule,
			sarg->protomod_key_string);
	} else {
		printf("TCP %s %s %s\n",
			endpoint_name,
			sarg->protomod,
			sarg->schedmod);
	}

	// Print sorry and QoS information when L7VSADM_OP_VERBOSE.
	if (l7vsadm_option_data.command_code == CMD_LIST_VERBOSE) {
		// Check sorry address.(Compare sorry_addr_0)
		memset(&sorry_addr_in_0, 0, sizeof(struct sockaddr_in));
		if (memcmp(&sarg->sorry_addr,
			   &sorry_addr_in_0,
			   sizeof(struct sockaddr_in)) == 0) {
			// sarg->sorry_addr not set.
			endpoint_name = NULL;
		} else {
			endpoint_name = 
				get_endpoint_name((struct sockaddr_storage *)(&sarg->sorry_addr));
		}
		// If endpoint_name is NULL then string is "none".
		if (endpoint_name == NULL) {
			endpoint_name = "none";
		}
		printf("    %s %d %d\n",
			endpoint_name,
			sarg->sorry_cc,
			sarg->sorry_flag);
		// QoS value and throughput convert from byte/s to bps.
		printf("    %lld %lld\n",
			sarg->qos_threshold_up * 8,
			sarg->throughput_to_server * 8);
		printf("    %lld %lld\n",
			sarg->qos_threshold_down * 8,
			sarg->throughput_to_client * 8);
	}

RETRY:
	// Alloc RealServer list memory.
	darg_buffer = (struct l7vs_dest_arg *)malloc(rs_num * sizeof(struct l7vs_dest_arg));

	/*-------- DEBUG LOG --------*/
	if (log_level_common == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,4,
			"malloc darg_buffer : "
			"allocate address = %p "
			"allocate size = %d",
			darg_buffer,
			int(rs_num * sizeof(struct l7vs_dest_arg)));
	}
	/*------ DEBUG LOG END ------*/

	if (darg_buffer == NULL) {
		LOGGER_PUT_LOG_FATAL(LOG_CAT_L7VSADM_COMMON,5,
			"Could not allocate darg_buffer memory.");
		fprintf(stderr, "COMMON ERROR : "
			"Could not allocate darg_buffer memory.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,55,
				"out_function : static int list_vs"
				"(struct l7vs_service_arg *sarg) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	// Get all RealServer list on VirtualService.
	ret_num = get_rs(darg_buffer, rs_num, sarg);
	// If return L7VSADM_RECV_TOO_LARGE (-2) when extend darg_buffer size and retry.
	if (ret_num < 0) {
		free(darg_buffer);

		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,5,
				"free darg_buffer : "
				"free address = %p",
				darg_buffer);
		}
		/*------ DEBUG LOG END ------*/

		if (ret_num == L7VSADM_RECV_TOO_LARGE) {
			rs_num += L7VSADM_DEFAULT_LIST_RS_RSNUM;
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_OPERATE,4,
				"Receive RealServer data too large. retrying.");
			goto RETRY;
		}
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,42,
			"Get RealServer data error.");
		fprintf(stderr, "OPERATE ERROR : "
			"Get RealServer data error.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,56,
				"out_function : static int list_vs"
				"(struct l7vs_service_arg *sarg) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	// List all RealServer.
	for (i = 0; i < ret_num; i++) {
		ret = list_rs(&darg_buffer[i]);
		if (ret < 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,43,
				"List RealServer data error.");
			fprintf(stderr, "OPERATE ERROR : "
				"List RealServer data error.\n");
			free(darg_buffer);

			/*-------- DEBUG LOG --------*/
			if (log_level_common == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,6,
					"free darg_buffer : "
					"free address = %p",
					darg_buffer);
			}
			/*------ DEBUG LOG END ------*/

			/*-------- DEBUG LOG --------*/
			if (log_level_operate == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,57,
					"out_function : static int list_vs"
					"(struct l7vs_service_arg *sarg) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}
	}

	free(darg_buffer);

	/*-------- DEBUG LOG --------*/
	if (log_level_common == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,7,
			"free darg_buffer : "
			"free address = %p",
			darg_buffer);
	}
	/*------ DEBUG LOG END ------*/

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,58,
			"out_function : static int list_vs"
			"(struct l7vs_service_arg *sarg) : "
			"return = 0");
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * List one of RealServer information.
 * @param[in]	*darg	l7vs_dest_arg pointer
 * @return	int	list result OK=0, NG=-1
 */
static int
list_rs(struct l7vs_dest_arg *darg)
{
	struct sockaddr_in rs_addr_0;
	char *endpoint_name;

	// Argment check.
	if (darg == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,44,
			"Argument darg is NULL.");
		fprintf(stderr, "OPERATE ERROR : "
			"Argument darg is NULL.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,59,
				"out_function : static int list_rs"
				"(struct l7vs_dest_arg *darg) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		char darg_str[DEBUG_STR_LEN] = {0};
		l7vs_darg_c_str(darg_str, darg);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,60,
			"in_function : static int list_rs"
			"(struct l7vs_dest_arg *darg) : "
			"darg = (%s)",
			darg_str);
	}
	/*------ DEBUG LOG END ------*/

	// Check RealServer address.(Compare rs_addr_0)
	memset(&rs_addr_0, 0, sizeof(struct sockaddr_in));
	if (memcmp(&darg->addr, &rs_addr_0, sizeof(struct sockaddr_in)) == 0) {
		// darg.addr not set.
		endpoint_name = NULL;
	} else {
		endpoint_name = 
			get_endpoint_name((struct sockaddr_storage *)(&darg->addr));
	}
	if (endpoint_name == NULL) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_OPERATE,5,
			"Invalid RealServer address for list_rs.");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,61,
				"out_function : static int list_rs"
				"(struct l7vs_dest_arg *darg) : "
				"return = 0");
		}
		/*------ DEBUG LOG END ------*/

		return 0;
	}

	// Display RealServer.
	printf("  -> %-28s %-7s %-6d %-10d %-10d\n",
		endpoint_name, "Masq", darg->weight, darg->nactive, darg->ninact);

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,62,
			"out_function : static int list_rs"
			"(struct l7vs_dest_arg *darg) : "
			"return = 0");
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * Get endpoint name string.
 * Translate form address to endpoint string(hostname:portname)
 * @param[in]	*addr	endpoint data
 * @return	char *	endpoint string, NG=NULL
 */
static char *
get_endpoint_name(struct sockaddr_storage *addr)
{
	char hostname[NI_MAXHOST], portname[NI_MAXSERV];
	int flags;
	int ret;

	// Argment check.
	if (addr == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,45,
			"Argument addr is NULL.");
		fprintf(stderr, "OPERATE ERROR : "
			"Argument addr is NULL.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,63,
				"out_function : static char *get_endpoint_name"
				"(struct sockaddr_storage *addr) : "
				"return = NULL");
		}
		/*------ DEBUG LOG END ------*/

		return NULL;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		char ep_str[DEBUG_STR_LEN] = {0};
		endpoint_c_str(ep_str, (void *)addr);
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,64,
			"in_function : static char *get_endpoint_name"
			"(struct sockaddr_storage *addr) : "
			"addr = (%s)",
			ep_str);
	}
	/*------ DEBUG LOG END ------*/


	if (l7vsadm_option_data.isnumeric != 0 && 
	    l7vsadm_option_data.isnumeric != 1) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,46,
			"Invalid isnumeric value.");
		fprintf(stderr, "OPERATE ERROR : "
			"Invalid isnumeric value.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,65,
				"out_function : static char *get_endpoint_name"
				"(struct sockaddr_storage *addr) : "
				"return = NULL");
		}
		/*------ DEBUG LOG END ------*/

		return NULL;
	}

	flags = 0;
	if (l7vsadm_option_data.isnumeric) {
		flags = NI_NUMERICHOST | NI_NUMERICSERV;
	}

	ret = getnameinfo((struct sockaddr *)addr, sizeof(*addr),
			  hostname, sizeof(hostname),
			  portname, sizeof(portname), flags);
	if (ret == EAI_AGAIN) {
		flags = flags | NI_NUMERICHOST;
		ret = getnameinfo((struct sockaddr *)addr, sizeof(*addr),
				hostname, sizeof(hostname),
				portname, sizeof(portname), flags);
		if (ret != 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,2,
				"getnameinfo() error: %s.", gai_strerror(ret));
			fprintf(stderr, "COMMON ERROR : "
				"getnameinfo() error: %s.\n", gai_strerror(ret));

			/*-------- DEBUG LOG --------*/
			if (log_level_operate == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,66,
					"out_function : static char *get_endpoint_name"
					"(struct sockaddr_storage *addr) : "
					"return = NULL");
			}
			/*------ DEBUG LOG END ------*/

			return NULL;
		}
	} else if (ret != 0) {	
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,3,
			"getnameinfo() error: %s.", gai_strerror(ret));
		fprintf(stderr, "COMMON ERROR : "
			"getnameinfo() error: %s.\n", gai_strerror(ret));

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,67,
				"out_function : static char *get_endpoint_name"
				"(struct sockaddr_storage *addr) : "
				"return = NULL");
		}
		/*------ DEBUG LOG END ------*/

		return NULL;
	}
	memset(tmp_endpoint_name,0x00,sizeof(tmp_endpoint_name));
	sprintf(tmp_endpoint_name, "%s:%s", hostname, portname);

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,68,
			"out_function : static char *get_endpoint_name"
			"(struct sockaddr_storage *addr) : "
			"return = %s",
			tmp_endpoint_name);
	}
	/*------ DEBUG LOG END ------*/

	return tmp_endpoint_name;
}

/*!
 * Create socket file for communicate.
 * @return	int	create result OK=0, NG=-1
 */
static int
create_socket(void)
{
	struct sockaddr_un addr;
	char sockname[sizeof(L7VS_CONFIG_SOCK_PATH) + sizeof(L7VSADM_NAME) + 10];
	int opt;
	int ret;
	int sock_fd;
	int retry_count = 0;

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,69,
			"in_function : static int create_socket(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	sprintf(sockname, "%s/%s-%d", L7VS_CONFIG_SOCK_PATH, L7VSADM_NAME, getpid());

	unlink(sockname);
	if (sizeof(sockname) > sizeof(addr.sun_path)) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,47,
			"Internal error. socket name too long.");
		fprintf(stderr, "OPERATE ERROR : "
			"Internal error. socket name too long.\n");
		ret = -1;
		goto END;
	}

	memset(&addr, 0, sizeof(addr));
	memcpy(addr.sun_path, sockname, sizeof(sockname));
	addr.sun_family = AF_LOCAL;

	sock_fd = socket(PF_LOCAL, SOCK_DGRAM, 0);
	if (sock_fd < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,4,
			"socket() error: %s.", strerror(errno));
		fprintf(stderr, "COMMON ERROR : "
			"socket() error: %s.\n", strerror(errno));
		ret = sock_fd;
		goto END;
	}

	opt = 1;
	ret = setsockopt(sock_fd, SOL_SOCKET, SO_PASSCRED, &opt, sizeof(opt));
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,5,
			"setsockopt() SO_PASSCRED error: %s.", strerror(errno));
		fprintf(stderr, "COMMON ERROR : "
			"setsockopt() SO_PASSCRED error: %s.\n", strerror(errno));
		goto END;
	}

	ret = bind(sock_fd, (struct sockaddr *)&addr, SUN_LEN(&addr));
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,6,
			"bind() on %s error: %s.", sockname, strerror(errno));
		fprintf(stderr, "COMMON ERROR : "
			"bind() on %s error: %s.\n", sockname, strerror(errno));
		goto END;
	}

	strcpy(local_sockname, sockname);

	memset(&addr, 0, sizeof(addr));
	addr.sun_family = AF_LOCAL;
	strcpy(addr.sun_path, L7VS_CONFIG_SOCKNAME);

	while (1) {
		// Check signal.
		if (signal_flg > 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,7,
				"Signal (%d) Received.", received_sig);
			fprintf(stderr, "COMMON ERROR : "
				"Signal (%d) Received.\n", received_sig);
			ret = -1;
			goto END;
		}

		// Try connect to config socket.
		ret = connect(sock_fd, (struct sockaddr *)&addr, SUN_LEN(&addr));
		if (ret == 0) {
			// Connect OK.
			break;
		}
		retry_count++;
		if (retry_count > connect_wait_count) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,8,
				"connect() to daemon timeout: %s.", strerror(errno));
			fprintf(stderr, "COMMON ERROR : "
				"connect() to daemon timeout: %s.\n", strerror(errno));
			goto END;
		}
		// Connect retrying.
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_COMMON,1,
			"connect() to daemon error (retry %d): %s.",
			retry_count, strerror(errno));
		sleep(connect_wait_interval);
	}

	// Save socket file discripter.
	local_sock_fd = sock_fd;
	ret = 0;

END:
	if (ret < 0) {
		if (sock_fd != -1) {
			close(sock_fd);
		}
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,70,
			"out_function : static int create_socket"
			"(void) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Destroy socket file for communicate.
 */
static void
destroy_socket(void)
{
	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,71,
			"in_function : static void destroy_socket(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	unlink(local_sockname);

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,72,
			"out_function : static void destroy_socket(void) : ");
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Communicate to l7vsd daemon.
 * Send request and receive response between l7vsd daemon
 * @param[in,out]	*iov_data	io-vector data pointer
 * @return		int		communicate result OK=byte, NG=-1
 */
static int
communicate_to_l7vsd(struct l7vsadm_iov_data *iov_data)
{
	int ret;

	// Argment check.
	if (iov_data == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMUNICATE,1,
			"Argument iov_data is NULL.");
		fprintf(stderr, "COMMUNICATE ERROR : "
			"Argument iov_data is NULL.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_communicate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,1,
				"out_function : static int communicate_to_l7vsd"
				"(struct l7vsadm_iov_data *iov_data) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_communicate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,2,
			"in_function : static int communicate_to_l7vsd"
			"(struct l7vsadm_iov_data *iov_data) : "
			"iov_data = %p",
			iov_data);
	}
	/*------ DEBUG LOG END ------*/

	// Check signal.
	if (signal_flg > 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,9,
			"Signal (%d) Received.", received_sig);
		fprintf(stderr, "COMMON ERROR : "
			"Signal (%d) Received.\n", received_sig);

		/*-------- DEBUG LOG --------*/
		if (log_level_communicate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,3,
				"out_function : static int communicate_to_l7vsd"
				"(struct l7vsadm_iov_data *iov_data) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	ret = send_request(iov_data->req_iov, iov_data->req_cnt);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMUNICATE,2,
			"Send Request error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_communicate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,4,
				"out_function : static int communicate_to_l7vsd"
				"(struct l7vsadm_iov_data *iov_data) : "
				"return = %d",
				ret);
		}
		/*------ DEBUG LOG END ------*/

		return ret;
	}

	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSADM_COMMUNICATE,1,
		"Send Request success.");

	// Check signal.
	if (signal_flg > 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,10,
			"Signal (%d) Received.", received_sig);
		fprintf(stderr, "COMMON ERROR : "
			"Signal (%d) Received.\n", received_sig);

		/*-------- DEBUG LOG --------*/
		if (log_level_communicate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,5,
				"out_function : static int communicate_to_l7vsd"
				"(struct l7vsadm_iov_data *iov_data) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	ret = recv_response(iov_data->rsp_iov, iov_data->rsp_cnt);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMUNICATE,3,
			"Receive Response error.");

		/*-------- DEBUG LOG --------*/
		if (log_level_communicate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,6,
				"out_function : static int communicate_to_l7vsd"
				"(struct l7vsadm_iov_data *iov_data) : "
				"return = %d",
				ret);
		}
		/*------ DEBUG LOG END ------*/

		return ret;
	}

	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSADM_COMMUNICATE,2,
		"Receive Response success.");

	/*-------- DEBUG LOG --------*/
	if (log_level_communicate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,7,
			"out_function : static int communicate_to_l7vsd"
			"(struct l7vsadm_iov_data *iov_data) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Send config request message.
 * @param[in]	*req_iov	request io-vector pointer
 * @param[in]	req_cnt		request io-vector count
 * @return	int		send result OK=byte, NG<0
 */
static int
send_request(struct iovec *req_iov, int req_cnt)
{
	int ret;

	// Argment check.
	if (req_iov == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMUNICATE,4,
			"Argument req_iov is NULL.");
		fprintf(stderr, "COMMUNICATE ERROR : "
			"Argument req_iov is NULL.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_communicate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,8,
				"out_function : static int send_request"
				"(struct iovec *req_iov, int req_cnt) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	if (req_cnt < 1) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMUNICATE,5,
			"Invalid req_cnt argument.");
		fprintf(stderr, "COMMUNICATE ERROR : "
			"Invalid req_cnt argument.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_communicate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,9,
				"out_function : static int send_request"
				"(struct iovec *req_iov, int req_cnt) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_communicate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,10,
			"in_function : static int send_request"
			"(struct iovec *req_iov, int req_cnt) : "
			"req_iov = %p "
			"req_cnt = %d",
			req_iov,
			req_cnt);
	}
	/*------ DEBUG LOG END ------*/

	if (req_cnt == 1) {
		if (req_iov->iov_base == NULL) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMUNICATE,6,
				"iov_base is NULL.");

			/*-------- DEBUG LOG --------*/
			if (log_level_communicate == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,11,
					"out_function : static int send_request"
					"(struct iovec *req_iov, int req_cnt) : "
					"return = -1");
			}
			/*------ DEBUG LOG END ------*/

			return -1;
		}
		ret = send(local_sock_fd,
			   req_iov->iov_base,
			   req_iov->iov_len,
			   0);
		if (ret < 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMUNICATE,7,
				"send() to daemon error: %s.", strerror(errno));
			fprintf(stderr, "COMMUNICATE ERROR : "
				"send() to daemon error: %s.\n", strerror(errno));
		}
	} else {
		ret = writev(local_sock_fd, req_iov, req_cnt);
		if (ret < 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMUNICATE,8,
				"writev() to daemon error: %s.", strerror(errno));
			fprintf(stderr, "COMMUNICATE ERROR : "
				"writev() to daemon error: %s.\n", strerror(errno));
		}
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_communicate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,12,
			"out_function : static int send_request"
			"(struct iovec *req_iov, int req_cnt) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Receive config response message.
 * @param[in,out]	*rsp_iov	response io-vector pointer
 * @param[in]		rsp_cnt		response io-vector count
 * @return		int		receive result OK=byte, NG<0
 */
static int
recv_response(struct iovec *rsp_iov, int rsp_cnt)
{
	struct msghdr msg;
	struct ucred *cred;
	struct cmsghdr *cmsg;
	unsigned char cbuf[CMSG_LEN(sizeof(struct ucred))];
	int ret;

	// Argment check.
	if (rsp_iov == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMUNICATE,9,
			"Argument rsp_iov is NULL.");
		fprintf(stderr, "COMMUNICATE ERROR : "
			"Argument rsp_iov is NULL.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_communicate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,13,
				"out_function : static int recv_response"
				"(struct iovec *rsp_iov, int rsp_cnt) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}
	if (rsp_cnt < 1) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMUNICATE,10,
			"Invalid rsp_cnt argument.");
		fprintf(stderr, "COMMUNICATE ERROR : "
			"Invalid rsp_cnt argument.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_communicate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,14,
				"out_function : static int recv_response"
				"(struct iovec *rsp_iov, int rsp_cnt) : "
				"return = -1");
		}
		/*------ DEBUG LOG END ------*/

		return -1;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_communicate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,15,
			"in_function : static int recv_response"
			"(struct iovec *rsp_iov, int rsp_cnt) : "
			"rsp_iov = %p "
			"rsp_cnt = %d",
			rsp_iov,
			rsp_cnt);
	}
	/*------ DEBUG LOG END ------*/

	memset(&msg, 0, sizeof(msg));
	msg.msg_control = cbuf;
	msg.msg_controllen = sizeof(cbuf);
	msg.msg_iov = rsp_iov;
	msg.msg_iovlen = (size_t)rsp_cnt;
	cmsg = (struct cmsghdr *)cbuf;

	ret = recvmsg(local_sock_fd, &msg, 0);
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMUNICATE,11,
			"recvmsg() to daemon error: %s.", strerror(errno));
		fprintf(stderr, "COMMUNICATE ERROR : "
			"recvmsg() to daemon error: %s.\n", strerror(errno));

		/*-------- DEBUG LOG --------*/
		if (log_level_communicate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,16,
				"out_function : static int recv_response"
				"(struct iovec *rsp_iov, int rsp_cnt) : "
				"return = L7VSADM_RECV_FAILED(-1)");
		}
		/*------ DEBUG LOG END ------*/

		return L7VSADM_RECV_FAILED;
	}

	if (msg.msg_flags & MSG_CTRUNC) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMUNICATE,12,
			"Invalid response from l7vsd.");
		fprintf(stderr, "COMMUNICATE ERROR : "
			"Invalid response from l7vsd.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_communicate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,17,
				"out_function : static int recv_response"
				"(struct iovec *rsp_iov, int rsp_cnt) : "
				"return = L7VSADM_RECV_FAILED(-1)");
		}
		/*------ DEBUG LOG END ------*/

		return L7VSADM_RECV_FAILED;
	}

	if (msg.msg_flags & MSG_TRUNC) {
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_COMMUNICATE,1,
			"Too large response from l7vsd.");

		/*-------- DEBUG LOG --------*/
		if (log_level_communicate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,18,
				"out_function : static int recv_response"
				"(struct iovec *rsp_iov, int rsp_cnt) : "
				"return = L7VSADM_RECV_TOO_LARGE(-2)");
		}
		/*------ DEBUG LOG END ------*/

		return L7VSADM_RECV_TOO_LARGE;
	}

	if (! (cmsg->cmsg_level == SOL_SOCKET &&
		cmsg->cmsg_type == SCM_CREDENTIALS)) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMUNICATE,13,
			"Could not receive a remote credential.");
		fprintf(stderr, "COMMUNICATE ERROR : "
			"Could not receive a remote credential.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_communicate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,19,
				"out_function : static int recv_response"
				"(struct iovec *rsp_iov, int rsp_cnt) : "
				"return = L7VSADM_RECV_FAILED(-1)");
		}
		/*------ DEBUG LOG END ------*/

		return L7VSADM_RECV_FAILED;
	}

	cred = (struct ucred *)CMSG_DATA(cmsg);
	if (cred->uid != 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMUNICATE,14,
			"Response from unprivileged user.");
		fprintf(stderr, "COMMUNICATE ERROR : "
			"Response from unprivileged user.\n");
		return L7VSADM_RECV_FAILED;

		/*-------- DEBUG LOG --------*/
		if (log_level_communicate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,20,
				"out_function : static int recv_response"
				"(struct iovec *rsp_iov, int rsp_cnt) : "
				"return = L7VSADM_RECV_FAILED(-1)");
		}
		/*------ DEBUG LOG END ------*/

	}

	/*-------- DEBUG LOG --------*/
	if (log_level_communicate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMUNICATE,21,
			"out_function : static int recv_response"
			"(struct iovec *rsp_iov, int rsp_cnt) : "
			"return = %d",
			ret);
	}
	/*------ DEBUG LOG END ------*/

	return ret;
}

/*!
 * Print l7vsadm command usage.
 * @param[in]	*fp	outoput file pointer
 */
static void
usage(FILE *fp)
{
	// Argment check.
	if (!fp) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,48,
			"Argument fp is NULL.");
		fprintf(stderr, "OPERATE ERROR : "
			"Argument fp is NULL.\n");

		/*-------- DEBUG LOG --------*/
		if (log_level_operate == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,73,
				"out_function : static void usage(FILE *fp) : ");
		}
		/*------ DEBUG LOG END ------*/

		return;
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,74,
			"in_function : static void usage(FILE *fp) : "
			"fp = %p",
			fp);
	}
	/*------ DEBUG LOG END ------*/

	fprintf(fp,
	"Usage: \n"
	"  l7vsadm -A|E -t service-address -m proto-module [module-args]\n"
	"          [-s scheduler] [-u connection-count] [-b sorry-server]\n"
	"          [-f sorry-flag] [-Q QoSval-up] [-q QoSval-down]\n"
	"  l7vsadm -D -t service-address -m proto-module [module-args]\n"
	"  l7vsadm -C\n"
	"  l7vsadm -a|e -t service-address -m proto-module [module-args]\n"
	"          -r server-address [-w weight]\n"
	"  l7vsadm -d -t service-address -m proto-module [module-args]\n"
	"          -r server-address\n"
	"  l7vsadm -R -s replication-switch\n"
	"  l7vsadm -R -f\n"
	"  l7vsadm -R -d\n"
	"  l7vsadm -L -c log-category -l log-level\n"
	"  l7vsadm -S -c log-category -l log-level\n"
	"  l7vsadm -P -r reload-parameter\n"
	"  l7vsadm -l [-n]\n"
	"  l7vsadm -V [-n]\n"
	"  l7vsadm -K [-n]\n"
	"  l7vsadm -h\n"
	"\n");

	/*-------- DEBUG LOG --------*/
	if (log_level_operate == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_OPERATE,75,
			"out_function : static void usage(FILE *fp) : ");
	}
	/*------ DEBUG LOG END ------*/

}

/*!
 * Set l7vsadm parameter data.
 * Get l7vsadm parameter data and set to global data
 */
static void
set_parameter(void)
{
	static struct l7vsadm_parameter_list l7vsadm_parameter_table[] = {
		{PARA_CMD_INTERVAL,		"cmd_interval"},
		{PARA_CMD_COUNT,		"cmd_count"},
		{PARA_CON_INTERVAL,		"con_interval"},
		{PARA_CON_COUNT,		"con_count"},
		{PARA_NONE,			NULL}
	};
	struct l7vsadm_parameter_list *list;
	int parameter_value = 0;

	/*-------- DEBUG LOG --------*/
	if (log_level_common == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,8,
			"in_function : static void set_parameter(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	// Get and Set l7vsadm all parameter value.
	for (list = l7vsadm_parameter_table; list->parameter_name != NULL; list++) {
		parameter_value = 0;
		if (parameter_is_int_exist(L7VSADM_COMPONENT, list->parameter_name)) {
			parameter_value = 
				parameter_get_int_value(L7VSADM_COMPONENT, list->parameter_name);
		} else {
			// When parameter not exist, use default parameter value.
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_COMMON,2,
				"Get %s parameter error. Use default value.",
				list->parameter_name);
			continue;
		}
		switch (list->parameter_code) {
		case PARA_CMD_INTERVAL:
			command_wait_interval = parameter_value;
			if (command_wait_interval < 0 ||
			    command_wait_interval > L7VSADM_MAX_WAIT) {
				// When illegal parameter value, use default parameter value.
				command_wait_interval = L7VSADM_DEFAULT_WAIT_INTERVAL;
				LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_COMMON,3,
					"Illegal %s parameter value. Use default value.",
					list->parameter_name);
			}
			break;
		case PARA_CMD_COUNT:
			command_wait_count = parameter_value;
			if (command_wait_count < 0 ||
			    command_wait_count > L7VSADM_MAX_WAIT) {
				// When illegal parameter value, use default parameter value.
				command_wait_count = L7VSADM_DEFAULT_WAIT_COUNT;
				LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_COMMON,4,
					"Illegal %s parameter value. Use default value.",
					list->parameter_name);
			}
			break;
		case PARA_CON_INTERVAL:
			connect_wait_interval = parameter_value;
			if (connect_wait_interval < 0 ||
			    connect_wait_interval > L7VSADM_MAX_WAIT) {
				// When illegal parameter value, use default parameter value.
				connect_wait_interval = L7VSADM_DEFAULT_WAIT_INTERVAL;
				LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_COMMON,5,
					"Illegal %s parameter value. Use default value.",
					list->parameter_name);
			}
			break;
		case PARA_CON_COUNT:
			connect_wait_count = parameter_value;
			if (connect_wait_count < 0 ||
			    connect_wait_count > L7VSADM_MAX_WAIT) {
				// When illegal parameter value, use default parameter value.
				connect_wait_count = L7VSADM_DEFAULT_WAIT_COUNT;
				LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_COMMON,6,
					"Illegal %s parameter value. Use default value.",
					list->parameter_name);
			}
			break;
		case PARA_NONE:
		default:
			break;
		}
	}

	if ((command_wait_interval * command_wait_count) > L7VSADM_MAX_WAIT) {
		// When wait value too long, use default parameter value.
		command_wait_interval = L7VSADM_DEFAULT_WAIT_INTERVAL;
		command_wait_count = L7VSADM_DEFAULT_WAIT_COUNT;
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_COMMON,7,
			"Command wait value too long. Use default value.");
	}
	if ((connect_wait_interval * connect_wait_count) > L7VSADM_MAX_WAIT) {
		// When wait value too long, use default parameter value.
		connect_wait_interval = L7VSADM_DEFAULT_WAIT_INTERVAL;
		connect_wait_count = L7VSADM_DEFAULT_WAIT_COUNT;
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_COMMON,8,
			"Connect wait value too long. Use default value.");
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_common == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,9,
			"out_function : static void set_parameter(void) : "
			"result "
			"command_wait_interval = %d "
			"command_wait_count = %d "
			"connect_wait_interval = %d "
			"connect_wait_count = %d",
			command_wait_interval,
			command_wait_count,
			connect_wait_interval,
			connect_wait_count);
	}
	/*------ DEBUG LOG END ------*/
}

/*!
 * Create config error string.
 * Translate from esponse code to error string
 * @param[in]	rsp_code	response code
 * @param[in]	*argv[]		array of argument string
 * @return	char *		error string
 */
static char *
config_strerror(int rsp_code)
{
	/*-------- DEBUG LOG --------*/
	if (log_level_common == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,10,
			"in_function : static char *config_strerror(int rsp_code) : "
			"rsp_code = %d",
			rsp_code);
	}
	/*------ DEBUG LOG END ------*/

	switch (rsp_code) {
	case L7VS_CONFIG_NOERR:
		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,11,
				"out_function : static char *config_strerror(int rsp_code) : "
				"return = No error");
		}
		/*------ DEBUG LOG END ------*/

		return "No error";
	case L7VS_CONFIG_ERR_INVALID_COMMAND:
		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,12,
				"out_function : static char *config_strerror(int rsp_code) : "
				"return = Invalid command");
		}
		/*------ DEBUG LOG END ------*/

		return "Invalid command";
	case L7VS_CONFIG_ERR_NOMEM:
		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,13,
				"out_function : static char *config_strerror(int rsp_code) : "
				"return = Could not allocate memory");
		}
		/*------ DEBUG LOG END ------*/

		return "Could not allocate memory";
	case L7VS_CONFIG_ERR_VS_EXISTS:
		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,14,
				"out_function : static char *config_strerror(int rsp_code) : "
				"return = Virtual service already exists");
		}
		/*------ DEBUG LOG END ------*/

		return "Virtual service already exists";
	case L7VS_CONFIG_ERR_RS_EXISTS:
		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,15,
				"out_function : static char *config_strerror(int rsp_code) : "
				"return = Real server already exists");
		}
		/*------ DEBUG LOG END ------*/

		return "Real server already exists";
	case L7VS_CONFIG_ERR_NOVS:
		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,16,
				"out_function : static char *config_strerror(int rsp_code) : "
				"return = No such virtual service");
		}
		/*------ DEBUG LOG END ------*/

		return "No such virtual service";
	case L7VS_CONFIG_ERR_NORS:
		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,17,
				"out_function : static char *config_strerror(int rsp_code) : "
				"return = No such real server");
		}
		/*------ DEBUG LOG END ------*/

		return "No such real server";
	case L7VS_CONFIG_ERR_NOSCHED:
		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,18,
				"out_function : static char *config_strerror(int rsp_code) : "
				"return = No such scheduler");
		}
		/*------ DEBUG LOG END ------*/

		return "No such scheduler";
	case L7VS_CONFIG_ERR_NOSOCK:
		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,19,
				"out_function : static char *config_strerror(int rsp_code) : "
				"return = Could not create a service socket");
		}
		/*------ DEBUG LOG END ------*/

		return "Could not create a service socket";
	case L7VS_CONFIG_ERR_QOSSET:
		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,20,
				"out_function : static char *config_strerror(int rsp_code) : "
				"return = QoS threshold setting failure");
		}
		/*------ DEBUG LOG END ------*/

		return "QoS threshold setting failure";
	case L7VS_CONFIG_ERR_SORRYSET:
		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,21,
				"out_function : static char *config_strerror(int rsp_code) : "
				"return = Sorry-Server modification failure");
		}
		/*------ DEBUG LOG END ------*/

		return "Sorry-Server modification failure";
	case L7VS_CONFIG_ERR_SENDMSG:
		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,22,
				"out_function : static char *config_strerror(int rsp_code) : "
				"return = Could not send response to l7vsadm");
		}
		/*------ DEBUG LOG END ------*/

		return "Could not send response to l7vsadm";
	case L7VS_CONFIG_ERR_MAXVS_EXISTS:
		/*------ DEBUG LOG -------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,23,
				"out_function : static char *config_strerror(int rep_code) : "
				"return = l7vsd is created max virtual service" );
		}
		 /*------ DEBUG LOG END ------*/

		return "Could not create virtual service to l7vsd because max limit it.";
	default:
		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,24,
				"out_function : static char *config_strerror(int rsp_code) : "
				"return = Unknown error");
		}
		/*------ DEBUG LOG END ------*/

		return "Unknown error";
		break;
	}
}

/*!
 * Signal handling function.
 * @param[in]	sig	signal number
 */
static void
sig_exit_handler(int sig)
{
	received_sig = sig;
	++signal_flg;
}

/*!
 * Set signal handler.
 * @param[in]	argc	number of argument
 * @param[in]	*argv[]	array of argument string
 * @return	int	set result OK=0, NG=-1
 */
static int
set_signal(void)
{
	static struct l7vsadm_sig_list sigs[] = {
		{SIGHUP,	sig_exit_handler},
		{SIGINT,	sig_exit_handler},
		{SIGQUIT,	sig_exit_handler},
		{SIGPIPE,	sig_exit_handler},
		{SIGTERM,	sig_exit_handler},
		{SIGUSR1,	sig_exit_handler},
		{SIGUSR2,	sig_exit_handler},
		{0,		NULL}
	};
	struct l7vsadm_sig_list *s;
	struct sigaction act, oact;
	int ret;

	/*-------- DEBUG LOG --------*/
	if (log_level_common == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,25,
			"in_function : static int set_signal(void) : ");
	}
	/*------ DEBUG LOG END ------*/

	memset(&act, 0x00, sizeof(struct sigaction));
	memset(&oact, 0x00, sizeof(struct sigaction));

	for (s = sigs; s->sig != 0; s++) {
		ret = sigaction(s->sig, NULL, &oact);
		if (ret < 0) {
			LOGGER_PUT_LOG_FATAL(LOG_CAT_L7VSADM_COMMON,6,
				"sigaction() error: %s.", strerror(errno));
			fprintf(stderr, "COMMON ERROR : "
				"sigaction() error: %s.\n", strerror(errno));

			/*-------- DEBUG LOG --------*/
			if (log_level_common == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,26,
					"out_function : static int set_signal(void) : "
					"return = %d",
					ret);
			}
			/*------ DEBUG LOG END ------*/

			return ret;
		}

		act = oact;
		act.sa_handler = s->sig_func;
		ret = sigaction(s->sig, &act, NULL);
		if (ret < 0) {
			LOGGER_PUT_LOG_FATAL(LOG_CAT_L7VSADM_COMMON,7,
				"sigaction() error: %s.", strerror(errno));
			fprintf(stderr, "COMMON ERROR : "
				"sigaction() error: %s.\n", strerror(errno));

			/*-------- DEBUG LOG --------*/
			if (log_level_common == LOG_LV_DEBUG) {
				LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,27,
					"out_function : static int set_signal(void) : "
					"return = %d",
					ret);
			}
			/*------ DEBUG LOG END ------*/

			return ret;
		}
	}

	/*-------- DEBUG LOG --------*/
	if (log_level_common == LOG_LV_DEBUG) {
		LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,28,
			"out_function : static int set_signal(void) : "
			"return = 0");
	}
	/*------ DEBUG LOG END ------*/

	return 0;
}

/*!
 * L7vsadm command main function.
 * Parse argument and execute command
 * @param[in]	argc	number of argument
 * @param[in]	*argv[]	array of argument string
 */
void
l7vsadm_main(int argc, char *argv[])
{
	char l7vsadm_file_path[256];
	int l7vsadm_fd = -1;
	int l7vsadm_lock = -1;
	int retry_count = 0;
	int ret;
	int error_f = 0;

	// Get and Set l7vsadm all log level value.
	log_level_parse = logger_get_log_level(LOG_CAT_L7VSADM_PARSE);
	log_level_operate = logger_get_log_level(LOG_CAT_L7VSADM_OPERATE);
	log_level_communicate = logger_get_log_level(LOG_CAT_L7VSADM_COMMUNICATE);
	log_level_config_result = logger_get_log_level(LOG_CAT_L7VSADM_CONFIG_RESULT);
	log_level_common = logger_get_log_level(LOG_CAT_L7VSADM_COMMON);

	// L7vsadm command start.
	char arg_str[DEBUG_STR_LEN] = {0};
	argc_argv_c_str(arg_str, argc, argv);
	LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSADM_COMMON,1,
		"l7vsadm command start. : "
		"argc = %d, %s",
		argc,
		arg_str);

	// Set signal handler.
	ret = set_signal();
	if (ret < 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,11,
			"Set signal handler error.");
		error_f = 1;
		goto END;
	}

	// l7vsadm_option_data initialize.
	memset(&l7vsadm_option_data, 0, sizeof(struct l7vsadm_option_data));

	// Check signal.
	if (signal_flg > 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,12,
			"Signal (%d) Received.", received_sig);
		fprintf(stderr, "COMMON ERROR : "
			"Signal (%d) Received.\n", received_sig);
		error_f = 1;
		goto END;
	}

	// Set l7vsadm parameter.
	set_parameter();

	// Get l7vsadm execute file path from /proc/(pid)/exe (symbolic link)
	memset(l7vsadm_file_path, 0, sizeof(l7vsadm_file_path));
	readlink("/proc/self/exe", l7vsadm_file_path, sizeof(l7vsadm_file_path));

	// L7vsadm command conflict check. (Try l7vsadm execute file lock)
	l7vsadm_fd = open(l7vsadm_file_path, O_RDONLY);
	if (l7vsadm_fd == -1) {
		LOGGER_PUT_LOG_FATAL(LOG_CAT_L7VSADM_COMMON,8,
			"L7vsadm execute file open error. file:%s", l7vsadm_file_path);
		fprintf(stderr, "COMMON ERROR : "
			"L7vsadm execute file open error. file:%s\n", l7vsadm_file_path);
		error_f = 1;
		goto END;
	}
	while (1) {
		// Check signal.
		if (signal_flg > 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,13,
				"Signal (%d) Received.", received_sig);
			fprintf(stderr, "COMMON ERROR : "
				"Signal (%d) Received.\n", received_sig);
			error_f = 1;
			goto END;
		}

		// Try lock l7vsadm file.
		l7vsadm_lock = flock(l7vsadm_fd, LOCK_EX | LOCK_NB);
		if (l7vsadm_lock == 0) {
			// l7vsadm file lock OK.
			break;
		}
		retry_count++;
		if (retry_count > command_wait_count) {
			// L7vsadm file lock error. (l7vsadm is executing)
			LOGGER_PUT_LOG_FATAL(LOG_CAT_L7VSADM_COMMON,9,
				"L7vsadm file lock timeout. (l7vsadm is already executing)");
			fprintf(stderr, "COMMON ERROR : "
				"L7vsadm file lock timeout. (l7vsadm is already executing)\n");
			error_f = 1;
			goto END;
		}
		// Lock retrying.
		LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_COMMON,9,
			"L7vsadm file lock error. (l7vsadm is already executing) (retry %d)",
			retry_count);
		sleep(command_wait_interval);
	}

	if (argc > 1) {
		// Check signal.
		if (signal_flg > 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,14,
				"Signal (%d) Received.", received_sig);
			fprintf(stderr, "COMMON ERROR : "
				"Signal (%d) Received.\n", received_sig);
			error_f = 1;
			goto END;
		}

		// Parse command option. (1st argument)
		ret = parse_command(argc, argv);
		if (ret < 0) {
			// Command option parse NG.
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,84,
				"Command option parse error.");
			usage(stderr);
			error_f = 1;
			goto END;
		}

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			char parse_str[DEBUG_STR_LEN] = {0};
			l7vsadm_option_data_c_str(parse_str);
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,112,
				"function : void l7vsadm_main(int argc, char *argv[]) : "
				"parse_command result (%s)",
				parse_str);
		}
		/*------ DEBUG LOG END ------*/

		// Check signal.
		if (signal_flg > 0) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,15,
				"Signal (%d) Received.", received_sig);
			fprintf(stderr, "COMMON ERROR : "
				"Signal (%d) Received.\n", received_sig);
			error_f = 1;
			goto END;
		}

		// Check parse function pointer.
		if (l7vsadm_option_data.parse_func == NULL) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_PARSE,7,
				"Parse function pointer is NULL.");
			error_f = 1;
			goto END;
		}

		// Parse xxx command option. (2nd argument and later)
		ret = l7vsadm_option_data.parse_func(argc, argv);
		if (ret < 0) {
			// xxx command option parse NG.
			LOGGER_PUT_LOG_WARN(LOG_CAT_L7VSADM_PARSE,85,
				"(%s) command option parse error.", argv[1]);
			usage(stderr);
			error_f = 1;
			goto END;
		}

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			char parse_str[DEBUG_STR_LEN] = {0};
			l7vsadm_option_data_c_str(parse_str);
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,113,
				"function : void l7vsadm_main(int argc, char *argv[]) : "
				"parse_option result (%s)",
				parse_str);
		}
		/*------ DEBUG LOG END ------*/

	} else {
		// No argument. Set default command (-l).
		l7vsadm_option_data.command_code = CMD_LIST;
		l7vsadm_option_data.operate_func = operate_list;

		/*-------- DEBUG LOG --------*/
		if (log_level_parse == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_PARSE,114,
				"function : void l7vsadm_main(int argc, char *argv[]) : "
				"no option");
		}
		/*------ DEBUG LOG END ------*/

	}

	// Check signal.
	if (signal_flg > 0) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_COMMON,16,
			"Signal (%d) Received.", received_sig);
		fprintf(stderr, "COMMON ERROR : "
			"Signal (%d) Received.\n", received_sig);
		error_f = 1;
		goto END;
	}


	// Check operate function pointer.
	if (l7vsadm_option_data.operate_func == NULL) {
		LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,49,
			"Operate function pointer is NULL.");
		error_f = 1;
		goto END;
	}

	// Operate l7vsadm command.
	ret = l7vsadm_option_data.operate_func();
	if (ret < 0) {
		// xxx command operate NG.
		if (argc > 1) {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,50,
				"(%s) command operate error.", argv[1]);
		} else {
			LOGGER_PUT_LOG_ERROR(LOG_CAT_L7VSADM_OPERATE,51,
				"(-l) command operate error.");
		}
		error_f = 1;
		goto END;
	}

END:
	if (l7vsadm_lock != -1) {
		// L7vsadm fd unlock.
		flock(l7vsadm_fd, LOCK_UN);
	}

	if (l7vsadm_fd != -1) {
		// L7vsadm fd close.
		close(l7vsadm_fd);
	}

	// Error check.
	if (error_f == 1) {
		// Abnormal end.
		if (argc > 1) {
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSADM_COMMON,2,
				"L7vsadm command (%s) is failure.", argv[1]);
		} else {
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSADM_COMMON,3,
				"L7vsadm command (-l) is failure.");
		}

		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,29, 
				"out_function : void l7vsadm_main(int argc, char *argv[]) : "
				"exit = %d",
				EXIT_FAILURE);
		}
		/*------ DEBUG LOG END ------*/

		fflush(stdout);
		fsync(1);
		_exit(EXIT_FAILURE);
	} else {
		// Normal end.
		if (argc > 1) {
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSADM_COMMON,4,
				"L7vsadm command (%s) is successful.", argv[1]);
		} else {
			LOGGER_PUT_LOG_INFO(LOG_CAT_L7VSADM_COMMON,5,
				"L7vsadm command (-l) is successful.");
		}

		/*-------- DEBUG LOG --------*/
		if (log_level_common == LOG_LV_DEBUG) {
			LOGGER_PUT_LOG_DEBUG(LOG_CAT_L7VSADM_COMMON,30, 
				"out_function : void l7vsadm_main(int argc, char *argv[]) : "
				"exit = %d",
				EXIT_SUCCESS);
		}
		/*------ DEBUG LOG END ------*/

		fflush(stdout);
		fsync(1);
		_exit(EXIT_SUCCESS);
	}
}

/*!
 * Serialize endpoint for debug log.
 * @param[out]	*ep_str	endpoint string
 * @param[in] 	*addr	endpoint structure
 */
void
endpoint_c_str(char* ep_str, void *addr)
{
	struct sockaddr_in *p;

	p = (struct sockaddr_in *)addr;
	if (p == NULL) {
		sprintf(ep_str, "NULL");
	} else {
		sprintf(ep_str, "sin_family = %d", p->sin_family);
		sprintf(ep_str, "%s sin_port = %d", ep_str, p->sin_port);
		if (memcmp(&p->sin_addr, "", 1) == 0) {
			sprintf(ep_str, "%s sin_addr = (NULL)", ep_str);
		} else {
			sprintf(ep_str, "%s sin_addr = (s_addr = %d)", ep_str, p->sin_addr.s_addr);
		}
		if (p->sin_zero == NULL) {
			sprintf(ep_str, "%s sin_zero = NULL", ep_str);
		} else {
			sprintf(ep_str, "%s sin_zero = %s", ep_str, p->sin_zero);
		}
	}
}

/*!
 * Serialize argc *argv[] for debug log.
 * @param[out]	*arg_str	serialized string
 * @param[in] 	argc		argument count
 * @param[in] 	*argv[]		argument string array
 */
void
argc_argv_c_str(char* arg_str, int argc, char *argv[])
{
	int i;

	for (i = 0; i < argc; i++) {
		if (argv == NULL) {
			if (strlen(arg_str) + 20 > DEBUG_STR_LEN) {
				break;
			}
			sprintf(arg_str, "%s argv[%d] = NULL", arg_str, i);
		} else {
			if (strlen(arg_str) + strlen(argv[i]) + 16 > DEBUG_STR_LEN) {
				break;
			}
			sprintf(arg_str, "%s argv[%d] = %s", arg_str, i, argv[i]);
		}
	}
}

/*!
 * Serialize l7vsadm_option_data (global data) for debug log.
 * @param[out]	*parse_str	serialized string
 */
void
l7vsadm_option_data_c_str(char* parse_str)
{
	sprintf(parse_str, "command_code = %d",
		l7vsadm_option_data.command_code);
	sprintf(parse_str, "%s parse_func = %p", parse_str,
		l7vsadm_option_data.parse_func);
	sprintf(parse_str, "%s operate_func = %p", parse_str,
		l7vsadm_option_data.operate_func);
	if (memcmp(&l7vsadm_option_data.vs_addr, "", 1) == 0) {
		sprintf(parse_str, "%s vs_addr = NULL", parse_str);
	} else {
		sprintf(parse_str, "%s vs_addr = %s", parse_str, 
			get_endpoint_name((struct sockaddr_storage *)(&l7vsadm_option_data.vs_addr)));
	}
	if (memcmp(&l7vsadm_option_data.rs_addr, "", 1) == 0) {
		sprintf(parse_str, "%s rs_addr = NULL", parse_str);
	} else {
		sprintf(parse_str, "%s rs_addr = %s", parse_str, 
			get_endpoint_name((struct sockaddr_storage *)(&l7vsadm_option_data.rs_addr)));
	}
	if (memcmp(&l7vsadm_option_data.protomod_name, "", 1) == 0) {
		sprintf(parse_str, "%s protomod_name = NULL", parse_str);
	} else {
		sprintf(parse_str, "%s protomod_name = %s", parse_str, 
			l7vsadm_option_data.protomod_name);
	}
	if (memcmp(&l7vsadm_option_data.schedmod_name, "", 1) == 0) {
		sprintf(parse_str, "%s schedmod_name = NULL", parse_str);
	} else {
		sprintf(parse_str, "%s schedmod_name = %s", parse_str, 
			l7vsadm_option_data.schedmod_name);
	}
	sprintf(parse_str, "%s protomod = %p", parse_str,
		l7vsadm_option_data.protomod);
	sprintf(parse_str, "%s persist = %d", parse_str,
		l7vsadm_option_data.persist);
	sprintf(parse_str, "%s backlog = %d", parse_str,
		l7vsadm_option_data.backlog);
	sprintf(parse_str, "%s weight = %d", parse_str,
		l7vsadm_option_data.weight);
	sprintf(parse_str, "%s isnumeric = %d", parse_str,
		l7vsadm_option_data.isnumeric);
	if (l7vsadm_option_data.protomod_options == NULL) {
		sprintf(parse_str, "%s protomod_options = %p", parse_str,
			l7vsadm_option_data.protomod_options);
	} else {
		char arg_str[DEBUG_STR_LEN] = {0};
		argc_argv_c_str(arg_str,
				l7vsadm_option_data.protomod_options->len,
				(char **)l7vsadm_option_data.protomod_options->data);
		sprintf(parse_str, "%s protomod_options = %s", parse_str,
			arg_str);
	}
	sprintf(parse_str, "%s protomod_sarg = %p", parse_str,
		&l7vsadm_option_data.protomod_sarg);
	sprintf(parse_str, "%s max_connection = %d", parse_str,
		l7vsadm_option_data.max_connection);
	if (memcmp(&l7vsadm_option_data.sorry_addr, "", 1) == 0) {
		sprintf(parse_str, "%s sorry_addr = NULL", parse_str);
	} else {
		sprintf(parse_str, "%s sorry_addr = %s", parse_str, 
			get_endpoint_name(&l7vsadm_option_data.sorry_addr));
	}
	sprintf(parse_str, "%s sorry_flag = %d", parse_str,
		l7vsadm_option_data.sorry_flag);
	sprintf(parse_str, "%s QoS_to_server = %lld", parse_str,
		l7vsadm_option_data.qos_threshold_up);
	sprintf(parse_str, "%s QoS_to_client = %lld", parse_str,
		l7vsadm_option_data.qos_threshold_down);
	sprintf(parse_str, "%s control_code = %d", parse_str,
		l7vsadm_option_data.control_code);
	sprintf(parse_str, "%s log_category = %d", parse_str,
		l7vsadm_option_data.log_category);
	sprintf(parse_str, "%s log_level = %d", parse_str,
		l7vsadm_option_data.log_level);
	sprintf(parse_str, "%s category_all_flag = %d", parse_str,
		l7vsadm_option_data.category_all_flag);
	sprintf(parse_str, "%s replication_start_flag = %d", parse_str,
		l7vsadm_option_data.replication_start_flag);
	sprintf(parse_str, "%s reload_param = %d", parse_str,
		l7vsadm_option_data.reload_param);
	return;
}
