/*
 * @file  parameter.cpp
 * @brief parameter module creation class.
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2008  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include "parameter_enum.h"
#include "parameter.h"
#include "parameter_impl.h"
#include "logger_wrapper.h"
#include <stdlib.h>

LOG_CATEGORY_TAG param_cat = LOG_CAT_SSLPROXY_PARAMETER;

/*!
 * Constructor of Parameter class
 */
l7vs::Parameter::Parameter()
{
	if( !ParameterImpl::getInstance().init() ){
		LOGGER_PUT_LOG_FATAL( param_cat, 1, "Parameter initialize failure" );
		/*-------- DEBUG LOG for sslproxy --------*/
		if (LOG_LV_DEBUG == logger_get_log_level(param_cat)) {
			LOGGER_PUT_LOG_DEBUG(param_cat, 1,
				"function : Constructor l7vs::Parameter::Parameter() : "
				"Initialize error.");
		}
		/*------ DEBUG LOG END for sslproxy ------*/
		exit(1);
	}
}

/*!
 * Destructor of Parameter class
 */
l7vs::Parameter::~Parameter()
{
}

/*!
 * reload config file
 * @param[in]   comp    section TAG
 * @param[in]   filename    config file name
 * @return	true = success read file / false = failure read file
 */
bool
l7vs::Parameter::rereadFile(PARAMETER_COMPONENT_TAG comp, const std::string& filename)
{
	/*-------- DEBUG LOG for sslproxy --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(param_cat)) {
		LOGGER_PUT_LOG_DEBUG(param_cat, 2,
			"function : bool l7vs::Parameter::rereadFile("
			"PARAMETER_COMPONENT_TAG comp, "
			"const std::string& filename) : "
			"Call ParameterImpl.rereadFile("
			"comp = %d, "
			"filename = %s)",
			comp,
			filename.c_str());
	}
	/*------ DEBUG LOG END for sslproxy ------*/
	return ParameterImpl::getInstance().rereadFile(comp, filename);
}

/*!
 * check whether integer data exists.
 * @param[in]	comp	section TAG
 * @param[in]	key	key string
 * @return	true = exist setting value / false = non exist setting value
 */
bool
l7vs::Parameter::isIntExist(const PARAMETER_COMPONENT_TAG comp, const std::string& key)
{
	return ParameterImpl::getInstance().isIntExist(comp, key);
}

/*!
 * check whether character data exists.
 * @param[in]	comp	section TAG
 * @param[in]	key	key string
 * @return	true = exist setting value / false = non exist setting value
 */
bool
l7vs::Parameter::isStringExist(const PARAMETER_COMPONENT_TAG comp, const std::string& key)
{
	return ParameterImpl::getInstance().isStringExist(comp, key);
}

/*!
 * get integer data.
 * @param[in]	comp	section TAG
 * @param[in]	key	key string
 * @return	value
 */
int
l7vs::Parameter::getIntValue(const PARAMETER_COMPONENT_TAG comp, const std::string& key)
{
	return ParameterImpl::getInstance().getIntValue(comp, key);
}

/*!
 * get character data.
 * @param[in]	comp	section TAG
 * @param[in]	key	key string
 * @return	value
 */
std::string
l7vs::Parameter::getStringValue(const PARAMETER_COMPONENT_TAG comp, const std::string& key)
{
	return ParameterImpl::getInstance().getStringValue(comp, key);
}

/*!
 * get character map data.
 * @param[in]	comp	section TAG
 * @param[in]	key	key string
 * @param[out]	retMap	map data
 * @return	value
 */
void
l7vs::Parameter::getStringMapValue(const PARAMETER_COMPONENT_TAG comp, const std::string& key, std::multimap<std::string, std::string>& retMap)
{
	/*-------- DEBUG LOG for sslproxy --------*/
	if (LOG_LV_DEBUG == logger_get_log_level(param_cat)) {
		LOGGER_PUT_LOG_DEBUG(param_cat, 3,
			"function : void l7vs::Parameter::getStringMapValue("
			"const PARAMETER_COMPONENT_TAG comp, "
			"const std::string& key, "
			"std::multimap<std::string, std::string>& retMap) : "
			"Call ParameterImpl.getStringMapValue("
			"comp = %d, "
			"key = %s)",
			comp,
			key.c_str());
	}
	/*------ DEBUG LOG END for sslproxy ------*/
	return ParameterImpl::getInstance().getStringMapValue(comp, key, retMap);
}

/*!
 * set-parameter function pointer relates component-tag
 * @param[in]	comp	section TAG
 * @param[in]	p_func	function pointer
 */
void
l7vs::Parameter::registerFunctionPointer(const PARAMETER_COMPONENT_TAG comp, void(*p_func)())
{
	ParameterImpl::getInstance().registerFunctionPointer(comp, p_func);
}

