#ifndef TSREMUXCPP_UTILS_H_
#define TSREMUXCPP_UTILS_H_

#include <cstring>
#include <ctime>
#include <list>
#include <iosfwd>
#include <vector>
#include <stdexcept>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/shared_array.hpp>
#include <boost/filesystem.hpp>
#include <boost/iostreams/stream.hpp>
#include <boost/thread.hpp>

#define readonly const

namespace TsRemux {

/**   tsremuxcpp original   **/
typedef unsigned char byte;
typedef unsigned short ushort;
typedef unsigned int uint;
typedef unsigned int UInt32;
typedef signed int Int32;
typedef unsigned long long UInt64;
typedef signed long long Int64;
typedef boost::posix_time::time_duration TimeSpan;
typedef boost::shared_array<byte> pByte;
typedef boost::thread BackgroundWorker;
typedef std::map Dictionary;
    
class Stream {
 public:
  Stream(void);
  ~Stream(void);
 private:
  std::ifstream&   input_file;
  std::ofstream&   output_file;
};
        
class FileStream {
 public:
  FileStream(void);
  ~FileStream(void);
 private:
  std::ifstream&   input_file;
  std::ofstream&   output_file;
};
    
enum TsFileType {
    UNKNOWN = 0,
    TS,
    M2TS,
    EVOB,
    ELEMENTARY,
    PES_ELEMENTARY,
    SUP_ELEMENTARY,
    BLU_RAY,
    MKV
};

enum DtcpCci {
    CopyFree = 0,
    NoMoreCopies,
    CopyOnce,
    CopyNever
};
 
enum ElementaryStreamTypes {
    INVALID = 0,
    VIDEO_STREAM_MPEG1 = 0x01,
    VIDEO_STREAM_MPEG2 = 0x02,
    AUDIO_STREAM_MPEG1 = 0x03, // all layers including mp3
    AUDIO_STREAM_MPEG2 = 0x04,
    VIDEO_STREAM_H264 = 0x1b,
    AUDIO_STREAM_LPCM = 0x80,
    AUDIO_STREAM_AC3 = 0x81,
    AUDIO_STREAM_DTS = 0x82,
    AUDIO_STREAM_AC3_TRUE_HD = 0x83,
    AUDIO_STREAM_AC3_PLUS = 0x84,
    AUDIO_STREAM_DTS_HD = 0x85,
    AUDIO_STREAM_DTS_HD_MASTER_AUDIO = 0x86,
    PRESENTATION_GRAPHICS_STREAM = 0x90,
    INTERACTIVE_GRAPHICS_STREAM = 0x91,
    SUBTITLE_STREAM = 0x92,
    SECONDARY_AUDIO_AC3_PLUS = 0xa1,
    SECONDARY_AUDIO_DTS_HD = 0xa2,
    VIDEO_STREAM_VC1 = 0xea
};

enum VideoFormat {
    VF_Reserved = 0,
    i480,
    i576,
    p480,
    i1080,
    p720,
    p1080,
    p576
};

enum FrameRate {
    FR_Reserved = 0,
    f23_976,
    f24,
    f25,
    f29_97,
    f50 = 6,
    f59_94
};

enum AspectRatio {
    AR_Reserved = 0,
    a4_3 = 2,
    a16_9
};

enum AudioPresentationType {
    AP_Reserved = 0,
    mono,
    stereo = 3,
    multi = 6,
    combo = 12
};

enum SamplingFrequency {
    SF_Reserved = 0,
    kHz48,
    kHz96 = 4,
    kHz192,
    kHz48_192 = 12,
    kHz48_96 = 14
};

struct EpElement {
 public:
  signed long long pts_;
  unsigned long spn_;
  EpElement(signed long long pts, unsigned long spn);
};

class Utility {
 public:
  static pByte ToString(std::vector<byte> vector);
  static void AddRange(std::vector<byte>& new_vctr, const byte* str);
  static void AddRange(std::vector<byte>& new_vctr, const pByte str);
  static void AddRange(std::vector<byte>& new_vctr, const std::vector<byte>& old_vctr);
};

class StreamInfo {
 public:
  static const int DATA_HEADER;
  StreamInfo(pByte data, int index)throw(std::invalid_argument);
  StreamInfo(ElementaryStreamTypes streamType, ushort elementaryPid);
  VideoFormat GetVideoFormat(void);
  void SetVideoFormat(VideoFormat videoformat);
  AspectRatio GetAspectRatio(void);
  void SetAspectRatio(AspectRatio aspectratio);
  FrameRate GetFrameRate(void);
  void SetFrameRate(FrameRate frameRate);
  AudioPresentationType GetAudioPresentationType(void);
  void SetAudioPresentationType(AudioPresentationType audioPresentationTyp);
  SamplingFrequency GetSamplingFrequency(void);
  void SetSamplingFrequency(SamplingFrequency samplingFrequency);
  ElementaryStreamTypes GetElementaryStreamTypes(void);
  void SetElementaryStreamTypes(ElementaryStreamTypes stream_type);
  ushort GetElementaryPID(void);
  void SetElementaryPID(ushort pid);
  pByte GetElementaryDescriptors(void);
  void SetElementaryDescriptors(pByte value)throw(std::invalid_argument);
  pByte GetByteData(void);
 private:
  pByte mData;
  VideoFormat mVideoFormat;
  AspectRatio mAspectRatio;
  FrameRate mFrameRate;
  AudioPresentationType mAudioPresentationType;
  SamplingFrequency mSamplingFrequency;
//  ElementaryStreamTypes StreamType;
//  ushort ElementaryPID;
//  pByte ElementaryDescriptors;
};

class BluRayOutput {
 public:
  static readonly byte index_bdmv[];
  static readonly byte MovieObject_bdmv[];
  static readonly byte PlayList_00000_mpls[];
  static readonly byte AppInfoPlayList[];
  static readonly byte PgStreamAttributes[];
  static readonly byte ClipInfo_0000_clpi[];
  static readonly byte TsTypeInfoBlock[];
  BluRayOutput(std::string path, TimeSpan chapterLen);
  void Author(std::vector<EpElement>& EpInfo, std::vector<StreamInfo>& sis, UInt32 numOfSourcePackets);
  pByte BuildPlayList(std::vector<pByte> PlayItems);
  pByte BuildFirstPlayItem(byte stc_id, UInt32 start, UInt32 end, pByte StnTable);
  pByte BuildFirstPlayMarks(UInt32 start, UInt32 end, UInt32 interval);
  pByte Build_clpi(pByte ClipInfo, pByte SequenceInfo, pByte ProgramInfo, pByte CPI);
  pByte BuildClipInfo(UInt32 numOfSourcePackets, std::vector<EpElement> EpInfo);
  pByte BuildSequenceInfo(UInt32 start, UInt32 end);
  pByte BuildProgramInfo(std::vector<ushort> pids, std::vector<pByte> StreamCodingInfos);
  pByte BuildVideoStreamCodingInfo(ElementaryStreamTypes type, VideoFormat format, FrameRate rate, AspectRatio ratio);
  pByte BuildAudioStreamCodingInfo(ElementaryStreamTypes type, AudioPresentationType format, SamplingFrequency rate);
  pByte BuildPgStreamCodingInfo(void);
  pByte BuildCpi(pByte EpMap);
  pByte BuildEpMap(std::vector<EpElement>& EpInfo);
 private:
  pByte BuildStreamEntry(ushort pid);
  pByte BuildVideoStreamAttributes(byte type, VideoFormat vf, FrameRate fr)throw(std::invalid_argument);
  pByte BuildAudioStreamAttributes(byte type, AudioPresentationType vf, SamplingFrequency fr)
        throw(std::invalid_argument);
  pByte BuildStnTable(pByte VideoEntry, pByte VideoAttributes,
        std::vector<pByte>& AudioEntries, std::vector<pByte>& AudioAttributes,
        std::vector<pByte>& PgEntries, std::vector<pByte>& PgAttributes);
  pByte UintToByteArraryNetwork(UInt32 value);
  pByte Build_mlps(pByte PlayList, pByte PlayListMark);
  boost::filesystem::path path;
  TimeSpan chapterLen; 
};

class Descriptor {
 public:
  Descriptor(pByte data, int startIndex)throw(std::invalid_argument);
  pByte GetData(void);
  byte GetTag(void);
  byte GetLength(void);
 protected:
  pByte mData;
};

class DTCP_Descriptor : public Descriptor {
 public:
  DTCP_Descriptor(pByte data, int startIndex)
    throw(std::invalid_argument);
  DtcpCci GetCopyStatus(void);
  bool GetAnalogConstrain(void);
  bool GetMacrovision(void);
};

class Constants {
 public:
  static const int TS_PAYLOAD_SIZE;
  static const int TS_SIZE;
  static const int STRIDE_SIZE;
  static const int DISK_BUFFER;  
  static const byte SYNC_BYTE;
  static const byte PAT_PID;
  static const byte SIT_PID;
  static const byte PAT_TABLE_ID;
  static const byte PMT_TABLE_ID;
  static const byte DTCP_DESCRIPTOR_TAG;
  static const byte PACK_ID;
  static const byte SYS_ID;
  static const byte MAP_ID;
  static const byte DIR_ID;
  static const byte PAD_ID;

  // defaults
  static const ushort DEFAULT_PMT_PID;
  static const ushort DEFAULT_VIDEO_PID;
  static const ushort MAX_VIDEO_PID;
  static const ushort DEFAULT_AUDIO_PID;
  static const ushort MAX_AUDIO_PID;
  static const ushort DEFAULT_PCR_PID;
  static const ushort DEFAULT_SUBTITLE_PID;
  static const ushort DEFAULT_PRESENTATION_GRAPHICS_PID;
  static const ushort DEFAULT_INTERACTIVE_GRAPHICS_PID;
  static const ushort DEFAULT_PROGRAM_NUMBER;
  static const int MAX_BUFFER_COUNT;
  static const int MIN_BUFFER_COUNT;
  static const Int64 AUDIO_DELAY;
  static const UInt32 MKVCLUSTER_START;
  static const UInt32 MKVFILE_START;
  static const UInt32 MKVSEGMENT_START;
  static const UInt32 MKVTRACKINFO_START;
  static const byte MKVTIMECODE_START;

  // stream types
  static const byte PES_VIDEO;
  static const byte PES_AUDIO_MPEG;
  static const byte PES_PRIVATE1;
  static const byte PES_PADDING;
  static const byte PES_PRIVATE2;
  static const byte PES_VIDEO_VC1;
  static const byte PES_PRIVATE_AC3;
  static const byte PES_PRIVATE_AC3_PLUS;
  static const byte PES_PRIVATE_DTS_HD;
  static const byte PES_PRIVATE_LPCM;
  static const byte PES_PRIVATE_AC3_TRUE_HD;
  static const UInt32 VC1_SEQ_SC;
  static const UInt32 VC1_END_OF_STREAM;
  static const ushort AC3_SYNC;
  static const UInt32 H264_PREFIX;
  static const UInt32 H264_END_OF_STREAM;
  static const UInt32 DTS_SYNC;
  static const UInt32 DTS_EXT_SYNC;
  static const UInt32 MLP_SYNC;
  static const UInt32 MPEG2_SEQ_CODE;
  static const UInt32 MPEG2_SEQ_EXT;
  static const UInt32 MPEG2_SEQ_END;

  // clocks
  static const Int64 MPEG2TS_CLOCK_RATE;
  static const Int64 MAX_MPEG2TS_CLOCK;
  static const Int64 MAX_BLURAY_CLOCK;
  static const Int64 MAX_FIREWIRE_CLOCK;
  static const Int64 MAX_PTS_CLOCK;
  static const Int64 PTS_CLOCK_RATE;
  static const int MAX_OFFSET;
  static const int MAX_COUNT;

  // descriptors
  static readonly byte hdmv_registration_descriptor[];
  static readonly byte copy_control_descriptor[];
  static readonly byte vc1_descriptor[];
  static readonly byte ac3_registration_descriptor[];
  static readonly byte DefaultSitTableOne[];
  static readonly uint crc_table[];
  static uint ComputeCrc(pByte data);
  static uint ComputeCrc(pByte data, int length);
  static uint ComputeCrc(pByte data, int length, int startIndex);
};

class ProgramInfo {
 public:
  ProgramInfo(pByte data, int index)throw(std::invalid_argument);
  ProgramInfo(ushort programNumber, ushort programPid);
  ushort GetProgramNumber(void);
  void SetProgramNumber(ushort programNumber);
  ushort GetProgramPID(void);
  void SetProgramPID(ushort programPID);
  pByte GetData(void);
 private:
  pByte mData;
};

class TsPacket {
 public:
  TsPacket(void);
  bool GetPriority(void);
  void SetPriority(bool priority);
  ushort GetPID(void);
  void SetPID(ushort pid);
  byte GetPointerSize(void);
  pByte GetData(void);
  void SetData(pByte data, int startIndex)throw(std::invalid_argument);
  bool HasPcr(void);
  Int64 GetPcr(void)throw(std::out_of_range);
  pByte Payload(void);
  void IncrementContinuityCounter(void);
  byte GetContinuityCounter(void);
  void SetContinuityCounter(byte value)throw(std::out_of_range);
  bool HasPesHeader(void);
 protected:
  pByte mData;
 private:
  bool Priority;
  ushort PID;
};

class PcrPacket : TsPacket {
 public:
  PcrPacket(Int64 pcr, byte counter, ushort pid);
};

class TsTable : public TsPacket {
 public:
  TsTable();
  TsTable(pByte data);
  void AddData(pByte data, int offset, int len);
  bool Complete(void);
  byte GetTableId(void);
  void SetTableId(byte value);
  ushort GetNumberId(void);
  void SetNumberId(ushort value);
  ushort GetLength(void);
  void SetLength(ushort value);
 protected:
  void RefreshCrc(void);
};

class PatPacket : public TsTable {
 public:
  PatPacket(void);
  PatPacket(pByte data)throw(std::invalid_argument);
  ushort GetTransportStreamId(void);
  void SetTransportStreamId(ushort TSId);
  std::vector<boost::shared_ptr<ProgramInfo> > GetPrograms(void);
  void SetPrograms(std::vector<boost::shared_ptr<ProgramInfo> > programinfo)
      throw(std::invalid_argument);
  ushort GetProgramInfoLength(void);
};

class SitPacket : TsTable {
 public:
  SitPacket(void);
  SitPacket(pByte data);
};

class PmtPacket : TsTable {
 public:
  PmtPacket(void);
  PmtPacket(pByte data)throw(std::invalid_argument);
  boost::shared_array<DTCP_Descriptor> GetDtcpInfo(void);
  pByte GetProgramDescriptorsData(void);
  void SetProgramDescriptorsData(pByte value);
  ushort GetProgramDescriptorsLength(void);
  void SetProgramDescriptorsLength(ushort value);
  ushort GetProgramNumber(void);
  void SetProgramNumber(ushort id);
  ushort GetPcrPID(void);
  void SetPcrPID(ushort pid);
  ushort GetStreamInfoLength(void);
 private:
  ushort PcrPID;
};

class PesHeader {
 public:
  PesHeader(pByte data)throw(std::invalid_argument);
  byte GetStreamId(void);
  byte GetByte(int i);
  void SetByte(int i, byte dat);
  byte GetHeaderLength(void);
  int GetTotalHeaderLength(void);
  ushort GetPacketLength(void);
  bool HasPts(void);
  bool HasDts(void);
  Int64 GetPts(void)throw(std::invalid_argument);
  void SetPts(Int64 value)throw(std::invalid_argument);
  Int64 GetDts(void)throw(std::invalid_argument);
  void SetDts(Int64 value)throw(std::invalid_argument);
  byte GetExtention2(void);
  pByte GetData(void);
 private:
  pByte mData;
};

class PesPacket {
 public:
  PesPacket(pByte buff, int offset, int length, ushort pid);
  bool GetPriority(void);
  void SetPriority(bool priority);
  pByte GetData(void);
  pByte GetPayload(void);
  byte GetByte(int i);
  void SetByte(int i, byte dat);
  ushort GetPID(void);
  void SetPID(ushort id);
  bool GetComplete(void);
  void SetComplete(bool value);
  boost::shared_ptr<PesHeader> GetHeader(void);
  void AddData(pByte moredata);
  void AddData(pByte buff, int offset, int length);
  byte GetBaseId(void);
  byte GetExtendedId(void);
  UInt32 GetExtendedType(void);
 private:
  pByte mData;
  bool mPriority;
  ushort mPID;
};

class VC1SequenceInfo {
 public:
  VC1SequenceInfo(pByte data, int offset);
  int GetHeight(void);
  int GetWidth(void);
  bool Valid(void);
  bool Interlaced(void);
  bool DisplayExt(void);
  bool AspectFlag(void);
  byte GetVc1AspectRatio(void);
  bool FrameFlag(void);
  bool FrameRateIndicatorFlag(void);
  AspectRatio GetAspectRatio(void);
  VideoFormat GetVideoFormat(void);
  FrameRate GetFrameRate(void);
 private:
  pByte mData;
};

enum Ac3SyntaxType {
        Invalid = 0,
        Standard = 8,
        Alternative = 6,
        Enhanced = 16
};

class ElementaryParse {
 public:
  ElementaryParse(void);
  virtual VideoFormat GetVideoFormat(void);
  virtual void SetVideoFormat(VideoFormat videoformat);
  virtual AspectRatio GetAspectRatio(void);
  virtual void SetAspectRatio(AspectRatio aspectratio);
  virtual FrameRate GetFrameRate(void);
  virtual void SetFrameRate(FrameRate frameRate);
  virtual AudioPresentationType GetAudioPresentationType(void);
  virtual void SetAudioPresentationType(AudioPresentationType audioPresentationTyp);
  virtual SamplingFrequency GetSamplingFrequency(void);
  virtual void SetSamplingFrequency(SamplingFrequency samplingFrequency);
  virtual pByte GetElementaryDescriptors(void);
  bool mValid;
 protected:
  byte GetNextBit(void);
  pByte mData;
  int indicator;
  VideoFormat mVideoFormat;
  FrameRate mFrameRate;
  AspectRatio mAspectRatio;
  SamplingFrequency mSamplingFrequency;
  AudioPresentationType mAudioPresentationType;
  pByte ElementaryDescriptors;
};

class H264Info : ElementaryParse {
 public:
  H264Info(pByte data, int offset);
  pByte GetElementaryDescriptors(void);
  VideoFormat GetVideoFormat(void);
  AspectRatio GetAspectRatio(void);
  FrameRate GetFrameRate(void);
  AudioPresentationType GetAudioPresentationType(void);
  SamplingFrequency GetSamplingFrequency(void);
  UInt32 GetNextExpGolomb();
  void ScalingListSkip(int skip);
  UInt32 GetWidth(void);
  UInt32 GetHeigth(void);
  pByte GetHdmvVideoRegistrationDescriptor(void);
};

class AC3Info : ElementaryParse {
 public:
  int GetMaxFrameLength(void);
  int GetFrameLength(void);
  Ac3SyntaxType GetSyntaxType(void);
  byte GetBsid(void);
  byte GetBsmod(void);
  byte GetAcmod(void);
  bool IsIndependentStream(void);
  AC3Info(pByte data, int offset);
  VideoFormat GetVideoFormat(void);
  AspectRatio GetAspectRatio(void);
  FrameRate GetFrameRate(void);
  AudioPresentationType GetAudioPresentationType(void);
  SamplingFrequency GetSamplingFrequency(void);
  pByte GetElementaryDescriptors(void);
  pByte AC3Info::GetAC3AudioDescriptor(void);
  byte GetSampleRateCode(void);
  bool Valid(void);
 private:
  static readonly int len48k[];
  static readonly int len44k[];
  static readonly int len32k[];
};

class DtsInfo : ElementaryParse {
 public:
  bool GetExtAudio(void);
  DtsInfo(pByte data, int offset);
  VideoFormat GetVideoFormat(void);
  AspectRatio GetAspectRatio(void);
  FrameRate GetFrameRate(void);
  AudioPresentationType GetAudioPresentationType(void);
  byte GetSampleFreq(void);
  pByte GetElementaryDescriptors(void);
  ushort GetFrameSize(void);
  byte GetExtAudioId(void);
  byte GetAmode(void);
  SamplingFrequency GetSamplingFrequency(void);
};

class MlpInfo : ElementaryParse {
 public:
  MlpInfo(pByte data, int offset);
  VideoFormat GetVideoFormat(void);
  AspectRatio GetAspectRatio(void);
  FrameRate GetFrameRate(void);
  AudioPresentationType GetAudioPresentationType(void);
  SamplingFrequency GetSamplingFrequency(void);
  pByte GetElementaryDescriptors(void);
  pByte GetAC3AudioDescriptor(void);
  byte GetSampleRateCode(void);
};

class Mpeg2Info : ElementaryParse {
 public:
  Mpeg2Info(void);
  Mpeg2Info(pByte data, int offset);
  VideoFormat GetVideoFormat(void);
  AspectRatio GetAspectRatio(void);
  FrameRate GetFrameRate(void);
  AudioPresentationType GetAudioPresentationType(void);
  SamplingFrequency GetSamplingFrequency(void);
  pByte GetElementaryDescriptors(void);
  pByte GetMpeg2VideoRegistrationDescriptor(void);
  ushort GetHorizontal(void);
  ushort GetVertical(void);
  byte GetAspect(void);
  byte GetFrameRateCode(void);
  bool GetProgressive(void);
 private:
  class Mpeg2Ext : public ElementaryParse {
   public:
    Mpeg2Ext(void);
    Mpeg2Ext(pByte data, int offset) {
        UInt32 marker = 0xffffffff;
        for (; offset < sizeof(data.get()) - 1; offset++)
        {
            marker = (UInt32)marker << 8;
            marker &= 0xffffff00;
            marker += data[offset];
            if (marker == Constants::MPEG2_SEQ_EXT)
            {
                if((data[offset + 1] & 0xf0) == 0x10)
                break;
            }
        }
        offset++;
        if (offset < sizeof(data.get()))
        {
            // sequence header
            mData = pByte(new byte[sizeof(data.get()) - offset]);
            for (int i = 0; offset < sizeof(data.get()); i++, offset++)
                mData[i] = data[offset];
        }
        else
            mData.reset();
    };
      AspectRatio GetAspectRatio(void){
          throw std::invalid_argument("The method or operation is not implemented.");
      };
      AudioPresentationType GetAudioPresentationType(void){
          throw std::invalid_argument("The method or operation is not implemented.");
      };
      pByte GetElementaryDescriptors(void){
          throw std::invalid_argument("The method or operation is not implemented.");
      };
      FrameRate GetFrameRate(void){
          throw std::invalid_argument("The method or operation is not implemented.");
      };
      SamplingFrequency GetSamplingFrequency(void){
          throw std::invalid_argument("The method or operation is not implemented.");
      };
      VideoFormat GetVideoFormat(void){
          throw std::invalid_argument("The method or operation is not implemented.");
      };
    bool GetProgressive(void){
        indicator = 12;
        if (GetNextBit() == 1)
           return true;
        return false;
    };
  };
  Mpeg2Ext mpgext;
};

} //namespace
#endif TSREMUXCPP_UTILS_H_
