#include <fstream>
#include "Utils.h"

namespace TsRemux
{
    const int Constants::TS_PAYLOAD_SIZE = 184;
    const int Constants::TS_SIZE = 188;
    const int Constants::STRIDE_SIZE = 192;
    const int Constants::DISK_BUFFER = 0x8D000 << 5;
    const byte Constants::SYNC_BYTE = 0x47;
    const byte Constants::PAT_PID = 0x00;
    const byte Constants::SIT_PID = 0x1f;
    const byte Constants::PAT_TABLE_ID = 0x00;
    const byte Constants::PMT_TABLE_ID = 0x02;
    const byte Constants::DTCP_DESCRIPTOR_TAG = 0x88;
    const byte Constants::PACK_ID = 0xba;
    const byte Constants::SYS_ID = 0xbb;
    const byte Constants::MAP_ID = 0xbc;
    const byte Constants::DIR_ID = 0xff;
    const byte Constants::PAD_ID = 0xbe;

    // defaults
    const ushort Constants::DEFAULT_PMT_PID = 0x0100;
    const ushort Constants::DEFAULT_VIDEO_PID = 0x1011;
    const ushort Constants::MAX_VIDEO_PID = 0x1019;
    const ushort Constants::DEFAULT_AUDIO_PID = 0x1100;
    const ushort Constants::MAX_AUDIO_PID = 0x111f;
    const ushort Constants::DEFAULT_PCR_PID = 0x1001;
    const ushort Constants::DEFAULT_SUBTITLE_PID = 0x1800;
    const ushort Constants::DEFAULT_PRESENTATION_GRAPHICS_PID = 0x1200;
    const ushort Constants::DEFAULT_INTERACTIVE_GRAPHICS_PID = 0x1400;
    const ushort Constants::DEFAULT_PROGRAM_NUMBER = 0x01;
    const int Constants::MAX_BUFFER_COUNT = 0xff;
    const int Constants::MIN_BUFFER_COUNT = 0x02;
    const Int64 Constants::AUDIO_DELAY = 30000;
    const UInt32 Constants::MKVCLUSTER_START = 0x1f43b675;
    const UInt32 Constants::MKVFILE_START = 0x1a45dfa3;
    const UInt32 Constants::MKVSEGMENT_START = 0x18538067;
    const UInt32 Constants::MKVTRACKINFO_START = 0x1654AE6B;

    // stream types
    const byte Constants::PES_VIDEO = 0xe0;
    const byte Constants::PES_AUDIO_MPEG = 0xc0;
    const byte Constants::PES_PRIVATE1 = 0xbd;
    const byte Constants::PES_PADDING = 0xbe;
    const byte Constants::PES_PRIVATE2 = 0xbf;
    const byte Constants::PES_VIDEO_VC1 = 0xfd;
    const byte Constants::PES_PRIVATE_AC3 = 0x80;
    const byte Constants::PES_PRIVATE_AC3_PLUS = 0xc0;
    const byte Constants::PES_PRIVATE_DTS_HD = 0x88;
    const byte Constants::PES_PRIVATE_LPCM = 0xa0;
    const byte Constants::PES_PRIVATE_AC3_TRUE_HD = 0xb0;
    const UInt32 Constants::VC1_SEQ_SC = 0x0000010f;
    const UInt32 Constants::VC1_END_OF_STREAM = 0x0000010a;
    const ushort Constants::AC3_SYNC = 0x0b77;
    const UInt32 Constants::H264_PREFIX = 0x00000107;
    const UInt32 Constants::H264_END_OF_STREAM = 0x0000010b;
    const UInt32 Constants::DTS_SYNC = 0x7ffe8001;
    const UInt32 Constants::DTS_EXT_SYNC = 0x64582025;
    const UInt32 Constants::MLP_SYNC = 0xF8726FBA;
    const UInt32 Constants::MPEG2_SEQ_CODE = 0x000001b3;
    const UInt32 Constants::MPEG2_SEQ_EXT = 0x000001b5;
    const UInt32 Constants::MPEG2_SEQ_END = 0x000001b7;

    // clocks
    const Int64 Constants::MPEG2TS_CLOCK_RATE = 27000000LL;
    const Int64 Constants::MAX_MPEG2TS_CLOCK = 0x25800000000LL;
    const Int64 Constants::MAX_BLURAY_CLOCK = 0x40000000LL;
    const Int64 Constants::MAX_FIREWIRE_CLOCK = 24576000LL;
    const Int64 Constants::MAX_PTS_CLOCK = 0x200000000LL;
    const Int64 Constants::PTS_CLOCK_RATE = 90000LL;
    const int Constants::MAX_OFFSET = 3072;
    const int Constants::MAX_COUNT = 8000;

   // descriptors
    readonly byte Constants::hdmv_registration_descriptor[]
        = { 0x05, 0x04, 0x48, 0x44, 0x4d, 0x56 };
    readonly byte Constants::copy_control_descriptor[]
        = { 0x88, 0x04, 0x0f, 0xff, 0x84, 0xfc };
    readonly byte Constants::vc1_descriptor[]
        = { 0x05, 0x05, 0x56, 0x43, 0x2d, 0x31, 0xff };
    readonly byte Constants::ac3_registration_descriptor[] 
         = { 0x05, 0x04, 0x41, 0x43, 0x2d, 0x33 };
    readonly byte Constants::DefaultSitTableOne[] = {
            0x47, 0x40, 0x1f, 0x10, 0x00, 0x7f, 0xf0, 0x19,
            0xff, 0xff, 0xc1, 0x00, 0x00, 0xf0, 0x0a, 0x63,
            0x08, 0xc1, 0x5a, 0xae, 0xff, 0xff, 0xff, 0xff,
            0xff, 0x00, 0x01, 0x80, 0x00, 0x34, 0x1e, 0xe7,
            0x4e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff };   
    readonly uint Constants::crc_table[] = {
        0x00000000, 0x04c11db7, 0x09823b6e, 0x0d4326d9, 0x130476dc, 0x17c56b6b,
        0x1a864db2, 0x1e475005, 0x2608edb8, 0x22c9f00f, 0x2f8ad6d6, 0x2b4bcb61,
        0x350c9b64, 0x31cd86d3, 0x3c8ea00a, 0x384fbdbd, 0x4c11db70, 0x48d0c6c7,
        0x4593e01e, 0x4152fda9, 0x5f15adac, 0x5bd4b01b, 0x569796c2, 0x52568b75,
        0x6a1936c8, 0x6ed82b7f, 0x639b0da6, 0x675a1011, 0x791d4014, 0x7ddc5da3,
        0x709f7b7a, 0x745e66cd, 0x9823b6e0, 0x9ce2ab57, 0x91a18d8e, 0x95609039,
        0x8b27c03c, 0x8fe6dd8b, 0x82a5fb52, 0x8664e6e5, 0xbe2b5b58, 0xbaea46ef,
        0xb7a96036, 0xb3687d81, 0xad2f2d84, 0xa9ee3033, 0xa4ad16ea, 0xa06c0b5d,
        0xd4326d90, 0xd0f37027, 0xddb056fe, 0xd9714b49, 0xc7361b4c, 0xc3f706fb,
        0xceb42022, 0xca753d95, 0xf23a8028, 0xf6fb9d9f, 0xfbb8bb46, 0xff79a6f1,
        0xe13ef6f4, 0xe5ffeb43, 0xe8bccd9a, 0xec7dd02d, 0x34867077, 0x30476dc0,
        0x3d044b19, 0x39c556ae, 0x278206ab, 0x23431b1c, 0x2e003dc5, 0x2ac12072,
        0x128e9dcf, 0x164f8078, 0x1b0ca6a1, 0x1fcdbb16, 0x018aeb13, 0x054bf6a4,
        0x0808d07d, 0x0cc9cdca, 0x7897ab07, 0x7c56b6b0, 0x71159069, 0x75d48dde,
        0x6b93dddb, 0x6f52c06c, 0x6211e6b5, 0x66d0fb02, 0x5e9f46bf, 0x5a5e5b08,
        0x571d7dd1, 0x53dc6066, 0x4d9b3063, 0x495a2dd4, 0x44190b0d, 0x40d816ba,
        0xaca5c697, 0xa864db20, 0xa527fdf9, 0xa1e6e04e, 0xbfa1b04b, 0xbb60adfc,
        0xb6238b25, 0xb2e29692, 0x8aad2b2f, 0x8e6c3698, 0x832f1041, 0x87ee0df6,
        0x99a95df3, 0x9d684044, 0x902b669d, 0x94ea7b2a, 0xe0b41de7, 0xe4750050,
        0xe9362689, 0xedf73b3e, 0xf3b06b3b, 0xf771768c, 0xfa325055, 0xfef34de2,
        0xc6bcf05f, 0xc27dede8, 0xcf3ecb31, 0xcbffd686, 0xd5b88683, 0xd1799b34,
        0xdc3abded, 0xd8fba05a, 0x690ce0ee, 0x6dcdfd59, 0x608edb80, 0x644fc637,
        0x7a089632, 0x7ec98b85, 0x738aad5c, 0x774bb0eb, 0x4f040d56, 0x4bc510e1,
        0x46863638, 0x42472b8f, 0x5c007b8a, 0x58c1663d, 0x558240e4, 0x51435d53,
        0x251d3b9e, 0x21dc2629, 0x2c9f00f0, 0x285e1d47, 0x36194d42, 0x32d850f5,
        0x3f9b762c, 0x3b5a6b9b, 0x0315d626, 0x07d4cb91, 0x0a97ed48, 0x0e56f0ff,
        0x1011a0fa, 0x14d0bd4d, 0x19939b94, 0x1d528623, 0xf12f560e, 0xf5ee4bb9,
        0xf8ad6d60, 0xfc6c70d7, 0xe22b20d2, 0xe6ea3d65, 0xeba91bbc, 0xef68060b,
        0xd727bbb6, 0xd3e6a601, 0xdea580d8, 0xda649d6f, 0xc423cd6a, 0xc0e2d0dd,
        0xcda1f604, 0xc960ebb3, 0xbd3e8d7e, 0xb9ff90c9, 0xb4bcb610, 0xb07daba7,
        0xae3afba2, 0xaafbe615, 0xa7b8c0cc, 0xa379dd7b, 0x9b3660c6, 0x9ff77d71,
        0x92b45ba8, 0x9675461f, 0x8832161a, 0x8cf30bad, 0x81b02d74, 0x857130c3,
        0x5d8a9099, 0x594b8d2e, 0x5408abf7, 0x50c9b640, 0x4e8ee645, 0x4a4ffbf2,
        0x470cdd2b, 0x43cdc09c, 0x7b827d21, 0x7f436096, 0x7200464f, 0x76c15bf8,
        0x68860bfd, 0x6c47164a, 0x61043093, 0x65c52d24, 0x119b4be9, 0x155a565e,
        0x18197087, 0x1cd86d30, 0x029f3d35, 0x065e2082, 0x0b1d065b, 0x0fdc1bec,
        0x3793a651, 0x3352bbe6, 0x3e119d3f, 0x3ad08088, 0x2497d08d, 0x2056cd3a,
        0x2d15ebe3, 0x29d4f654, 0xc5a92679, 0xc1683bce, 0xcc2b1d17, 0xc8ea00a0,
        0xd6ad50a5, 0xd26c4d12, 0xdf2f6bcb, 0xdbee767c, 0xe3a1cbc1, 0xe760d676,
        0xea23f0af, 0xeee2ed18, 0xf0a5bd1d, 0xf464a0aa, 0xf9278673, 0xfde69bc4,
        0x89b8fd09, 0x8d79e0be, 0x803ac667, 0x84fbdbd0, 0x9abc8bd5, 0x9e7d9662,
        0x933eb0bb, 0x97ffad0c, 0xafb010b1, 0xab710d06, 0xa6322bdf, 0xa2f33668,
        0xbcb4666d, 0xb8757bda, 0xb5365d03, 0xb1f740b4 };

pByte Utility::ToString(const std::vector<byte> vctr)
{
    pByte str = pByte(new byte[vctr.size()]);
    for(int i = 0; i < vctr.size(); i++)
    {
        str[i] = vctr.at(i);
    } 
    return str;
}

void Utility::AddRange(std::vector<byte>& vctr, const byte* str)
{
    for(int i = 0; i < sizeof(str); i++)
       vctr.push_back(str[i]); 
}

void Utility::AddRange(std::vector<byte>& vctr, const pByte str)
{
    for(int i = 0; i < sizeof(str.get()); i++)
       vctr.push_back(str[i]); 
}

void Utility::AddRange(std::vector<byte>& new_vctr, const std::vector<byte>& old_vctr)
{
    for(int i = 0; i < old_vctr.size(); i++)
        new_vctr.push_back(old_vctr.at(i));
}

const int DATA_HEADER = 5;
StreamInfo::StreamInfo(pByte data, int index)throw(std::invalid_argument)
{
    if(!data)
        throw std::invalid_argument("stream data is NULL");

    if(sizeof(data.get()) + index < DATA_HEADER)
        throw std::invalid_argument("stream data too short");

    uint descLength = (data[3 + index] & 0x0f) << 8 + data[4 + index];

    if(descLength > Constants::TS_SIZE)
        throw std::invalid_argument("descriptors data too long");

    if(5 + descLength > sizeof(data.get()) - index)
    {
        throw std::invalid_argument("stream data too short");
    }

    mData = pByte(new byte[DATA_HEADER + descLength]);
    for(int i = 0; i < sizeof(mData.get()); i++)
    {
        mData[i] = data[i + index];
    }

    SetVideoFormat(VF_Reserved);
    SetAspectRatio(AR_Reserved);
    SetFrameRate(FR_Reserved);
    SetAudioPresentationType(AP_Reserved);
    SetSamplingFrequency(SF_Reserved);
}

VideoFormat StreamInfo::GetVideoFormat(void)
{
    return mVideoFormat;
}

void StreamInfo::SetVideoFormat(VideoFormat video_format)
{
    mVideoFormat = video_format;
}

AspectRatio StreamInfo::GetAspectRatio(void)
{
    return mAspectRatio;
}

void StreamInfo::SetAspectRatio(AspectRatio aspect_ratio)
{
    mAspectRatio = aspect_ratio;
}

FrameRate StreamInfo::GetFrameRate(void)
{
    return mFrameRate;
}

void StreamInfo::SetFrameRate(FrameRate frame_rate)
{
    mFrameRate = frame_rate;
}

AudioPresentationType StreamInfo::GetAudioPresentationType(void)
{
    return mAudioPresentationType;
}

void StreamInfo::SetAudioPresentationType(AudioPresentationType ap_type)
{
    mAudioPresentationType = ap_type;
}

SamplingFrequency StreamInfo::GetSamplingFrequency(void)
{
    return mSamplingFrequency;
}

void StreamInfo::SetSamplingFrequency(SamplingFrequency sampling_frequency)
{
    mSamplingFrequency = sampling_frequency;
}

StreamInfo::StreamInfo(ElementaryStreamTypes streamType, ushort elementaryPid)
{
    mData = pByte(new byte[DATA_HEADER]);
    SetElementaryStreamTypes(streamType);
    SetElementaryPID(elementaryPid);
    // reserved and descriptors length
    mData[3] = 0xf0;
    mData[4] = 0x00;
}

pByte StreamInfo::GetByteData(void)
{
    return mData;
}

ElementaryStreamTypes StreamInfo::GetElementaryStreamTypes(void)
{
    return (ElementaryStreamTypes)mData[0];
}

void StreamInfo::SetElementaryStreamTypes(ElementaryStreamTypes stream_type)
{
    mData[0] = (byte)stream_type;
}
ushort StreamInfo::GetElementaryPID(void)
{
    return (ushort)(((mData[1] & 0x1f) << 8) + mData[2]);
}

void StreamInfo::SetElementaryPID(ushort pid)
{
    mData[1] = (byte)(((pid >> 8) & 0x1f) | 0xe0);
    mData[2] = (byte)(pid & 0xff);
}

pByte StreamInfo::GetElementaryDescriptors(void)
{
    pByte descriptors;
    if(sizeof(mData.get()) == DATA_HEADER)
    {
        descriptors.reset();
    }
    else
    {
        descriptors = pByte(new byte[sizeof(mData.get()) - DATA_HEADER]);
        for (int i = 0; i < (sizeof(mData.get()) - DATA_HEADER); i++)
        {
            descriptors[i] = mData[i + DATA_HEADER];
        }
    }
    return descriptors;
}

void StreamInfo::SetElementaryDescriptors(pByte value)
    throw(std::invalid_argument)
{
    if(!value || 0 == sizeof(value.get()))
    {
        if(sizeof(mData.get()) > DATA_HEADER)
        {
            // need to remove existing descriptors
            pByte data = pByte(new byte[DATA_HEADER]);
            data[0] = mData[0];
            data[1] = mData[1];
            data[2] = mData[2];
            data[3] = 0xf0;
            data[4] = 0x00;
            mData = data;
        }
        else
        {
            // nothing to do
            return;
        }
    }
    else
    {
        if(sizeof(value.get()) > 180) {
            throw std::invalid_argument("descriptors data too long");
        }

        pByte data = pByte(new byte[DATA_HEADER + sizeof(value.get())]);
        data[0] = mData[0];
        data[1] = mData[1];
        data[2] = mData[2];
        data[3] = (byte)(0xf0 | (byte)((sizeof(value.get()) >> 8) & 0x0f));
        data[4] = (byte)(sizeof(value.get()) & 0xff);
        for (int i = 0; i < sizeof(value.get()); i++)
        {
            data[DATA_HEADER + i] = value[i];
        }
        mData = data;
    }
}

ProgramInfo::ProgramInfo(pByte data, int index)throw(std::invalid_argument)
{
    if(!data)
        throw std::invalid_argument("program data is null");

    if(sizeof(data.get()) + index < 4)
        throw std::invalid_argument("program data too short");

    mData = pByte(new byte[4]);
    for(int i = 0; i < sizeof(mData.get()); i++)
    {
        mData[i] = data[i + index];
    }
}

ProgramInfo::ProgramInfo(ushort programNumber, ushort programPid)
{
    mData = pByte(new byte[4]);
    SetProgramNumber(programNumber);
    SetProgramPID(programPid);
}

ushort ProgramInfo::GetProgramNumber(void)
{
    return (ushort)((mData[0] << 8) + mData[1]);
}

void ProgramInfo::SetProgramNumber(ushort value)
{
    mData[0] = (byte)((value >> 8) & 0xff);
    mData[1] = (byte)(value & 0xff);
}

ushort ProgramInfo::GetProgramPID(void)
{
    return (ushort)(((mData[2] & 0x1f) << 8) + mData[3]);
}

void ProgramInfo::SetProgramPID(ushort value)
{
    mData[2] = (byte)(((value >> 8) & 0x1f) | 0xe0);
    mData[3] = (byte)(value & 0xff);
}

pByte ProgramInfo::GetData(void)
{
    return mData;
}

/////////////////////////////////////
/**   implement ElementaryParse   **/
/////////////////////////////////////
ElementaryParse::ElementaryParse(void)
{
    indicator = 0;
}

byte ElementaryParse::GetNextBit(void)
{
    byte ret = (byte)(((mData[indicator / 8]) >> (7 - (indicator % 8))) & 1);
    indicator++;
    return ret;
}

//////////////////////////////
/**   implement TsPacket   **/
//////////////////////////////
TsPacket::TsPacket(void)
{
    // initialize the packet as a null packet
    mData = pByte(new byte[Constants::TS_SIZE]);
    mData[0] = Constants::SYNC_BYTE; // sync byte
    mData[1] = 0x1f; // PID = 0x1FFF
    mData[2] = 0xff; // PID == 0x1FFF
    mData[3] = 0x10; // no adaptation field
    for (int i = 4; i < Constants::TS_SIZE; i++)
        mData[i] = 0xff;
}

bool TsPacket::GetPriority(void)
{
    if ((mData[1] & 0x20) > 0)
        return true;
    else
        return false;
}

void TsPacket::SetPriority(bool priority)
{
    if (priority)
        mData[1] |= 0x20;
    else
        mData[1] &= 0xdf;
}

ushort TsPacket::GetPID(void)
{
    return (ushort)(((mData[1] & 0x1f) << 8) + mData[2]);
}
void TsPacket::SetPID(ushort pid)
{
    byte b = (byte)(mData[1] & 0xE0);
    b += (byte)((pid >> 8) & 0x1f);
    mData[1] = b;
    mData[2] = (byte)(pid & 0xff);
}

byte TsPacket::GetPointerSize(void)
{
    if ((mData[3] & 0x20) == 0) // No adaptation field present
        return mData[4];
    return (byte)(mData[4] + 1 + mData[mData[4] + DATA_HEADER]);
}

pByte TsPacket::GetData(void)
{
    return mData;
}

void TsPacket::SetData(pByte data, int startIndex)throw(std::invalid_argument)
{
    if (!data)
        throw std::invalid_argument("NULL packet");
    else if (Constants::TS_SIZE > sizeof(data.get()) - startIndex)
        throw std::invalid_argument("small packet");
    else if (Constants::SYNC_BYTE != data[0 + startIndex])
        throw std::invalid_argument("sync byte missing");
    for (int i = 0; i < Constants::TS_SIZE; i++) {
        mData[i] = data[i + startIndex];
    }
}

bool TsPacket::HasPcr(void)
{
    if ((mData[3] & 0x20) > 0  // Adaptation field present
     && (mData[4] > 0)         // length > 0
     && (mData[DATA_HEADER] & 0x10) > 0) // and a PCR
        return true;
    return false;
}

Int64 TsPacket::GetPcr(void)throw(std::out_of_range)
{
    if (false == this->HasPcr())
        throw std::out_of_range("PCR not present in this packet");
    Int64 mpeg2tsClock = mData[6];
    mpeg2tsClock <<= 25;
    mpeg2tsClock += (mData[7] << 17);
    mpeg2tsClock += (mData[8] << 9);
    mpeg2tsClock += (mData[9] << 1);
    mpeg2tsClock += ((mData[10] & 0x80) >> 7);
    mpeg2tsClock *= 300;
    mpeg2tsClock += ((mData[10] & 0x1) << 8);
    mpeg2tsClock += (mData[11]);
    return mpeg2tsClock;
}

bool TsPacket::HasPesHeader(void)
{
    if ((mData[1] & 0x40) == 0)
        return false;
    int offset = 4;
    if ((mData[3] & 0x20) > 0)
    {
        // adaptation field present
        offset += (1 + mData[4]);
        if (offset >= Constants::TS_SIZE)
            return false;
    }
    if ((mData[3] & 0x10) > 0)
    {
        // payload present
        int len = Constants::TS_SIZE - offset;
        if (len < 10)
            return false;

        if(mData[offset] == 0
        && mData[offset + 1] == 0
        && mData[offset + 2] == 1
        && len >= 9 + mData[offset + 8])
            return true;
        else if(mData[offset] == 0
             && mData[offset + 1] == 0
             && mData[offset + 2] == 1)
            return false;
    }
    return false;
}

pByte TsPacket::Payload(void)
{
    pByte ret;
    ret.reset();
    int offset = 4;
    if ((mData[3] & 0x20) > 0)
    {
        // adaptation field present
        offset += (1 + mData[4]);
        if (offset >= Constants::TS_SIZE)
            return ret;
    }
    if ((mData[3] & 0x10) > 0)
    {
        // payload present
        ret = pByte(new byte[Constants::TS_SIZE - offset]);
        for (int i = 0; i < sizeof(ret); i++)
            ret[i] = mData[i + offset];
        ret[Constants::TS_SIZE - offset] = '\0';
        return ret;
    }
    return ret;
}

void TsPacket::IncrementContinuityCounter(void)
{
    if (0xf == (mData[3] & 0xf))
        mData[3] &= 0xf0;
    else
        mData[3]++;
}

byte TsPacket::GetContinuityCounter(void)
{
    return (byte)(mData[3] & 0xf);
}

void TsPacket::SetContinuityCounter(byte value)throw(std::out_of_range)
{
    if (value > 0x0f)
        throw std::out_of_range("Invalid continuity counter");
    mData[3] &= 0xf0;
    mData[3] |= value;
}

/////////////////////////////
/**   implement TsTable   **/
/////////////////////////////
TsTable::TsTable(void)
{
    // error = 0, payload = 1, priority = 0
    mData[1] = 0x40;
    // scrambling = 0, adaptation = 01, continuity = 0
    mData[3] = 0x10;
    // pointer = 00
    mData[4] = 0x0;
    // reserved, version, current/next
    mData[10] = 0xc1;
    // section
    mData[11] = 0x0;
    // last section
    mData[12] = 0x0;
}

TsTable::TsTable(pByte data)
{
    SetData(data, 0);
}

void TsTable::AddData(pByte value, int offset, int len)
{
    std::vector<byte> data;
    Utility::AddRange(data, mData);
    Utility::AddRange(data, &value[offset]);
    mData = Utility::ToString(data);
}

bool TsTable::Complete(void)
{
    int currentLen = sizeof(mData.get()) - (GetPointerSize() + 8);
    if (GetLength() > currentLen)
        return false;
    return true;
}

byte TsTable::GetTableId(void)
{
    return mData[DATA_HEADER + GetPointerSize()];
}

void TsTable::SetTableId(byte value)
{
    mData[DATA_HEADER + GetPointerSize()] = value;
}

ushort TsTable::GetNumberId(void)
{
    return (ushort)((mData[8 + GetPointerSize()] << 8)
        + mData[9 + GetPointerSize()]);
}

void TsTable::SetNumberId(ushort value)
{
    mData[8 + GetPointerSize()] = (byte)((value >> 8) & 0xff);
    mData[9 + GetPointerSize()] = (byte)(value & 0xff);
}

ushort TsTable::GetLength(void)
{
    return (ushort)(((mData[6 + GetPointerSize()] & 0x0f) << 8)
        + mData[7 + GetPointerSize()]);
}

void TsTable::SetLength(ushort value)
{
    // syntax, reserved, length
    mData[6 + GetPointerSize()]
        = (byte)(0xb0 | (byte)((value >> 8) & 0x0f));
    mData[7 + GetPointerSize()] = (byte)(value & 0xff);
}

void TsTable::RefreshCrc(void)
{
    uint crc = Constants::ComputeCrc(
        mData, GetLength() - 1, DATA_HEADER + GetPointerSize());
    mData[GetLength() + 4 + GetPointerSize()]
        = (byte)((crc >> 24) & 0xff);
    mData[GetLength() + DATA_HEADER + GetPointerSize()]
        = (byte)((crc >> 16) & 0xff);
    mData[GetLength() + 6 + GetPointerSize()]
        = (byte)((crc >> 8) & 0xff);
    mData[GetLength() + 7 + GetPointerSize()]
        = (byte)(crc & 0xff);
    for (int i = GetLength() + 8 + GetPointerSize();
        i < Constants::TS_SIZE; i++) {
        mData[i] = 0xff;
    }
}

////////////////////////////////
/**   implement Descriptor   **/
////////////////////////////////
Descriptor::Descriptor(pByte data, int startIndex)throw(std::invalid_argument)
{
    if (sizeof(data.get()) < 2)
        throw std::invalid_argument("Invalid descriptor");

    if (startIndex + 2 + data[startIndex + 1]
        > strlen(reinterpret_cast<char*>(data.get())))
        throw std::invalid_argument("Invalid descriptor");

    mData = pByte(new byte[2 + data[startIndex + 1]]);
    for (int i = 0; i < sizeof(mData.get()); i++)
        mData[i] = data[i + startIndex];
}

byte Descriptor::GetTag(void)
{
    return mData[0];
}

byte Descriptor::GetLength(void)
{
    return mData[1];
}

pByte Descriptor::GetData(void)
{
    return mData;
}

///////////////////////////////
/**   implement PcrPacket   **/
///////////////////////////////
PcrPacket::PcrPacket(Int64 pcr, byte counter, ushort pid)
{
    SetPID(pid);
    SetContinuityCounter(counter);
    mData[3] &= 0x0f; // adaptation field only, no payload
    mData[3] |= 0x20; // adaptation field only, no payload
    mData[4] = 183; // length
    mData[DATA_HEADER] = 0x10; // only PCR present
    Int64 tsClockValue = pcr / 300;
    Int64 tsOffsetValue = pcr % 300;
    mData[6] = (byte)((tsClockValue & 0x1fe000000LL) >> 25);
    mData[7] = (byte)((tsClockValue & 0x1fe0000) >> 17);
    mData[8] = (byte)((tsClockValue & 0x1fe00) >> 9);
    mData[9] = (byte)((tsClockValue & 0x1fe) >> 1);
    if ((tsClockValue & 0x1) == 0)
        mData[10] &= 0x7f;
    if ((tsOffsetValue & 0x100) == 0)
        mData[10] &= 0xfe;
    mData[11] = (byte)(tsOffsetValue & 0xff);
    for (int i = 12; i < Constants::TS_SIZE; i++)
        mData[i] = 0xff;
}

///////////////////////////////
/**   implement PesPacket   **/
///////////////////////////////
PesPacket::PesPacket(pByte buff, int offset, int length, ushort pid)
{
    mData = pByte(new byte[length]);
    SetPID(pid);
    AddData(buff, offset, length);
    SetPriority(false);
}

bool PesPacket::GetPriority(void)
{
    return mPriority;
}

void PesPacket::SetPriority(bool value)
{
    mPriority = value;
}

pByte PesPacket::GetData(void)
{
    return mData;
}

pByte PesPacket::GetPayload(void)
{
    boost::shared_ptr<PesHeader> ph = GetHeader();
    if(!ph) return GetData();
    std::vector<byte> buff;
    Utility::AddRange(buff, &mData[9 + ph.get()->GetHeaderLength()]);
    return Utility::ToString(buff);
}

byte PesPacket::GetByte(int i)
{
    return mData[i];
}

void PesPacket::SetByte(int i, byte value)
{
    mData[i] = value;
}

ushort PesPacket::GetPID(void)
{
    return mPID;
}

void PesPacket::SetPID(ushort id)
{
    mPID = id;
}

bool PesPacket::GetComplete(void)
{
    if(sizeof(mData.get()) < 6) return false;
    ushort len = (ushort)((mData[4] << 8) + mData[DATA_HEADER]);
    if(len == 0) return false;
    if(sizeof(mData.get()) != len + 6) return false;
    return true;
}

void PesPacket::SetComplete(bool value)
{
    if(value)
    {
        ushort len = (ushort)(sizeof(mData.get()) - 6);
        if (sizeof(mData.get()) > (0xffff - 6))
            len = 0;
        mData[4] = (byte)((len >> 8) & 0xff);
        mData[DATA_HEADER] = (byte)(len & 0xff);
    }
}

boost::shared_ptr<PesHeader> PesPacket::GetHeader(void)
{
    boost::shared_ptr<PesHeader> ph;
    try
    {
        ph = boost::shared_ptr<PesHeader>(new PesHeader(mData));
        return ph;
    }
    catch(...)
    {
        // no valid header (yet)
        ph.reset();
        return ph;
    }
}

void PesPacket::AddData(pByte moredata)
{
    std::vector<byte> data;
    Utility::AddRange(data, mData);
    Utility::AddRange(data, moredata);
    mData = Utility::ToString(data);
}

void PesPacket::AddData(pByte buff, int offset, int len)
{
    std::vector<byte> data;
    Utility::AddRange(data, mData);
    for(int i = 0; i < len; i++)
        data.push_back(buff[offset + i]);
    mData = Utility::ToString(data);
}

byte PesPacket::GetBaseId(void)
{
    if (sizeof(mData.get()) > 3)
        return mData[3];
    return 0;
}

byte PesPacket::GetExtendedId(void)
{
    if ((sizeof(mData.get()) > 8) && sizeof(mData.get()) > (8 + mData[8]))
        return mData[9 + mData[8]];
    return 0;
}

UInt32 PesPacket::GetExtendedType(void)
{
    if ((sizeof(mData.get()) > 8) && sizeof(mData.get()) > (11 + mData[8]))
    {
        UInt32 format = (UInt32)mData[9 + mData[8]] << 24;
        format += (UInt32)mData[10 + mData[8]] << 16;
        format += (UInt32)mData[11 + mData[8]] << 8;
        format += (UInt32)mData[12 + mData[8]];
        return format;
    }
    return 0;
}

PesHeader::PesHeader(pByte data)throw(std::invalid_argument)
{
    if (sizeof(data.get()) < 9)
        throw std::invalid_argument("Invalid PES header length");
    if (data[0] != 0x00 || data[1] != 0x00 || data[2] != 0x01)
        throw std::invalid_argument("Invalid PES prefix");
    int hlen = 9 + data[8];
    int plen = 6 + (data[4] << 8) + data[DATA_HEADER];
    if (plen != 6 && hlen > plen)
        throw new std::invalid_argument("Invalid PES header/packet length");
    if (sizeof(data.get()) < hlen)
        throw std::invalid_argument("PES Header too short");
    mData = data;
}

byte PesHeader::GetStreamId(void)
{
    return mData[3];
}

byte PesHeader::GetByte(int i)
{
    return mData[i];
}

void PesHeader::SetByte(int i, byte data)
{
    mData[i] = data;
}

byte PesHeader::GetHeaderLength(void)
{
    return mData[8];
}

int PesHeader::GetTotalHeaderLength(void)
{
    return 9 + GetHeaderLength();
}

ushort PesHeader::GetPacketLength(void)
{
    return (ushort)((mData[4] << 8) + mData[DATA_HEADER]);
}

bool PesHeader::HasPts(void)
{
    if(sizeof(mData.get()) > 13)
        return (mData[7] & 0x80) > 0;
    return false;
}

bool PesHeader::HasDts(void)
{
    if(sizeof(mData.get()) > 18 )
        return (mData[7] & 0x40) > 0;
    return false;
}

Int64 PesHeader::GetPts(void)throw(std::invalid_argument)
{
    if (HasPts() == false)
        throw std::invalid_argument("No Pts available");
    Int64 ret = 0;
    ret += ((Int64)(mData[9] & 0x0e)) << 29;
    ret += ((Int64)(mData[10])) << 22;
    ret += ((Int64)(mData[11] & 0xfe)) << 14;
    ret += ((Int64)(mData[12])) << 7;
    ret += ((Int64)(mData[13] & 0xfe)) >> 1;
    return ret;
}

void PesHeader::SetPts(Int64 value)throw(std::invalid_argument)
{
    if (HasPts() == false)
        throw std::invalid_argument("No Pts available");
    byte old = (byte)(mData[9] & 0xf1);
    mData[9] = (byte)(((value & 0x1C0000000LL) >> 29) | old);
    mData[10] = (byte)((value & 0x3fC00000LL) >> 22);
    mData[11] = (byte)(((value & 0x3f8000LL) >> 14) | 0x01);
    mData[12] = (byte)((value & 0x7f80LL) >> 7);
    mData[13] = (byte)(((value & 0x7fLL) << 1) | 0x01);
}

Int64 PesHeader::GetDts(void)throw(std::invalid_argument)
{
    if (HasDts() == false)
        throw std::invalid_argument("No Dts available");
    Int64 ret = 0;
    ret += ((Int64)(mData[14] & 0x0e)) << 29;
    ret += ((Int64)(mData[15])) << 22;
    ret += ((Int64)(mData[16] & 0xfe)) << 14;
    ret += ((Int64)(mData[17])) << 7;
    ret += ((Int64)(mData[18] & 0xfe)) >> 1;
    return ret;
}

void PesHeader::SetDts(Int64 value)throw(std::invalid_argument)
{
    if (HasDts() == false)
        throw std::invalid_argument("No Dts available");
    byte old = (byte)(mData[14] & 0xf1);
    mData[14] = (byte)(((value & 0x1C0000000LL) >> 29) | old);
    mData[15] = (byte)((value & 0x3fC00000LL) >> 22);
    mData[16] = (byte)(((value & 0x3f8000LL) >> 14) | 0x01);
    mData[17] = (byte)((value & 0x7f80LL) >> 7);
    mData[18] = (byte)(((value & 0x7fLL) << 1) | 0x01);
}

byte PesHeader::GetExtention2(void)
{
    int offset = 6;
    if((mData[offset] & 0xc0) != 0x80)
        return 0; // first two bits must be '10'
    byte PTS_DTS_flags = (byte)(mData[offset + 1] & 0xc0);
    byte ESCR_flag = (byte)(mData[offset + 1] & 0x20);
    byte ES_rate_flag = (byte)(mData[offset + 1] & 0x10);
    byte DSM_trick_mode_flag = (byte)(mData[offset + 1] & 0x08);
    byte additional_copy_info_flag = (byte)(mData[offset + 1] & 0x04);
    byte PES_CRC_flag = (byte)(mData[offset + 1] & 0x02);
    byte PES_extension_flag = (byte)(mData[offset + 1] & 0x01);
    if(mData[offset + 2] == 0)
        return 0;
    int length = offset + mData[offset + 2] + 3;
    if(sizeof(mData.get()) < length)
        return 0;
    offset += 3;
    if(PTS_DTS_flags == 0x80)
        offset += 5;
    if(PTS_DTS_flags == 0xc0)
        offset += 10;
    if(ESCR_flag > 0)
        offset += 6;
    if(ES_rate_flag > 0)
        offset += 3;
    if(DSM_trick_mode_flag > 0)
        offset += 1;
    if(additional_copy_info_flag > 0)
        offset += 1;
    if(PES_CRC_flag > 0)
        offset += 2;
    if(PES_extension_flag == 0)
        return 0;
    byte PES_private_data_flag = (byte)(mData[offset] & 0x80);
    byte pack_header_field_flag = (byte)(mData[offset] & 0x40);
    byte program_packet_sequence_counter_flag
        = (byte)(mData[offset] & 0x20);
    byte PSTD_mDataer_flag = (byte)(mData[offset] & 0x10);
    byte PES_extension_flag_2 = (byte)(mData[offset] & 0x01);
    offset++;
    if(PES_private_data_flag > 0)
        offset += 25;
    if(pack_header_field_flag > 0)
        offset += (mData[offset] + 1);
    if(program_packet_sequence_counter_flag > 0)
        offset += 2;
    if(PSTD_mDataer_flag > 0)
        offset += 2;
    if(PES_extension_flag_2 == 0)
        return 0;
    if(mData[offset] != 0x81)
        return 0;
    return mData[offset + 1];
}

pByte PesHeader::GetData(void)
{
    return mData;
}

/////////////////////////////////////
/**   implement VC1SequenceInfo   **/
/////////////////////////////////////
VC1SequenceInfo::VC1SequenceInfo(pByte data, int offset)
{
    UInt32 marker = 0xffffffff;
    for(; offset < sizeof(data.get()); offset++)
    {
        marker = marker << 8;
        marker &= 0xffffff00;
        marker += data[offset];
        if(marker == Constants::VC1_SEQ_SC)
            break;
    }
    offset++;
    if(offset < sizeof(data.get()))
    {
        // sequence header
        mData = pByte(new byte[sizeof(data.get()) - offset]);
        for(int i = 0; offset < sizeof(data.get()); i++, offset++)
            mData[i] = data[offset];
    }
    else
        mData.reset();
}

int VC1SequenceInfo::GetHeight(void)
{
    if(!mData && sizeof(mData.get()) > 4)
        return ((((mData[3] & 0x0f) << 8) + mData[4]) << 1) + 2;
    else
        return -1;
}

int VC1SequenceInfo::GetWidth(void)
{
    if(!mData && sizeof(mData.get()) > 3)
        return (((mData[2] << 4) + ((mData[3] & 0xf0) >> 4)) << 1) + 2;
    else
        return -1;
}

bool VC1SequenceInfo::Valid(void)
{
    return !mData;
}

bool VC1SequenceInfo::Interlaced(void)
{
    if(!mData && sizeof(mData.get()) > DATA_HEADER)
        return ((mData[DATA_HEADER] & 0x40) > 0);
    return false;
}

bool VC1SequenceInfo::DisplayExt(void)
{
    if(!mData && sizeof(mData.get()) > DATA_HEADER)
        return ((mData[DATA_HEADER] & 0x02) > 0);
    return false;
}

bool VC1SequenceInfo::AspectFlag(void)
{
    if(DisplayExt() && sizeof(mData.get()) > 9)
        return ((mData[9] & 0x10) > 0);
    return false;
}

byte VC1SequenceInfo::GetVc1AspectRatio(void)
{
    if(AspectFlag() && sizeof(mData.get()) > 9)
        return (byte)(mData[9] & 0x0f);
    return 0;
}

bool VC1SequenceInfo::FrameFlag(void)
{
    if(AspectFlag())
    {
         if(GetVc1AspectRatio() == 15 && sizeof(mData.get()) > 12)
             return ((mData[12] & 0x80) > 0);
         else if(GetVc1AspectRatio() != 15 && sizeof(mData.get()) > 10)
             return ((mData[10] & 0x80) > 0);
         else
             return false;
    }
    else if(sizeof(mData.get()) > 9)
         return ((mData[9] & 0x08) > 0);
    else
         return false;
}

bool VC1SequenceInfo::FrameRateIndicatorFlag(void)
{
    if(FrameFlag())
    {
        if(AspectFlag())
        {
            if (GetVc1AspectRatio() == 15 && sizeof(mData.get()) > 12)
                return ((mData[12] & 0x40) > 0);
            else if (GetVc1AspectRatio() != 15 && sizeof(mData.get()) > 10)
                return ((mData[10] & 0x40) > 0);
            else
                return false;
        }
        else if (sizeof(mData.get()) > 9)
            return ((mData[9] & 0x04) > 0);
        else
            return false;
    }
    else
        return false;
}

AspectRatio VC1SequenceInfo::GetAspectRatio(void)
{
    if(GetVc1AspectRatio() == 1)
    {
        if(GetWidth() == 1920 && GetHeight() == 1080)
            return a16_9;
        if(GetWidth() == 1280 && GetHeight() == 720)
            return a16_9;
        if(GetWidth() == 640 && GetHeight() == 480)
            return a4_3;
    }
    if(GetVc1AspectRatio() >= 2 && GetVc1AspectRatio() <= 5)
        return a4_3;
    if(GetVc1AspectRatio() >= 6 && GetVc1AspectRatio() <= 9)
        return a16_9;
    if(GetVc1AspectRatio() >= 10 && GetVc1AspectRatio() <= 11)
        return a4_3;
    if(GetVc1AspectRatio() >= 12 && GetVc1AspectRatio() <= 13)
        return a16_9;
    return AR_Reserved;
}

VideoFormat VC1SequenceInfo::GetVideoFormat(void)
{
    if (GetHeight() == 480 && Interlaced() == true)
        return i480;
    else if (GetHeight() == 480 && Interlaced() == false)
        return p480;
    else if (GetHeight() == 576 && Interlaced() == true)
        return i576;
    else if (GetHeight() == 576 && Interlaced() == false)
        return p576;
    else if (GetHeight() == 720 && Interlaced() == false)
        return p720;
    else if (GetHeight() == 1080 && Interlaced() == true)
        return i1080;
    else if (GetHeight() == 1080 && Interlaced() == false)
        return p1080;
    return VF_Reserved;
}

FrameRate VC1SequenceInfo::GetFrameRate(void)
{
    if(false == FrameFlag())
        return FR_Reserved;
    if(false == FrameRateIndicatorFlag())
    {
        byte FrameRateNr = 0;
        byte FrameRateDr = 0;
        if(AspectFlag())
        {
            if(GetVc1AspectRatio() == 15 && sizeof(mData.get()) > 13)
            {
                FrameRateNr = (byte)(((mData[12] & 0x3f) << 2)
                    + ((mData[13] & 0xc0) >> 6));
                FrameRateDr = (byte)((mData[13] & 0x3c) >> 2);
            }
            else if(GetVc1AspectRatio() != 15 && sizeof(mData.get()) > 11)
            {
                FrameRateNr = (byte)(((mData[10] & 0x3f) << 2)
                    + ((mData[11] & 0xc0) >> 6));
                FrameRateDr = (byte)((mData[11] & 0x3c) >> 2);
            }
        }
        else if(sizeof(mData.get()) > 11)
        {
            FrameRateNr = (byte)(((mData[9] & 0x03) << 6)
                + ((mData[10] & 0xfc) >> 2));
            FrameRateDr = (byte)(((mData[10] & 0x03) << 2)
                + ((mData[11] & 0xc0) >> 6));
        }

        if(FrameRateNr == 1 && FrameRateDr == 2)
                return f23_976;
        else if(FrameRateNr == 1 && FrameRateDr == 1)
                return f24;
        else if(FrameRateNr == 2 && FrameRateDr == 1)
                return f25;
        else if(FrameRateNr == 3 && FrameRateDr == 2)
                return f29_97;
        else if(FrameRateNr == 4 && FrameRateDr == 1)
                return f50;
        else if(FrameRateNr == 5 && FrameRateDr == 2)
                return f59_94;
    }
    return FR_Reserved;
}

///////////////////////////////
/**   implement PatPacket   **/
///////////////////////////////
PatPacket::PatPacket(void)
{
    SetPID(Constants::PAT_PID);
    SetTableId(Constants::PAT_TABLE_ID);
    SetLength(9);
    SetTransportStreamId(1);
    RefreshCrc();
}

PatPacket::PatPacket(pByte data)throw(std::invalid_argument)
{
    if(GetTableId() != Constants::PAT_TABLE_ID)
        throw std::invalid_argument(
        "packet does not contain a valid PAT table ID");
    if (0 != GetPID())
        throw std::invalid_argument("packet does not contain a valid PAT PID");
}

ushort PatPacket::GetTransportStreamId(void)
{
    return GetNumberId();
}

void PatPacket::SetTransportStreamId(ushort value)
{
    SetNumberId(value);
    RefreshCrc();
}

std::vector<boost::shared_ptr<ProgramInfo> > PatPacket::GetPrograms(void)
{
    std::vector<boost::shared_ptr<ProgramInfo> > programs;
    if (GetProgramInfoLength() == 0)
        return programs;
    boost::shared_ptr<ProgramInfo> program;
    for (int i = 0; i < GetProgramInfoLength(); i += 4)
    {
        program = boost::shared_ptr<ProgramInfo>(
            new ProgramInfo(mData, 13 + GetPointerSize() + i));
        programs.push_back(program);
    }
    return programs;
}

void PatPacket::SetPrograms(std::vector<boost::shared_ptr<ProgramInfo> > value)
    throw(std::invalid_argument)
{
    if (value.size() == 0)
    {
        if (GetProgramInfoLength() == 0)
            return;
        SetLength(9);
        RefreshCrc();
    }
    else
    {
        if ((value.size() * 4) + 17 + GetPointerSize()
            > Constants::TS_SIZE)
            throw std::invalid_argument("program info data too long");
        SetLength((ushort)(9 + value.size() * 4));
        int index = 13 + GetPointerSize();
        for(int pi = 0; pi < GetProgramInfoLength() / 4; pi++)
        {
            for (int i = 0; i < 4; i++)
                mData[index + i] = value[pi]->GetData()[i];
            index += 4;
        }
        RefreshCrc();
    }
}

ushort PatPacket::GetProgramInfoLength(void)
{
    return (ushort)(GetLength() - 9);
}

DTCP_Descriptor::DTCP_Descriptor(pByte data, int startIndex)
    throw(std::invalid_argument):Descriptor(data, startIndex)
{
    if (sizeof(data.get()) < 6)
        throw std::invalid_argument("Invalid DTCP descriptor");
    if (GetTag() != Constants::DTCP_DESCRIPTOR_TAG)
        throw std::invalid_argument("Invalid DTCP descriptor tag");
    if (GetLength() < 4)
        throw std::invalid_argument("Invalid DTCP descriptor length");
    if (data[startIndex + 2] != 0x0f || data[startIndex + 3] != 0xff)
        throw std::invalid_argument("Invalid DTCP descriptor CA system ID");
}

DtcpCci DTCP_Descriptor::GetCopyStatus(void)
{
    return (DtcpCci)(mData[4] & 0x3);
}

bool DTCP_Descriptor::GetAnalogConstrain(void)
{
    return ((mData[5] & 0x4) == 0);
}

bool DTCP_Descriptor::GetMacrovision(void)
{
    return ((mData[5] & 0x3) > 0);
}

/////////////////////////////
/**   implement MlpInfo   **/
/////////////////////////////
MlpInfo::MlpInfo(pByte data, int offset)
{
    UInt32 marker = 0xffffffff;
    for (; offset < sizeof(data.get()); offset++)
    {
        marker = (UInt32)marker << 8;
        marker &= 0xffffff00;
        marker += data[offset];
        if (marker == Constants::MLP_SYNC)
            break;
    }
    offset++;
    if (offset < sizeof(data.get()))
    {
        // sequence header
        mData = pByte(new byte[sizeof(data.get()) - offset]);
        for (int i = 0; offset < sizeof(data.get()); i++, offset++)
            mData[i] = data[offset];
    }
    else
        mData.reset();
}

AspectRatio MlpInfo::GetAspectRatio(void)
{
    return AR_Reserved;
}

FrameRate MlpInfo::GetFrameRate(void)
{
    return FR_Reserved;
}

VideoFormat MlpInfo::GetVideoFormat(void)
{
    return VF_Reserved;
}

pByte MlpInfo::GetElementaryDescriptors(void)
{
    std::vector<byte> descriptors;
    for(int i = 0; i < sizeof(Constants::ac3_registration_descriptor); i++)
    {
        descriptors.push_back(Constants::ac3_registration_descriptor[i]);
    }
    pByte ad = GetAC3AudioDescriptor();
    for(int i = 0; i < sizeof(ad.get()); i++)
    {
       descriptors.push_back(ad[i]); 
    }
    return Utility::ToString(descriptors);
}

AudioPresentationType MlpInfo::GetAudioPresentationType(void)
{
    return multi;
}

SamplingFrequency MlpInfo::GetSamplingFrequency(void)
{
    switch (GetSampleRateCode())
    {
    case 0:
        return kHz48;
    case 1:
        return kHz96;
    case 2:
        return kHz192;
    case 8: // 44.1kHz
    case 9: // 88.2kHz
    case 10: // 176.4kHz
    default:
        return SF_Reserved;
    }
}

pByte MlpInfo::GetAC3AudioDescriptor(void)
{
    std::vector<byte> desc;
    desc.push_back(0x81);
    desc.push_back(0x00);
    desc.push_back((byte)((GetSampleRateCode() << 5) | 0x08));
    desc.push_back(200);
    desc.push_back((byte)(0x0f));
    desc[1] = (byte)(desc.size() - 2);
    return Utility::ToString(desc);
}

byte MlpInfo::GetSampleRateCode(void)
{
    if (sizeof(mData.get()) > 0)
        return (byte)(mData[0] >> 4);
    return 0x00;
}

///////////////////////////////////
/**   implement class AC3Info   **/
///////////////////////////////////
readonly int AC3Info::len48k[] = {
             128,  128,  160,  160,  192,  192,  224,  224,
             256,  256,  320,  320,  384,  384,  448,  448,
             512,  512,  640,  640,  768,  768,  896,  896,
            1024, 1024, 1280, 1280, 1536, 1536, 1792, 1792,
            2048, 2048, 2304, 2304, 2560, 2560 };

readonly int AC3Info::len44k[] = {
             138,  140,  174,  176,  208,  210,  242,  244,
             278,  280,  348,  350,  416,  418,  486,  488,
             556,  558,  696,  698,  834,  836,  974,  976,
            1114, 1116, 1392, 1394, 1670, 1672, 1950, 1952,
            2228, 2230, 2506, 2508, 2786, 2788 };

readonly int AC3Info::len32k[] = { 
             192,  192,  240,  240,  288,  288,  336,  336,
             384,  384,  480,  480,  576,  576,  672,  672,
             768,  768,  960,  960, 1152, 1152, 1344, 1344,
            1536, 1536, 1920, 1920, 2304, 2304, 2688, 2688,
            3072, 3072, 3456, 3456, 3840, 3840 };

int AC3Info::GetMaxFrameLength(void)
{
    return len32k[sizeof(len32k) - 1];
}

int AC3Info::GetFrameLength(void)
{
    if(GetSyntaxType() == Standard
    || GetSyntaxType() == Alternative)
    {
        byte index = (byte)(mData[2] & 0x3f);
        if (index < 38)
        {
            switch (GetSampleRateCode())
            {
            case 00:
                return len48k[mData[2] & 0x3f];
            case 01:
                return len44k[mData[2] & 0x3f];
            case 02:
                return len32k[mData[2] & 0x3f];
            }
        }
    }
    else if (GetSyntaxType() == Enhanced)
        return (((mData[0] & 0x03) << 8) + mData[1] + 1) << 1;
    return 0;
}

bool AC3Info::Valid(void)
{
    return mValid && GetSyntaxType() != Invalid;
}

byte AC3Info::GetSampleRateCode(void)
{
    if (sizeof(mData.get()) > 2)
        return (byte)(mData[2] >> 6);
    return 0x03;
}

byte AC3Info::GetBsid(void)
{
    if (sizeof(mData.get()) > 3)
        return (byte)(mData[3] >> 3);
    return 0;
}

byte AC3Info::GetBsmod(void)
{
    if (sizeof(mData.get()) > 3
    && (GetSyntaxType() == Standard
    || GetSyntaxType() == Alternative))
        return (byte)(mData[3] & 0x07);
    return 0;
}

byte AC3Info::GetAcmod(void)
{
    if (sizeof(mData.get()) > 4
    && (GetSyntaxType() == Standard
    || GetSyntaxType() == Alternative))
        return (byte)(mData[4] >> 5);
    else if (sizeof(mData.get()) > 2
    && GetSyntaxType() == Enhanced)
        return (byte)((mData[2] >> 1) & 0x07);
    return 0x07;
}

bool AC3Info::IsIndependentStream(void)
{
    if (Enhanced != GetSyntaxType())
        return true;
    byte res = (byte)(mData[0] >> 6);
    if (res != 1)
        return true;
    return false;
}

Ac3SyntaxType AC3Info::GetSyntaxType(void)
{
    switch (GetBsid())
    {
    case (byte)Standard:
        return Standard;
    case (byte)Alternative:
        return Alternative;
    case (byte)Enhanced:
        return Enhanced;
    }
    return Invalid;
}

pByte AC3Info::GetAC3AudioDescriptor(void)
{
    std::vector<byte> desc;
    desc.push_back(0x81);
    desc.push_back((byte)(desc.size() - 2));
    desc.push_back((byte)((GetSampleRateCode() << 5) | GetBsid()));
    desc.push_back(200);
    desc.push_back((byte)((GetBsmod() << 5) | (GetAcmod() << 1) | 1));
    return Utility::ToString(desc);
}

AC3Info::AC3Info(pByte data, int offset)
{
    ushort marker = 0xffff;
    for (; offset < sizeof(data.get()); offset++)
    {
        marker = (ushort)(marker << 8);
        marker &= 0xff00;
        marker += data[offset];
        if (marker == Constants::AC3_SYNC)
            break;
    }
    offset++;
    if (offset < sizeof(data.get()))
    {
        // sequence header
        mData = pByte(new byte[sizeof(data.get()) - offset]);
        for (int i = 0; offset < sizeof(data.get()); i++, offset++)
            mData[i] = data[offset];
    }
    else
        mData.reset();
}

AudioPresentationType AC3Info::GetAudioPresentationType(void)
{
    switch (GetAcmod())
    {
    case 0x00:
        return stereo;
    case 0x01:
        return mono;
    case 0x02:
        return stereo;
    default:
        return multi;
    }
}

SamplingFrequency AC3Info::GetSamplingFrequency(void)
{
    switch (GetSampleRateCode())
    {
    case 0x00:
        return kHz48;
    }
    return SF_Reserved;
}

pByte AC3Info::GetElementaryDescriptors(void)
{
    std::vector<byte> descriptors;
    for(int i = 0; i < sizeof(Constants::ac3_registration_descriptor); i++)
    {
        descriptors.push_back(Constants::ac3_registration_descriptor[i]);
    }
    pByte ac3ad = GetAC3AudioDescriptor();
    for(int j = 0; j < sizeof(ac3ad.get()); j++)
    {
        descriptors.push_back(ac3ad[j]);
    }
    return Utility::ToString(descriptors);
}

AspectRatio AC3Info::GetAspectRatio(void)
{
    return AR_Reserved;
}

FrameRate AC3Info::GetFrameRate(void)
{
    return FR_Reserved;
}

VideoFormat AC3Info::GetVideoFormat(void)
{
    return VF_Reserved;
}

/////////////////////////////////
/** implement class DtsInfo   **/
/////////////////////////////////
DtsInfo::DtsInfo(pByte data, int offset)
{
    UInt32 marker = 0xffffffff;
    for(; offset < sizeof(data.get()); offset++)
    {
        marker = (UInt32)marker << 8;
        marker &= 0xffffff00;
        marker += data[offset];
        if(marker == Constants::DTS_SYNC)
            break;
    }
    offset++;
    if(offset < sizeof(data.get()))
    {
        // sequence header
        mData = pByte(new byte[sizeof(data.get()) - offset]);
        for(int i = 0; offset < sizeof(data.get()); i++, offset++)
            mData[i] = data[offset];
    }
    else
        mData.reset();
}

ushort DtsInfo::GetFrameSize(void)
{
    indicator = 14;
    ushort ret = 0;
    for (int i = 0; i < 14; i++)
    {
        ret <<= 1;
        ret |= GetNextBit();
    }
    ret++;
    return ret;
}

byte DtsInfo::GetAmode(void)
{
    indicator = 28;
    byte ret = 0;
    for (int i = 0; i < 6; i++)
    {
        ret <<= 1;
        ret |= GetNextBit();
    }
    return ret;
}

byte DtsInfo::GetSampleFreq(void)
{
    indicator = 34;
    byte ret = 0;
    for (int i = 0; i < 4; i++)
    {
        ret <<= 1;
        ret |= GetNextBit();
    }
    return (SamplingFrequency)ret;
}

bool DtsInfo::GetExtAudio(void)
{
    indicator = 51;
    if (1 == GetNextBit())
        return true;
    return false;
}

byte DtsInfo::GetExtAudioId(void)
{
    if (false == GetExtAudio())
        return 0xff;
    indicator = 48;
    byte ret = 0;
    for (int i = 0; i < 3; i++)
    {
        ret <<= 1;
        ret |= GetNextBit();
    }
    return ret;
}

AudioPresentationType DtsInfo::GetAudioPresentationType(void)
{
    switch (GetAmode())
    {
    case 0x00:
        return mono;
    case 0x01:
    case 0x02:
    case 0x03:
    case 0x04:
        return stereo;
    default:
        return multi;
    }
}

SamplingFrequency DtsInfo::GetSamplingFrequency(void)
{
    switch (mSamplingFrequency)
    {
    case 0xd:
        return kHz48;
    }
    return SF_Reserved;
}

pByte DtsInfo::GetElementaryDescriptors(void)
{
    // DTS registration descriptor
    pByte regdesc = pByte(new byte[6]);
    regdesc[0] = 0x05;
    regdesc[1] = 0x04;
    regdesc[2] = 0x44;
    regdesc[3] = 0x54;
    regdesc[4] = 0x53;
    if(sizeof(mData.get()) < 0x400)
        regdesc[5] = 0x31;
    else if(sizeof(mData.get()) < 0x800)
        regdesc[5] = 0x32;
    else
        regdesc[5] = 0x33;
    return regdesc;
}

AspectRatio DtsInfo::GetAspectRatio(void)
{
    return AR_Reserved;
}

FrameRate DtsInfo::GetFrameRate(void)
{
    return FR_Reserved;
}

VideoFormat DtsInfo::GetVideoFormat(void)
{
    return VF_Reserved;
}

///////////////////////////////////
/** implement class Mpeg2Info   **/
///////////////////////////////////
Mpeg2Info::Mpeg2Info(pByte data, int offset)
{
    UInt32 marker = 0xffffffff;
    int oldOffset = offset;
    for (; offset < sizeof(data.get()); offset++)
    {
        marker = (UInt32)marker << 8;
        marker &= 0xffffff00;
        marker += data[offset];
        if (marker == Constants::MPEG2_SEQ_CODE)
            break;
    }
    offset++;
    if(offset < sizeof(data.get()))
    {
        // sequence header
        mData = pByte(new byte[sizeof(data.get()) - offset]);
        for(int i = 0; offset < sizeof(data.get()); i++, offset++)
            mData[i] = data[offset];
        mpgext = Mpeg2Ext(data, oldOffset);
    }
    else
        mData.reset();
}

AspectRatio Mpeg2Info::GetAspectRatio(void)
{
    switch (GetAspect())
    {
    case 0x01:
        if (GetVertical() == 1080 || GetVertical() == 1088 || GetVertical() == 720)
            return a16_9;
        else
            return a4_3;
    case 0x02:
        return a4_3;
    case 0x03:
        return a16_9;
    default:
        return AR_Reserved;
    }
}

FrameRate Mpeg2Info::GetFrameRate(void)
{
    switch (GetFrameRateCode())
    {
    case 0x01:
        return f23_976;
    case 0x02:
        return f24;
    case 0x03:
        return f25;
    case 0x04:
        return f29_97;
    case 0x06:
        return f50;
    case 0x07:
        return f59_94;
    default:
        return FR_Reserved;
    }
}

VideoFormat Mpeg2Info::GetVideoFormat(void)
{
    if (GetVertical() == 1080 || GetVertical() == 1088)
    {
        if(GetProgressive())
            return p1080;
        else
            return i1080;
    }
    else if (GetVertical() == 576)
    {
    if (GetProgressive())
        return p576;
    else
        return i576;
    }
    else if (GetVertical() == 720)
        return p720;
    else if (GetVertical() == 480)
    {
    if (GetProgressive())
        return p480;
    else
        return i480;
    }
    return VF_Reserved;
}

pByte Mpeg2Info::GetElementaryDescriptors(void)
{
    return GetMpeg2VideoRegistrationDescriptor();
}

AudioPresentationType Mpeg2Info::GetAudioPresentationType(void)
{
    return AP_Reserved;
}

SamplingFrequency Mpeg2Info::GetSamplingFrequency(void)
{
    return SF_Reserved;
}

pByte Mpeg2Info::GetMpeg2VideoRegistrationDescriptor(void)
{
    pByte data = pByte(new byte[10]);
    data[0] = 0x05;
    data[1] = 0x08;
    data[2] = 0x48;
    data[3] = 0x44;
    data[4] = 0x4d;
    data[5] = 0x56;
    data[6] = 0xff;
    data[7] = 0x02;
    data[8] = (byte)((((byte)GetVideoFormat()) << 4) | ((byte)GetFrameRate()));
    data[9] = (byte)((((byte)GetAspectRatio()) << 4) | 0x0f);
    return data;
}

ushort Mpeg2Info::GetHorizontal(void)
{
    indicator = 0;
    ushort ret = 0;
    for (int i = 0; i < 12; i++)
    {
        ret <<= 1;
        ret |= GetNextBit();
    }
    return ret;
}

ushort Mpeg2Info::GetVertical(void)
{
    indicator = 12;
    ushort ret = 0;
    for (int i = 0; i < 12; i++)
    {
        ret <<= 1;
        ret |= GetNextBit();
    }
    return ret;
}

byte Mpeg2Info::GetAspect(void)
{
    indicator = 24;
    byte ret = 0;
    for (int i = 0; i < 4; i++)
    {
        ret <<= 1;
        ret |= GetNextBit();
    }
    return ret;
}

byte Mpeg2Info::GetFrameRateCode(void)
{
    indicator = 28;
    byte ret = 0;
    for (int i = 0; i < 4; i++)
    {
        ret <<= 1;
        ret |= GetNextBit();
    }
    return ret;
}

bool Mpeg2Info::GetProgressive(void)
{
    if (mpgext.mValid && mpgext.GetProgressive())
        return true;
    return false;
}

// implement class H264Info : ElementaryParse
H264Info::H264Info(pByte data, int offset)
{
    UInt32 marker = 0xffffffff;
    for (; offset < sizeof(data.get()); offset++)
    {
        marker = marker << 8;
        marker &= 0xffffff00;
        marker += data[offset];
        if ((marker & 0xffffff9f) == Constants::H264_PREFIX)
        {
                    break;
        }
    }
    offset++;
    if (offset < sizeof(data.get()))
    {
        // sequence parameter set
        mData = pByte(new byte[sizeof(data.get()) - offset]);
        for (int i = 0; offset < sizeof(data.get()); i++, offset++)
                    mData[i] = data[offset];
    }
    else
        mData.reset();
}

UInt32 H264Info::GetNextExpGolomb(void)
{
    int leadingZeroBits = -1;
    byte b = 0;
    for (; b == 0; leadingZeroBits++)
        b = GetNextBit();
    UInt32 codeNum = (UInt32)(1 << leadingZeroBits);
    codeNum -= 1;
    UInt32 part2 = 0;
    for (; leadingZeroBits > 0; leadingZeroBits-- )
    {
        b = GetNextBit();
        part2 = part2 << 1;
        part2 |= b;
    }
    codeNum += part2;
    return codeNum;
}

void H264Info::ScalingListSkip(int skip)
{
    int lastScale = 8;
    int nextScale = 8;
    for (int i = 0; i < skip; i++)
    {
        if (nextScale != 0)
        {
            int deltaScale = (int)GetNextExpGolomb();
            nextScale = (lastScale + deltaScale) % 256;
        }
    }
}

UInt32 H264Info::GetWidth(void)
{
    indicator = 24;
    GetNextExpGolomb();
    if (mData[0] == 100 || mData[0] == 110 || mData[0] == 122 || mData[0] == 144)
    {
        UInt32 chroma = GetNextExpGolomb();
        if (chroma == 3)
        GetNextBit();
        GetNextExpGolomb();
        GetNextExpGolomb();
        GetNextBit();
        if (GetNextBit() == 1)
        {
            for (int i = 0; i < 6; i++)
            {
                if (GetNextBit() == 1)
                {
                    ScalingListSkip(16);
                }
            }
            for (int i = 6; i < 8; i++)
            {
                if (GetNextBit() == 1)
                {
                    ScalingListSkip(64);
                }
            }
        }
    }
    GetNextExpGolomb();
    UInt32 pic = GetNextExpGolomb();
    if (pic == 0)
        GetNextExpGolomb();     
    else if (pic == 1)
    {
        GetNextBit();
        GetNextExpGolomb();
        GetNextExpGolomb();
        UInt32 numFrame = GetNextExpGolomb();
        for (int i = 0; i < numFrame; i++)
            GetNextExpGolomb();
    }
    GetNextExpGolomb();
    GetNextBit();
    UInt32 wid = GetNextExpGolomb();
    wid++;
    wid <<= 4;
    return wid;
}

UInt32 H264Info::GetHeigth(void)
{
    UInt32 width = GetWidth();
    UInt32 height = GetNextExpGolomb();
    height++;
    height <<= 4;
    return height;
}

pByte H264Info::GetHdmvVideoRegistrationDescriptor(void)
{
    pByte data = pByte(new byte[10]);
    data[0] = 0x05;
    data[1] = 0x08;
    data[2] = 0x48;
    data[3] = 0x44;
    data[4] = 0x4d;
    data[5] = 0x56;
    data[6] = 0xff;
    data[7] = 0x1b;
    data[8] = (byte)((((byte)GetVideoFormat()) << 4) | ((byte)GetFrameRate()));
    data[9] = (byte)((((byte)GetAspectRatio()) << 4) | 0x0f);
    return data;
}

VideoFormat H264Info::GetVideoFormat(void)
{
    UInt32 h = GetHeigth();
    if (h == 1080 || h == 1088)
        return p1080;
    else if (h == 720)
        return p720;
    else if (h == 576)
        return p576;
    else if (h == 480)
        return p480;
    else if (h == 540 || h == 544)
        return i1080;
    else if (h == 288)
        return i576;
    else if (h == 240)
        return i480;
    else
        return VF_Reserved;
}

AspectRatio H264Info::GetAspectRatio(void)
{
    if (GetVideoFormat() == i480 || GetVideoFormat() == i576)
        return a4_3;
    return a16_9;
}

FrameRate H264Info::GetFrameRate(void)
{
    if (GetVideoFormat() == p720)
        return f59_94;
    else if (GetVideoFormat() == p1080 || GetVideoFormat() == p480)
        return f23_976;
    else if (GetVideoFormat() == p576 || GetVideoFormat() == i576)
        return f25;
    else
        return f29_97;
}

pByte H264Info::GetElementaryDescriptors(void)
{
    return GetHdmvVideoRegistrationDescriptor();
}

AudioPresentationType H264Info::GetAudioPresentationType(void)
{
    return AP_Reserved;
}

SamplingFrequency H264Info::GetSamplingFrequency(void)
{
    return SF_Reserved;
}

// implement class SitPacket : TsTable
SitPacket::SitPacket(void)
{
}

SitPacket::SitPacket(pByte data)
{
}

// implement class PmPacket : TsTable
PmtPacket::PmtPacket(void)
{
    // table id = 02
    SetTableId(Constants::PMT_TABLE_ID);
    SetLength(13);
    // program id = 1
    SetProgramNumber(1);
    // reserved, PcrPID
    SetPcrPID(Constants::DEFAULT_PCR_PID);
    // reserved, program info length
    SetProgramDescriptorsLength(0);
    SetPID(Constants::DEFAULT_PMT_PID);
    RefreshCrc();
}

PmtPacket::PmtPacket(pByte data)throw(std::invalid_argument)
{
    if (GetTableId() != Constants::PMT_TABLE_ID)
        throw std::invalid_argument("packet does not contain a valid PMT table ID");
}

boost::shared_array<DTCP_Descriptor> PmtPacket::GetDtcpInfo(void)
{
    pByte descriptors = GetProgramDescriptorsData();
    boost::shared_array<DTCP_Descriptor> dt;
    if (!descriptors)
        return dt;

    for (int i = 0; i < sizeof(descriptors.get()); )
    {
        try
        {
            dt = boost::shared_array<DTCP_Descriptor>(new DTCP_Descriptor(descriptors, i));
            break;
        }
        catch (std::invalid_argument)
        {
            i += (2 + descriptors[i + 1]);
        }
    }

    return dt;
}

pByte PmtPacket::GetProgramDescriptorsData(void)
{
    pByte descriptors;
    if (GetProgramDescriptorsLength() == 0)
        return descriptors;
    descriptors = pByte(new byte[GetProgramDescriptorsLength()]);
    for (int i = 0; i < sizeof(descriptors.get()); i++)
    {
        descriptors[i] = mData[i + 17 + GetPointerSize()];
    }
    return descriptors;
}

void PmtPacket::SetProgramDescriptorsData(pByte value)
{
    if (!value || 0 == sizeof(value.get()))
    {
        if (GetProgramDescriptorsLength() > 0)
        {
            // need to remove existing descriptors
            pByte data = pByte(new byte[GetStreamInfoLength()]);
            int index = 17 + GetProgramDescriptorsLength() + GetPointerSize();
            // copy data between descriptor and crc
            for (int i = 0; i < sizeof(data.get()); i++)
            {
                data[i] = mData[index + i];
                mData[17 + i + GetPointerSize()] = data[i];
            }
            SetLength(GetLength() - GetProgramDescriptorsLength());
            SetProgramDescriptorsLength(0);
            RefreshCrc();
        }
        else
        {
            // nothing to do
            return;
        }
    }
    else
    {
        if (sizeof(value.get()) + GetLength() + GetPointerSize() + 5
            - GetProgramDescriptorsLength() > Constants::TS_SIZE)
            throw std::invalid_argument("program descriptors data too long");
        // need to remove existing descriptors
        pByte data = pByte(new byte[GetStreamInfoLength()]);
        int index = 17 + GetProgramDescriptorsLength() + GetPointerSize();
        // copy data between descriptor and crc
        for (int i = 0; i < sizeof(data.get()); i++)
            data[i] = mData[index + i];
        SetLength(GetLength() - GetProgramDescriptorsLength());
        SetLength(GetLength() + sizeof(value.get()));
        SetProgramDescriptorsLength(sizeof(value.get()));
        // copy the new descriptor
        for (int i = 0; i < sizeof(value.get()); i++)
            mData[17 + i + GetPointerSize()] = value[i];
        // recover data between descriptor and crc
        for (int i = 0; i < sizeof(data.get()); i++)
            mData[17 + sizeof(value.get()) + i + GetPointerSize()] = data[i];
        RefreshCrc();
    }
}
ushort PmtPacket::GetProgramNumber(void)
{
    return GetNumberId();
}

void PmtPacket::SetProgramNumber(ushort id)
{
    SetNumberId(id);
    return;
}

ushort PmtPacket::GetPcrPID(void)
{
    return PcrPID;
}

void PmtPacket::SetPcrPID(ushort pid)
{
    PcrPID = pid;
    return;
}

ushort PmtPacket::GetProgramDescriptorsLength(void)
{
    return (ushort)(((mData[15 + GetPointerSize()] & 0x0f) << 8) + mData[16 + GetPointerSize()]);
}

void PmtPacket::SetProgramDescriptorsLength(ushort value)
{
    mData[15 + GetPointerSize()] = (byte)(0xf0 | (byte)((value >> 8) & 0x0f));
    mData[16 + GetPointerSize()] = (byte)(value & 0xff);
}

ushort PmtPacket::GetStreamInfoLength(void)
{
    return (ushort)(GetLength() - 13 - GetProgramDescriptorsLength());
}

//////////////////////////////////////
/** implement class BluRayOutput   **/
//////////////////////////////////////
readonly byte BluRayOutput::index_bdmv[] = {
    0x49, 0x4e, 0x44, 0x58, 0x30, 0x31, 0x30, 0x30, 0x00, 0x00, 0x00, 0x4e, 0x00, 0x00, 0x00, 0x78,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x26, 0x40, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00,
    0x00, 0x00, 0x40, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x40, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x92, 0x00, 0x00, 0x00, 0x18,
    0x00, 0x00, 0x00, 0x01, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x01, 0x7e,
    0x49, 0x44, 0x45, 0x58, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x62, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x36, 0x10, 0x13, 0x00, 0x01,
    0x54, 0x52, 0x20, 0x30, 0x2e, 0x30, 0x2e, 0x31, 0x2e, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x06, 0xff, 0xff, 0x42, 0x20, 0x07, 0x08, 0x08, 0x00, 0x54, 0x53, 0x00, 0x90, 0x0a, 0x54,
    0x52, 0x20, 0x30, 0x2e, 0x30, 0x2e, 0x31, 0x2e, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02,
    0x00, 0x00, 0x00, 0x01, 0x30, 0x30, 0x30, 0x30, 0x30, 0x01, 0x00, 0x00, 0x00, 0x00 };

readonly byte BluRayOutput::MovieObject_bdmv[] = {
    0x4d, 0x4f, 0x42, 0x4a, 0x30, 0x31, 0x30, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xea, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x03, 0x80, 0x00, 0x00, 0x04, 0x50, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00,
    0x00, 0x00, 0x50, 0x40, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x82,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x21, 0x81, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x09, 0x50, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x0a, 0x00, 0x00, 0x00, 0x03, 0x50, 0x40, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00,
    0xff, 0xff, 0x48, 0x40, 0x03, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0xff, 0xff, 0x22, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x0a, 0x00, 0x00, 0x00, 0x04, 0x50, 0x40, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x00, 0x00, 0x48, 0x40, 0x03, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x21, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x21, 0x81, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x05, 0x50, 0x40, 0x00, 0x01, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x50, 0x40, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00,
    0x00, 0x01, 0x50, 0x40, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0xff, 0xff, 0x50, 0x40,
    0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x21, 0x81, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

readonly byte BluRayOutput::PlayList_00000_mpls[] = { 0x4d, 0x50, 0x4c, 0x53, 0x30, 0x31, 0x30, 0x30 };

readonly byte BluRayOutput::AppInfoPlayList[] = {
    0x00, 0x00, 0x00, 0x0e, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00 };

readonly byte BluRayOutput::PgStreamAttributes[] = { 0x05, 0x90, 0x65, 0x6e, 0x67, 0x00 };

readonly byte BluRayOutput::ClipInfo_0000_clpi[] = { 0x48, 0x44, 0x4d, 0x56, 0x30, 0x31, 0x30, 0x30 };

readonly byte BluRayOutput::TsTypeInfoBlock[] =  {
    0x00, 0x1e, 0x80, 0x48, 0x44, 0x4d, 0x56, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

BluRayOutput::BluRayOutput(std::string arg_path, TimeSpan arg_chapterLen)
{
    chapterLen = arg_chapterLen;
    path = arg_path;
    if(boost::filesystem::exists(path))
        boost::filesystem::remove(path);
    boost::filesystem::create_directories(path);
    boost::filesystem::create_directories(path / "BDMV");
    boost::filesystem::create_directories(path / "BDMV" / "AUXDATA");
    boost::filesystem::create_directories(path / "BDMV" / "BACKUP");
    boost::filesystem::create_directories(path / "BDMV" / "BACKUP" / "BDJO");
    boost::filesystem::create_directories(path / "BDMV" / "BACKUP" / "CLIPINF");
    boost::filesystem::create_directories(path / "BDMV" / "BACKUP" / "PLAYLIST");
    boost::filesystem::create_directories(path / "BDMV" / "BDJO");
    boost::filesystem::create_directories(path / "BDMV" / "CLIPINF");
    boost::filesystem::create_directories(path / "BDMV" / "JAR");
    boost::filesystem::create_directories(path / "BDMV" / "META");
    boost::filesystem::create_directories(path / "BDMV" / "PLAYLIST");
    boost::filesystem::create_directories(path / "BDMV" / "STREAM");
    boost::filesystem::create_directories(path / "CERTIFICATE");
    boost::filesystem::create_directories(path / "CERTIFICATE" / "BACKUP");

    std::string file_name;
    file_name = path.string() + "/BDMV/index.bdmv";
    std::ofstream file_index_bdmv(file_name.data());
    file_index_bdmv << index_bdmv;
    file_name = path.string() + "/BDMV/BACKUP/index.bdmv";
    std::ofstream file_index_bdmv_backup(file_name.data());
    file_index_bdmv_backup << index_bdmv;

    file_name = path.string() + "/BDMV/MovieObject.bdmv";
    std::ofstream file_movieobject_bdmv(file_name.data());
    file_movieobject_bdmv << MovieObject_bdmv;
    file_name = path.string() + "/BDMV/BACKUP/MovieObject.bdmv";
    std::ofstream file_movieobject_bdmv_backup(file_name.data());
    file_movieobject_bdmv_backup << MovieObject_bdmv;
}

void BluRayOutput::Author(std::vector<EpElement>& EpInfo, std::vector<StreamInfo>& sis, UInt32 numOfSourcePackets)
{
    std::vector<ushort> Pids;
    std::vector<pByte> StreamCodingInfos;
    std::vector<pByte> AudioEntries;
    std::vector<pByte> AudioAttributes;
    std::vector<pByte> PgEntries;
    std::vector<pByte> PgAttributes;
    pByte VideoEntry;
    pByte VideoAttribute;
    pByte VideoCodingInfo;
    pByte AudioEntry;
    pByte AudioAttribute;
    pByte AudioCodingInfo;
    pByte PgEntry;
    pByte PgAttribute;
    pByte PgCodingInfo;
    ElementaryStreamTypes VideoType = INVALID;
    for(int i = 0; i < sis.size(); i++)
    {
        StreamInfo si = sis.at(i); 
        switch (si.GetElementaryStreamTypes())
        {
        case AUDIO_STREAM_AC3:
        case AUDIO_STREAM_AC3_PLUS:
        case AUDIO_STREAM_AC3_TRUE_HD:
        case AUDIO_STREAM_DTS:
        case AUDIO_STREAM_DTS_HD:
        case AUDIO_STREAM_DTS_HD_MASTER_AUDIO:
        case AUDIO_STREAM_LPCM:
        case AUDIO_STREAM_MPEG1:
        case AUDIO_STREAM_MPEG2:
            AudioEntry = BuildStreamEntry(si.GetElementaryPID());
            AudioAttribute = BuildAudioStreamAttributes(
                (byte)si.GetElementaryStreamTypes(), si.GetAudioPresentationType(), si.GetSamplingFrequency());
            AudioEntries.push_back(AudioEntry);
            AudioAttributes.push_back(AudioAttribute);
            AudioCodingInfo = BuildAudioStreamCodingInfo(
                si.GetElementaryStreamTypes(), si.GetAudioPresentationType(), si.GetSamplingFrequency());
            Pids.push_back(si.GetElementaryPID());
            StreamCodingInfos.push_back(AudioCodingInfo);
            break;
        case VIDEO_STREAM_H264:
        case VIDEO_STREAM_MPEG1:
        case VIDEO_STREAM_MPEG2:
        case VIDEO_STREAM_VC1:
            VideoType = si.GetElementaryStreamTypes();
            VideoEntry = BuildStreamEntry(si.GetElementaryPID());
            VideoAttribute = BuildVideoStreamAttributes(
                (byte)si.GetElementaryStreamTypes(), si.GetVideoFormat(), si.GetFrameRate());
            VideoCodingInfo = BuildVideoStreamCodingInfo(
                si.GetElementaryStreamTypes(), si.GetVideoFormat(), si.GetFrameRate(), si.GetAspectRatio());
            Pids.push_back(si.GetElementaryPID());
            StreamCodingInfos.push_back(VideoCodingInfo);
            break;
        case PRESENTATION_GRAPHICS_STREAM:
            PgEntry = BuildStreamEntry(si.GetElementaryPID());
            PgEntries.push_back(PgEntry);
            pByte pgsa = pByte(new byte[6]);
            memcpy(pgsa.get(), &PgStreamAttributes[0], sizeof(PgStreamAttributes));
            PgAttributes.push_back(pgsa);
            PgCodingInfo = BuildPgStreamCodingInfo();
            Pids.push_back(si.GetElementaryPID());
            StreamCodingInfos.push_back(PgCodingInfo);
            break;
        }
    }
    std::vector<pByte> PlayItems;
    UInt32 Start = (UInt32)((EpInfo.at(0).pts_ >> 1) & 0xffffffff);
    UInt32 End = (UInt32)((EpInfo.at(EpInfo.size() - 1).pts_ >> 1) & 0xffffffff);
    UInt32 Interval = ((UInt32)(chapterLen.total_seconds() / 60)) * 2700000;
    pByte StnTable = BuildStnTable( VideoEntry, VideoAttribute,
        AudioEntries, AudioAttributes, PgEntries, PgAttributes);
    PlayItems.push_back(BuildFirstPlayItem(0, Start, End, StnTable));
    pByte PlayList = BuildPlayList(PlayItems);
    pByte PlayListMark = BuildFirstPlayMarks(Start, End, Interval);
    pByte mlps = Build_mlps(PlayList, PlayListMark);

    std::string file_name;
    file_name = path.string() + "/BDMV/PLAYLIST/00000.mpls";
    std::ofstream file_mlps(file_name.data());
    file_mlps << mlps.get();
    file_name = path.string() + "/BDMV/BACKUP/PLAYLIST/00000.mlps";
    std::ofstream file_mlps_backup(file_name.data());
    file_mlps_backup << mlps.get();

    pByte ClipInfo = BuildClipInfo(numOfSourcePackets,EpInfo);
    pByte SequenceInfo = BuildSequenceInfo(Start, End);
    pByte ProgramInf = BuildProgramInfo(Pids, StreamCodingInfos);
    pByte EpMap = BuildEpMap(EpInfo);
    pByte CPI = BuildCpi(EpMap);
    pByte clpi = Build_clpi(ClipInfo, SequenceInfo, ProgramInf, CPI);

    file_name = path.string() + "/BDMV/CLIPINF/00001.clpi";
    std::ofstream file_clpi(file_name.data());
    file_clpi << clpi.get();
    file_name = path.string() + "/BDMV/BACKUP/CLIPINF/00001.clpi";
    std::ofstream file_clpi_backup(file_name.data());
    file_clpi_backup << clpi.get();

}

pByte BluRayOutput::BuildStreamEntry(ushort pid)
{
    byte entry[] = {
        0x09, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
    pByte StreamEntry = pByte(entry);
    StreamEntry[2] = (byte)((pid >> 8) & 0xff);
    StreamEntry[3] = (byte)(pid & 0xff);
    return StreamEntry;
}

pByte BluRayOutput::BuildVideoStreamAttributes(byte type, VideoFormat vf, FrameRate fr)
    throw(std::invalid_argument)
{
    if (type != (byte)VIDEO_STREAM_VC1
        && type != (byte)VIDEO_STREAM_MPEG2
        && type != (byte)VIDEO_STREAM_H264)
        throw std::invalid_argument("Video stream of type is not supported by Blu Ray");
    pByte attributes = pByte(new byte[6]);
    attributes[0] = 0x05;
    attributes[1] = type;
    attributes[2] = (byte)((((byte)vf) << 4) & 0xf0);
    attributes[2] |= (byte)(((byte)fr) & 0x0f);
    attributes[3] = attributes[4] = attributes[5] = 0;
    return attributes;
}

pByte BluRayOutput::BuildAudioStreamAttributes(byte type, AudioPresentationType vf, SamplingFrequency fr)
    throw(std::invalid_argument)
{
    if (type != (byte)AUDIO_STREAM_AC3
        && type != (byte)AUDIO_STREAM_AC3_PLUS
        && type != (byte)AUDIO_STREAM_AC3_TRUE_HD
        && type != (byte)AUDIO_STREAM_DTS
        && type != (byte)AUDIO_STREAM_DTS_HD
        && type != (byte)AUDIO_STREAM_DTS_HD_MASTER_AUDIO
        && type != (byte)AUDIO_STREAM_LPCM)
        throw std::invalid_argument("Audio stream of type is not supported by Blu Ray");
    pByte attributes = pByte(new byte[6]);
    attributes[0] = 0x05;
    attributes[1] = type;
    attributes[2] = (byte)((((byte)vf) << 4) & 0xf0);
    attributes[2] |= (byte)(((byte)fr) & 0x0f);
    attributes[3] = 0x65;
    attributes[4] = 0x6e;
    attributes[5] = 0x67;
    return attributes;
}

pByte BluRayOutput::BuildStnTable(pByte VideoEntry, pByte VideoAttributes,
    std::vector<pByte>& AudioEntries, std::vector<pByte>& AudioAttributes,
    std::vector<pByte>& PgEntries, std::vector<pByte>& PgAttributes)
{
    std::vector<byte> table;
    std::vector<byte> temp;
    table.push_back(0x00);
    table.push_back(0x00);
    table.push_back(0x01);
    if (AudioEntries.size() > 0)
        table.push_back((byte)AudioEntries.size());
    else
        table.push_back(0x00);
    if (PgEntries.size() > 0)
        table.push_back((byte)PgEntries.size());
    else
        table.push_back(0x00);
    for (int i = 0; i < 9; i++)
        table.push_back(0x00);
    Utility::AddRange(table, VideoEntry);
    Utility::AddRange(table, VideoAttributes);
    for (int i = 0; i < AudioEntries.size(); i++)
    {
        Utility::AddRange(table, AudioEntries.at(i));
        Utility::AddRange(table, AudioAttributes.at(i));
    }
    for (int i = 0; i < PgEntries.size(); i++)
    {
        Utility::AddRange(table, PgEntries.at(i));
        Utility::AddRange(table, PgAttributes.at(i));
    }
    UInt32 len = (UInt32)table.size();
    temp.push_back((byte)((len >> 8) & 0xff));
    temp.push_back((byte)(len & 0xff));
    Utility::AddRange(temp, table);
    return Utility::ToString(temp);
}

pByte BluRayOutput:: UintToByteArraryNetwork(UInt32 value)
{
    pByte ret = pByte(new byte[4]);
    ret[0] = (byte)((value >> 24) & 0xff);
    ret[1] = (byte)((value >> 16) & 0xff);
    ret[2] = (byte)((value >> 8) & 0xff);
    ret[3] = (byte)(value & 0xff);
    return ret;
}

pByte BluRayOutput::Build_mlps(pByte PlayList, pByte PlayListMark)
{
    std::vector<byte> mlps;
    Utility::AddRange(mlps, PlayList_00000_mpls);
    UInt32 temp = sizeof(AppInfoPlayList);
    temp += 40;
    Utility::AddRange(mlps, UintToByteArraryNetwork(temp));
    temp += sizeof(PlayList.get());
    Utility::AddRange(mlps, UintToByteArraryNetwork(temp));
    for (int i = 0; i < 24; i++)
        mlps.push_back(0x00);
    Utility::AddRange(mlps, AppInfoPlayList);
    Utility::AddRange(mlps, PlayList);
    Utility::AddRange(mlps, PlayListMark);
    return Utility::ToString(mlps);
}

pByte BluRayOutput::BuildPlayList(std::vector<pByte> PlayItems)
{
    std::vector<byte> playList;
    UInt32 playItemNum = PlayItems.size();
    playItemNum <<= 16;
    playItemNum &= 0xffff0000;
    UInt32 len = PlayItems.size() + 6;
    Utility::AddRange(playList, UintToByteArraryNetwork(len));
    playList.push_back(0x00);
    playList.push_back(0x00);
    Utility::AddRange(playList, UintToByteArraryNetwork(playItemNum));
    for (int i = 0; i < PlayItems.size(); i++)
        Utility::AddRange(playList, PlayItems.at(i));
    return Utility::ToString(playList);
}

pByte BluRayOutput::BuildFirstPlayItem(byte stc_id, UInt32 start, UInt32 end, pByte StnTable)
{
    std::vector<byte> playItem;
    std::vector<byte> playTemp;
    for (int i = 0; i < 4; i++)
        playItem.push_back(0x30);
    playItem.push_back(0x31);
    playItem.push_back(0x4d);
    playItem.push_back(0x32);
    playItem.push_back(0x54);
    playItem.push_back(0x53);
    playItem.push_back(0x00);
    playItem.push_back(0x01);
    playItem.push_back(stc_id);
    Utility::AddRange(playItem, UintToByteArraryNetwork(start));
    Utility::AddRange(playItem, UintToByteArraryNetwork(end));
    for (int i = 0; i < 12; i++)
        playItem.push_back(0x00);
    Utility::AddRange(playItem, StnTable);
    UInt32 len = (UInt32)playItem.size();
    playTemp.push_back((byte)((len >> 8) & 0xff));
    playTemp.push_back((byte)(len & 0xff));
    Utility::AddRange(playTemp, playItem);
    return Utility::ToString(playTemp);
}

pByte BluRayOutput::BuildFirstPlayMarks(UInt32 start, UInt32 end, UInt32 interval)
{
    std::vector<byte> marks;
    std::vector<byte> temp;
    UInt32 num = 0;
    for (UInt32 i = start; i < end; i += interval, num++)
    {
        marks.push_back(0x00);
        marks.push_back(0x01);
        marks.push_back(0x00);
        marks.push_back(0x00);
        UInt32 time = i;
        Utility::AddRange(marks, UintToByteArraryNetwork(time));
        marks.push_back(0xff);
        marks.push_back(0xff);
        time = 0;
        Utility::AddRange(marks, UintToByteArraryNetwork(time));
    }
    UInt32 len = (UInt32)marks.size();
    len += 2;
    Utility::AddRange(temp, UintToByteArraryNetwork(len));
    temp.push_back((byte)((num >> 8) & 0xff));
    temp.push_back((byte)(num & 0xff));
    Utility::AddRange(temp, marks);
    return Utility::ToString(temp);
}

pByte BluRayOutput::Build_clpi(pByte ClipInfo, pByte SequenceInfo, pByte ProgramInfo, pByte CPI)
{
    std::vector<byte> clpi;
    Utility::AddRange(clpi, ClipInfo_0000_clpi);
    UInt32 len = 40;
    len += sizeof(ClipInfo.get());
    Utility::AddRange(clpi, UintToByteArraryNetwork(len));
    len += sizeof(SequenceInfo.get());
    Utility::AddRange(clpi, UintToByteArraryNetwork(len));
    len += sizeof(ProgramInfo.get());
    Utility::AddRange(clpi, UintToByteArraryNetwork(len));
    len += sizeof(CPI.get());
    Utility::AddRange(clpi, UintToByteArraryNetwork(len));
    for (int i = 0; i < 16; i++)
        clpi.push_back(0x00);
    Utility::AddRange(clpi, ClipInfo);
    Utility::AddRange(clpi, SequenceInfo);
    Utility::AddRange(clpi, ProgramInfo);
    Utility::AddRange(clpi, CPI);
    Utility::AddRange(clpi, UintToByteArraryNetwork(0x00000000));
    return Utility::ToString(clpi);
}

pByte BluRayOutput::BuildClipInfo(UInt32 numOfSourcePackets, std::vector<EpElement> EpInfo)
{
    std::vector<byte> clip;
    std::vector<byte> temp;

    clip.push_back(0x00);
    clip.push_back(0x00);
    clip.push_back(0x01);
    clip.push_back(0x01);
    clip.push_back(0x00);
    clip.push_back(0x00);
    clip.push_back(0x00);
    clip.push_back(0x00);
    Int64 rate = EpInfo.at(EpInfo.size() - 1).spn_ - EpInfo.at(0).spn_;
    rate *= 192;
    rate /= ((EpInfo.at(EpInfo.size()- 1).pts_ - EpInfo.at(0).pts_) / 90000);
    Utility::AddRange(clip, UintToByteArraryNetwork((UInt32)rate));
    Utility::AddRange(clip, UintToByteArraryNetwork(numOfSourcePackets));
    for (int i = 0; i < 128; i++)
        clip.push_back(0x00);
    Utility::AddRange(clip, TsTypeInfoBlock);
    UInt32 len = (UInt32)clip.size();
    Utility::AddRange(temp, UintToByteArraryNetwork(len));
    Utility::AddRange(temp, clip);
    return Utility::ToString(temp);
}

pByte BluRayOutput::BuildSequenceInfo(UInt32 start, UInt32 end)
{
    std::vector<byte> seq;
    std::vector<byte> temp;

    seq.push_back(0x00);
    seq.push_back(0x01);
    seq.push_back(0x00);
    seq.push_back(0x00);
    seq.push_back(0x00);
    seq.push_back(0x00);
    seq.push_back(0x01);
    seq.push_back(0x00);
    seq.push_back((byte)((Constants::DEFAULT_PCR_PID >> 8) & 0xff));
    seq.push_back((byte)(Constants::DEFAULT_PCR_PID & 0xff));
    seq.push_back(0x00);
    seq.push_back(0x00);
    seq.push_back(0x00);
    seq.push_back(0x00);
    Utility::AddRange(seq, UintToByteArraryNetwork(start));
    Utility::AddRange(seq, UintToByteArraryNetwork(end));
    UInt32 len = (UInt32)seq.size();
    Utility::AddRange(temp, UintToByteArraryNetwork(len));
    Utility::AddRange(temp, seq);
    return Utility::ToString(temp);
}

pByte BluRayOutput::BuildProgramInfo(std::vector<ushort> pids, std::vector<pByte> StreamCodingInfos)
{
    std::vector<byte> info;
    std::vector<byte> temp;
    info.push_back(0x00);
    info.push_back(0x01);
    info.push_back(0x00);
    info.push_back(0x00);
    info.push_back(0x00);
    info.push_back(0x00);
    info.push_back((byte)((Constants::DEFAULT_PMT_PID >> 8) & 0xff));
    info.push_back((byte)(Constants::DEFAULT_PMT_PID & 0xff));
    info.push_back((byte)pids.size());
    info.push_back(0x00);
    for (int i = 0; i < pids.size(); i++)
    {
        info.push_back((byte)((pids.at(i) >> 8) & 0xff));
        info.push_back((byte)(pids.at(i) & 0xff));
        Utility::AddRange(info, StreamCodingInfos.at(i));
    }

    UInt32 len = info.size();
    Utility::AddRange(temp, UintToByteArraryNetwork(len));
    Utility::AddRange(temp, info);
    return Utility::ToString(temp);
}

pByte BluRayOutput::BuildVideoStreamCodingInfo(
    ElementaryStreamTypes type, VideoFormat format, FrameRate rate, AspectRatio ratio)
{
    std::vector<byte> info;
    info.push_back(0x15);
    info.push_back((byte)type);
    info.push_back((byte)((((byte)format) << 4) | (byte)rate));
    info.push_back((byte)(((byte)(ratio)) << 4));
    for(int i = 0; i < 18; i++)
        info.push_back(0x00);
    return Utility::ToString(info);
}

pByte BluRayOutput::BuildAudioStreamCodingInfo(
    ElementaryStreamTypes type, AudioPresentationType format, SamplingFrequency rate)
{
    std::vector<byte> info;
    info.push_back(0x15);
    info.push_back((byte)type);
    info.push_back((byte)((((byte)format) << 4) | (byte)rate));
    info.push_back(0x65);
    info.push_back(0x6e);
    info.push_back(0x67);
    for (int i = 0; i < 16; i++)
        info.push_back(0x00);
    return Utility::ToString(info);
}

pByte BluRayOutput::BuildPgStreamCodingInfo(void)
{
    std::vector<byte> info;
    info.push_back(0x15);
    info.push_back(0x90);
    info.push_back(0x65);
    info.push_back(0x6e);
    info.push_back(0x67);
    for (int i = 0; i < 17; i++)
        info.push_back(0x00);
    return Utility::ToString(info);
}

pByte BluRayOutput::BuildCpi(pByte EpMap)
{
    std::vector<byte> info;
    UInt32 len = sizeof(EpMap.get()) + 2;
    Utility::AddRange(info, UintToByteArraryNetwork(len));
    info.push_back(0x00);
    info.push_back(0x01);
    Utility::AddRange(info, EpMap);
    return Utility::ToString(info);
}

pByte BluRayOutput::BuildEpMap(std::vector<EpElement>& EpInfo)
{
    UInt32 lastepfine = 0x7ff;
    UInt32 lastspnfine = 0x1ffff;
    UInt32 numofcoarse = 0;
    std::vector<byte> coarseloop;
    std::vector<byte> fineloop;
    std::vector<byte> EpMap;

    for (int i = 0; i < EpInfo.size(); i++)
    {
        UInt32 epfine = (UInt32)((EpInfo.at(i).pts_ >> 9) % 0x800);
        UInt32 epcoarse = (UInt32)((EpInfo.at(i).pts_ >> 19) % 0x4000);
        UInt32 spnfine = EpInfo.at(i).spn_ % 0x20000;
        if (lastepfine > epfine || lastspnfine > spnfine)
        {
            UInt32 reftofine = (UInt32)i;
            reftofine <<= 14;
            reftofine |= epcoarse;
            Utility::AddRange(coarseloop, UintToByteArraryNetwork(reftofine));
            Utility::AddRange(coarseloop, UintToByteArraryNetwork(EpInfo.at(i).spn_));
            numofcoarse++;
        }
        UInt32 value = 0x1000;
        value |= (epfine << 17);
        value |= spnfine;
        Utility::AddRange(fineloop, UintToByteArraryNetwork(value));
        lastepfine = epfine;
        lastspnfine = spnfine;
    }

    EpMap.push_back(0x00);
    EpMap.push_back(0x01);
    EpMap.push_back((byte)((Constants::DEFAULT_VIDEO_PID >> 8) & 0xff));
    EpMap.push_back((byte)(Constants::DEFAULT_VIDEO_PID & 0xff));
    EpMap.push_back(0x00);
    byte btemp = 4;
    btemp |= (byte)((numofcoarse >> 14) & 0xff);
    EpMap.push_back(btemp);
    btemp = (byte)((numofcoarse >> 6) & 0xff);
    EpMap.push_back(btemp);
    btemp = (byte)((numofcoarse & 0x3f) << 2);
    btemp |= (byte)((EpInfo.size() >> 16) & 0x03);
    EpMap.push_back(btemp);
    btemp = (byte)((EpInfo.size() >> 8) & 0xff);
    EpMap.push_back(btemp);
    btemp = (byte)(EpInfo.size() & 0xff);
    EpMap.push_back(btemp);
    UInt32 count = 4 + (UInt32)EpMap.size();
    Utility::AddRange(EpMap, UintToByteArraryNetwork(count));
    UInt32 start = 4 + (UInt32)coarseloop.size();
    Utility::AddRange(EpMap, UintToByteArraryNetwork(start));
    Utility::AddRange(EpMap, coarseloop);
    Utility::AddRange(EpMap, fineloop);
    return Utility::ToString(EpMap);
}



} // namespace


