/*
 *  TLV - Trace Log Visualizer
 *
 *  Copyright (C) 2008-2010 by Nagoya Univ., JAPAN
 *
 *  上記著作権者は，以下の(1)〜(4)の条件を満たす場合に限り，本ソフトウェ
 *  ア（本ソフトウェアを改変したものを含む．以下同じ）を使用・複製・改
 *  変・再配布（以下，利用と呼ぶ）することを無償で許諾する．
 *  (1) 本ソフトウェアをソースコードの形で利用する場合には，上記の著作
 *      権表示，この利用条件および下記の無保証規定が，そのままの形でソー
 *      スコード中に含まれていること．
 *  (2) 本ソフトウェアを，ライブラリ形式など，他のソフトウェア開発に使
 *      用できる形で再配布する場合には，再配布に伴うドキュメント（利用
 *      者マニュアルなど）に，上記の著作権表示，この利用条件および下記
 *      の無保証規定を掲載すること．
 *  (3) 本ソフトウェアを，機器に組み込むなど，他のソフトウェア開発に使
 *      用できない形で再配布する場合には，次のいずれかの条件を満たすこ
 *      と．
 *    (a) 再配布に伴うドキュメント（利用者マニュアルなど）に，上記の著
 *        作権表示，この利用条件および下記の無保証規定を掲載すること．
 *    (b) 再配布の形態を，別に定める方法によって，TOPPERSプロジェクトに
 *        報告すること．
 *  (4) 本ソフトウェアの利用により直接的または間接的に生じるいかなる損
 *      害からも，上記著作権者およびTOPPERSプロジェクトを免責すること．
 *      また，本ソフトウェアのユーザまたはエンドユーザからのいかなる理
 *      由に基づく請求からも，上記著作権者およびTOPPERSプロジェクトを
 *      免責すること．
 *
 *  本ソフトウェアは，無保証で提供されているものである．上記著作権者お
 *  よびTOPPERSプロジェクトは，本ソフトウェアに関して，特定の使用目的
 *  に対する適合性も含めて，いかなる保証も行わない．また，本ソフトウェ
 *  アの利用により直接的または間接的に生じたいかなる損害に関しても，そ
 *  の責任を負わない．
 *
 *  @(#) $Id$
 */
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml.Serialization;
using System.Collections.ObjectModel;
using System.IO;
using NU.OJL.MPRTOS.TLV.Base;

namespace NU.OJL.MPRTOS.TLV.Core
{
	public class TraceLog : ICloneable
	{
		private string _Time;
		private string _Object;
		private string _ObjectName;
		private string _ObjectType;
		private string _Behavior;
		private string _Attribute;
		private string _Value;
		private string _Arguments;
		private bool _HasTime;
		private bool _HasObjectName;
		private bool _HasObjectType;
		private TraceLogType _Type;

		public string Time { get { return _Time; } private set { _Time = value; } }
		public string Object { get { return _Object; } private set { _Object = value; } }
		public string ObjectName { get { return _ObjectName; } private set { _ObjectName = value; } }
		public string ObjectType { get { return _ObjectType; } private set { _ObjectType = value; } }
		public string Behavior { get { return _Behavior; } private set { _Behavior = value; } }
		public string Attribute { get { return _Attribute; } private set { _Attribute = value; } }
		public string Value { get { return _Value; } private set { _Value = value; } }
		public string Arguments { get { return _Arguments; } private set { _Arguments = value; } }
		public bool HasTime { get { return _HasTime; } private set { _HasTime = value; } }
		public bool HasObjectName { get { return _HasObjectName; } private set { _HasObjectName = value; } }
		public bool HasObjectType { get { return _HasObjectType; } private set { _HasObjectType = value; } }
		public TraceLogType Type { get { return _Type; } private set { _Type = value; } }

		private string _log;

		private static TraceLogParser _parser = new TraceLogParser();

		public TraceLog(string log)
		{
			_log = Regex.Replace(log, @"\s", "");

			_parser.Parse(_log);

			Time = _parser.TimeValue;
			Object = _parser.ObjectValue;
			ObjectName = _parser.ObjectNameValue;
			ObjectType = _parser.ObjectTypeValue;
			Behavior = _parser.BehaviorValue;
			Attribute = _parser.AttributeValue;
			Value = _parser.ValueValue;
			Arguments = _parser.ArgumentsValue;
			HasTime = _parser.HasTimeValue;
			HasObjectName = _parser.HasObjectNameValue;
			HasObjectType = _parser.HasObjectTypeValue;

			if (Behavior != null && Attribute != null)
				throw new Exception("不正なトレースログです。\n" + _log);
			else if (Behavior == null && Attribute != null)
				Type = TraceLogType.AttributeChange;
			else if (Behavior != null && Attribute == null)
				Type = TraceLogType.BehaviorHappen;
		}

		public static implicit operator string(TraceLog stdlog)
		{
			return stdlog.ToString();
		}
		public static implicit operator TraceLog(string str)
		{
			return new TraceLog(str);
		}

		public override string ToString()
		{
			return _log;
		}

		public Json GetValue(ResourceData resourceData)
		{
			if (!HasObjectType || HasObjectName)
				ObjectType = resourceData.Resources[ObjectName].Type;

			switch (resourceData.ResourceHeaders[ObjectType].Attributes[Attribute].VariableType)
			{
				case JsonValueType.String:
					return new Json(Value);
				case JsonValueType.Number:
					return new Json(Convert.ToDecimal(Value));
				case JsonValueType.Boolean:
					return new Json(Convert.ToBoolean(Value));
				default:
					return null;
			}
		}

		public ArgumentList GetArguments(ResourceData resourceData)
		{
			if (!HasObjectType || HasObjectName)
				ObjectType = resourceData.Resources[ObjectName].Type;

			if (Arguments == string.Empty)
				return new ArgumentList();

			string[] args = Arguments.Split(',');
			ArgumentList argList = new ArgumentList();

			int i = 0;
			foreach (ArgumentType argType in resourceData.ResourceHeaders[ObjectType].Behaviors[Behavior].Arguments)
			{
				if (args[i] == string.Empty)
					argList.Add(null);
				else
					switch (argType.Type)
					{
						case JsonValueType.String:
							argList.Add(new Json(args[i]));
							break;
						case JsonValueType.Number:
							argList.Add(new Json(Convert.ToDecimal(args[i])));
							break;
						case JsonValueType.Boolean:
							argList.Add(new Json(Convert.ToBoolean(args[i])));
							break;
						default:
							argList.Add(null);
							break;
					}

				i++;
				if (args.Length <= i)
					break;
			}

			return argList;
		}

		public object Clone()
		{
			TraceLog log = new TraceLog(_log);
			return log;
		}
	}
}
