package jp.osoite.tomu.xml.jaxb.util;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import jp.osoite.tomu.xml.jaxb.object.Head;
import jp.osoite.tomu.xml.jaxb.object.SensorSourceDataSet;
import jp.osoite.tomu.xml.jaxb.object.Sfeer;
import jp.osoite.tomu.xml.jaxb.object.SfeerRequestSet;
import jp.osoite.tomu.xml.jaxb.object.StaticSensorData;
import jp.osoite.tomu.xml.jaxb.object.TomuMessage;
import jp.osoite.tomu.xml.jaxb.wrapper.InitRequestSfeerWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.InitResponseSfeerWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.RequestSfeerWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.ResponseAppsWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.UploadCo2Wrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.UploadHumidityWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.UploadKito2Wrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.UploadPedestrianWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.UploadTempWrapper;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.InitRequestSfeerListener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.InitResponseSfeerListener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.RequestSfeerListener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.ResponseAppsListener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.UploadCo2Listener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.UploadHumidityListener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.UploadKito2Listener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.UploadPedestrianListener;
import jp.osoite.tomu.xml.jaxb.wrapper.listener.UploadTempListener;

/**
 * TomuMessageを各種ラッパーへスイッチするためのクラスです．
 * @author shima
 */
public class TomuMessageSwitchManager {

    private CopyOnWriteArrayList<UploadPedestrianListener> uploadPedestrianList;
    private CopyOnWriteArrayList<UploadKito2Listener> uploadKito2List;
    private CopyOnWriteArrayList<UploadTempListener> uploadTempList;
    private CopyOnWriteArrayList<UploadCo2Listener> uploadCo2List;
    private CopyOnWriteArrayList<UploadHumidityListener> uploadHumidityList;
    private CopyOnWriteArrayList<InitRequestSfeerListener> initReqSfeerList;
    private CopyOnWriteArrayList<InitResponseSfeerListener> initRespSfeerList;
    private CopyOnWriteArrayList<RequestSfeerListener> reqSfeerList;
    private CopyOnWriteArrayList<ResponseAppsListener> respAppsList;

    public TomuMessageSwitchManager() {
    }

    public void switchMessage(TomuMessage tomuMsg) {
        Head head = tomuMsg.getHead();
        String messageType = head.getMsgType();
        if (messageType.equals(TomuMessageType.APPS_REQUEST.toString())) {
            switchAppsRequest(tomuMsg);
        } else if (messageType.equals(TomuMessageType.APPS_RESPONSE.toString())) {
            switchAppsResponse(tomuMsg);
        } else if (messageType.equals(TomuMessageType.APPS_INIT_REQUEST.toString())) {
            switchAppsInitRequest(tomuMsg);
        } else if (messageType.equals(TomuMessageType.APPS_INIT_RESPONSE.toString())) {
            switchAppsInitResponse(tomuMsg);
        } else if (messageType.equals(TomuMessageType.SOURCE_UPLOAD.toString())) {
            switchSourceUpload(tomuMsg);
        }
    }

    private void switchAppsRequest(TomuMessage tomuMsg) {
        SfeerRequestSet sfeerReq = tomuMsg.getRequest().getSfeerRequestSet();
        if (sfeerReq != null) {
            notifyRequestSfeerListener(new RequestSfeerWrapper(tomuMsg));
        }
    }

    private void switchAppsResponse(TomuMessage tomuMsg) {
        notifyResponseAppsListener(new ResponseAppsWrapper(tomuMsg));
    }

    private void switchAppsInitRequest(TomuMessage tomuMsg) {
        Sfeer sfeer = tomuMsg.getInitRequest().getSfeer();
        if (sfeer != null) {
            notifyInitRequestSfeerListener(new InitRequestSfeerWrapper(tomuMsg));
        }
    }

    private void switchAppsInitResponse(TomuMessage tomuMsg) {
        Sfeer sfeer = tomuMsg.getInitResponse().getSfeer();
        if (sfeer != null) {
            notifyInitResponseSfeerListener(new InitResponseSfeerWrapper(tomuMsg));
        }
    }

    private void switchSourceUpload(TomuMessage tomuMsg) {
        SensorSourceDataSet ssds = tomuMsg.getUpload().getSensorSourceDataSet();
        if (ssds.getStaticSensorData() != null) {
            List<StaticSensorData> ssdList = ssds.getStaticSensorData();
            if (ssdList.size() != 0) {
                StaticSensorData ssd = ssdList.get(0);
                String sensorType = ssd.getType();
                if (sensorType.equals("TEMP")) {
                    notifyUploadTempListener(new UploadTempWrapper(tomuMsg));
                } else if (sensorType.equals("CO2")) {
                    notifyUploadCo2Listener(new UploadCo2Wrapper(tomuMsg));
                } else if (sensorType.equals("HUMIDITY")) {
                    notifyUploadHumidityListener(new UploadHumidityWrapper(tomuMsg));
                } else if (sensorType.contains("PEDESTRIAN")) {
                    notifyUploadPedestrianListener(new UploadPedestrianWrapper(tomuMsg));
                }
            } else if (ssds.getKitokitoSensorData() != null) {
                notifyUploadKito2Listener(new UploadKito2Wrapper(tomuMsg));
            }
        } else if (ssds.getDynamicSensorData() != null) {
        }
    }

    // -------------------------------------------------------------------------
    // Listeners
    // -------------------------------------------------------------------------
    public void addUploadPedestrianListener(UploadPedestrianListener listener) {
        if (uploadPedestrianList == null) {
            uploadPedestrianList = new CopyOnWriteArrayList<UploadPedestrianListener>();
        }
        uploadPedestrianList.add(listener);
    }

    private void notifyUploadPedestrianListener(UploadPedestrianWrapper wrapper) {
        if (uploadPedestrianList == null) {
            return;
        }
        for (UploadPedestrianListener listener : uploadPedestrianList) {
            listener.receiveUploadPedestrian(wrapper);
        }
    }

    public void addUploadKito2Listener(UploadKito2Listener listener) {
        if (uploadKito2List == null) {
            uploadKito2List = new CopyOnWriteArrayList<UploadKito2Listener>();
        }
        uploadKito2List.add(listener);
    }

    private void notifyUploadKito2Listener(UploadKito2Wrapper wrapper) {
        if (uploadKito2List == null) {
            return;
        }
        for (UploadKito2Listener listener : uploadKito2List) {
            listener.receiveUploadKito2(wrapper);
        }
    }

    public void addInitRequestSfeerListener(InitRequestSfeerListener listener) {
        if (initReqSfeerList == null) {
            initReqSfeerList = new CopyOnWriteArrayList<InitRequestSfeerListener>();
        }
        initReqSfeerList.add(listener);
    }

    private void notifyInitRequestSfeerListener(InitRequestSfeerWrapper wrapper) {
        if (initReqSfeerList == null) {
            return;
        }
        for (InitRequestSfeerListener listener : initReqSfeerList) {
            listener.receiveInitRequestSfeer(wrapper);
        }
    }

    public void addInitResponseSfeerListener(InitResponseSfeerListener listener) {
        if (initRespSfeerList == null) {
            initRespSfeerList = new CopyOnWriteArrayList<InitResponseSfeerListener>();
        }
        initRespSfeerList.add(listener);
    }

    private void notifyInitResponseSfeerListener(InitResponseSfeerWrapper wrapper) {
        if (initRespSfeerList == null) {
            return;
        }
        for (InitResponseSfeerListener listener : initRespSfeerList) {
            listener.receiveInitResponseSfeer(wrapper);
        }
    }

    public void addRequestSfeerListener(RequestSfeerListener listener) {
        if (reqSfeerList == null) {
            reqSfeerList = new CopyOnWriteArrayList<RequestSfeerListener>();
        }
        reqSfeerList.add(listener);
    }

    private void notifyRequestSfeerListener(RequestSfeerWrapper wrapper) {
        if (reqSfeerList == null) {
            return;
        }
        for (RequestSfeerListener listener : reqSfeerList) {
            listener.receiveRequestSfeer(wrapper);
        }
    }

    public void addUploadTempListener(UploadTempListener listener) {
        if (uploadTempList == null) {
            uploadTempList = new CopyOnWriteArrayList<UploadTempListener>();
        }
        uploadTempList.add(listener);
    }

    private void notifyUploadTempListener(UploadTempWrapper wrapper) {
        if (uploadTempList == null) {
            return;
        }
        for (UploadTempListener listener : uploadTempList) {
            listener.receiveUploadTemp(wrapper);
        }
    }

    public void addUploadCo2Listener(UploadCo2Listener listener) {
        if (uploadCo2List == null) {
            uploadCo2List = new CopyOnWriteArrayList<UploadCo2Listener>();
        }
        uploadCo2List.add(listener);
    }

    private void notifyUploadCo2Listener(UploadCo2Wrapper wrapper) {
        if (uploadCo2List == null) {
            return;
        }
        for (UploadCo2Listener listener : uploadCo2List) {
            listener.receiveUploadCo2(wrapper);
        }
    }

    public void addUploadHumidityListener(UploadHumidityListener listener) {
        if (uploadHumidityList == null) {
            uploadHumidityList = new CopyOnWriteArrayList<UploadHumidityListener>();
        }
        uploadHumidityList.add(listener);
    }

    private void notifyUploadHumidityListener(UploadHumidityWrapper wrapper) {
        if (uploadHumidityList == null) {
            return;
        }
        for (UploadHumidityListener listener : uploadHumidityList) {
            listener.receiveUploadHumidity(wrapper);
        }
    }

    public void addResponseAppsListener(ResponseAppsListener listener) {
        if (respAppsList == null) {
            respAppsList = new CopyOnWriteArrayList<ResponseAppsListener>();
        }
        respAppsList.add(listener);
    }

    private void notifyResponseAppsListener(ResponseAppsWrapper wrapper) {
        if (respAppsList == null) {
            return;
        }
        for (ResponseAppsListener listener : respAppsList) {
            listener.receiveResponseApps(wrapper);
        }
    }
}
