/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Script/Expressions.cs#7 $
 * $DateTime: 2008/05/14 13:05:12 $
 * \؂̊evf݂NXBantlrR[hō\z
 */

using System;
using System.IO;
using System.Diagnostics;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using antlr;

using Travis.ORT;
using Bellagio.Script.Parser;

namespace Bellagio.Script {
    //TESTINFO ExpressionParserTests.cs
    public class ScriptException : Exception {
        public ScriptException(string msg) : base(msg) { }
    }

	//Expression Tree
	public abstract class Expression {
		public abstract T Apply<T>(ExpressionVisitor<T> visitor);

        private delegate object TopLevelParseMethod(ExpressionParser p);

        //p[X{́BExpressionParser̃gbvxp[T̂ꂩĂԂ̂ŁA߂lobjectBeXgP[XĂԂƂȂǁAtextnull̂ƂB
        private static object ParseInternal(TopLevelParseMethod parse, IORText text, string expression) { 
            try {
                Debug.Assert(expression!=null);
                ExpressionParser p = new ExpressionParser(new ExpressionLexer(new StringReader(expression)));
                return parse(p);
            }
            catch(antlr.TokenStreamRecognitionException ex) {
                throw ThrowParseException(expression, text, ex.recog.line, ex.recog.column, ex.Message);
            }
            catch(antlr.NoViableAltException ex) {
                throw ThrowParseException(expression, text, ex.line, ex.column, ex.Message);
            }
            catch(antlr.MismatchedTokenException ex) {
                throw ThrowParseException(expression, text, ex.line, ex.column, ex.Message);
            }

        }

        public static Expression ParseExpression(string expression) {
            return (Expression)ParseInternal(delegate(ExpressionParser p) { return p.top_expr(); }, null, expression);
        }
        public static Expression ParseExpression(IORText text) {
            return (Expression)ParseInternal(delegate(ExpressionParser p) { return p.top_expr(); }, text, text.Value);
        }
        public static Expression[] ParseExpressionList(IORText text) {
            return (Expression[])ParseInternal(delegate(ExpressionParser p) { return p.top_expr_list(); }, text, text.Value);
        }
        public static ParameterListDefinition ParseParameterListDefinition(IORText text) {
            if(text.IsOmitted)
                return new ParameterListDefinition();
            else
                return (ParameterListDefinition)ParseInternal(delegate(ExpressionParser p) { return p.top_parameter_list_def(); }, text, text.Value);
        }
        public static ParameterListDefinition ParseParameterListDefinition(string text) {
            return (ParameterListDefinition)ParseInternal(delegate(ExpressionParser p) { return p.top_parameter_list_def(); }, null, text);
        }
        public static FunctionDefinitionList ParseFunctionDefinitionList(IORText text) {
            return (FunctionDefinitionList)ParseInternal(delegate(ExpressionParser p) { return p.top_function_definition_list(); }, text, text.Value);
        }
        public static TypeDesc ParseTypeDesc(string expression) {
            return (TypeDesc)ParseInternal(delegate(ExpressionParser p) { return p.top_type_desc(); }, null, expression);
        }
        public static TypeDesc ParseTypeDesc(IORText text) {
            return (TypeDesc)ParseInternal(delegate(ExpressionParser p) { return p.top_type_desc(); }, text, text.Value);
        }
        private static BellagioException ThrowParseException(string expr, IORText text, int line, int column, string msg) {
            string filename = null;
            if(text!=null) {
                filename = text.ContainerLocation.FileName;
                line += text.Line-1;
                column += text.Column-1;
            }
            
            return new BellagioException(String.Format("{0}s {1}  {2}\n{3} ̃p[XɎs܂B{4})", filename==null? "" : "t@C"+filename, line, column, expr, msg));
        }
    }

	public class LiteralNumberExpression : Expression {
		private string _value;

		public LiteralNumberExpression(string value) { //the parse calls directly
            _value = value;
		}

        public string Value {
            get {
                return _value;
            }
        }

        public override T Apply<T>(ExpressionVisitor<T> visitor) {
			return visitor.LiteralNumber(this);
		}
	}
    public class LiteralBooleanExpression : Expression {
        private bool _value;

        public LiteralBooleanExpression(bool value) { //the parse calls directly
            _value = value;
        }
        public bool Value {
            get {
                return _value;
            }
        }

        public override T Apply<T>(ExpressionVisitor<T> visitor) {
            return visitor.LiteralBoolean(this);
        }
    }
    public class LiteralTimeExpression : Expression {
        private string _value;

        public LiteralTimeExpression(string value) { //the parse calls directly
            _value = value;
        }

        public string Value {
            get {
                return _value;
            }
        }

        public override T Apply<T>(ExpressionVisitor<T> visitor) {
            return visitor.LiteralTime(this);
        }
    }
    public class LiteralDateExpression : Expression {
        private string _value;

        public LiteralDateExpression(string value) { //the parse calls directly
            _value = value;
        }

        public string Value {
            get {
                return _value;
            }
        }
        public override T Apply<T>(ExpressionVisitor<T> visitor) {
            return visitor.LiteralDate(this);
        }
    }
    public class LiteralStringExpression : Expression {
        private string _value;
        public LiteralStringExpression(string value) {
            _value = value;
        }
        public string Value {
            get {
                return _value;
            }
        }
        public override T Apply<T>(ExpressionVisitor<T> visitor) {
            return visitor.LiteralString(this);
        }
    }

    public class ConditionalExpression : Expression {
        private Expression _condition;
        private Expression _trueCase;
        private Expression _falseCase;

        public ConditionalExpression(Expression cond, Expression tc, Expression fc) {
            _condition = cond;
            _trueCase = tc;
            _falseCase = fc;
        }

        public Expression Condition {
            get {
                return _condition;
            }
        }
        public Expression TrueCase {
            get {
                return _trueCase;
            }
        }
        public Expression FalseCase {
            get {
                return _falseCase;
            }
        }

        public override T Apply<T>(ExpressionVisitor<T> visitor) {
            return visitor.Condition(this);
        }
    }

    public class SymbolExpression : Expression {
        private string _symbol;
        private object _valueCache; //V{ēlLbV

        public SymbolExpression(string symbol) {
            _symbol = symbol;
        }
        public string Symbol {
            get {
                return _symbol;
            }
        }
        public object ValueCache {
            get {
                return _valueCache;
            }
            set {
                _valueCache = value;
            }
        }


        public override T Apply<T>(ExpressionVisitor<T> visitor) {
            return visitor.Symbol(this);
        }
    }


    public class ParameterExpression : Expression {
		private string _symbol;
        public string Symbol {
			get {
				return _symbol;
			}
		}
        public ParameterExpression(string symbol) {
			_symbol = symbol;
        }
        public override T Apply<T>(ExpressionVisitor<T> visitor) {
			return visitor.Parameter(this);
		}
	}

	public enum BinaryOperator {
		And,
		Or,
		Predication,

		Plus,
		Minus,
		Multiply,
		Divide,
        Mod,
		
		Eq,
		NotEq,
		Smaller,
		Greater,
		SmallerEq,
		GreaterEq
	}

	public class BinaryOpExpression : Expression {
		private BinaryOperator _operator;
        private string _stringOp;
		private Expression _left;
		private Expression _right;

		public BinaryOpExpression(BinaryOperator t, Expression left, Expression right) {
			_operator = t;
			_left = left;
			_right = right;
		}
        public BinaryOpExpression(IToken tok, Expression left, Expression right) {
            _operator = FromANTLRType(tok.Type);
            _stringOp = tok.getText();
            _left = left;
            _right = right;
        }
        public BinaryOperator Operator {
            get {
				return _operator;
			}
		}
		public Expression Left {
			get{
				return _left;
			}
		}
		public Expression Right {
			get {
				return _right;
			}
		}
        public string OperatorAsString {
            get {
                return _stringOp;
            }
        }
		public override T Apply<T>(ExpressionVisitor<T> visitor) {
			return visitor.BinaryOp(this);
		}

        public static BinaryOperator FromANTLRType(int antlr_type) {
            switch(antlr_type) {
                case ExpressionParserTokenTypes.ARITH_OP_ADD:
                    return BinaryOperator.Plus;
                case ExpressionParserTokenTypes.ARITH_OP_DIV:
                    return BinaryOperator.Divide;
                case ExpressionParserTokenTypes.ARITH_OP_MUL:
                    return BinaryOperator.Multiply;
                case ExpressionParserTokenTypes.ARITH_OP_SUB:
                    return BinaryOperator.Minus;
                case ExpressionParserTokenTypes.ARITH_OP_MOD:
                    return BinaryOperator.Mod;
                case ExpressionParserTokenTypes.COMP_OP_EQ:
                    return BinaryOperator.Eq;
                case ExpressionParserTokenTypes.COMP_OP_GT:
                    return BinaryOperator.Greater;
                case ExpressionParserTokenTypes.COMP_OP_GTE:
                    return BinaryOperator.GreaterEq;
                case ExpressionParserTokenTypes.COMP_OP_LT:
                    return BinaryOperator.Smaller;
                case ExpressionParserTokenTypes.COMP_OP_LTE:
                    return BinaryOperator.SmallerEq;
                case ExpressionParserTokenTypes.COMP_OP_NEQ:
                    return BinaryOperator.NotEq;
                case ExpressionParserTokenTypes.LOGICAL_AND:
                    return BinaryOperator.And;
                case ExpressionParserTokenTypes.LOGICAL_OR:
                    return BinaryOperator.Or;
                default:
                    throw new ArithmeticException("sȓ񍀉Zq " + antlr_type);
            }
        }

    }

	public enum UnaryOperator {
		Inverse,
        LogicalNot
	}

	public class UnaryOpExpression : Expression {

		private UnaryOperator _operator;
        private string _stringOp;
		private Expression _expr;

		public UnaryOpExpression(UnaryOperator t, Expression expr) {
			_operator = t;
			_expr = expr;
		}
        public UnaryOpExpression(IToken token, Expression expr) {
            _operator = FromANTLRType(token.Type);
            _stringOp = token.getText();
            _expr = expr;
        }
		public UnaryOperator Operator {
			get {
				return _operator;
			}
		}
		public Expression Content {
			get{
				return _expr;
			}
		}
		public override T Apply<T>(ExpressionVisitor<T> visitor) {
			return visitor.UnaryOp(this);
		}
        public string OperatorAsString {
            get {
                return _stringOp;
            }
        }

        public static UnaryOperator FromANTLRType(int antlr_type) {
            switch(antlr_type) {
                case ExpressionParserTokenTypes.ARITH_OP_SUB:
                    return UnaryOperator.Inverse;
                case ExpressionParserTokenTypes.LOGICAL_NOT:
                    return UnaryOperator.LogicalNot;
                default:
                    throw new ArithmeticException("sȒPZq " + antlr_type);
            }
        }
    }

    public class ReferenceExpression : Expression {
        private Expression _left;
        private Expression _right;

        public ReferenceExpression(Expression left, Expression right) {
            _left = left;
            _right = right;
        }

        public Expression Left {
            get {
                return _left;
            }
        }
        public Expression Right {
            get {
                return _right;
            }
        }

        public override T Apply<T>(ExpressionVisitor<T> visitor) {
            return visitor.Reference(this);
        }
    }

    public class InvokeExpression : Expression, IEnumerable<Expression> {
        private Expression _target;
		private List<Expression> _args;
		public InvokeExpression(Expression target) {
            _target = target;
			_args = new List<Expression>();
		}
        public Expression Target {
            get {
                return _target;
            }
        }
       
        public int ArgCount {
            get {
                return _args.Count;
            }
        }
        public InvokeExpression AddArg(Expression arg) {
            _args.Add(arg);
            return this;
        }
        public Expression this[int i] {
            get {
                return _args[i];
            }
        }
        public IEnumerator<Expression> GetEnumerator() {
            return _args.GetEnumerator();
        }
        IEnumerator IEnumerable.GetEnumerator() {
            return _args.GetEnumerator();
        }
        public override T Apply<T>(ExpressionVisitor<T> visitor) {
			return visitor.Invoke(this);
		}
	}
    public class ArrayExpression : Expression {
        private Expression _expr;
        private Expression _index;

        public ArrayExpression(Expression expr, Expression index) {
            _expr = expr;
            _index = index;
        }
        public Expression Target {
            get {
                return _expr;
            }
        }
        public Expression Index {
            get {
                return _index;
            }
        }
        public override T Apply<T>(ExpressionVisitor<T> visitor) {
            return visitor.Array(this);
        }
    }
    public class LambdaExpression : Expression {
        private ParameterListDefinition _args;
        private Expression _body;
        public LambdaExpression(ParameterListDefinition args, Expression body) {
            _args = args;
            _body = body;
        }
        public override T Apply<T>(ExpressionVisitor<T> visitor) {
            return visitor.Lambda(this);
        }

        public ParameterListDefinition Args {
            get {
                return _args;
            }
        }
        public Expression Body {
            get {
                return _body;
            }
        }
    }
    public class LetExpression : Expression {
        private List<string> _localNames;
        private List<Expression> _localDefs;
        private Expression _body;
        public LetExpression() {
            _localNames = new List<string>();
            _localDefs = new List<Expression>();
        }
        public void DefLocal(string name, Expression def) {
            _localNames.Add(name);
            _localDefs.Add(def);
        }
        public void SetBody(Expression body) {
            _body = body;
        }

        public int LocalVarCount {
            get {
                return _localNames.Count;
            }
        }
        public string LocalNameAt(int index) {
            return _localNames[index];
        }
        public Expression LocalExprAt(int index) {
            return _localDefs[index];
        }
        public Expression Body {
            get {
                return _body;
            }
        }
        public override T Apply<T>(ExpressionVisitor<T> visitor) {
            return visitor.Let(this);
        }
    }

    //֐`̌^錾p[g
    public class ParameterListDefinition {
        private List<TypeDesc> _types;
        private List<string> _names;

        public ParameterListDefinition() {
            _types = new List<TypeDesc>();
            _names = new List<string>();
        }
        public void Add(TypeDesc t, string n) {
            _types.Add(t);
            _names.Add(n);
        }
        public int Count {
            get {
                return _types.Count;
            }
        }
        public TypeDesc TypeDescAt(int index) {
            return _types[index];
        }
        public string NameAt(int index) {
            return _names[index];
        }
    }

    public abstract class TypeDesc {
        public abstract string Format();

        public static NamedTypeDesc FromName(string s) {
            return new NamedTypeDesc(s);
        }
    }
    public class NamedTypeDesc : TypeDesc {
        private string _name;

        public NamedTypeDesc(string name) {
            _name = name;
        }
        public override string Format() {
            return _name;
        }
        public string Name {
            get {
                return _name;
            }
        }
    }
    public class ArrayTypeDesc : TypeDesc {
        private TypeDesc _elem;
        public ArrayTypeDesc(TypeDesc t) {
            _elem = t;
        }
        public TypeDesc Element {
            get {
                return _elem;
            }
        }
        public override string Format() {
            return _elem.Format() + "[]";
        }
    }
    public class LambdaTypeDesc : TypeDesc {
        private TypeDesc _retType;
        private List<TypeDesc> _args;
        public LambdaTypeDesc() {
            _args = new List<TypeDesc>();
        }
        public void AddArg(TypeDesc t) {
            _args.Add(t);
        }
        public void Fix(TypeDesc t) {
            _retType = t;
        }
        public int ArgCount {
            get {
                return _args.Count;
            }
        }
        public TypeDesc ArgAt(int index) {
            return _args[index];
        }
        public TypeDesc RetType {
            get {
                return _retType;
            }
        }
        public override string Format() {
            StringBuilder bld = new StringBuilder();
            bld.Append("(");
            for(int i=0; i<_args.Count; i++) {
                if(i>0) bld.Append(", ");
                bld.Append(_args[i].Format());
            }
            bld.Append(") -> ");
            bld.Append(_retType.Format());
            return bld.ToString();
        }
    }

    //[U`֐ vɖOt_
    public class FunctionDefinition {
        private string _name;
        private TypeDesc _returnType;
        private ParameterListDefinition _parameters;
        private Expression _body;
        private int _line; //`Xg͒Ȃ̂ňʒuĂ

        public FunctionDefinition(string name, TypeDesc rt, ParameterListDefinition parameters, Expression body, int line) {
            _name = name;
            _returnType = rt;
            _parameters = parameters;
            _body = body;
            _line = line;
        }
        public string Name {
            get {
                return _name;
            }
        }
        public TypeDesc ReturnType {
            get {
                return _returnType;
            }
        }
        public ParameterListDefinition Parameters {
            get {
                return _parameters;
            }
        }
        public Expression BodyExpression {
            get {
                return _body;
            }
        }
        public int LineNumber {
            get {
                return _line;
            }
        }
    }

    public class FunctionDefinitionList : IEnumerable<FunctionDefinition> {
        private List<FunctionDefinition> _functions;
        public FunctionDefinitionList() {
            _functions = new List<FunctionDefinition>();
        }
        public void Add(FunctionDefinition fd) {
            _functions.Add(fd);
        }
        IEnumerator<FunctionDefinition> IEnumerable<FunctionDefinition>.GetEnumerator() {
            return _functions.GetEnumerator();
        }
        IEnumerator IEnumerable.GetEnumerator() {
            return _functions.GetEnumerator();
        }
    }
}
