/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Chart/ChartPosition.cs#12 $
 * $DateTime: 2008/02/22 14:51:26 $
 * 
 * `[g̈ʒuNX
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

#if UNITTEST
using NUnit.Framework;
#endif


namespace Bellagio.Chart {
    public enum ChartSectionType {
        Backward,  //ŏs̈
        Data,      //f[^̈
        Space,     //x݂Ȃǂ̋
        Forward    //swWp[g
    }
    public class ChartSectionInfo {
        public class Tag {
            public ChartSectionType type;
            public int index;
            public int length;
            public int displayIndex;
            public int dataIndex; //f[^ZNVł̂ݗLBg̊Jnʒuf[^̒ʎZłǂ̈ʒu

            public Tag(ChartSectionType t, int i, int l, int di, int da) {
                type = t;
                index = i;
                length = l;
                displayIndex = di;
                dataIndex = da;
            }
        }
        private Tag[] _tags;

        private bool _dirty; //_[eBtO
        private int _totalDisplayLength;
        private int _totalDataLength;

        //쐬͂قǕpɂł͂ȂB
        public ChartSectionInfo(ChartSectionType[] types, int[] lengths) {
            Debug.Assert(types.Length==lengths.Length);
            _tags = new Tag[types.Length];
            int display_acc = 0;
            int data_acc = 0;

            for(int i=0; i<types.Length; i++) {
                Tag t = new Tag(types[i], i, lengths[i], display_acc, data_acc);
                _tags[i] = t;

                Debug.Assert(t.length > 0); //݂ZNV͂Pȏ̒
                display_acc += t.length;
                if(t.type==ChartSectionType.Data) data_acc += t.length;
            }

            _totalDisplayLength = display_acc;
            _totalDataLength = data_acc;
            _dirty = false;
        }
        //VOZNV
        public ChartSectionInfo(ChartSectionType type, int length) {
            _tags = new Tag[1];

            Tag t = new Tag(type, 0, length, 0, 0);
            _tags[0] = t;

            Debug.Assert(t.length > 0); //݂ZNV͂Pȏ̒

            _totalDisplayLength = length;
            _totalDataLength = type==ChartSectionType.Data? length : 0;
            _dirty = false;
        }
        
        //IɊĝ͂܂gÂ炢
        public void AppendLast(ChartSectionType type, int length) {
            Tag t = new Tag(type, _tags.Length, length, 0, 0);
            Tag[] na = new Tag[_tags.Length+1];
            Array.Copy(_tags, na, _tags.Length);
            na[na.Length-1] = t;
            _tags = na;
            _dirty = true;
        }


        public static ChartSectionInfo Empty {
            get {
                return new ChartSectionInfo(new ChartSectionType[0], new int[0]);
            }
        }
        public bool IsEmpty {
            get {
                return _tags.Length==0;
            }
        }
        //݌v̂ݍXV
        private void CalcLengths() {
            int display_acc = 0;
            int data_acc = 0;

            for(int i=0; i<_tags.Length; i++) {
                Tag t = _tags[i];
                t.displayIndex = display_acc;
                t.dataIndex = data_acc;

                display_acc += t.length;
                if(t.type!=ChartSectionType.Space) data_acc += t.length;
            }

            _totalDisplayLength = display_acc;
            _totalDataLength = data_acc;
            _dirty = false;
        }

        public int SectionCount {
            get {
                return _tags.Length;
            }
        }
        public int DataSectionCount {
            get {
                int t = 0;
                foreach(Tag tag in _tags)
                    if(tag.type==ChartSectionType.Data) t++;
                return t;
            }
        } 

        public void UpdateLength(int index, int length) {
            if(_tags[index].length != length) {
                _tags[index].length = length;
                CalcLengths(); //]
            }
        }

        public int GetDisplayLength() {
            if(_dirty) CalcLengths();
            return _totalDisplayLength;
        }
        public int GetDataLength() {
            if(_dirty) CalcLengths();
            return _totalDataLength;
        }

        public CHARTPOSITION FirstPosition {
            get {
                if(_dirty) CalcLengths();
                if(_tags.Length==0)
                    return CHARTPOSITION.Empty(this);
                else
                    return new CHARTPOSITION(this, 0, 0);
            }
        }
        public CHARTPOSITION LastPosition {
            get {
                if(_dirty) CalcLengths();
                if(_tags.Length==0)
                    return CHARTPOSITION.Empty(this);
                else
                    return new CHARTPOSITION(this, _tags.Length-1, _tags[_tags.Length-1].length-1);
            }
        }
        public Tag GetTagAt(int index) {
            Debug.Assert(index>=0 && index < _tags.Length);
            if(_dirty) CalcLengths();
            return _tags[index];
        }
        public Tag FirstData {
            get {
                for(int i=0; i<_tags.Length; i++) {
                    if(_tags[i].type==ChartSectionType.Data) return GetTagAt(i);
                }
                return null;
            }
        }
        public Tag LastData {
            get {
                for(int i=_tags.Length-1; i>=0; i--) {
                    if(_tags[i].type==ChartSectionType.Data) return GetTagAt(i);
                }
                return null;
            }
        }

        public CHARTPOSITION GetChartPositionByData(int data_index) {
            if(_dirty) CalcLengths();

            if(_tags.Length==0 || data_index<0)
                return CHARTPOSITION.Empty(this);
            //t߂邱Ƃ͂
            for(int i=_tags.Length-1; i>=0; i--) {
                Tag t = _tags[i];
                if(t.type==ChartSectionType.Data && t.dataIndex <= data_index && data_index <t.dataIndex+t.length)
                    return new CHARTPOSITION(this, i, data_index-t.dataIndex);
            }
            Debug.Assert(false, "must not come"); //ɗ邱Ƃ͂Ȃ
            return CHARTPOSITION.Empty(this);
        }

        public CHARTPOSITION GetChartPosition(int display_index) {
            if(_dirty) CalcLengths();

            if(_tags.Length==0 || display_index<0 || display_index >= _totalDisplayLength)
               return CHARTPOSITION.Empty(this);

            Debug.Assert(display_index < _totalDisplayLength);

            //t߂邱Ƃ͂
            for(int i=_tags.Length-1; i>=0; i--) {
                Tag t = _tags[i];
                if(t.displayIndex <= display_index && display_index <t.displayIndex+t.length)
                    return new CHARTPOSITION(this, i, display_index-t.displayIndex);
            }
            Debug.Assert(false, "must not come"); //ɗ邱Ƃ͂Ȃ
            return CHARTPOSITION.Empty(this);
        }

        //̈ʒu̎Bg͔jIŁAɐi߂ƂtrueԂ
        public bool MoveNext(ref CHARTPOSITION position) {
            //Debug.Assert(position.Owner==this); //TODO Assert𕜊̂́AChartDocumentCX^XChartSectionInfog܂킷悤ɂƂɌ
            if(position.IsEmpty) return false;

            int si = position.SectionIndex;
            int pi = position.PositionInSection;
            Tag t = _tags[si];

            if(pi < t.length-1) { //ZNVړ
                position.InternalSet(si, pi+1);
                return true;
            }
            if(si < _tags.Length-1) { //̃ZNV̐擪
                position.InternalSet(si+1, 0);
                return true;
            }

            position = CHARTPOSITION.Empty(this);
            return false;
        }
        public bool MovePrev(ref CHARTPOSITION position) {
            //Debug.Assert(position.Owner==this);
            if(position.IsEmpty) return false;

            int si = position.SectionIndex;
            int pi = position.PositionInSection;
            Tag t = _tags[si];

            if(pi > 0) { //ZNVړ
                position.InternalSet(si, pi-1);
                return true;
            }
            if(si > 0) { //ÕZNṼXg
                position.InternalSet(si-1, _tags[si-1].length-1);
                return true;
            }

            position = CHARTPOSITION.Empty(this);
            return false;
        }

    }

    //`[g̈ʒuw肷
    public struct CHARTPOSITION {
        private ChartSectionInfo _sections;
        private int _sectionIndex;
        private int _positionInSection; //sZNVłindex

        public CHARTPOSITION(ChartSectionInfo s, int index, int pos) {
            Debug.Assert(pos >= 0);
            _sections = s;
            _sectionIndex = index;
            _positionInSection = pos;
        }
        //ChartSectionInfô݌Ă
        public void InternalSet(int index, int pos) {
            Debug.Assert(pos >= 0);
            _sectionIndex = index;
            _positionInSection = pos;
        }
        public bool IsEmpty {
            get {
                return _sectionIndex==-1;
            }
        }
        public ChartSectionInfo Owner {
            get {
                return _sections;
            }
        }
        public int SectionIndex {
            get {
                return _sectionIndex;
            }
        }
        public int PositionInSection {
            get {
                return _positionInSection;
            }
        }
        public ChartSectionType SectionType {
            get {
                return _sections.GetTagAt(_sectionIndex).type;
            }
        }
        public bool IndicatesData {
            get {
                return this.SectionType==ChartSectionType.Data;
            }
        }
        public int ToDataIndex() {
            ChartSectionInfo.Tag t = _sections.GetTagAt(_sectionIndex);
            Debug.Assert(t.type==ChartSectionType.Data);
            Debug.Assert(t.length > _positionInSection);
            return t.dataIndex + _positionInSection;
        }
        public int ToDataIndexSoft() {
            if(this.IsEmpty) return -1;

            ChartSectionInfo.Tag t = _sections.GetTagAt(_sectionIndex);
            if(t.type==ChartSectionType.Data)
                return t.dataIndex + _positionInSection;
            else
                return -1;
        }
        public int ToDisplayIndex() {
            ChartSectionInfo.Tag t = _sections.GetTagAt(_sectionIndex);
            return t.displayIndex + _positionInSection;
        }
        public ChartSectionType ToChartSectionType() {
            ChartSectionInfo.Tag t = _sections.GetTagAt(_sectionIndex);
            return t.type;
        }

        //{NVOȂŔr
        public bool FastEq(ref CHARTPOSITION pos) {
            //Debug.Assert(_sections==pos._sections);
            return _sectionIndex==pos._sectionIndex && _positionInSection==pos._positionInSection;
        }

        //
        public static CHARTPOSITION Empty(ChartSectionInfo sec) {
            return new CHARTPOSITION(sec, -1, 0);
        }
    }
#if UNITTEST
    [TestFixture]
    public class ChartTests {
        [Test]
        public void SectionInfoTest() {
            ChartSectionInfo si = new ChartSectionInfo(
                new ChartSectionType[] { ChartSectionType.Data, ChartSectionType.Space, ChartSectionType.Data },
                new int[] { 120, 90, 150});

            Assert.AreEqual(2, si.DataSectionCount);
            Assert.AreEqual(360, si.GetDisplayLength());
            ChartSectionInfo.Tag t = si.GetTagAt(2);
            Assert.AreEqual(150, t.length);
            Assert.AreEqual(120, t.dataIndex);
            Assert.AreEqual(210, t.displayIndex);

            si.AppendLast(ChartSectionType.Forward, 30);
            Assert.AreEqual(4, si.SectionCount);
            Assert.AreEqual(2, si.DataSectionCount);
            t = si.GetTagAt(3);
            Assert.AreEqual(ChartSectionType.Forward, t.type);
            Assert.AreEqual(30, t.length);
            Assert.AreEqual(270, t.dataIndex);
            Assert.AreEqual(360, t.displayIndex);

            
        }

        [Test]
        public void ChartPositionTest1() {
            //Ƃ΁AO120Ax90A150Ä30̍\
            ChartSectionInfo si = new ChartSectionInfo(
                new ChartSectionType[] { ChartSectionType.Data, ChartSectionType.Space, ChartSectionType.Data, ChartSectionType.Forward },
                new int[] { 120, 90, 150, 30 });

            CHARTPOSITION pos;
            pos = si.GetChartPosition(0);
            Assert.AreEqual(0, pos.ToDataIndex());
            Assert.AreEqual(0, pos.ToDisplayIndex());
            Assert.AreEqual(0, pos.SectionIndex);
            Assert.AreEqual(0, pos.PositionInSection);

            pos = si.GetChartPosition(30);
            Assert.AreEqual(30, pos.ToDataIndex());
            Assert.AreEqual(30, pos.ToDisplayIndex());
            Assert.AreEqual(0, pos.SectionIndex);
            Assert.AreEqual(30, pos.PositionInSection);

            pos = si.GetChartPosition(119); //OꃉXg
            Assert.AreEqual(119, pos.ToDataIndex());
            Assert.AreEqual(119, pos.ToDisplayIndex());
            Assert.AreEqual(0, pos.SectionIndex);
            Assert.AreEqual(119, pos.PositionInSection);

            pos = si.GetChartPosition(150); //xݒ
            Assert.AreEqual(-1, pos.ToDataIndexSoft());
            Assert.AreEqual(150, pos.ToDisplayIndex());
            Assert.AreEqual(1, pos.SectionIndex);
            Assert.AreEqual(30, pos.PositionInSection);

            pos = si.GetChartPosition(210); //Jn
            Assert.AreEqual(120, pos.ToDataIndex());
            Assert.AreEqual(210, pos.ToDisplayIndex());
            Assert.AreEqual(2, pos.SectionIndex);
            Assert.AreEqual(0, pos.PositionInSection);

            pos = si.GetChartPosition(270); //r
            Assert.AreEqual(180, pos.ToDataIndex());
            Assert.AreEqual(270, pos.ToDisplayIndex());
            Assert.AreEqual(2, pos.SectionIndex);
            Assert.AreEqual(60, pos.PositionInSection);

            pos = si.GetChartPosition(359); //ꂨ
            Assert.AreEqual(269, pos.ToDataIndex());
            Assert.AreEqual(359, pos.ToDisplayIndex());
            Assert.AreEqual(2, pos.SectionIndex);
            Assert.AreEqual(149, pos.PositionInSection);

            pos = si.GetChartPosition(360); //Xg
            Assert.AreEqual(-1, pos.ToDataIndexSoft());
            Assert.AreEqual(360, pos.ToDisplayIndex());
            Assert.AreEqual(3, pos.SectionIndex);
            Assert.AreEqual(0, pos.PositionInSection);

            pos = si.GetChartPosition(389); //Xg
            Assert.AreEqual(-1, pos.ToDataIndexSoft());
            Assert.AreEqual(389, pos.ToDisplayIndex());
            Assert.AreEqual(3, pos.SectionIndex);
            Assert.AreEqual(29, pos.PositionInSection);

            Assert.AreEqual(390, si.GetDisplayLength());
        }
        [Test]
        public void ChartPositionTest2() {
            //Ƃ΁AO120Ax90A150Ä30̍\
            ChartSectionInfo si = new ChartSectionInfo(
                new ChartSectionType[] { ChartSectionType.Data, ChartSectionType.Space, ChartSectionType.Data, ChartSectionType.Forward },
                new int[] { 120, 90, 150, 30 });

            CHARTPOSITION pos;
            pos = si.GetChartPosition(0);
            bool r;

            r = si.MoveNext(ref pos);
            Assert.IsTrue(r);
            Assert.AreEqual(1, pos.ToDataIndex());
            Assert.AreEqual(1, pos.ToDisplayIndex());
            Assert.AreEqual(0, pos.SectionIndex);
            Assert.AreEqual(1, pos.PositionInSection);

            //119񂷂ƑOꂪI
            for(int i=1; i<120; i++) {
                Assert.AreEqual(i, pos.ToDataIndex());
                r = si.MoveNext(ref pos);
                Assert.IsTrue(r);
            }
            Assert.AreEqual(-1, pos.ToDataIndexSoft());
            Assert.AreEqual(120, pos.ToDisplayIndex());
            Assert.AreEqual(1, pos.SectionIndex);
            Assert.AreEqual(0, pos.PositionInSection);

            //x
            for(int i=0; i<90; i++) {
                Assert.AreEqual(-1, pos.ToDataIndexSoft());
                r = si.MoveNext(ref pos);
                Assert.IsTrue(r);
            }
            Assert.AreEqual(120, pos.ToDataIndex());
            Assert.AreEqual(210, pos.ToDisplayIndex());
            Assert.AreEqual(2, pos.SectionIndex);
            Assert.AreEqual(0, pos.PositionInSection);

            //
            for(int i=0; i<150; i++) {
                Assert.AreEqual(120 + i, pos.ToDataIndex());
                r = si.MoveNext(ref pos);
                Assert.IsTrue(r);
            }
            Assert.AreEqual(-1, pos.ToDataIndexSoft());
            Assert.AreEqual(360, pos.ToDisplayIndex());
            Assert.AreEqual(3, pos.SectionIndex);
            Assert.AreEqual(0, pos.PositionInSection);

            //Xg
            for(int i=0; i<29; i++) {
                r = si.MoveNext(ref pos);
                Assert.IsTrue(r);
            }
            r = si.MoveNext(ref pos);
            Assert.IsFalse(r);

        }
    }
#endif
}
