/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Values/Candle.cs#7 $
 * $DateTime: 2008/04/17 18:17:59 $
 * 
 * S{l
 */
using System;
using System.Diagnostics;

using System.Collections;
using System.Collections.Generic;
using System.Text;

using Bellagio.Evaluators;

#if UNITTEST
using NUnit.Framework;
#endif

namespace Bellagio.Values {

    [Flags]
    public enum CandleFlags {
        None = 0,
        Nil  = 1
    }

    public class Candle : BV {
        private Quote _quote;
        private int _index;
        private int _time;
        private int _open;
        private int _high;
        private int _low;
        private int _close;
        private int _volume;
        private int _creditLong;
        private int _creditShort;
        private CandleFlags _flags;

        public Candle() {
        }
        public Candle(Quote quote, int index, int time, int price, int volume) {
            Set(quote, index, time, price, volume);
        }
        public Candle(Quote quote, int index, int time, int open, int high, int low, int close, int volume, CandleFlags flags) {
            Set(quote, index, time, open, high, low, close, volume, flags);
        }
        public Candle(Candle src) {
            Set(src._quote, src._index, src._time, src._open, src._high, src._low, src._close, src._volume, src._flags);
            _creditLong = src._creditLong;
            _creditShort = src._creditShort;
        }
        public int Index {
            get {
                return _index;
            }
        }
        public Quote BaseQuote {
            get {
                return _quote;
            }
        }

        public int Time {
            get {
                return _time;
            }
            set {
                _time = value;
            }
        }
        public int Open {
            get {
                return _open;
            }
            set {
                _open = value;
            }
        }
        public int High {
            get {
                return _high;
            }
            set {
                _high = value;
            }
        }
        public int Low {
            get {
                return _low;
            }
            set {
                _low = value;
            }
        }
        public int Close {
            get {
                return _close;
            }
            set {
                _close = value;
            }
        }
        public int Volume {
            get {
                return _volume;
            }
            set {
                _volume = value;
            }
        }
        public CandleFlags Flags {
            get {
                return _flags;
            }
            set {
                _flags = value;
            }
        }
        public int CreditLong {
            get {
                return _creditLong;
            }
            set {
                _creditLong = value;
            }
        }
        public int CreditShort {
            get {
                return _creditShort;
            }
            set {
                _creditShort = value;
            }
        }

        public void Set(Quote quote, int index, int time, int value, int volume) {
            Debug.Assert(volume>=0);
            _quote = quote;
            _index = index;
            _open = value;
            _high = value;
            _low = value;
            _close = value;
            _volume = volume;
            _time = time;
            _flags  = CandleFlags.None;
        }
        public void Set(Quote quote, int index, int time, int open, int high, int low, int close, int volume, CandleFlags flags) {
            Debug.Assert(volume>=0);
            _quote = quote;
            _index = index;
            _open = open;
            _high = high;
            _low = low;
            _close = close;
            _volume = volume;
            _time = time;
            _flags = flags;
        }
        //lX^[gāAXɍXVX^C
        public void Update(int value, int volume) {
            if(this.IsNil) {
                Set(_quote, _index, _time, value, value, value, value, volume, CandleFlags.None);
            }
            else {
                _high = Math.Max(_high, value);
                _low = Math.Min(_low, value);
                _close = value;
                _volume += volume;
            }
        }
        public void Update(Candle y) {
            if(y.IsNil || y.Volume==0) return;

            if(this.IsNil || _volume==0)
                Set(_quote, _index, _time, y._open, y._high, y._low, y._close, y._volume, y._flags);
            else {
                _high = Math.Max(_high, y.High);
                _low = Math.Min(_low, y.Low);
                _close = y.Close;
                _volume += y.Volume;
            }
        }
        public void SetQuoteAndIndex(Quote q, int index) {
            _quote = q;
            _index = index;
        }

        public override BT BT {
            get {
                return CandleType.instance;
            }
        }
        public override bool IsNil {
            get {
                return (_flags & CandleFlags.Nil)!=CandleFlags.None;
            }
        }
        public override void Format(BVFormatter formatter) {
            formatter.Candle(this);
        }
        public override void Let(BV value) {
            Candle y = (Candle)value;
            _quote = y._quote;
            _index = y._index;
            _time = y._time;
            _open = y._open;
            _high = y._high;
            _low = y._low;
            _close = y._close;
            _volume = y._volume;
            _flags = y._flags;
        }

        //vfodelegate : MemoryMappedFileŎwItZbgƃRr
        public delegate double ElementDelegate(Candle e);
        public class ElementProfile {
            public ElementDelegate _delegate;
            public int _offset;
            public ElementProfile(ElementDelegate dg, int offset) {
                _delegate = dg;
                _offset = offset;
            }
            public int OffsetByIntPtr {
                get {
                    return _offset >> 2;
                }
            }
        }

        public static unsafe Candle FromRawPointer(Quote parent, int index, byte* p) {
            int* ip = (int*)p;
            Candle c = new Candle(parent, index, *ip, *(ip+1), *(ip+2), *(ip+3), *(ip+4), *(ip+5), CandleFlags.None);
            c.CreditLong = *(ip+6);
            c.CreditShort = *(ip+7);
            return c;
        }

        public const int ELEMENT_SIZE_IN_MEMORY = 32;
        public static ElementProfile OpenDelegate   = new ElementProfile(delegate(Candle y) { return y.Open; }, 4);
        public static ElementProfile HighDelegate   = new ElementProfile(delegate(Candle y) { return y.High; }, 8);
        public static ElementProfile LowDelegate    = new ElementProfile(delegate(Candle y) { return y.Low; }, 12);
        public static ElementProfile CloseDelegate  = new ElementProfile(delegate(Candle y) { return y.Close; }, 16);
        public static ElementProfile VolumeDelegate = new ElementProfile(delegate(Candle y) { return y.Volume; }, 20);
        public static ElementProfile CreditLongDelegate = new ElementProfile(delegate(Candle y) { return y.CreditLong; }, 24);
        public static ElementProfile CreditShortDelegate = new ElementProfile(delegate(Candle y) { return y.CreditShort; }, 28);
        public static ElementDelegate TradeAmountDelegate = delegate(Candle y) { return ((double)y.Volume) * y.Close; };

        //intłɂTradeAmountȂ
        public delegate int ElementDelegateI(Candle e);
        public static ElementDelegateI OpenDelegateI   = delegate(Candle y) { return y.Open; };
        public static ElementDelegateI CloseDelegateI  = delegate(Candle y) { return y.Close; };
        public static ElementDelegateI HighDelegateI   = delegate(Candle y) { return y.High; };
        public static ElementDelegateI LowDelegateI    = delegate(Candle y) { return y.Low; };
        public static ElementDelegateI VolumeDelegateI = delegate(Candle y) { return y.Volume; };
        public static ElementDelegateI CreditLongDelegateI = delegate(Candle y) { return y.CreditLong; };
        public static ElementDelegateI CreditShortDelegateI = delegate(Candle y) { return y.CreditShort; };

        //nullłUpdateł悤ȃ^Cv
        public static Candle Update(Candle target, Candle value) {
            if(target==null)
                return new Candle(value);
            else {
                target.Update(value);
                return target;
            }

        }
    }
    //S{l
    public class CandleType : BObjectT {
        public static CandleType instance;
        static CandleType() {
            instance =  new CandleType();
        }
        private static Candle MakeNil() {
            Candle c = new Candle();
            c.Flags = CandleFlags.Nil;
            return c;
        }

        private CandleType()
            : base("Candle", MakeNil()) {
        }

        public override BV CreateInstance() {
            return new Candle();
        }

        public override void RegisterFunctionsTo(FunctionLibrary lib, SystemFunctionLibrary sys) {
            BT[] noArg = FunctionLibrary.noArg;
            BT isCandle = instance;

            lib.DefineInternalFunction("open", isCandle, BT.Int, noArg, new Element(Candle.OpenDelegateI).Body);
            lib.DefineInternalFunction("high", isCandle, BT.Int, noArg, new Element(Candle.HighDelegateI).Body);
            lib.DefineInternalFunction("low", isCandle, BT.Int, noArg, new Element(Candle.LowDelegateI).Body);
            lib.DefineInternalFunction("close", isCandle, BT.Int, noArg, new Element(Candle.CloseDelegateI).Body);
            lib.DefineInternalFunction("volume", isCandle, BT.Int, noArg, new Element(Candle.VolumeDelegateI).Body);
            lib.DefineInternalFunction("creditlong", isCandle, BT.Int, noArg, new Element(Candle.CreditLongDelegateI).Body);
            lib.DefineInternalFunction("creditshort", isCandle, BT.Int, noArg, new Element(Candle.CreditShortDelegateI).Body);
            lib.DefineInternalFunction("tradeAmount", isCandle, BT.Double, noArg, new BInternalExecution(TradeAmount));
            lib.DefineInternalFunction("quote", isCandle, QuoteType.instance, noArg, new BInternalExecution(Quote));
            lib.DefineInternalFunction("rawTime", isCandle, BT.Int, noArg, new BInternalExecution(RawTime));
            lib.DefineInternalFunction("distanceTo", isCandle, BT.Int, new BT[] { instance }, new BInternalExecution(DistanceTo));
            lib.DefineInternalFunction("shift", isCandle, instance, FunctionLibrary.oneIntArg, new BInternalExecution(Shift));
        }

        private static bool IsCandle(BT value) {
            return value==instance;
        }
        private static Candle ToCandle(BV value) {
            Debug.Assert(IsCandle(value.BT));
            return (Candle)value;
        }
        private class Element {
            private Candle.ElementDelegateI _del;
            public Element(Candle.ElementDelegateI d) {
                _del = d;
            }
            public ExecResult Body(BV target, EvalContext extra, BV[] args, BV result) {
                Candle v = ToCandle(target);
                ((BInt)result).Value = _del(v);
                return ExecResult.OK;
            }
        }
        private static ExecResult TradeAmount(BV target, EvalContext extra, BV[] args, BV result) {
            Candle v = ToCandle(target);
            double a = v.Close;
            a *= v.Volume;
            ((BDouble)result).Value = a;
            return ExecResult.OK;
        }
        private static ExecResult Quote(BV target, EvalContext extra, BV[] args, BV result) {
            Candle v = ToCandle(target);
            ((SubseqQuote)result).Set(v.BaseQuote);
            return ExecResult.OK;
        }
        private static ExecResult RawTime(BV target, EvalContext extra, BV[] args, BV result) {
            Candle v = ToCandle(target);
            ((BInt)result).Value = v.Time;
            return ExecResult.OK;
        }

        //x.distanceTo(x.shift(n)) == n ƂȂ悤
        private static ExecResult DistanceTo(BV target, EvalContext extra, BV[] args, BV result) {
            Candle v = ToCandle(target);
            Candle a = ToCandle(args[0]);

            ((BInt)result).Value = v.Index - a.Index;
            return ExecResult.OK;
        }
        private static ExecResult Shift(BV target, EvalContext extra, BV[] args, BV result) {
            Candle v = ToCandle(target);
            int shift = ((BInt)args[0]).Value;
            Candle c = v.BaseQuote.CandleAtOrNull(v.Index - shift);
            if(c==null || c.IsNil)
                return ExecResult.Nil;

            ((Candle)result).Let(c);
            return ExecResult.OK;
        }
    }
}
