/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Screening/ScreeningExecution.cs#3 $
 * $DateTime: 2008/05/14 13:05:12 $
 */
using System;
using System.Collections;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;

using Bellagio.Data;
using Bellagio.Values;
using Bellagio.Evaluators;
using Bellagio.Environment;

//XN[jOvfAʂ̃f[^p[g
namespace Bellagio.Screening {
    public class ScreeningQuery {
        private ScreeningSchema _source;

        private string _title; //XN[jÕ^Cg

        public enum SpecialColumn {
            None, Rank
        }
        public class Column {
            public string label;
            public SpecialColumn specialColumn;
            public StandAloneEvaluator evaluator;
            public NumberFormatter formatter;
            public Column(string l, StandAloneEvaluator e, NumberFormatter f) {
                label = l;
                specialColumn = SpecialColumn.None;
                evaluator = e;
                formatter = f;
            }
            public Column(string l, SpecialColumn sc, NumberFormatter f) {
                label = l;
                specialColumn = sc;
                evaluator = null;
                formatter = f;
            }
        }
        private Column[] _columns;
        private int _mainColumnIndex;
        private int _rankColumnIndex;
        private StandAloneEvaluator _filter;
        private int _originDate;
        private ScreeningFilterType _staticFilter;
        private ScreeningSortType _sortType;
        private ScreeningResultCountType _countType;

        //Ɗ{tB^Aw肪

        public ScreeningQuery() {
        }
        public ScreeningSchema SourceSchema {
            get {
                return _source;
            }
        }
        public Column[] Columns {
            get {
                return _columns;
            }
        }
        public StandAloneEvaluator FilterEvaluator {
            get {
                return _filter;
            }
        }
        public string Title {
            get {
                return _title;
            }
        }
        public int MainColumnIndex {
            get {
                return _mainColumnIndex;
            }
        }
        public int RankColumnIndex {
            get {
                return _rankColumnIndex;
            }
        }

        //ȉ͎sɃXL[}Ƃ͊֌WȂݒ肷
        public int OriginDate {
            get {
                return _originDate;
            }
            set {
                _originDate = value;
            }
        }
        public ScreeningSortType SortType {
            get {
                return _sortType;
            }
            set {
                _sortType = value;
            }
        }
        public ScreeningResultCountType CountType {
            get {
                return _countType;
            }
            set {
                _countType = value;
            }
        }
        public ScreeningFilterType StaticFilter {
            get {
                return _staticFilter;
            }
            set {
                _staticFilter = value;
            }
        }

        public void ConstructWithSchema(ScreeningSchema screening) {
            _source = screening;
            ScreeningExecutorBase.RegisterFunctions();

            StringWithParameterParser string_parser = new StringWithParameterParser(screening);
            NumberFormatter defaultformatter = new NumberFormatter("");

            _title = string_parser.Parse(screening.title);
            int c = screening.column.Count;
            if(c==0) throw new FormatException(String.Format("{0} columnvf͂PȏKvł", screening.title.LocationString));
            _columns = new Column[c];
            _mainColumnIndex = -1;
            _rankColumnIndex = -1;

            for(int i=0; i<screening.column.Count; i++) {
                ScreeningSchemaColumn col = screening.column[i];
                string label = string_parser.Parse(col.label);
                Column runtimecol;
                if(col.expression.ParseOptionalString("")=="rank") { //JFN
                    runtimecol = new Column(label, SpecialColumn.Rank, new NumberFormatter(""));
                    _rankColumnIndex = i;
                }
                else {
                    EvaluatorBuildContext bc = new EvaluatorBuildContext(screening);
                    bc.Initialize(QuoteType.instance, LocalVariable.EmptyArray);
                    IEvaluator ev = EvaluatorBuilderFromText.Build(col.expression, bc);
                    if(ev.BT!=BT.Double && ev.BT!=BT.Int) throw new FormatException(String.Format("{0} ^͐lłKv܂B", col.expression));
                    NumberFormatter formatter = NumberFormatter.Parse(col.format, defaultformatter);
                    runtimecol = new Column(label, new StandAloneEvaluator(ev, bc, BV.EmptyArray), formatter);

                    if(_mainColumnIndex==-1) _mainColumnIndex = i;
                }
                _columns[i] = runtimecol;

            }

            if(screening.filter.IsOmitted)
                _filter = null;
            else {
                _filter = EvaluatorBuilderFromText.BuildStandAlone(screening.filter, screening, QuoteType.instance, LocalVariable.EmptyArray, BV.EmptyArray);
                if(_filter.BT!=BT.Bool) throw new FormatException(String.Format("{0} bool^łKv܂B", screening.filter));
            }

        }

    }

    public class ScreeningExecutor : ScreeningExecutorBase {
        private ScreeningQuery _query; //\[gΏ
        private ScreeningResult _result;

        public ScreeningExecutor(ScreeningQuery query)
            : base() {
            _query = query;
        }
        public ScreeningQuery Query {
            get {
                return _query;
            }
        }
        public ScreeningResult Result {
            get {
                return _result;
            }
        }

        private void AfterExecute() {
            _result.SetExcludedStocks(_insufficientDataStocks, _filteredStocks, _errorStocks);
            _result.Sort();
            //ʃJ̒g
            int ri = _query.RankColumnIndex;
            if(ri!=-1) {
                for(int i=0; i<_result.Count; i++) {
                    _result[i].Results[ri] = new BInt(i + 1);
                }
            }
        }
        protected override void ExecuteMain() {
            //ɓoς̃f[^ȂΒ~
            Quote quote = ScreeningPlugin.Instance.DailyDataCache.Request(BellagioRoot.GlobalStockCollection.FindExact("NK225"));
            if(quote==null || quote.LastCandle.Time < _query.OriginDate) { //Ō̓OȂ̂͂ŏO
                throw new BellagioException("̊f[^܂");
            }

            _result = new ScreeningResult(_query);

            //LbVAbvf[gĂ邩`FbNAłȂ΍XVJn
            //f[^XVAŌɍXVł낤ŊmF
            if(!ScreeningPlugin.Instance.DailyDataCache.CheckUpdated(BellagioRoot.GlobalStockCollection.FindExact("9997"))) {
                ScreeningPlugin.Instance.DailyDataCache.Clear();
                UpdateCache();
            }

            if(_progressNotifier!=null)
                _progressNotifier.InitProgress("sĂ܂", GetTargetStockCount());

            int count = 0;
            foreach(AbstractStockProfile prof in GetStockEnumerable()) {
                try {
                    ExecuteBrandResult t = ExecuteBrand(prof);
                    if(t==ExecuteBrandResult.Stop) break;
                    else if(t==ExecuteBrandResult.Filtered) _filteredStocks.Add(prof);
                    else if(t==ExecuteBrandResult.InsufficientData) _insufficientDataStocks.Add(prof);

                    if(_progressNotifier!=null)
                        _progressNotifier.OnProgress(count);

                }
                catch(Exception ex) {
                    Debug.WriteLine(ex.Message);
                    Debug.WriteLine(ex.StackTrace);
                    _errorStocks.Add(prof);
                }

                count++;
            }

            AfterExecute();
        }

        private ExecuteBrandResult ExecuteBrand(AbstractStockProfile prof_) {
            
            BasicStockProfile prof = prof_ as BasicStockProfile;
            if(prof==null || (prof.Primary.StockFlags & StockFlags.Obsolete)!=StockFlags.None)
                return ExecuteBrandResult.Ignored; //hAp~Ȃǂ͖

            int origin_date = _query.OriginDate;

            if(ExcludesByStaticFilter(prof, _query.StaticFilter)) return ExecuteBrandResult.Ignored;

            //XN[jOƂ̓LbVǂސƂ
            Quote quote = ScreeningPlugin.Instance.DailyDataCache.Request(prof);
            if(quote==null || quote.Count==0 || quote.LastCandle.Time < origin_date) { //Ō̓OȂ̂͂ŏO
                return ExecuteBrandResult.InsufficientData;
            }

            while(true) {
            mainloop:
                EvalContext context;
                if(quote.LastCandle.Time==origin_date) //͂悭P[X
                    context = new EvalContext(new SubseqQuote(quote, 0, quote.Count));
                else {
                    int index = quote.FindExactly(origin_date);
                    if(index < 0) return ExecuteBrandResult.InsufficientData;

                    context = new EvalContext(new SubseqQuote(quote, 0, index+1)); //wtŎ擾
                }
                context.CurrentStock = new StockRef(prof.Primary);

                //tB^[`FbN
                if(_query.FilterEvaluator!=null) {
                    BV filter = _query.FilterEvaluator.Eval(context);
                    if(filter.IsNil) {
                        if(ScreeningPlugin.Instance.DailyDataCache.ExtendDataIfPossible(prof, ref quote))
                            continue; //gf[^ŃgC
                    }

                    if(filter.IsNil || ((BBoolean)filter).Value==false)
                        return ExecuteBrandResult.Filtered;
                }

                BV[] values = new BV[_query.Columns.Length];
                for(int i=0; i<values.Length; i++) {
                    BV value = null;
                    ScreeningQuery.Column col = _query.Columns[i];
                    if(col.specialColumn==ScreeningQuery.SpecialColumn.None) {
                        BV.Let(ref value, col.evaluator.Eval(context));
                        if(value.IsNil) {
                            if(ScreeningPlugin.Instance.DailyDataCache.ExtendDataIfPossible(prof, ref quote))
                                goto mainloop; //gf[^ŃgC
                            else if(_query.MainColumnIndex==i) //CJ̃f[^Ȃ΍ڂ𖳎
                                return ExecuteBrandResult.Ignored;
                        }
                        values[i] = value;
                    }
                }

                _result.AddResult(new ScreeningResultItem(prof, values));
                break; //słA[vEo
            }

            return ExecuteBrandResult.Succeeded;
        }

    }

    public class ScreeningResultItem : ScreeningResultItemBase {
        private BV[] _results;

        public ScreeningResultItem(AbstractStockProfile prof, BV[] results)
            : base(prof) {
            _results = results;
        }

        public BV[] Results {
            get {
                return _results;
            }
        }

    }

    public class ScreeningResult : ScreeningResultBase<ScreeningResultItem> {
        private ScreeningQuery _query;

        public ScreeningResult(ScreeningQuery query)
            : base() {
            _query = query;
        }

        protected override Comparison<ScreeningResultItem> GetSortComparison() {
            if(_query.SortType==ScreeningSortType.Ascending)
                return delegate(ScreeningResultItem i1, ScreeningResultItem i2) {
                    return GetPrimaryValueAsDouble(i1).CompareTo(GetPrimaryValueAsDouble(i2));
                };
            else
                return delegate(ScreeningResultItem i1, ScreeningResultItem i2) {
                    return GetPrimaryValueAsDouble(i2).CompareTo(GetPrimaryValueAsDouble(i1));
                };
        }
        private double GetPrimaryValueAsDouble(ScreeningResultItem i) {
            BV v = i.Results[_query.MainColumnIndex];
            return ScreeningUtil.BVToDouble(v);
        }

        public ScreeningQuery Query {
            get {
                return _query;
            }
        }


    }
}
